/*
 *  client/read-line.c
 *		Routines to read one line from file.
 *		Automatically (re)allocates enough space for
 *		line.  These routines are system dependant, and
 *		will probably need to be fixed when porting.
 *
 *
 *  Copyright (C) 1990	Lysator Computer Club,
 *			Linkoping University,  Sweden
 *
 *  Everyone is granted permission to copy, modify and redistribute
 *  this code, provided the people they give it to can.
 *
 *
 *  Author:	Thomas Bellman
 *		Lysator Computer Club
 *		Linkoping University
 *		Sweden
 *
 *  email:	Bellman@Lysator.LiU.SE
 *
 *
 *  Any opinions expressed in this code are the author's PERSONAL opinions,
 *  and does NOT, repeat NOT, represent any official standpoint of Lysator,
 *  even if so stated.
 */

#include <config.h>

#include <stdio.h>
#include <sys/types.h>

#if STDC_HEADERS || HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <errno.h>
#include <ctype.h>

#include "xoutput.h"

#include <s-string.h>
#include <zmalloc.h>

#include "internal.h"
#include "error.h"

#include "read-line.h"



#define Export


/**************************************
*     Local function declarations     *
**************************************/


static void
erase_backward_char (FILE   * terminal);

static void
move_cursor_backwards (int	no_of_chars,
		       FILE   * terminal   );

static void
erase_to_end_of_line (FILE   * terminal);

static void
alert_user (FILE   * terminal);




/*  ==================================================================  */
/*			    Exported functions				*/


/*
 *  Read a string from a terminal into a String with some fancy
 *  options and some editing, including delete character, delete
 *  word and kill line.  Automatically allocates all necessary
 *  storage.
 *  Parameters:
 *  -  INFILE:	the file to read the string from.
 *  -  ECHOFILE: the file to which echoing is done.
 *  -  LINE:	pointer to the String where the result should be placed.
 *		This String must be initialized to something useful,
 *		i e EMPTY_STRING or a zmalloc():ed area.
 *  -  MAXLEN: maximum number of characters to allow in input.
 *		Excess characters are ignored.  -1 means no limit.
 *		-64 means that linebreak is done at next space after
 *		64 read characters.
 *  -  QUIT_CHARS: a NUL-terminated string of characters that can
 *		be used to end the input.  Typically contains the
 *		newline character ('\n').  The character used to
 *		exit, will be part of the resulting string.
 *  -  INSERT_CHARS: a NUL-terminated string of characters that
 *		can build up the resulting String.  NULL means
 *		normal printable characters including tab.  ""
 *		means all characters without special meaning.
 *  -  ECHO_CHAR: the caracter that is to be echoed. NUL means no
 *		echoing is done, and EOF means echo the read
 *		characters.
 *  Returns values:
 *	    0	Something went wrong - probably out of memory
 *	  EOF	End of file reached (includes Ctrl-D)
 *	Other	The character that caused the exit
 *
 *  Even if input_line() returns FAILURE, the value of *LINE is
 *  valid, and can be zfree():ed or zrealloc():ed, though the contents
 *  is probably just gibberish.
 */
Export  int
input_line (FILE		* infile,
	    FILE		* echofile,
	    String		* line,
	    String_size		  maxlen,
	    char		* quit_chars,
	    char		* insert_chars,
	    int			  echo_char)

{
#define FUNCTION "input_line()"
    char		* work_string;
    char		* temp_string; 		/* Temporary storage	*/
    String_size		  charno;		/* Current character no */
    String_size		  alloced_size;		/* Allocated size	*/
    String_size		  i;			/* Loop counter		*/
    String_size		  linelen;              /* The length of the line */
#if defined( __linux__ ) || defined( __svr4__ )
    unsigned
#endif
    int			  c;			/* Read character	*/
    Bool		  end;			/* Time to exit?	*/
    int			  exit_value = 0;	/* Value to return	*/
    struct  emacs_tty	  tty_param;
    int			  raw_terminal = 0;

#define		  chunk_size	32



    if (!raw_terminal 
	&& emacs_get_tty(fileno(infile), &tty_param) == -1)
    {
	/* This *shouldn't* fail, since init_terminal()
	 * didn't abort...				*/
	/* Well this could have failed in init_terminal.
	   Lets ignore it.
    	perror("Oops!  You shouldn't see this.  Anyway, here's the error:\n"
	       "Fetching tty parameters: ");
	fatal2 (CLIENT_ERROR_TOO_boring, "Fetching tty parameters. errno:", errno);
	 */
	raw_terminal = 1;
    }

    /* Free the old line, and allocate new work-string */
    work_string = zrealloc (line->string, alloced_size = chunk_size);
    if (work_string == NULL)
    {
	/*  We seem to have a problem, Dave.  We need more core.  */
	return  0;
    }

    end = FALSE;
    charno = 0;
    linelen = 0;
    while (! end)
    {
	c = getc (infile);

	switch(charset)
	{
	    case 0: break;      /* Already ISO 8859-1 */
	    case 1: c=pc8_to_iso(c); break;
	    case 2: c=pc7_to_iso(c); break;
	    case 3: c=mac_to_iso(c); break;
	    default: break;
	}

	if (!raw_terminal &&
#ifdef VERASE
	    c == tty_param.main.c_cc[VERASE]
#else
	    c == tty_param.main.sg_erase
#endif
	    )	/* Delete one char */
	{
	    if (charno != 0)
	    {
		if (echo_char != '\000')
		    erase_backward_char (echofile);
		charno--;
		linelen--;
	    }
	}
        else if (!raw_terminal && c == 8)
        /* Delete one char */
        {
            if (charno != 0)
            {
                if (echo_char != '\000')
                    erase_backward_char (echofile);
                charno--;
                linelen--;
            }
        }
	else if	(!raw_terminal &&
#ifdef VKILL
		 c == tty_param.main.c_cc[VKILL]
#else
		 c == tty_param.main.sg_kill
#endif
		 )	/* Kill entire line */
	{
	    if (echo_char != '\000')
	    {
		move_cursor_backwards (charno, echofile);
		erase_to_end_of_line (echofile);
		linelen = 0;
	    }
	    charno = 0;
	}
	else if (!raw_terminal &&
#ifdef VWERASE
		 c == tty_param.main.c_cc[VWERASE]
#else
#ifdef HAVE_LTCHARS
		 c == tty_param.ltchars.t_werasc
#else
		 c == 027 /* Ctrl-W */
#endif
#endif
		 )	/* Kill word */
	{
	    if (charno != 0)
	    {
		while ((charno > 0) && (isspace (work_string[charno - 1])))
		{
		    if (echo_char != '\000')
			erase_backward_char (echofile);
		    charno--;
		    linelen--;
		}
		while ((charno > 0) && (! isspace (work_string[charno - 1])))
		{
		    if (echo_char != '\000')
			erase_backward_char (echofile);
		    charno--;
		    linelen--;
		}
	    }
	}
	else if (!raw_terminal &&
#ifdef VREPRINT
		 c == tty_param.main.c_cc[VREPRINT]
#else
#ifdef HAVE_LTCHARS
		 c == tty_param.ltchars.t_rprntc
#else
		 c == 022 /* Ctrl-R to reprint. */
#endif
#endif
		 )	/* Reprint line */
	{
	    if (echo_char != '\000')
	    {
		putc('\n', echofile);
		for ( i = 0 ;  i < charno ;  i++ )
		    putc(work_string[i], echofile);
	    }
	}
	else if (!raw_terminal && (
#if defined (HAVE_TERMIO) || defined (HAVE_TERMIOS)
		 /* The VEOF field is occupied by a char counter and
		    overwritten. */
		 c == original_eof_char
#else
#ifdef HAVE_TCHARS
		 c == tty_param.tchars.t_eofc
#else
		 c == 004 /* Ctrl-D, EOF */
#endif
#endif
		 || c == EOF)) /* End of file */
	{
	    exit_value = EOF;
	    end = TRUE;
	}
	else if (raw_terminal && c == EOF) {
	    /* Well, this is it. */
	    if (infile == stdin)
		exit(0);
	    else
	    {
		exit_value = EOF;
		end = TRUE;
	    }
	}

	/* 			* Character to be inserted in string */
	else if (  (strchr (quit_chars, c) != NULL)
		 || (  (insert_chars == NULL
			? (!iscntrl(c) 
			   || c == '\t'
#ifdef BUGGY_LOCALE
			   || c == 0344 || c == 0345 || c == 0366
			   || c == 0304 || c == 0305 || c == 0326
#endif
			   )
			: (  *insert_chars == '\000'
			   || strchr (insert_chars, c) != NULL))
		     && ((maxlen < 0) || (charno < maxlen-1))))
	{
	    /* Do we need a larger work-string? */
	    if (charno >= alloced_size)
	    {
		temp_string = zrealloc (work_string,
					alloced_size += chunk_size);
		if (temp_string == NULL)
		{
		    /* Memory is tired - it is exhausted */
		    end = TRUE;
		    exit_value = 0;
		}
		work_string = temp_string;
	    }

	    linelen++;		/* BUG should handle tabs. */

	    if (maxlen < -1)	/* Break at whitespace enabled. */
	    {
		if (linelen  > -maxlen
		    && strchr(" \t", c)) /* Whitespace. */ {
		    c = '\n';
		    linelen = 0;
		}
	    }

	    work_string [charno++] = c;

	    if	(strchr (quit_chars, c) != NULL) /* Return or equivalent */
	    {
		exit_value = c;
		end = TRUE;
	    }
	    else
	    {
		if (echo_char == -1)
		    switch(charset)
		    {
		        case 0: putc (c, echofile); break;
			case 1: putc (iso_to_pc8(c), echofile); break;
			case 2: putc (iso_to_pc7(c), echofile); break;
			case 3: putc (iso_to_mac(c), echofile); break;
			default: putc (c, echofile); break;
		    }
		else if (echo_char != '\000')
		    putc (echo_char, echofile);
	    }
	}
	else					/* Bad input */
	{
	    alert_user (echofile);
	}

    }	/**  while (! end)  **/

    line->string = (unsigned char *)work_string;
    line->len = charno;

    return  exit_value;
#undef FUNCTION
}




Export  int
read_line (String	* line)
{
    int		  retval;

    retval = input_line (stdin, stdout, line, -1, "\n", NULL, EOF);
    if (retval == '\n')
	line->len--;
    return  retval;
}



Export  int
read_password (String	* line)
{
    int		  retval;

    retval = input_line (stdin, stdout, line, -1, "\n", NULL, '*');
    if (retval == '\n')
	line->len--;
    return  retval;
}



/*************************************
*     Local function definitions     *
*************************************/

/* These tables are far from complete */

Export int
pc8_to_iso(int c)
{
    switch(c)
    {
        case 144: c=201; break;
	case 142: c=196; break;
	case 153: c=214; break;
	case 143: c=197; break;
	case 154: c=220; break;
	case 130: c=233; break;
	case 132: c=228; break;
	case 148: c=246; break;
	case 134: c=229; break;
	case 129: c=252; break;
	default: break;
    }
    return(c);
}

Export int
iso_to_pc8(int c)
{
    switch(c)
    {
        case 201: c=144; break;
	case 196: c=142; break;
	case 214: c=153; break;
	case 197: c=143; break;
	case 220: c=154; break;
	case 233: c=130; break;
	case 228: c=132; break;
	case 246: c=148; break;
	case 229: c=134; break;
	case 252: c=129; break;
	default: break;
    }
    return(c);
}

Export int
iso_to_pc7(int c)
{
    switch(c)
    {
        case 201: c=64; break;
	case 196: c=91; break;
	case 214: c=92; break;
	case 197: c=93; break;
	case 220: c=94; break;
	case 233: c=96; break;
	case 228: c=123; break;
	case 246: c=124; break;
	case 229: c=125; break;
	case 252: c=126; break;
	default: break;
    }
    return(c);
}

Export int
pc7_to_iso(int c)
{
    switch(c)
    {
        case 64: c=201; break;
	case 91: c=196; break;
	case 92: c=214; break;
	case 93: c=197; break;
	case 94: c=220; break;
	case 96: c=233; break;
	case 123: c=228; break;
	case 124: c=246; break;
	case 125: c=229; break;
	case 126: c=252; break;
	default: break;
    }
    return(c);
}

Export int
mac_to_iso(int c)
{
    switch(c)
    {
        case 131: c=201; break;
	case 128: c=196; break;
	case 133: c=214; break;
	case 129: c=197; break;
	case 134: c=220; break;
	case 142: c=233; break;
	case 138: c=228; break;
	case 154: c=246; break;
	case 140: c=229; break;
	case 159: c=252; break;
	default: break;
    }
    return(c);
}

Export int
iso_to_mac(int c)
{
    switch(c)
    {
        case 201: c=131; break;
	case 196: c=128; break;
	case 214: c=133; break;
	case 197: c=129; break;
	case 220: c=134; break;
	case 233: c=142; break;
	case 228: c=138; break;
	case 246: c=154; break;
	case 229: c=140; break;
	case 252: c=159; break;
	default: break;
    }
    return(c);
}


static  void
erase_backward_char (FILE   * terminal)
{
    fputs("\b \b", terminal);
}


static  void
move_cursor_backwards (int	  no_of_chars,
		       FILE	* terminal    )
{
    for ( ;  no_of_chars-- ; )
	putc('\b', terminal);
}


static  void
alert_user (FILE   * terminal)
{
    putc('\a', terminal);
}


/*  Hardcoded for vt100 compatible terminals... */
static void
erase_to_end_of_line (FILE   * terminal)
{
    fputs("\033[0K", terminal);
}



#if  0		/* Don't warn about unused functions */

static  void
cursor_at_start_of_line (FILE	* terminal)
{
    putc('\r', terminal);
}


/*  Hardcoded for vt100 compatible terminals... */
static  void
erase_entire_line (FILE	  * terminal)
{
    fputs("\033[2K\015", terminal);
}


/*  Hardcoded for vt100 compatible terminals... */
static  void
swedish_font (FILE   * terminal)
{
    fputs("\033<\033(B\033)1\033)H\016", terminal);
}


/*  Hardcoded for vt100 compatible terminals... */
static  void
american_font (FILE   * terminal)
{
    fputs("\033<\033(B\033)1\033)G\017", terminal);
}

#endif	/* 0 */
