/* -*- mode: c; c-basic-offset: 8; indent-tabs-mode: nil; -*-
 * vim:expandtab:shiftwidth=8:tabstop=8:
 */

#ifndef _LL_H
#define _LL_H

#if defined(__linux__)
#include <linux/lustre_lite.h>
#elif defined(__APPLE__)
#include <darwin/lustre_lite.h>
#elif defined(__WINNT__)
#include <winnt/lustre_lite.h>
#else
#error Unsupported operating system.
#endif

#include <obd_class.h>
#include <obd_ost.h>
#include <lustre_net.h>
#include <lustre_mds.h>
#include <lustre_ha.h>

#ifdef __KERNEL__

/* careful, this is easy to screw up */
#define PAGE_CACHE_MAXBYTES ((__u64)(~0UL) << CFS_PAGE_SHIFT)

#endif

#define LLAP_FROM_COOKIE(c)                                                    \
        (LASSERT(((struct ll_async_page *)(c))->llap_magic == LLAP_MAGIC),     \
         (struct ll_async_page *)(c))

// 4*1024*1024
#define LL_MAX_BLKSIZE_BITS     (22)
#define LL_MAX_BLKSIZE          (1UL<<LL_MAX_BLKSIZE_BITS)

#include <lustre/lustre_user.h>


struct lustre_rw_params {
        int                lrp_lock_mode;
        ldlm_policy_data_t lrp_policy;
        obd_flag           lrp_brw_flags;
        int                lrp_ast_flags;
};

/*
 * XXX nikita: this function lives in the header because it is used by both
 * llite kernel module and liblustre library, and there is no (?) better place
 * to put it in.
 */
static inline void lustre_build_lock_params(int cmd, unsigned long open_flags,
                                            __u64 connect_flags,
                                            loff_t pos, ssize_t len,
                                            struct lustre_rw_params *params)
{
        params->lrp_lock_mode = (cmd == OBD_BRW_READ) ? LCK_PR : LCK_PW;
        params->lrp_brw_flags = 0;

        params->lrp_policy.l_extent.start = pos;
        params->lrp_policy.l_extent.end = pos + len - 1;
        /*
         * for now O_APPEND always takes local locks.
         */
        if (cmd == OBD_BRW_WRITE && (open_flags & O_APPEND)) {
                params->lrp_policy.l_extent.start = 0;
                params->lrp_policy.l_extent.end   = OBD_OBJECT_EOF;
        } else if (LIBLUSTRE_CLIENT && (connect_flags & OBD_CONNECT_SRVLOCK)) {
                /*
                 * liblustre: OST-side locking for all non-O_APPEND
                 * reads/writes.
                 */
                params->lrp_lock_mode = LCK_NL;
                params->lrp_brw_flags = OBD_BRW_SRVLOCK;
        } else {
                /*
                 * nothing special for the kernel. In the future llite may use
                 * OST-side locks for small writes into highly contended
                 * files.
                 */
        }
        params->lrp_ast_flags = (open_flags & O_NONBLOCK) ?
                LDLM_FL_BLOCK_NOWAIT : 0;
}

/*
 * This is embedded into liblustre and llite super-blocks to keep track of
 * connect flags (capabilities) supported by all imports given mount is
 * connected to.
 */
struct lustre_client_ocd {
        /*
         * This is conjunction of connect_flags across all imports (LOVs) this
         * mount is connected to. This field is updated by ll_ocd_update()
         * under ->lco_lock.
         */
        __u64      lco_flags;
        spinlock_t lco_lock;
};

/*
 * This function is used as an upcall-callback hooked by liblustre and llite
 * clients into obd_notify() listeners chain to handle notifications about
 * change of import connect_flags. See llu_fsswop_mount() and
 * lustre_common_fill_super().
 *
 * Again, it is dumped into this header for the lack of a better place.
 */
static inline int ll_ocd_update(struct obd_device *host,
                                struct obd_device *watched,
                                enum obd_notify_event ev, void *owner)
{
        struct lustre_client_ocd *lco;
        struct client_obd        *cli;
        __u64 flags;
        int   result;

        ENTRY;
        if (!strcmp(watched->obd_type->typ_name, LUSTRE_OSC_NAME)) {
                cli = &watched->u.cli;
                lco = owner;
                flags = cli->cl_import->imp_connect_data.ocd_connect_flags;
                CDEBUG(D_SUPER, "Changing connect_flags: "LPX64" -> "LPX64"\n",
                       lco->lco_flags, flags);
                spin_lock(&lco->lco_lock);
                lco->lco_flags &= flags;
                spin_unlock(&lco->lco_lock);
                result = 0;
        } else {
                CERROR("unexpected notification from %s %s!\n",
                       watched->obd_type->typ_name,
                       watched->obd_name);
                result = -EINVAL;
        }
        RETURN(result);
}

#endif
