/* -*- mode: c; c-basic-offset: 8; indent-tabs-mode: nil; -*-
 * vim:expandtab:shiftwidth=8:tabstop=8:
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <libgen.h>
#include <errno.h>
#include <sys/wait.h>

const char *progname;
const char usage_fmt[] = "Usage: %s <mountpoint>\n";
#define INAME_LEN (PATH_MAX + 1)

#define CHECK_IT(exp, pstr) \
if (!(exp)) { \
    fprintf(stderr, "%s: at %s:%d: ", progname, __FILE__, __LINE__); \
    perror((pstr)); \
    exit(1); \
}

#define CHECK_SNPRINTF(rc, len) \
    CHECK_IT((rc) > 0 && (rc) <= (len), "snprintf() failed")

static char *get_iname(char *fname, const char *mtpt)
{
        char *iname;
        int fd, rc;
        struct stat buf;

        iname = malloc(INAME_LEN);
        CHECK_IT(iname, "malloc() failed");

        fd = open(fname, O_CREAT, 0644);
        if (fd < 0 && errno != EISDIR) {
                fprintf(stderr, "%s:%d: open(%s) failed: %s\n", __FILE__,
                        __LINE__, fname, strerror(errno));
                exit(1);
        }

        if (fd >= 0)
                close(fd);

        rc = stat(fname, &buf);
        if (rc != 0) {
                fprintf(stderr, "%s:%d: stat(%s) failed: %s\n", __FILE__,
                        __LINE__, fname, strerror(errno));
                exit(1);
        }

        rc = snprintf(iname, INAME_LEN,
                      "%s/__iopen__/%lu", mtpt, (unsigned long)buf.st_ino);
        CHECK_SNPRINTF(rc, INAME_LEN);

        return iname;
}

int main(int argc, char *argv[])
{
        char *fname, *mtpt, *pname;
        char *fname_iname, *dir;
        char *dir_iname = NULL, *foo = NULL, *bar = NULL;
        int rc, fd, i, thread = 0;
        int pidlist[10];

        pname = strdup(argv[0]);
        progname = basename(argv[0]);

        if (argc != 2) {
                fprintf(stderr, usage_fmt, progname);
                return 1;
        }
 
        for (i = 1; i <= 10; i++) {
                rc = fork();
                if (rc < 0) {
                        fprintf(stderr, "error: %s: #%d - %s\n", argv[0], i,
                                strerror(rc = errno));
                        break;
                } else if (rc == 0) {
                        thread = i;
                        break;
                }
                printf("%s: thread #%d (PID %d) started\n", argv[0], i, rc);
                pidlist[i-1] = rc;
                rc = 0;
        }

        if (thread != 0) {
                mtpt  = argv[1];
                fname = malloc(INAME_LEN);
                CHECK_IT(fname, "malloc() failed");

                rc = snprintf(fname, INAME_LEN,
                              "%s/%d", mtpt, getpid());
                CHECK_SNPRINTF(rc, INAME_LEN);

                rc = mkdir(fname, 0644);
                if (rc != 0) {
                        fprintf(stderr, "%s:%d: mkdir(%s) failed: %s\n",
                                __FILE__, __LINE__, fname, strerror(errno));
                        exit(1);
                }

                fname_iname = get_iname(fname, mtpt);

                dir = malloc(INAME_LEN);
                CHECK_IT(dir, "malloc() failed");

                rc = snprintf(dir, INAME_LEN, "%s/dir", fname_iname);
                CHECK_SNPRINTF(rc, INAME_LEN);

                foo = malloc(INAME_LEN);
                CHECK_IT(foo, "malloc() failed");

                bar = malloc(INAME_LEN);
                CHECK_IT(bar, "malloc() failed");

                for (i = 0; i < 1000; i++) {
                        rc = mkdir(dir, 0644);
                        if (rc != 0) {
                                fprintf(stderr, "%s:%d: mkdir(%s) failed: %s\n",
                                        __FILE__, __LINE__, dir,
                                        strerror(errno));
                                exit(1);
                        }

                        dir_iname = get_iname(dir, mtpt);

                        rc = snprintf(foo, INAME_LEN, "%s/bar", dir_iname);
                        CHECK_SNPRINTF(rc, INAME_LEN);

                        rc = snprintf(bar, INAME_LEN, "%s/bar", dir_iname);
                        CHECK_SNPRINTF(rc, INAME_LEN);

                        fd = open(foo, O_CREAT, 0644);
                        if (fd < 0) {
                                fprintf(stderr, "%s:%d: open(%s) failed: %s\n",
                                        __FILE__, __LINE__, foo,
                                        strerror(errno));
                                exit(1);
                        }
                        rc = close(fd);
                        if (rc != 0) {
                                fprintf(stderr, "%s:%d: close() failed: %s\n",
                                        __FILE__, __LINE__, strerror(errno));
                                exit(1);
                        }

                        rc = rename(foo, bar);
                        if (rc != 0) {
                                fprintf(stderr, "%s:%d: rename(%s, %s) failed: "
                                        "%s\n", __FILE__, __LINE__, foo, bar,
                                        strerror(errno));
                                exit(1);
                        }

                        rc = unlink(bar);
                        if (rc != 0) {
                                fprintf(stderr, "%s:%d: unlink(%s) failed: "
                                        "%s\n", __FILE__, __LINE__, bar,
                                        strerror(errno));
                                exit(1);
                        }
                        rc = rmdir(dir);
                        if (rc != 0) {
                                fprintf(stderr, "%s:%d: rmdir(%s) failed: %s\n",
                                        __FILE__, __LINE__, dir,
                                        strerror(errno));
                                exit(1);
                        }

                        free(dir_iname);
                }
        } else {
                        for ( i=0; i<10; i++)
                                waitpid(pidlist[i], NULL, 0);
        }
        return 0;
}
