/*
 * Luola - 2D multiplayer cavern-flying game
 * Copyright (C) 2003 Calle Laakkonen
 *
 * File        : special.c
 * Description : Level special (eg. turrets and jump-gates) handling and animation
 * Author(s)   : Calle Laakkonen
 *
 * Luola is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Luola is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <math.h>
#include <SDL.h>
#include "defines.h"
#include "fs.h"
#include "player.h"
#include "level.h"
#include "console.h"
#include "weapon.h"
#include "special.h"
#include "vector.h"

#if HAVE_LIBSDL_MIXER
#include "audio.h"
#endif

struct Special_list {
  SpecialObj *special;
  struct Special_list *next;
  struct Special_list *prev;
};

/* Internally used globals */
static struct Special_list *specials;

static SDL_Surface *fx_jumpgate[1];
static SDL_Surface *fx_turret[3][TURRET_FRAMES]; /* Normal turrets, upside down turrets and SAM-Sites */
static SDL_Surface *fx_jumppoint_entry[WARP_FRAMES];
static SDL_Surface *fx_jumppoint_exit[WARP_FRAMES];

/* Internally used function prototypes */
static inline void draw_special(SpecialObj *special);
static void ship_hit_special(struct Special_list *list);

/* Internally used function */
static char hitsolid_rect(int x,int y,int w,int h) {
  int tx,ty,tx2,ty2;
  tx2=x+w; if(tx2>lev_level.width) return 1;
  ty2=y+h; if(ty2>lev_level.height) return 1;
  for(tx=x;tx<tx2;tx++)
    for(ty=y;ty<ty2;ty++)
      if(lev_level.solid[tx][ty]!=TER_FREE&&lev_level.solid[tx][ty]!=TER_WATER) return 1;
  return 0;
}
static int find_turret_y(int x,int y) {
  int pos;
  char a1,a2;
  a1=lev_level.solid[x][y];
  a2=a1;
  for(pos=0;pos<300;pos++) {
    if(y+pos<lev_level.height-30) {
      if(lev_level.solid[x][y+pos]==TER_GROUND && a1==TER_FREE) return y+pos;
      a1=lev_level.solid[x][y+pos];
    }
    if(y-pos>30) {
      if(lev_level.solid[x][y-pos]==TER_GROUND && a2==TER_FREE) return y-pos;
      a2=lev_level.solid[x][y-pos];
    }
  }
  return -1;
}

void init_specials(void) {
  int r;
  LDAT *datafile;
  specials=NULL;
  datafile=ldat_open_file((char*)getfullpath(GFX_DIRECTORY,"special.ldat"));
  for(r=0;r<WARP_FRAMES;r++) {
    fx_jumppoint_entry[r]=load_image_ldat(datafile,0,1,"WARP",r);
    fx_jumppoint_exit[r]=load_image_ldat(datafile,0,1,"WARPE",r);
  }
  for(r=0;r<TURRET_FRAMES;r++) {
    fx_turret[0][r]=load_image_ldat(datafile,0,1,"TURRET",r);
    fx_turret[1][r]=load_image_ldat(datafile,0,1,"SAMSITE",r);
    fx_turret[2][r]=flip_surface(fx_turret[0][r]);
  }
  fx_jumpgate[0]=load_image_ldat(datafile,0,1,"JUMPGATE",0);
  ldat_free(datafile);
  things_loaded[TL_SPECIAL]=1;
}

void deinit_specials(char just_clear) {
  int r;
  struct Special_list *next;
  if(just_clear==0) {
    for(r=0;r<WARP_FRAMES;r++) {
      SDL_FreeSurface(fx_jumppoint_entry[r]);
      SDL_FreeSurface(fx_jumppoint_exit[r]);
    }
    for(r=0;r<TURRET_FRAMES;r++) {
      SDL_FreeSurface(fx_turret[0][r]);
      SDL_FreeSurface(fx_turret[1][r]);
    }
  }
  while(specials) {
    next=specials->next;
    free(specials->special);
    free(specials);
    specials=next;
  }
}

void prepare_specials(LevelSettings *settings) {
  SpecialObj *gate1,*gate2,*turret,*obj;
  struct Special_list *list,*list2;
  int r,loops,tmpi;
  LSB_Objects *object=NULL;
  /* Initialize jump gates */
  for(r=0;r<level_settings.jumpgates;r++) {
    gate1=(SpecialObj*)malloc(sizeof(SpecialObj));
    gate1->age=-1;
    gate1->frame=0;
    gate1->type=JumpGate;
    gate1->owner=-1;
    gate1->timer=0;
    gate1->var1=0;
    loops=0;
    do {
      gate1->x=rand()%lev_level.width;
      gate1->y=rand()%lev_level.height;
      loops++;
      if(loops>1000) break;
    } while(hitsolid_rect(gate1->x,gate1->y,32,32));
    if(loops<1000) {
     gate2=(SpecialObj*)malloc(sizeof(SpecialObj));
     gate2->age=-1;
     gate2->frame=0;
     gate2->type=JumpGate;
     gate2->owner=-1;
     gate2->timer=0;
     gate2->var1=0;
     loops=0;
     do {
       gate2->x=rand()%lev_level.width;
       gate2->y=rand()%lev_level.height;
       loops++;
       if(loops>1000) break;
     } while(hitsolid_rect(gate2->x,gate2->y,32,32) || hypot(gate2->x-gate1->x,gate2->y-gate1->y)<200);
     if(loops<10000) {
       gate1->link=gate2;
       gate2->link=gate1;
       gate1->health=-1;
       gate2->health=-1;
       addspecial(gate2);
       addspecial(gate1);
     }
    }
  }
  /* Initialize turrets */
  for(r=0;r<level_settings.turrets;r++) {
    turret=(SpecialObj*)malloc(sizeof(SpecialObj));
    loops=0;
    while(1) {
      do {
        turret->x=30+rand()%(lev_level.width-60);
        turret->y=30+rand()%(lev_level.height-60);
        loops++;
        if(loops>5000) break;
      } while(lev_level.solid[turret->x][turret->y]!=TER_FREE);
      if(loops>5000) break;
        tmpi=find_turret_y(turret->x,turret->y);
	if(tmpi!=-1) { turret->y=tmpi; break; }
    }
    if(loops<5000) {
      for(tmpi=turret->y+1;tmpi<turret->y+7;tmpi++) if(lev_level.solid[turret->x][tmpi]!=TER_FREE&&lev_level.solid[turret->x][tmpi]!=TER_TUNNEL) break;
      if(tmpi<turret->y+6) {
        turret->y-=7;
	turret->var1=rand()%3;	/* Normal turret, Grenade launcher,SAM site */
      } else { /* Turret is upside down */
        turret->var1=3+(rand()&0x01);
      }
      turret->type=Turret;
      turret->owner=-1;
      turret->age=-1;
      turret->frame=1+rand()%(TURRET_FRAMES-2);
      turret->timer=0;
      turret->link=NULL;
      turret->health=1;
      turret->var2=rand()&0x01;
      addspecial(turret);
    }
  }
  /* Initialize manually placed specials */
  if(settings) object=settings->objects;
  while(object) {
    if(object->type<0x10) {
      obj=(SpecialObj*)malloc(sizeof(SpecialObj));
      if(object->type==1) obj->type=Turret;
      else if(object->type==2) obj->type=JumpGate;
      else {
	printf("Unrecognized special object type %d\n",object->type);
	object=object->next;
	continue;
      }
      obj->x=object->x;
      obj->y=object->y;
      obj->owner=-1;
      obj->age=-1;
      obj->timer=0;
      obj->link=NULL;
      obj->health=1;
      if(obj->type==Turret) {
        obj->var1=object->value;
	obj->var2=rand()&0x01;
	obj->frame=1+rand()%(TURRET_FRAMES-2);
      } else {
        obj->var1=object->id; /* We borrow this */
	obj->link=(SpecialObj*)(int)object->link;
	obj->frame=0;
      }
      if(object->ceiling_attach==0) {
        if(obj->type==Turret) {
	  obj->y-=fx_turret[0][0]->h;
	  obj->var1=object->value;
	} else {
	  obj->y-=fx_jumpgate[0]->h;
	}
      } else if(obj->type==Turret) obj->var1=3+object->value;
      addspecial(obj);
    }
    object=(LSB_Objects*)object->next;
  }
  /* Pair up the jumpgates */
  list=specials;
  while(list) {
    if(list->special->type==JumpGate)
      if((int)list->special->link<=255) {
        list2=list->next;
	while(list2) {
	  if(list2->special->type==JumpGate && (int)list2->special->link<=255)
	    if(list2->special->var1==(int)list->special->link) {
	      list->special->var1=0;
	      list->special->link=list2->special;
	      list2->special->var1=0;
	      list2->special->link=list->special;
	      break;
	    }
        list2=list2->next;
        }
	if(list2==NULL) {
	  printf("Warning ! Pairless jumpgate with id %d and link %d (removed)\n",list->special->var1,(int)list->special->link);
	  list2=list->next;
	  if(list->prev) list->prev->next=list->next;
	  if(list->next) list->next->prev=list->prev;
	  free(list->special);
	  free(list);
	  list=list2;
	  continue;
	}
      }
    list=list->next;
  }
}

void addspecial(SpecialObj *special) {
  struct Special_list *list=NULL;
  struct Special_list *newentry;
  if(specials!=NULL) {
    list=specials;
    while(list->next) list=list->next;
  }
  newentry=(struct Special_list*)malloc(sizeof(struct Special_list));
  newentry->next=NULL;
  newentry->prev=list;
  newentry->special=special;
  if(specials==NULL)
    specials=newentry;
  else
    list->next=newentry;
}

void drop_jumppoint(int x,int y,char player) {
  struct Special_list *list=specials;
  SpecialObj *jump,*exit=NULL;
  jump=(SpecialObj*)malloc(sizeof(SpecialObj));
  jump->x=x-16;
  jump->y=y-16;
  jump->owner=player;
  jump->frame=0;
  jump->timer=(game_settings.jumplife==0)?0:WARP_LOOP/2;
  jump->health=-1;
  while(list) {
    if(list->special->type==WarpExit && list->special->owner==player)
      if(list->special->link==NULL) exit=list->special;
    list=list->next;
  }
  if(exit==NULL) {
    jump->type=WarpExit;
    jump->link=NULL;
    jump->age=-1;
  } else {
    jump->type=WarpEntry;
    jump->link=exit;
    exit->link=jump;
    jump->age=(game_settings.jumplife==0)?JPSHORTLIFE:(game_settings.jumplife==1)?JPMEDIUMLIFE:JPLONGLIFE;
    exit->age=jump->age;
#if HAVE_LIBSDL_MIXER
    playwave(WAV_JUMP);
#endif
  }
  addspecial(jump);
}

void animate_specials(void) {
  struct Special_list *list=specials;
  struct Special_list *next;
  int i1;
  double f1,f2;
  while(list) {
    /* Animations */
    switch(list->special->type) {
      case WarpExit:
      case WarpEntry:
	list->special->frame++;
	if(list->special->age<0) list->special->frame=0;
	else if(list->special->frame>=WARP_FRAMES) list->special->frame=WARP_LOOP;
	break;
      case Turret:
	if(list->special->timer!=0) break;
	if(list->special->var2<5) {
	  list->special->var2++;
	  if(list->special->var2==4) {
	    list->special->var2=0;
	    list->special->frame++;
	    if(list->special->frame==TURRET_FRAMES-1) list->special->var2=5;
	  }
	} else if(list->special->var2>=5 && list->special->var2<=10) {
	  list->special->var2++;
	  if(list->special->var2==10) {
	    list->special->var2=5;
	    list->special->frame--;
	    if(list->special->frame==0) list->special->var2=0;
	  }
	}
	break;
      default: break;
    }
    /* Turrets shoot ! */
    if(list->special->type==Turret && list->special->timer==0) {
      Ship *targ=NULL;
      int ti;
      if(list->special->var1==2) i1=190; else i1=140;
      ti=find_nearest_player(list->special->x,list->special->y,-1,&f1);
      if(ti>=0) targ=players[ti].ship;
      if(targ && f1<i1 && (targ->visible || targ->tagged)) {
	 Projectile *p;
	 f2=atan2(targ->x-list->special->x,targ->y-list->special->y);
	 if(((f2<-M_PI_2||f2>M_PI_2)&&list->special->var1<2) || list->special->var1==2 || ((f2>-M_PI_2&&f2<M_PI_2)&&list->special->var1>2)) {
	   if(list->special->var1>2) { /* Upside down */
	     if(f2<0)
	       list->special->frame=(int)(14+(f2-M_PI_2/(M_PI/15.0)));
	     else
	       list->special->frame=(int)((f2+M_PI_2/(M_PI/15.0)));
	   } else { /* Normal */
	     if(f2>-M_PI_2 && f2<M_PI_2) {
	       if(f2<0) f2-=M_PI_2; else f2+=M_PI_2;
	     }
	     if(f2<0)
	       list->special->frame=(int)(0-(f2/(2.0*M_PI/15.0)));
	     else
	       list->special->frame=(int)(14-(f2/(2.0*M_PI/15.0)));
	   }
	   p=make_projectile(list->special->x+sin(f2)*fx_turret[0][0]->w,list->special->y+cos(f2)*fx_turret[0][0]->h,makeVector(-sin(f2),-cos(f2)));
	   p->owner=-1;
	   if(list->special->var1>2) p->y+=fx_turret[0][0]->h+3;
	   if(list->special->var1==1) { p->type=Grenade; p->color=col_grenade; list->special->timer=25; }
	   else if(list->special->var1==2) { p->type=Missile; p->color=col_gray; list->special->timer=45; p->angle=f2; }
	   else list->special->timer=25;
#if HAVE_LIBSDL_MIXER
	   if(p->type==Missile) playwave_3d(WAV_MISSILE,p->x,p->y); else playwave_3d(WAV_NORMALWEAP,p->x,p->y);
#endif
	   add_projectile(p);
	 }
      }
    }
    /* Draw the special */
    draw_special(list->special);
    /* Decrement age if positive */
    if(list->special->age>0) list->special->age--;
    if(list->special->timer>0) list->special->timer--;
    /* Check if the object has expired. If age is <0, the object has no time limit */
    if(list->special->age==0) {
      next=list->next;
      free(list->special);
      if(list->prev)
        list->prev->next=list->next;
      else
        specials=list->next;
      if(list->next)
        list->next->prev=list->prev;
      free(list);
      list=next;
    } else { /* Check for player collisions */
      next=list->next;
      ship_hit_special(list);
      list=next;
    }
  }
}

static inline void draw_special(SpecialObj *special) {
  SDL_Surface **sprite=NULL;
  SDL_Rect rect,rect2;
  signed char showme=-1;
  int p;
  switch(special->type) {
    case Unknown: sprite=NULL; break;
    case WarpEntry: sprite=fx_jumppoint_entry; if(special->frame==0) showme=special->owner; break;
    case WarpExit: sprite=fx_jumppoint_exit; if(special->frame==0) showme=special->owner; break;
    case JumpGate: sprite=fx_jumpgate; break;
    case Turret: sprite=fx_turret[(special->var1==2)+(special->var1>2?2:0)]; break;
  }
  for(p=0;p<4;p++) {
#ifdef DONT_UPDATE_DEAD
    if(players[p].active && players[p].pilot->dead==0 && (showme==p||showme==-1) ) {
#else
    if(players[p].active && (showme==p||showme==-1)) {
#endif
      rect.w=sprite[0]->w;
      rect.h=sprite[0]->h;
      rect.x=special->x-cam_rects[p].x+lev_rects[p].x;
      rect.y=special->y-cam_rects[p].y+lev_rects[p].y;
      if((rect.x>lev_rects[p].x-rect.w && rect.x<lev_rects[p].x+lev_rects[p].w) && (rect.y>lev_rects[p].y-rect.h && rect.y<lev_rects[p].y+lev_rects[p].h)) {
        rect2=cliprect(rect.x,rect.y,rect.w,rect.h,lev_rects[p].x,lev_rects[p].y,lev_rects[p].x+lev_rects[p].w,lev_rects[p].y+lev_rects[p].h);
        if(rect.x<lev_rects[p].x) rect.x=lev_rects[p].x;
        if(rect.y<lev_rects[p].y) rect.y=lev_rects[p].y;
        SDL_BlitSurface(sprite[special->frame],&rect2,screen,&rect);
      }
    }
  }
}

char hit_special(Projectile *proj) {
  struct Special_list *list=specials;
  int w,h;
  while(list) {
    w=16;
    h=16;
    if(list->special->type!=JumpGate) /* Currently, jumpgate is the only special that doesn't collide with projectiles */
      if(proj->x>=list->special->x && proj->y>=list->special->y)
        if(proj->x<=list->special->x+w && proj->y<=list->special->y+h) {
	  switch(list->special->type) {
	    case WarpEntry:
	    case WarpExit:
	      if(proj->type!=MegaBomb&&proj->type!=Rocket) { list=list->next; continue; }
	      if(list->special->link && list->special->timer==0) {
	        proj->x=list->special->link->x+w/2;
		proj->y=list->special->link->y+h/2;
		list->special->timer=10;
		list->special->link->timer=10;
	      }
	      return 0;
	      break;
	    default:
	      list->special->health-=0.2;
	      if(list->special->health<=0) {
	    	spawn_clusters(list->special->x,list->special->y,10,-1,Cannon);
	    	spawn_clusters(list->special->x,list->special->y,16,-1,FireStarter);
	    	add_explosion(list->special->x,list->special->y,Noproj);
	    	free(list->special);
	    	if(list->prev) list->prev->next=list->next; else specials=list->next;
	    	if(list->next) list->next->prev=list->prev;
	    	free(list);
	      } break;
	  }
	  return 1;
	}
    list=list->next;
  }
  return 0;
}

static void ship_hit_special(struct Special_list *list) {
  ShipList *ships;
  Ship *ship;
  int w;
  w=32;
  ships=getshiplist();
  while(ships) {
    ship=ships->ship;
    if(ship->x>=list->special->x && ship->x<=list->special->x+w)
      if(ship->y>=list->special->y && ship->y<=list->special->y+w) {
        switch(list->special->type) {
	  case WarpEntry:
	  case WarpExit:
	    if(list->special->link && list->special->timer==0) {
	      ship->x=list->special->link->x+w/2;
	      ship->y=list->special->link->y+w/2;
	      list->special->timer=25;
	      list->special->link->timer=25;
	    } break;
	  case JumpGate:
	    if(list->special->timer==0) {
	      list->special->timer=(game_settings.jumplife==0)?JPSHORTLIFE:(game_settings.jumplife==1)?JPMEDIUMLIFE:JPLONGLIFE;
	      list->special->link->timer=list->special->timer;
	      drop_jumppoint(list->special->x+16,list->special->y+16,find_player(ship)+10);
	      drop_jumppoint(list->special->link->x+16,list->special->link->y+16,find_player(ship)+10);
	    } break;
	  case Turret:
	    if(ship->darting) {
	      spawn_clusters(list->special->x,list->special->y,10,-1,Cannon);
	      add_explosion(list->special->x,list->special->y,Noproj);
	      free(list->special);
	      if(list->prev) list->prev->next=list->next; else specials=list->next;
	      if(list->next) list->next->prev=list->prev;
	      free(list);
	      return;
	    } break;
	  default: break;
        }
      }
      ships=ships->next;
  }
}
