/*
 * Luola - 2D multiplayer cavern-flying game
 * Copyright (C) 2003 Calle Laakkonen
 *
 * File        : lconf.c
 * Description : Level configuration file parsing
 * Author(s)   : Calle Laakkonen
 *
 * Luola is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Luola is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <SDL_image.h>

#include "stringutil.h"
#include "lconf.h"
#include "game.h"	/* Use game_settings.ls to get default values */
#include "sbmp.h"
/* Internally used functions */
static char *read_line(FILE *fp);
static LSB_Main *readMainBlock(FILE *fp);
static LSB_Override *readOverrideBlock(FILE *fp);
static LSB_Objects *readObjectsBlock(FILE *fp);
static LSB_Palette *readPaletteBlock(FILE *fp);
static SDL_Surface *readIconBlock(FILE *fp);
static int readMainBlockRW(LevelSettings *settings,SDL_RWops *rw,int len);
static int readOverrideBlockRW(LevelSettings *settings,SDL_RWops *rw,int len);
static int readObjectsBlockRW(LevelSettings *settings,SDL_RWops *rw,int len);
static int readPaletteBlockRW(LevelSettings *settings,SDL_RWops *rw,int len);
static int readIconRW(LevelInfo *info,SDL_RWops *rw,int len);

/* Load the configuration file from file */
LevelSettings *load_level_config(const char *filename,int loadextra) {
  FILE *fp;
  char *line;
  LevelSettings *settings;
  /* Initialize */
  settings=(LevelSettings*)malloc(sizeof(LevelSettings));
  memset(settings,0,sizeof(LevelSettings));
  /* Open file */
  fp=fopen(filename,"r");
  if(!fp) {
    printf("Error ! Could not open configuration file \"%s\" !\n",filename);
    exit(1);
  }
  /* Read */
  while((line=read_line(fp))) {
    if(line[0]=='\0') continue;
    if(strcmp(line,"[main]")==0) settings->mainblock=readMainBlock(fp);
    else if(strcmp(line,"[override]")==0) settings->override=readOverrideBlock(fp);
    else if(strcmp(line,"[objects]")==0) settings->objects=readObjectsBlock(fp);
    else if(strcmp(line,"[palette]")==0) settings->palette=readPaletteBlock(fp);
    else if(strcmp(line,"[icon]")==0 && loadextra) settings->icon=readIconBlock(fp);
    free(line);
  }
  /* Finished */
  fclose(fp);
  return settings;
}

/* Load the (binary) configuration file from RWops */
LevelSettings *load_level_config_rw(SDL_RWops *rw,int len) {
  LevelSettings *settings;
  int read=0;
  Uint16 l;
  Uint8 type;
  /* Initialize */
  settings=(LevelSettings*)malloc(sizeof(LevelSettings));
  memset(settings,0,sizeof(LevelSettings));
  /* Read */
  while(read<len) {
    read+=SDL_RWread(rw,&l,2,1)*2;
    read+=SDL_RWread(rw,&type,1,1);
    switch(type) {
      case 0x01: read+=readMainBlockRW(settings,rw,l); break;
      case 0x02: read+=readOverrideBlockRW(settings,rw,l); break;
      case 0x03: read+=readObjectsBlockRW(settings,rw,l); break;
      case 0x04: read+=readPaletteBlockRW(settings,rw,l); break;
      case 0x05: read+=l; SDL_RWseek(rw,l,SEEK_CUR); break; /* We (currently) don't need to load the icon here */
      default:
        printf("Warning: Unrecognized toplevel block 0x%x ! (skipping %d bytes...)\n",type,l);
	SDL_RWseek(rw,l,SEEK_CUR);
	read+=l;
	break;
    }
  }
  if(read>len)
    printf("Warning: read %d bytes past the limit in load_level_config_rw() !\n",read-len);
  /* Finished */
  return settings;
}

/* Read the icon from a textmode level settings file */
static SDL_Surface *readIconBlock(FILE *fp) {
  SDL_Surface *icon;
  SDL_RWops *rw;
  rw=SDL_RWFromFP(fp,0);
  icon=IMG_LoadTyped_RW(rw,0,"XPM");
  if(icon==NULL)
    printf("Error occured while tried to read level icon:\n%s\n",SDL_GetError());
  SDL_FreeRW(rw);
  return icon;
}

/* Read the icon from a binary level settings file */
static int readIconRW(LevelInfo *info,SDL_RWops *rw,int len) {
  Uint32 colorkey;
  Uint8 hascolorkey;
  char *data;
  int read;
  data=malloc(len);
  read=SDL_RWread(rw,&hascolorkey,1,1);
  read+=SDL_RWread(rw,&colorkey,4,1)*4;
  read+=SDL_RWread(rw,data,1,len);
  len-=read;
  if(len) SDL_RWseek(rw,len,SEEK_CUR);
  info->icon=sbmp_to_surface(data);
  if(hascolorkey)
    SDL_SetColorKey(info->icon,SDL_SRCCOLORKEY,colorkey);
  free(data);
  return read+len;
}


/* Load just the main block and icon from a text mode configuration file */
LevelInfo *quickload_levelsettings(const char *filename) {
  LevelInfo *info;
  FILE *fp;
  char *line;
  fp=fopen(filename,"r");
  if(!fp) {
    printf("Error ! Could not open configuration file \"%s\" !\n",filename);
    exit(1);
  }
  info=(LevelInfo*)malloc(sizeof(LevelInfo));
  memset(info,0,sizeof(LevelInfo));
  /* Read */
  while((line=read_line(fp))) {
    if(line[0]=='\0') continue;
    if(strcmp(line,"[main]")==0) info->mainblock=readMainBlock(fp);
    else if(strcmp(line,"[icon]")==0) info->icon=readIconBlock(fp);
    free(line);
  }
  fclose(fp);
  return info;
}

/* Load just the main block and icon from a binary configuration file */
LevelInfo *quickload_levelsettings_rw(SDL_RWops *rw,int len) {
  LevelSettings settings;
  LevelInfo *info;
  Uint16 l;
  Uint8 type;
  int read=0;
  info=(LevelInfo*)malloc(sizeof(LevelInfo));
  memset(info,0,sizeof(LevelInfo));
  /* Read */
  while(read<len) {
    read+=SDL_RWread(rw,&l,2,1)*2;
    read+=SDL_RWread(rw,&type,1,1);
    switch(type) {
      case 0x01:
        read+=readMainBlockRW(&settings,rw,l);
	info->mainblock=settings.mainblock;
	break;
      case 0x05:
        read+=readIconRW(info,rw,l);
	break;
      default:
	SDL_RWseek(rw,l,SEEK_CUR);
	read+=l;
	break;
    }
  }
  /* Finished */
  return info;
}

/* Read the main block */
static LSB_Main *readMainBlock(FILE *fp) {
  LSB_Main *mblock;
  char *left,*right;
  char *line;
  mblock=(LSB_Main*)malloc(sizeof(LSB_Main));
  memset(mblock,0,sizeof(LSB_Main));
  mblock->smooth_scale=1;
  while((line=read_line(fp))) {
    if(line[0]=='\0') continue;
    if(strcmp(line,"[end]")==0) {free(line); break;}
    split_string(line,'=',&left,&right);
    if(left==NULL||right==NULL) {
      printf("Warning: Malformed line:\n \"%s\"\n",line);
      continue;
    }
    if(strcmp(left,"modified")==0) {
      if(strlen(right)!=12) printf("Warning: illegal modification date !\n");
      else strncpy(mblock->modified,right,12);
      free(right);
    } else if(strcmp(left,"collisionmap")==0) mblock->collmap=right;
    else if(strcmp(left,"artwork")==0) mblock->artwork=right;
    else if(strcmp(left,"name")==0) mblock->name=right;
    else if(strcmp(left,"type")==0) {
      if(strcmp(right,"luola")==0) mblock->type=LuolaLevel;
      else if(strcmp(right,"luola_8bit")==0) mblock->type=Luola8bitLevel;
      else printf("Warning: Unrecognized level type \"%s\"\n",right);
      free(right);
    } else if(strcmp(left,"smoothscale")==0) {
      mblock->smooth_scale=atoi(right);
      free(right);
    } else if(strcmp(left,"scalex")==0) {
      mblock->scalex=atoi(right);
      free(right);
    } else if(strcmp(left,"scaley")==0) {
      mblock->scaley=atoi(right);
      free(right);
    } else if(strcmp(left,"music")==0) {
      LevelBgMusic *mentry=NULL;
      mentry=(LevelBgMusic*)malloc(sizeof(LevelBgMusic));
      memset(mentry,0,sizeof(LevelBgMusic));
      mentry->file=right;
      if(mblock->music==NULL) mblock->music=mentry;
      else {
        LevelBgMusic *mtmp;
	mtmp=mblock->music;
	while(mtmp->next) mtmp=mtmp->next;
	mtmp->next=mentry;
      }
    }
    free(line);
    free(left);
  }
  return mblock;
}
/* Read the mainblock (binary) */
static int readMainBlockRW(LevelSettings *settings,SDL_RWops *rw,int len) {
  LSB_Main *mblock;
  int read=0;
  Uint8 type,u8;
  mblock=(LSB_Main*)malloc(sizeof(LSB_Main));
  memset(mblock,0,sizeof(LSB_Main));
  mblock->smooth_scale=1;
  while(read<len) {
    read+=SDL_RWread(rw,&type,1,1);
    switch(type) {
      case 0x01: {/* Modification date */
        Uint16 year;
	Uint8 m,d,hour,min;
	read+=SDL_RWread(rw,&year,2,1)*2;
	read+=SDL_RWread(rw,&m,1,1);
	read+=SDL_RWread(rw,&d,1,1);
	read+=SDL_RWread(rw,&hour,1,1);
	read+=SDL_RWread(rw,&min,1,1);
	sprintf(mblock->modified,"%d%d%d%d%d",year,m,d,hour,min);
      } break;
      case 0x02: { /* Level name */
        char tmps[512],chr='a';
	int l=0;
	while(chr!='\0' && l<512) {
	  read+=SDL_RWread(rw,&chr,1,1);
	  tmps[l]=chr;
	  l++;
	}
	mblock->name=malloc(l);
	strncpy(mblock->name,tmps,l);
      } break;
      case 0x03: /* Level type */
        read+=SDL_RWread(rw,&u8,1,1);
	if(u8==0x01) mblock->type=LuolaLevel;
	else if(u8==0x02) mblock->type=Luola8bitLevel;
	else printf("Warning ! Unrecognized level type 0x%x !\n",u8);
        break;
      case 0x04: /* Horizontal scaling */
        read+=SDL_RWread(rw,&mblock->scalex,1,1);
        break;
      case 0x05: /* Vertical scaling */
        read+=SDL_RWread(rw,&mblock->scaley,1,1);
        break;
      case 0x06: /* Smooth scaling */
        read+=SDL_RWread(rw,&mblock->smooth_scale,1,1);
        break;
      default:
        printf("WARNING !! Unknown main block item 0x%x ! (skipping one byte...)\n",type);
	SDL_RWseek(rw,1,SEEK_CUR); read++;
        break;
    }
  }
  settings->mainblock=mblock;
  return read;
}

/* Read the override block */
static LSB_Override *readOverrideBlock(FILE *fp) {
  LSB_Override *override;
  char *left,*right;
  char *line;
  override=(LSB_Override*)malloc(sizeof(LSB_Override));
  override->indstr_base=game_settings.ls.indstr_base;
  override->critters_enabled=0;
  override->stars=game_settings.ls.stars;
  override->snowfall=game_settings.ls.snowfall;
  override->turrets=game_settings.ls.turrets;
  override->jumpgates=game_settings.ls.jumpgates;
  override->cows=game_settings.ls.cows;
  override->fish=game_settings.ls.birds;
  override->birds=game_settings.ls.fish;
  override->bats=game_settings.ls.bats;
  override->soldiers=game_settings.ls.soldiers;
  override->helicopters=game_settings.ls.helicopters;
  while((line=read_line(fp))) {
    if(line[0]=='\0') continue;
    if(strcmp(line,"[end]")==0) {free(line); break;}
    split_string(line,'=',&left,&right);
    if(left==NULL||right==NULL) {
      printf("Warning: Malformed line:\n \"%s\"\n",line);
      continue;
    }
    if(strcmp(left,"critters_enabled")==0) override->critters_enabled=atoi(right);
    else if(strcmp(left,"bases_indestructable")==0) override->indstr_base=atoi(right);
    else if(strcmp(left,"turrets")==0) override->turrets=atoi(right);
    else if(strcmp(left,"jumpgates")==0) override->jumpgates=atoi(right);
    else if(strcmp(left,"cows")==0) override->cows=atoi(right);
    else if(strcmp(left,"fish")==0) override->fish=atoi(right);
    else if(strcmp(left,"birds")==0) override->birds=atoi(right);
    else if(strcmp(left,"bats")==0) override->bats=atoi(right);
    else if(strcmp(left,"soldiers")==0) override->soldiers=atoi(right);
    else if(strcmp(left,"helicopters")==0) override->soldiers=atoi(right);
    else printf("Unidentified options \"%s\" with parameter \"%s\"\n",left,right);
    free(line);
    free(left);
    free(right);
  }
  return override;
}

/* Read override block (binary) */
static int readOverrideBlockRW(LevelSettings *settings,SDL_RWops *rw,int len) {
  LSB_Override *override;
  int read=0;
  Uint8 type;
  override=(LSB_Override*)malloc(sizeof(LSB_Override));
  override->indstr_base=game_settings.ls.indstr_base;
  override->critters_enabled=0;
  override->stars=game_settings.ls.stars;
  override->snowfall=game_settings.ls.snowfall;
  override->turrets=game_settings.ls.turrets;
  override->jumpgates=game_settings.ls.jumpgates;
  override->cows=game_settings.ls.cows;
  override->fish=game_settings.ls.birds;
  override->birds=game_settings.ls.fish;
  override->bats=game_settings.ls.bats;
  override->soldiers=game_settings.ls.soldiers;
  override->helicopters=game_settings.ls.helicopters;
  while(read<len) {
    read+=SDL_RWread(rw,&type,1,1);
    switch(type) {
      case 0x01: /* Critters enabled */
        read+=SDL_RWread(rw,&override->critters_enabled,1,1);
	break;
      case 0x02: /* Bases indestructable */
        read+=SDL_RWread(rw,&override->indstr_base,1,1);
	break;
      case 0x03: /* Stars */
        read+=SDL_RWread(rw,&override->stars,1,1);
	break;
      case 0x04: /* Snowfall */
        read+=SDL_RWread(rw,&override->snowfall,1,1);
	break;
      case 0x05: /* Turrets */
        read+=SDL_RWread(rw,&override->turrets,1,1);
	break;
      case 0x06: /* Jumpgates */
        read+=SDL_RWread(rw,&override->jumpgates,1,1);
	break;
      case 0x07: /* Cows */
        read+=SDL_RWread(rw,&override->cows,1,1);
	break;
      case 0x08: /* Fish */
        read+=SDL_RWread(rw,&override->fish,1,1);
	break;
      case 0x09: /* Birds */
        read+=SDL_RWread(rw,&override->birds,1,1);
	break;
      case 0x0a: /* Bats */
        read+=SDL_RWread(rw,&override->bats,1,1);
	break;
      default:
        printf("WARNING !! Unknown override block item 0x%x ! (skipping one byte...)\n",type);
	SDL_RWseek(rw,1,SEEK_CUR); read++;
        break;
    }
  }
  settings->override=override;
  return read;
}

/* Read the objects block */
static LSB_Objects *readObjectsBlock(FILE *fp) {
  LSB_Objects *objects=NULL,*first=NULL,*prev=NULL;
  char *left,*right;
  char *line;
  while((line=read_line(fp))) {
    if(line[0]=='\0') continue;
    if(strcmp(line,"[endsub]")==0) {
      prev=objects;
      objects=NULL;
    } else if(strcmp(line,"[end]")==0) { free(line); break; }
    else if(strcmp(line,"[object]")==0) {
      free(line);
      objects=(LSB_Objects*)malloc(sizeof(LSB_Objects));
      if(objects==NULL) {
        printf("Malloc error at readObjectsBlock !\n");
	exit(1);
      }
      memset(objects,0,sizeof(LSB_Objects));
      if(!first) first=objects;
      if(prev) prev->next=(struct LSB_Objects*)objects;
      continue;
    }
    if(!objects) { free(line); continue; }
    split_string(line,'=',&left,&right);
    if(left==NULL||right==NULL) {
      printf("Warning: Malformed line:\n \"%s\"\n",line);
      continue;
    }
    if(strcmp(left,"type")==0) {
      if(strcmp(right,"turret")==0) objects->type=1;
      else if(strcmp(right,"jumpgate")==0) objects->type=2;
      else if(strcmp(right,"cow")==0) objects->type=0x10;
      else if(strcmp(right,"fish")==0) objects->type=0x11;
      else if(strcmp(right,"bird")==0) objects->type=0x12;
      else if(strcmp(right,"bat")==0) objects->type=0x13;
      else if(strcmp(right,"ship")==0) objects->type=0x20;
      else printf("Warning: Unidentified type \"%s\"\n",right);
    } else if(strcmp(left,"x")==0) objects->x=atoi(right);
    else if(strcmp(left,"y")==0) objects->y=atoi(right);
    else if(strcmp(left,"ceiling_attach")==0) objects->ceiling_attach=atoi(right);
    else if(strcmp(left,"value")==0) objects->value=atoi(right);
    else if(strcmp(left,"link")==0) objects->link=atoi(right);
    else if(strcmp(left,"id")==0) objects->id=atoi(right);
    else printf("Unidenified option \"%s\" with parameter \"%s\"\n",left,right);
    free(left);
    free(right);
    free(line);
  }
  return first;
}

static LSB_Palette *readPaletteBlock(FILE *fp) {
  LSB_Palette *palette;
  char *left,*right;
  char *line;
  char mode=0;
  Uint8 value=0;
  int begin=-1,end=-1;
  
  palette=(LSB_Palette*)malloc(sizeof(LSB_Palette));
  memset(palette->entries,0,256); /* In case the levelmaker forgot to put in the default value */
  while((line=read_line(fp))) {
    if(line[0]=='\0') continue;
    if(strcmp(line,"[endsub]")==0) {
      if(mode==0) {
        printf("Misplaced [endsub] in palette block !\n");
      } else {
        mode=0;
	if(begin==-1) {
	  printf("Error ! No first colour !\n");
	  continue;
	}
	if(end==-1) end=begin;
	memset(palette->entries+begin,value,end-begin+1);
      }
      continue;
    } else if(strcmp(line,"[end]")==0) {
      free(line);
      break;
    } else if(strcmp(line,"[terrain]")==0) {
      if(mode) printf("Nested [terrains] !\n");
      mode=1;
      end=-1;
      begin=-1;
      continue;
    }
    split_string(line,'=',&left,&right);
    if(left==NULL||right==NULL) {
      printf("Warning: Malformed line:\n \"%s\"\n",line);
      continue;
    }
    if(mode==0) {
      if(strcmp(left,"default")==0) {
	int def=atoi(right);
	if(def>255) { printf("Warning, default terrain value is %d, but the 255 is the maximium !\n",def); def=255; }
	memset(palette->entries,def,256);
      } else {
	printf("Unrecognized option \"%s\"\n",left);
      }
    } else {
      int val;
      val=atoi(right);
      if(val>255) { printf("Warning ! Parameter '%d' for option \"%s\" is too large, maximium is 255 !\n",val,left); val=255; }
      if(strcmp(left,"type")==0) value=val;
      else if(strcmp(left,"begin")==0) begin=val;
      else if(strcmp(left,"end")==0) end=val;
      else printf("Unrecognized option \"%s\"\n",left);
    }
    free(left);
    free(right);
    free(line);
  }
  return palette;
}

/* Read objects block (binary) */
static int readObjectsBlockRW(LevelSettings *settings,SDL_RWops *rw,int len) {
  LSB_Objects *objects=NULL,*first=NULL,*prev;
  int read=0;
  Uint16 l;
  while(read<len) {
    prev=objects;
    objects=(LSB_Objects*)malloc(sizeof(LSB_Objects));
    if(objects==NULL) {
      printf("Malloc error at readObjectsBlockRW !\n");
      exit(1);
    }
    memset(objects,0,sizeof(LSB_Objects));
    if(!first) first=objects;
    if(prev) prev->next=(struct LSB_Objects*)objects;
    read+=SDL_RWread(rw,&l,2,1)*2;
    read+=l;
    l-=SDL_RWread(rw,&objects->type,1,1);
    l-=SDL_RWread(rw,&objects->x,4,1)*4;
    l-=SDL_RWread(rw,&objects->y,4,1)*4;
    l-=SDL_RWread(rw,&objects->ceiling_attach,1,1);
    l-=SDL_RWread(rw,&objects->value,1,1);
    l-=SDL_RWread(rw,&objects->id,4,1)*4;
    l-=SDL_RWread(rw,&objects->link,4,1)*4;
    if(l) SDL_RWseek(rw,l,SEEK_CUR);
  }
  settings->objects=first;
  return read;
}

/* Read the palette block (binary) */
static int readPaletteBlockRW(LevelSettings *settings,SDL_RWops *rw,int len) {
  LSB_Palette *palette;
  int read;
  palette=(LSB_Palette*)malloc(sizeof(LSB_Palette));
  if(palette==NULL) {
    printf("Malloc error in readPaletteBlockRW !\n");
    exit(1);
  }
  if(len!=256) printf("Warning: Palette block has strange length (%d)\n",len);
  if(len<256) printf("Error: Palette block length is less than 256 bytes !\n");
  read=SDL_RWread(rw,&palette->entries,1,256);
  if(read!=256) printf("Warning ! Tried to read 256 bytes but read only %d !\n",read);
  len-=read;
  if(len) SDL_RWseek(rw,len,SEEK_CUR);
  settings->palette=palette;
  return read+len;
}

/* Read a single line from a file, discarding whitespace and comment lines */
static char *read_line(FILE *fp) {
  char tmps[512],*line;
  if(fgets(tmps,512,fp)==NULL) return NULL;
  line=strip_white_space(tmps);
  if(line==NULL) return "\0";
  if(line[0]=='#') {free(line); return "\0";}
  return line;
}
