/*
 * Luola - 2D multiplayer cavern-flying game
 * Copyright (C) 2003 Calle Laakkonen
 *
 * File        : game.c
 * Description : Game configuration and initialization
 * Author(s)   : Calle Laakkonen
 *
 * Luola is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Luola is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <dirent.h>
#include <SDL.h>

#include "defines.h"
#include "stringutil.h"
#include "console.h"
#include "fs.h"
#include "game.h"
#include "levelfile.h"
#include "player.h"
#include "intro.h"
#include "font.h"
#include "animation.h"
#include "startup.h"

#if HAVE_LIBSDL_MIXER
#include "audio.h"
#endif

/* Some globals */
SDL_Surface *gam_filler;
GameInfo game_settings;
PerLevelSettings level_settings;
GameStatus game_status;
Uint8 game_loop;

void init_game(void) {
  int p;
  LDAT *datafile;
  datafile=ldat_open_file((char*)getfullpath(GFX_DIRECTORY,"misc.ldat"));
  gam_filler=load_image_ldat(datafile,1,0,"FILLER",0);
  ldat_free(datafile);
  if(gam_filler) recolor(gam_filler,0.4,0.4,0.4,1.0); /* Think the background logo is too bright or dark ? Then adjust these values ! */

  /* Set up the default keys */
  game_settings.buttons[0][0]=SDLK_UP;
  game_settings.buttons[0][1]=SDLK_DOWN;
  game_settings.buttons[0][2]=SDLK_LEFT;
  game_settings.buttons[0][3]=SDLK_RIGHT;
  game_settings.buttons[0][4]=SDLK_RSHIFT;
  game_settings.buttons[0][5]=SDLK_RCTRL;

  game_settings.buttons[1][0]=SDLK_w;
  game_settings.buttons[1][1]=SDLK_s;
  game_settings.buttons[1][2]=SDLK_a;
  game_settings.buttons[1][3]=SDLK_d;
  game_settings.buttons[1][4]=SDLK_LSHIFT;
  game_settings.buttons[1][5]=SDLK_LCTRL;

  game_settings.buttons[2][0]=SDLK_i;
  game_settings.buttons[2][1]=SDLK_k;
  game_settings.buttons[2][2]=SDLK_j;
  game_settings.buttons[2][3]=SDLK_l;
  game_settings.buttons[2][4]=SDLK_y;
  game_settings.buttons[2][5]=SDLK_h;

  game_settings.buttons[3][0]=SDLK_KP8;
  game_settings.buttons[3][1]=SDLK_KP5;
  game_settings.buttons[3][2]=SDLK_KP4;
  game_settings.buttons[3][3]=SDLK_KP6;
  game_settings.buttons[3][4]=SDLK_KP1;
  game_settings.buttons[3][5]=SDLK_KP2;

  for(p=0;p<4;p++) {
    game_settings.controller[p]=Keyboard;
  }
  game_settings.ship_collisions=1;
  game_settings.ls.indstr_base=1;
  game_settings.coll_damage=1;
  game_settings.enable_smoke=1;
  game_settings.ls.jumpgates=0;
  game_settings.jumplife=1;
  game_settings.ls.turrets=0;
  game_settings.ls.critters=1;
  game_settings.ls.cows=5;
  game_settings.ls.fish=5;
  game_settings.ls.birds=5;
  game_settings.ls.bats=5;
  game_settings.ls.soldiers=15;
  game_settings.ls.helicopters=5;
  game_settings.ls.snowfall=0;
  game_settings.ls.stars=1;
  game_settings.endmode=0;
  game_settings.levelcount=0;
  game_settings.levels=NULL;
  game_settings.first_level=NULL;
  game_settings.last_level=NULL;
  game_settings.gravity_bullets=1;
  game_settings.wind_bullets=0;
  game_settings.weapon_switch=0;
  game_settings.eject=1;
  game_settings.explosions=1;
  game_settings.holesize=2;
  game_settings.recall=0;
  game_settings.criticals=0;

  game_settings.sounds=0;
  game_settings.music=0;
  game_settings.playlist=0;
  game_settings.music_vol=128;

  /* Set the temporary settings */
  game_settings.mbg_anim=luola_options.mbg_anim;
  things_loaded[TL_GAME]=1;
}

void reset_game(void) {
  int p;
  for(p=0;p<4;p++) {
    game_settings.players_in[p]=' ';
    game_status.wins[p]=0;
    game_status.lifetime[p]=0;
    player_teams[p]=p;
  }
  game_settings.rounds=5;
  game_settings.playmode=Normal;
  game_status.lastwin=0;
}

void deinit_game(void) {
  if(gam_filler) SDL_FreeSurface(gam_filler);
}

void prematch_game(void) {
  /* Randomize level */
  int r,n;
  n=rand()%(game_settings.levelcount);
  for(r=-1;r<n;r++)
    if(game_settings.levels->next==NULL) game_settings.levels=game_settings.first_level; else game_settings.levels=game_settings.levels->next;
}

void apply_per_level_settings(LevelSettings *settings) {
  if(settings && settings->override) {
    level_settings.critters=game_settings.ls.critters||settings->override->critters_enabled;
    level_settings.stars=settings->override->stars;
    level_settings.snowfall=settings->override->snowfall;
    level_settings.turrets=settings->override->turrets;
    level_settings.jumpgates=settings->override->jumpgates;
    level_settings.cows=settings->override->cows;
    level_settings.fish=settings->override->fish;
    level_settings.birds=settings->override->birds;
    level_settings.bats=settings->override->bats;
    level_settings.soldiers=settings->override->soldiers;
    level_settings.helicopters=settings->override->helicopters;
    level_settings.indstr_base=settings->override->indstr_base;
  } else {
    level_settings=game_settings.ls;
  }
}

void game_statistics(void) {
  Uint32 color,sep_col,team_col,plr_col;
  SDL_Event event;
  SDL_Rect rect1,rect2;
  int t,p,teams[4];
  char tmps[256];
  char data[8][3][32];	/* Four teams and four players */
  char summary[128];
  char data_format[8];	/* Some formatting instructions */
  int data_line,team_data_line;
  long int avglife;
  int mostwins;
  char tiegame;
  int x,y,x2,y2;	/* Variables used when outputting data to screen */
#if HAVE_LIBSDL_GFX
  color=0xC85050FF;
  sep_col=0x5050FFF0;
  team_col=0x0A0A40F0;
  plr_col=0x000032A0;
#else
  color=SDL_MapRGB(screen->format,200,80,80);
  sep_col=SDL_MapRGB(screen->format,80,80,255);
  team_col=SDL_MapRGB(screen->format,10,10,80);
  plr_col=SDL_MapRGB(screen->format,0,0,50);
#endif
  /* Draw background */
  rect1.x=SCREEN_W/2-300; rect1.y=SCREEN_H/2-200;
  rect1.w=600; rect1.h=400;
  rect2.x=rect1.x+2; rect2.y=rect1.y+2;
  rect2.w=rect1.w-4; rect2.h=rect1.h-4;
#if HAVE_LIBSDL_GFX
  draw_box(rect1.x,rect1.y,rect1.w,rect1.h,2,0xC85050BF);
  boxColor(screen,rect2.x,rect2.y,rect2.x+rect2.w-1,rect2.y+rect2.h-1,0x000000BF);
#else
  SDL_FillRect(screen,&rect1,color);
  SDL_FillRect(screen,&rect2,col_black);
#endif
  if(game_status.total_rounds==1)
    sprintf(tmps,_("Game over after one round"));
  else
    sprintf(tmps,_("Game over after %d rounds"),game_status.total_rounds);
  centered_string(screen,Bigfont,rect2.y+10,tmps,font_color_white);
  /* Generate game statistics */
  memset(teams,0,sizeof(int)*4);
  data_line=0;
  for(p=0;p<4;p++) if(players[p].active) teams[player_teams[p]]=1;
  for(t=0;t<4;t++) {
    data_format[data_line]=0;
    sprintf(data[data_line][0],_("Team %d"),t+1);
    if(teams[t]) {
      team_data_line=data_line;
      mostwins=0;
      avglife=0;
      for(p=0;p<4;p++) {
        if(players[p].active && player_teams[p]==t) {
	  data_line++;
          data_format[data_line]=1;
          sprintf(data[data_line][0],_("Player %d"),p+1);
	  sprintf(data[data_line][1],"%d",game_status.wins[p]);
	  sprintf(data[data_line][2],_("%ld seconds"),game_status.lifetime[p]/25/game_status.total_rounds);
	  if(game_status.wins[p]>mostwins) mostwins=game_status.wins[p];
	  avglife+=game_status.lifetime[p]/25/game_status.total_rounds;
	}
      }
      avglife/=(data_line-team_data_line);
      sprintf(data[team_data_line][1],"%d",mostwins);
      sprintf(data[team_data_line][2],_("%ld seconds"),avglife);
      teams[t]=mostwins; // Recycle the 'teams' variable
    } else {
      data[data_line][1][0]='\0';
      data[data_line][2][0]='\0';
      teams[t]=-1;
    }
    data_line++;
  }
  /* Generate summary */
  tiegame=0;
  mostwins=-1;
  for(t=0;t<4;t++) {
    if(teams[t]==mostwins && mostwins>0) { tiegame=1; break; }
    if(teams[t]>mostwins) mostwins=teams[t];
  }
  if(mostwins==0 || tiegame) {
    sprintf(summary,_("Tie game between teams"));
    for(t=0;t<4;t++)
      if(teams[t]==mostwins)
        sprintf(summary,"%s %d ",summary,t+1);
  } else {
    for(t=0;t<4;t++)
      if(teams[t]==mostwins) {
        sprintf(summary,_("Team %d wins !"),t+1);
	break;
      }
  }
  /* Draw game statistics */
  x=rect2.x+10;
  x2=rect2.x+rect2.w-10;
  y=rect2.y+75;
  y2=y+30;
  putstring_direct(screen,Bigfont,rect2.x+200,rect2.y+50,_("Wins"),font_color_green);
  putstring_direct(screen,Bigfont,rect2.x+300,rect2.y+50,_("Average lifetime"),font_color_green);
  for(p=0;p<data_line;p++) {
    if(data_format[p]==0) {		/* Draw team line */
      SDL_Rect fr;
      fr.x=x; fr.y=y;
      fr.w=x2-x; fr.h=y2-y;
#if HAVE_LIBSDL_GFX
      boxColor(screen,fr.x,fr.y,fr.x+fr.w,fr.y+fr.h,team_col);
#else
      SDL_FillRect(screen,&fr,team_col);
#endif
      draw_line(screen,x,y,x2,y,sep_col);
      putstring_direct(screen,Bigfont,x,y,data[p][0],font_color_white);
    } else {
      SDL_Rect fr;
      fr.x=x; fr.y=y;
      fr.w=x2-x; fr.h=y2-y;
#if HAVE_LIBSDL_GFX
      boxColor(screen,fr.x,fr.y,fr.x+fr.w,fr.y+fr.h,plr_col);
#else
      SDL_FillRect(screen,&fr,plr_col);
#endif
      putstring_direct(screen,Bigfont,x+40,y,data[p][0],font_color_white);
    }
    putstring_direct(screen,Bigfont,rect2.x+200,y,data[p][1],font_color_white);
    putstring_direct(screen,Bigfont,rect2.x+300,y,data[p][2],font_color_white);
    y+=30;
    y2+=30;
  }
  /* Draw summary line */
  putstring_direct(screen,Bigfont,x,y,summary,font_color_white);
  /* Draw the "press enter to continue" message */
  centered_string(screen,Bigfont,rect2.y+rect2.h-30,_("Press enter to continue"),font_color_white);
  SDL_UpdateRect(screen,rect1.x,rect1.y,rect1.w,rect1.h);
  /* Wait for Enter to be pressed */
  while(1) {
    SDL_WaitEvent(&event);
    if(event.type==SDL_KEYDOWN && event.key.keysym.sym==SDLK_RETURN) break;
    if(event.type==SDL_JOYBUTTONDOWN) break;
  }
}

void fill_unused_player_screens(void) {
  SDL_Rect rect;
  int p;
  rect.w=320;
  rect.h=320;
  rect.x=0;
  rect.y=0;
  for(p=0;p<4;p++) {
    if(gam_filler) SDL_BlitSurface(gam_filler,NULL,screen,&rect); else SDL_FillRect(screen,&rect,0);
    rect.x+=320;
    if(rect.x==640) {rect.y+=240; rect.x=0;}
  }
  SDL_UpdateRect(screen,0,0,0,0);
}

/* Ingame event loop */
void game_eventloop(void) {
  SDL_Event Event;
  Uint32 lasttime = SDL_GetTicks(),delay;
  Uint8 is_not_paused=1;
  game_loop=1;
  while(game_loop) {
    while(SDL_PollEvent(&Event)) {
      switch(Event.type) {
        case SDL_KEYDOWN:
        case SDL_KEYUP:
#ifdef CHEAT_POSSIBLE
  	      if(cheat && Event.type==SDL_KEYDOWN) cheatcode(Event.key.keysym.sym);
#endif
          if(Event.key.keysym.sym==SDLK_ESCAPE && Event.type==SDL_KEYDOWN) return;
          else if(Event.key.keysym.sym==SDLK_F1 && Event.type==SDL_KEYDOWN) radars_visible=!radars_visible;
    	  else if(Event.key.keysym.sym==SDLK_F11 && Event.type==SDL_KEYDOWN) screenshot();
	      else if(Event.key.keysym.sym==SDLK_F12 && Event.type==SDL_KEYUP) SDL_WM_ToggleFullScreen(screen);
		  else if(Event.key.keysym.sym==SDLK_PAUSE && Event.type==SDL_KEYUP) is_not_paused=pause_game();
		  else player_keyhandler(&Event.key,Event.type);
		  break;
        case SDL_JOYBUTTONDOWN:
        case SDL_JOYBUTTONUP:
	      player_joybuttonhandler(&Event.jbutton); break;
        case SDL_JOYAXISMOTION:
		  player_joyaxishandler(&Event.jaxis); break;
		default: break;
      }
	}
    if(is_not_paused) {
      lasttime=SDL_GetTicks();
      animate_frame();
      delay=SDL_GetTicks()-lasttime;
      if(delay>=GAME_SPEED) delay=0; else delay=GAME_SPEED-delay;
      SDL_Delay(delay);
    }
  }
}

/* Save game configuration to file */
void save_game_config(void) {
  int p,k;
  FILE *fp;
  const char *filename;
  filename=getfullpath(HOME_DIRECTORY,CONF_FILE);
  fp=fopen(filename,"w");
  if(!fp) {
    printf("Error ! Cannot open file \"%s\" for writing !\n",filename);
    exit(1);
  }
  /* Write controllers */
  fprintf(fp,"[controllers]\n");
  for(p=0;p<4;p++) {
    fprintf(fp,"%d=%d\n",p,game_settings.controller[p]);
  }
  /* Write keys */
  for(p=0;p<4;p++) {
    fprintf(fp,"[keys%d]\n",p+1);
    for(k=0;k<6;k++) {
      fprintf(fp,"%d=%d\n",k,game_settings.buttons[p][k]);
    }
  }
  /* Write settings */
  fprintf(fp,"[settings]\n");
  fprintf(fp,"ship_collisions=%d\n",game_settings.ship_collisions);
  fprintf(fp,"indestructable_base=%d\n",game_settings.ls.indstr_base);
  fprintf(fp,"collision_damage=%d\n",game_settings.coll_damage);
  fprintf(fp,"jumpgates=%d\n",game_settings.ls.jumpgates);
  fprintf(fp,"jumplife=%d\n",game_settings.jumplife);
  fprintf(fp,"turrets=%d\n",game_settings.ls.turrets);
  fprintf(fp,"critters=%d\n",game_settings.ls.critters);
  fprintf(fp,"cows=%d\n",game_settings.ls.cows);
  fprintf(fp,"birds=%d\n",game_settings.ls.birds);
  fprintf(fp,"fish=%d\n",game_settings.ls.fish);
  fprintf(fp,"soldiers=%d\n",game_settings.ls.soldiers);
  fprintf(fp,"helicopters=%d\n",game_settings.ls.helicopters);
  fprintf(fp,"bats=%d\n",game_settings.ls.bats);
  fprintf(fp,"smoke=%d\n",game_settings.enable_smoke);
  fprintf(fp,"bullet_gravity=%d\n",game_settings.gravity_bullets);
  fprintf(fp,"bullet_wind=%d\n",game_settings.wind_bullets);
  fprintf(fp,"snowfall=%d\n",game_settings.ls.snowfall);
  fprintf(fp,"stars=%d\n",game_settings.ls.stars);
  fprintf(fp,"endmode=%d\n",game_settings.endmode);
  fprintf(fp,"weapon_switch=%d\n",game_settings.weapon_switch);
  fprintf(fp,"eject=%d\n",game_settings.eject);
  fprintf(fp,"explosions=%d\n",game_settings.explosions);
  fprintf(fp,"holesize=%d\n",game_settings.holesize);
  fprintf(fp,"recall=%d\n",game_settings.recall);
  fprintf(fp,"criticalhits=%d\n",game_settings.criticals);

  fprintf(fp,"sounds=%d\n",game_settings.sounds);
  fprintf(fp,"music=%d\n",game_settings.music);
  fprintf(fp,"playlist=%d\n",game_settings.playlist);
  fprintf(fp,"music_volume=%d\n",game_settings.music_vol);

  fclose(fp);
}

void load_game_config(void) {
  int readmode;
  FILE *fp;
  const char *filename;
  char tmps[256],*line=NULL,*left,*right;
  int r;
  filename=getfullpath(HOME_DIRECTORY,CONF_FILE);
  fp=fopen(filename,"r");
  if(!fp) {
    printf("No configuration file \"%s\". Using built in defaults.\n",filename);
    return;
  }
  readmode=-1;
  for (; fgets(tmps,sizeof(tmps)-1,fp); free(line)) {
    line=strip_white_space(tmps);
    if(line==NULL) continue;
    if(line[0]=='[') readmode=-1;
    if(strcmp(line,"[keys1]")==0) {readmode=0; continue;}
    else if(strcmp(line,"[keys2]")==0) {readmode=1; continue;}
    else if(strcmp(line,"[keys3]")==0) {readmode=2; continue;}
    else if(strcmp(line,"[keys4]")==0) {readmode=3; continue;}
    else if(strcmp(line,"[settings]")==0) {readmode=4; continue;}
    else if(strcmp(line,"[controllers]")==0) {readmode=5; continue;}
    left = right = NULL;
    if(readmode==0) {
      split_string(line,'=',&left,&right);
      game_settings.buttons[0][atoi(left)]=atoi(right);
    }
    else if(readmode==1) {
      split_string(line,'=',&left,&right);
      game_settings.buttons[1][atoi(left)]=atoi(right);
    }
    else if(readmode==2) {
      split_string(line,'=',&left,&right);
      game_settings.buttons[2][atoi(left)]=atoi(right);
    }
    else if(readmode==3) {
      split_string(line,'=',&left,&right);
      game_settings.buttons[3][atoi(left)]=atoi(right);
    }
    else if(readmode==4) {
      split_string(line,'=',&left,&right);
      if(strcmp(left,"ship_collisions")==0) game_settings.ship_collisions=atoi(right);
      else if(strcmp(left,"indestructable_base")==0) game_settings.ls.indstr_base=atoi(right);
      else if(strcmp(left,"collision_damage")==0) game_settings.coll_damage=atoi(right);
      else if(strcmp(left,"jumpgates")==0) game_settings.ls.jumpgates=atoi(right);
      else if(strcmp(left,"jumplife")==0) game_settings.jumplife=atoi(right);
      else if(strcmp(left,"turrets")==0) game_settings.ls.turrets=atoi(right);
      else if(strcmp(left,"critters")==0) game_settings.ls.critters=atoi(right);
      else if(strcmp(left,"cows")==0) game_settings.ls.cows=atoi(right);
      else if(strcmp(left,"birds")==0) game_settings.ls.birds=atoi(right);
      else if(strcmp(left,"fish")==0) game_settings.ls.fish=atoi(right);
      else if(strcmp(left,"bats")==0) game_settings.ls.bats=atoi(right);
      else if(strcmp(left,"soldiers")==0) game_settings.ls.soldiers=atoi(right);
      else if(strcmp(left,"helicopters")==0) game_settings.ls.helicopters=atoi(right);
      else if(strcmp(left,"smoke")==0) game_settings.enable_smoke=atoi(right);
      else if(strcmp(left,"bullet_gravity")==0) game_settings.gravity_bullets=atoi(right);
      else if(strcmp(left,"bullet_wind")==0) game_settings.wind_bullets=atoi(right);
      else if(strcmp(left,"snowfall")==0) game_settings.ls.snowfall=atoi(right);
      else if(strcmp(left,"stars")==0) game_settings.ls.stars=atoi(right);
      else if(strcmp(left,"endmode")==0) game_settings.endmode=atoi(right);
      else if(strcmp(left,"weapon_switch")==0) game_settings.weapon_switch=atoi(right);
      else if(strcmp(left,"eject")==0) game_settings.eject=atoi(right);
      else if(strcmp(left,"explosions")==0) game_settings.explosions=atoi(right);
      else if(strcmp(left,"holesize")==0) game_settings.holesize=atoi(right);
      else if(strcmp(left,"recall")==0) game_settings.recall=atoi(right);
      else if(strcmp(left,"criticalhits")==0) game_settings.criticals=atoi(right);
      else if(strcmp(left,"sounds")==0) game_settings.sounds=atoi(right);
      else if(strcmp(left,"music")==0) game_settings.music=atoi(right);
      else if(strcmp(left,"playlist")==0) game_settings.playlist=atoi(right);
      else if(strcmp(left,"music_volume")==0) game_settings.music_vol=atoi(right);
    }
    else if(readmode==5) {
      split_string(line,'=',&left,&right);
      game_settings.controller[atoi(left)]=atoi(right);
    }
    if (left && right) {
      free(right);
      free(left);
    }
  }
  fclose(fp);
  /* Some last minute checkups */
  for(r=0;r<4;r++)
    if(game_settings.controller[r]>=Controllers) game_settings.controller[r]=0;
}
