/*
 * Luola - 2D multiplayer cavern-flying game
 * Copyright (C) 2003 Calle Laakkonen
 *
 * File        : console.c
 * Description : 
 * Author(s)   : Calle Laakkonen
 *
 * Luola is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Luola is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>

#include "defines.h"
#include "fs.h"
#include "console.h"
#include "animation.h"

#include "game.h"
#include "player.h"
#include "particle.h"
#include "weapon.h"
#include "special.h"
#include "critter.h"
#include "hotseat.h"
#include "audio.h"
#include "font.h"
#include "demo.h"
#include "startup.h"

/** Globals **/
SDL_Surface *screen;

unsigned int Controllers;

Uint32 col_gray,col_grenade,col_snow,col_clay,col_clay2,col_default,col_yellow,col_black,col_red,col_white,col_rope,col_plrs[4];
Uint32 col_pause_backg,col_green,col_blue,col_transculent;

char things_loaded[12]={0,0,0,0,0,0,0,0,0,0,0,0};

/* Internally used globals */
static SDL_Joystick **pads;
static int jscount;

/** Functions used only in this module **/
static void con_cleanup(void);

/** Initalize SDL **/
void init_sdl() {
  Uint32 initflags;
  int j;
  initflags=SDL_INIT_VIDEO;
  if(luola_options.joystick) initflags=initflags|SDL_INIT_JOYSTICK;
#if HAVE_LIBSDL_MIXER
  if(luola_options.sounds) initflags=initflags|SDL_INIT_AUDIO;
#endif
  if(SDL_Init(initflags)) {
    printf("Error: Unable to initialize SDL: %s\n",SDL_GetError());
    exit(1);
  }
  pads=NULL;
  jscount=0;
  if(luola_options.joystick) {
    jscount=SDL_NumJoysticks();
    if(jscount==0) printf("Warning: No joysticks available\n");
    else pads=(SDL_Joystick**)malloc(sizeof(SDL_Joystick*)*jscount);
    memset(pads,0,sizeof(SDL_Joystick*)*jscount);
    for(j=0;j<jscount;j++) {
      pads[j]=SDL_JoystickOpen(j);
      if(pads[j]==NULL) printf("Warning: Unable to open joystick %d: %s\n",j,SDL_GetError());
    }
  }
  atexit(con_cleanup);
  Controllers=jscount+1;	/* One keyboard + joysticks */
}
/** Initialize video **/
void init_video() {
  char name[128];
  screen = SDL_SetVideoMode(SCREEN_W,SCREEN_H,SCREEN_DEPTH,luola_options.fullscreen?SDL_FULLSCREEN:0|SDL_HWSURFACE|SDL_HWACCEL);
  if(screen==NULL) {
    printf("Unable to set video mode: %s\n", SDL_GetError());
    exit(1);
  }
  if(luola_options.hidemouse) SDL_ShowCursor(SDL_DISABLE);
  /* Set window caption */
  sprintf(name,"%s %s",PACKAGE,VERSION);
  SDL_WM_SetCaption(name,PACKAGE);
  /* Initialize colours */
#if HAVE_LIBSDL_GFX
  col_black=0x000000FF;
  col_gray=0x808080FF;
  col_grenade=0xA0A0A0FF;
  col_snow=0xB0C1FFFF;
  col_clay=0xFFC880FF;
  col_default=0xFF6464FF;
  col_yellow=0xFFFF64FF;
  col_red=0xFF0000FF;
  col_white=0xFFFFFFFF;
  col_rope=0xB29F67FF;
  col_plrs[0]=0xFF0000FF;
  col_plrs[1]=0x0000FFFF;
  col_plrs[2]=0x00FF00FF;
  col_plrs[3]=0xFFFF00FF;
  col_pause_backg=0x80808080;
  col_green=0x009E00FF;
  col_blue=0x0000FFFF;
  col_transculent=0xFFFFFF80;
#else
  col_black=SDL_MapRGB(screen->format,0,0,0);
  col_gray=SDL_MapRGB(screen->format,128,128,128);
  col_grenade=SDL_MapRGB(screen->format,160,160,160);
  col_snow=SDL_MapRGB(screen->format,176,193,255);
  col_clay=SDL_MapRGB(screen->format,255,200,128);
  col_default=SDL_MapRGB(screen->format,255,100,100);
  col_yellow=SDL_MapRGB(screen->format,255,255,100);
  col_red=SDL_MapRGB(screen->format,255,0,0);
  col_blue=SDL_MapRGB(screen->format,0,0,255);
  col_white=SDL_MapRGB(screen->format,255,255,255);
  col_rope=SDL_MapRGB(screen->format,178,159,103);
  col_plrs[0]=SDL_MapRGB(screen->format,255,0,0);
  col_plrs[1]=SDL_MapRGB(screen->format,0,0,255);
  col_plrs[2]=SDL_MapRGB(screen->format,0,255,0);
  col_plrs[3]=SDL_MapRGB(screen->format,255,255,0);
  col_pause_backg=SDL_MapRGB(screen->format,128,128,128);
  col_green=SDL_MapRGB(screen->format,0,158,0);
  col_transculent=col_gray;
#endif
}

/** Cleanup **/
void con_cleanup(void) {
  int p;
  if(things_loaded[TL_SHIPS]) deinit_ships();
  if(things_loaded[TL_PILOTS]) deinit_pilots();
  if(things_loaded[TL_PLAYERS]) deinit_players();
  if(things_loaded[TL_PARTICLES]) deinit_particles();
  if(things_loaded[TL_WEAPONS]) deinit_weapons(0);
  if(things_loaded[TL_SPECIAL]) deinit_specials(0);
  if(things_loaded[TL_CRITTERS]) deinit_critters(0);
  if(things_loaded[TL_GAME]) deinit_game();
  if(things_loaded[TL_INTRO]) deinit_intro();
  if(things_loaded[TL_HOTSEAT]) deinit_hotseat();
  if(things_loaded[TL_FONT]) deinit_font();
  if(things_loaded[TL_DEMOS]) deinit_demos();
  for(p=0;p<jscount;p++)
    if(pads[p]) SDL_JoystickClose(pads[p]);
#if HAVE_LIBSDL_MIXER
  if(things_loaded[TL_AUDIO]) deinit_audio();
#endif
  SDL_Quit();
}

/**** Draw a box *****/
void draw_box(int x,int y,int w,int h,char width,Uint32 color) {
#if HAVE_LIBSDL_GFX
  int r;
  for(r=0;r<width;r++)
    rectangleColor(screen,x+r,y+r,x+w-r,y+h-r,color);
#else
  SDL_Rect rect;
  rect.x=x;
  rect.y=y;
  rect.w=w;
  rect.h=width;
  SDL_FillRect(screen,&rect,color);
  rect.y+=h-width;
  SDL_FillRect(screen,&rect,color);
  rect.x=x;
  rect.y=y;
  rect.w=width;
  rect.h=h;
  SDL_FillRect(screen,&rect,color);
  rect.x+=w-width;
  SDL_FillRect(screen,&rect,color);
#endif
}

/* Do a raw 32bit pixel copy */
void pixelcopy(Uint32 *srcpix, Uint32 *pixels,int w,int h,int srcpitch,int pitch) {
  int x,y;
  for(y=0;y<h;y++) {
    for(x=0;x<w;x++) {
      *pixels=*srcpix;
      srcpix++;
      pixels++;
    }
    pixels+=pitch-w;
    srcpix+=srcpitch-w;
  }
}

/* Returns a new surface that is a copy of the original surface upside down */
SDL_Surface *flip_surface(SDL_Surface *original) {
  Uint8 bpp,*src,*targ;
  SDL_Surface *flipped;
  int y;
  bpp=original->format->BytesPerPixel;
  flipped=SDL_CreateRGBSurface(original->flags,original->w,original->h,original->format->BitsPerPixel,original->format->Rmask,original->format->Gmask,original->format->Bmask,original->format->Amask);
  if(bpp==1) /* Copy the palette */
    SDL_SetPalette(flipped,SDL_LOGPAL,original->format->palette->colors,0,original->format->palette->ncolors);
  src = ((Uint8*)original->pixels)+(original->h-1)*original->pitch;
  targ = ((Uint8*)flipped->pixels);
  for(y=0;y<flipped->h;y++) {
    memcpy(targ,src,flipped->pitch);
    targ+=flipped->pitch;
    src-=original->pitch;
  }
  return flipped;
}

/* Returns a copy of the surface */
SDL_Surface *copy_surface(SDL_Surface *original) {
  SDL_Surface *newsurface;
  Uint8 bpp,*src,*targ;
  bpp=original->format->BytesPerPixel;
  newsurface=SDL_CreateRGBSurface(original->flags,original->w,original->h,original->format->BitsPerPixel,original->format->Rmask,original->format->Gmask,original->format->Bmask,original->format->Amask);
  if(bpp==1) /* Copy the palette */
    SDL_SetPalette(newsurface,SDL_LOGPAL,original->format->palette->colors,0,original->format->palette->ncolors);
  src = (Uint8*)original->pixels;
  targ = (Uint8*)newsurface->pixels;
  memcpy(targ,src,original->pitch*original->h);
  return newsurface;
}

/* Rectangle clipping */
SDL_Rect cliprect(int x1,int y1,int w1,int h1,int x2,int y2,int x3,int y3) {
  int d;
  SDL_Rect r;
  r.x=0; r.y=0;
  r.w=w1;
  r.h=h1;
  if(x1<x2) {
    d=x2-x1;
    r.x+=d;
    r.w-=d;
  }
  if(x1+w1>x3) {
    r.w-=x1+w1-x3;
  }
  if(y1<y2) {
    d=y2-y1;
    r.y+=d;
    r.h-=d;
  }
  if(y1+h1>y3) {
    r.h-=y1+h1-y3;
  }
  return r;
}

char clip_line(int *x1,int *y1,int *x2,int *y2,int left,int top,int right,int bottom) {
  double m;
  if((*x1<left && *x2<left) || (*x1>right && *x2>right)) return 0;
  if((*y1<top && *y2<top) || (*y1>bottom && *y2>bottom)) return 0;
  if(*x1!=*x2) {
    m=(double)(*y2-*y1)/(double)(*x2-*x1);
  } else {
    m=1.0;
  }
  if(*x1<left) {
    *y1+=(left-*x1)*m;
    *x1=left;
  }
  else if(*x2<left) {
    *y2+=(left-*x2)*m;
    *x2=left;
  }
  if(*x2>right) {
    *y2+=(right-*x2)*m;
    *x2=right;
  }
  else if(*x1>right) {
    *y1+=(right-*x1)*m;
    *x1=right;
  }
  if(*y1>bottom) {
    if(*x2!=*x1) {
      *x1+=(bottom-*y1)/m;
    }
    *y1=bottom;
  } else if (*y2>bottom) {
    if(*x2!=*x1) {
      *x2+=(bottom-*y2)/m;
    }
    *y2=bottom;
  }
  if(*y1<top) {
    if(*x2!=*x1) {
      *x1 += (top-*y1)/m;
    }
    *y1=top;
  } else if(*y2<top) {
    if(*x2!=*x1) {
      *x2 += (top-*y2)/m;
    }
    *y2=top;
  }
  return 1;
}

/* Modify the colors of a surface (Used to change the player colors in ship sprites and such) */
void recolor(SDL_Surface *surface,float red,float green,float blue,float alpha) {
  Uint8 *pos, *end;
  if(surface->format->BytesPerPixel != 4) {
    printf("Error ! Unsupported color depth (%d bytes per pixel), currently only 4 bps is supported ! (fix this at recolor(), console.c)\n", screen->format->BytesPerPixel);
    exit(1);
  }

  pos = (Uint8*)surface->pixels;
  end = pos + 4 * surface->w * surface->h;
  while (pos < end) {
     (*pos++) *= blue;
     (*pos++) *= green;
     (*pos++) *= red;
     (*pos++) *= alpha;
  }
}

#ifndef HAVE_LIBSDL_GFX
SDL_Surface *scale_surface(SDL_Surface *original,int sX,int sY,int smooth) { /* Note. smooth is unused. It is for API compatability with zoomSurface from SDL_gfx */
  SDL_Surface *scaled;
  Uint8 *bits,*targ,bpp;
  int x,y,xscale,yscale;
  bpp=original->format->BytesPerPixel;
  scaled=SDL_CreateRGBSurface(original->flags,original->w*sX,original->h*sY,original->format->BitsPerPixel,original->format->Rmask,original->format->Gmask,original->format->Bmask,original->format->Amask);
  if(bpp==1) /* Copy the palette */
    SDL_SetPalette(scaled,SDL_LOGPAL,original->format->palette->colors,0,original->format->palette->ncolors);
  bits = ((Uint8*)original->pixels);
  targ = ((Uint8*)scaled->pixels);
  for(y=0;y<original->h;y++) {
    for(x=0;x<original->w;x++) {
      switch(bpp) {
      case 1:
	for(yscale=0;yscale<sY;yscale++) {
          for(xscale=0;xscale<sX;xscale++) {
  	    *((Uint8 *)(targ)) = (*(Uint8*)(bits));
	    targ++;
          }
	  targ+=scaled->pitch-sX;
	}
	break;
      case 4:
	for(yscale=0;yscale<sY;yscale++) {
	  for(xscale=0;xscale<sX;xscale++) {
	    *((Uint32 *)(targ)) = (*(Uint32*)(bits));
	    targ+=4;
	  }
	  targ+=scaled->pitch-4*sX;
	}
      }
      bits+=bpp;
      targ-=(scaled->pitch*sY)-bpp*sX;
    }
    targ+=scaled->pitch*(sY-1);
  }
  return scaled;
}

#endif

void joystick_button(SDL_JoyButtonEvent *btn) {
  SDL_Event event;
  event.key.state=btn->state;
  event.key.type=(btn->type==SDL_JOYBUTTONDOWN)?SDL_KEYDOWN:SDL_KEYUP;
  if(btn->button==0||btn->button==3) event.key.keysym.sym=SDLK_RETURN; else return;
  SDL_PushEvent(&event);
}

void joystick_motion(SDL_JoyAxisEvent *axis,char plrmode) {
  SDL_Event event;
  int p,plr=-1;
  if(axis->axis>1) return; /* Only the first two axises are handled */
  if(abs(axis->value)>JSTRESHOLD) {
    event.key.state=SDL_PRESSED;
    event.key.type=SDL_KEYDOWN;
  } else {
    event.key.state=SDL_RELEASED;
    event.key.state=SDL_KEYUP;
  }
  for(p=0;p<4;p++)
    if(game_settings.controller[p]-1==axis->which) {plr=p; break; }
  if(axis->axis==0) p=2;
  else p=0;
  if(axis->value>0) p++;
  if(p==0) {
    if(plrmode && plr>-1)
      event.key.keysym.sym=game_settings.buttons[plr][0];
    else event.key.keysym.sym=SDLK_UP;
  } else if(p==1) {
    if(plrmode && plr>-1)
      event.key.keysym.sym=game_settings.buttons[plr][1];
    else event.key.keysym.sym=SDLK_DOWN;
  } else if(p==2) {
    if(plrmode && plr>-1)
      event.key.keysym.sym=game_settings.buttons[plr][2];
    else event.key.keysym.sym=SDLK_LEFT;
  } else {
    if(plrmode &&plr>-1)
      event.key.keysym.sym=game_settings.buttons[plr][3];
    else event.key.keysym.sym=SDLK_RIGHT;
  }
  SDL_PushEvent(&event);
}

#ifndef HAVE_LIBSDL_GFX
/* Draw a single pixel */
/* Note. The support code for other bit depths than 32bit is currently commented out.
 * this is because, this function is currently used only with 32bit surfaces.
 * Should this change, remove the comments. */

inline void putpixel(SDL_Surface *surface,int x,int y,Uint32 color) {
#if SCREEN_DEPTH != 32
#error update putpixel() with bitdepths other than 32 !
#endif
  Uint8 *bits,bpp;
  if(x<0 || y<0 || x>=surface->w || y>=surface->h) return;
  bpp = surface->format->BytesPerPixel;
  bits = ((Uint8 *)surface->pixels)+y*surface->pitch+x*bpp;
  /*switch(bpp) {
    case 1:
     *((Uint8 *)(bits)) = (Uint8)color;
     break;
    case 2:
      *((Uint16 *)(bits)) = (Uint16)color;
      break;
    case 3: {
      Uint8 r, g, b;
      r = (color>>surface->format->Rshift)&0xFF;
      g = (color>>surface->format->Gshift)&0xFF;
      b = (color>>surface->format->Bshift)&0xFF;
      *((bits)+surface->format->Rshift/8) = r;
      *((bits)+surface->format->Gshift/8) = g;
      *((bits)+surface->format->Bshift/8) = b;
      }
      break;
    case 4:
    */
      *((Uint32 *)(bits)) = (Uint32)color;
  /*}*/
}
#endif

#ifndef HAVE_LIBSDL_GFX
/* Draw a line from point A to point B */
void draw_line(SDL_Surface *screen,int x1,int y1,int x2,int y2,Uint32 pixel) {
    Uint8 *bits,bpp;
    Uint8 r,g,b;
    int dx,dy;
    int ax,ay;
    int sx,sy;
    int x,y;
    
    if(x1<0) x1=0; else if(x1>screen->w) x1=screen->w;
    if(x2<0) x2=0; else if(x2>screen->w) x2=screen->w;
    if(y1<0) y1=0; else if(y1>screen->h) y1=screen->h;
    if(y2<0) y2=0; else if(y2>screen->h) y2=screen->h;

    dx=x2-x1;
    dy=y2-y1;
    ax=abs(dx) << 1;
    ay=abs(dy) << 1;
    sx=(dx>=0)?1:-1;
    sy=(dy>=0)?1:-1;
    x=x1;
    y=y1;
    bpp = screen->format->BytesPerPixel;
    if (ax > ay) {
	int d = ay - (ax >> 1);
	while (x != x2) {
		/***  DRAW PIXEL HERE ***/	
	 	bits = ((Uint8 *)screen->pixels)+y*screen->pitch+x*bpp;
		switch(bpp) {
		  case 1:
			*((Uint8 *)(bits)) = (Uint8)pixel;
			break;
		  case 2:
			*((Uint16 *)(bits)) = (Uint16)pixel;
			break;
		  case 3: { /* Format/endian independent */
			r = (pixel>>screen->format->Rshift)&0xFF;
			g = (pixel>>screen->format->Gshift)&0xFF;
			b = (pixel>>screen->format->Bshift)&0xFF;
			*((bits)+screen->format->Rshift/8) = r;
			*((bits)+screen->format->Gshift/8) = g;
			*((bits)+screen->format->Bshift/8) = b;
			}
			break;
		  case 4:
			*((Uint32 *)(bits)) = (Uint32)pixel;
			break;
		}
	    	/*** END DRAWING PIXEL ***/
	    if (d > 0 || (d == 0 && sx == 1)) {
		y += sy;
		d -= ax;
	    }
	    x += sx;
	    d += ay;
	}
    } else {
	int d = ax - (ay >> 1);
	while (y != y2) {
		/*** DRAW PIXEL HERE ***/
	bits = ((Uint8 *)screen->pixels)+y*screen->pitch+x*bpp;
		switch(bpp) {
		  case 1:
			*((Uint8 *)(bits)) = (Uint8)pixel;
			break;
		  case 2:
			*((Uint16 *)(bits)) = (Uint16)pixel;
			break;
		  case 3: { /* Format/endian independent */
			r = (pixel>>screen->format->Rshift)&0xFF;
			g = (pixel>>screen->format->Gshift)&0xFF;
			b = (pixel>>screen->format->Bshift)&0xFF;
			*((bits)+screen->format->Rshift/8) = r;
			*((bits)+screen->format->Gshift/8) = g;
			*((bits)+screen->format->Bshift/8) = b;
			}
			break;
		  case 4:
			*((Uint32 *)(bits)) = (Uint32)pixel;
			break;
		}

	    	/*** END DRAWING PIXEL ***/

	    if (d > 0 || (d == 0 && sy == 1)) {
		x += sx;
		d -= ay;
	    }
	    y += sy;
	    d += ax;
	}
    }
    /*** DRAW PIXEL HERE ***/
    bits = ((Uint8 *)screen->pixels)+y*screen->pitch+x*bpp;
	switch(bpp) {
	case 1:
		*((Uint8 *)(bits)) = (Uint8)pixel;
		break;
	  case 2:
		*((Uint16 *)(bits)) = (Uint16)pixel;
		break;
	  case 3: { /* Format/endian independent */
		r = (pixel>>screen->format->Rshift)&0xFF;
		g = (pixel>>screen->format->Gshift)&0xFF;
		b = (pixel>>screen->format->Bshift)&0xFF;
		*((bits)+screen->format->Rshift/8) = r;
		*((bits)+screen->format->Gshift/8) = g;
		*((bits)+screen->format->Bshift/8) = b;
		}
		break;
	  case 4:
		*((Uint32 *)(bits)) = (Uint32)pixel;
		break;
	}
    /*** END DRAWING PIXEL HERE ***/
}

#endif

#ifdef DONT_UPDATE_DEAD
void kill_plr_screen(unsigned char plr) {
  SDL_Rect rect,rect2;
  rect.x=(plr==1||plr==3)*320; rect2.x=rect.x+160-plr_messages[plr]->w/2;
  rect.y=(plr>1)*240; rect2.y=rect.y+120-plr_messages[plr]->h/2;
  rect.w=320; rect.h=240;
  SDL_FillRect(screen,&rect,0);
  SDL_BlitSurface(plr_messages[plr],NULL,screen,&rect2);
  SDL_UpdateRect(screen,rect.x,rect.y,rect.w,rect.h);
}

#endif

