/* --------------------------luckybackupwindow.cpp -----------------------------------------------------------------------------
cpp file that does all stuff needed when a signal is transmitted (eg button pressed) from the main app window

===============================================================================================================================
===============================================================================================================================
    This file is part of "luckyBackup" project
    Copyright 2008-2009, Loukas Avgeriou
    luckyBackup is distributed under the terms of the GNU General Public License
    luckyBackup is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    luckyBackup is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with luckyBackup.  If not, see <http://www.gnu.org/licenses/>.


project version	: Please see "main.cpp" for project version

developer 	: luckyb 
last modified 	: xx May 2009
===============================================================================================================================
===============================================================================================================================
*/

#include <QtGui>

#include "global.cpp"

//inluded headers ---------------------------------------------------------------
#include "luckybackupwindow.h"

//inluded sources ---------------------------------------------------------------
#include "modifyDialog.cpp"
#include "textDialog.cpp"
#include "rsyncDialog.cpp"
#include "helpBrowser.cpp"
#include "scheduleDialog.cpp"
#include "about.cpp"
#include "manage.cpp"
#include "functions.cpp"


//class luckyBackupWindow constructor=============================================================================================
luckyBackupWindow::luckyBackupWindow(QMainWindow *parent) : QMainWindow(parent)
{
	ui.setupUi(this);

	InitializeVariables();

	//create directory ~/.luckyBackup including folders "logs", "profiles" & "schedule" if it does not exist for the current user
	QDir profiledir(profileDir), logdir(logDir), scheduledir(scheduleDir);
	if (!profiledir.exists())
		profiledir.mkpath(profileDir);
	if (!logdir.exists())
		logdir.mkpath(logDir);
	if (!scheduledir.exists())
		scheduledir.mkpath(scheduleDir);

	// load the user settings file. If it does not exist create it
	QFile settingsfile(settingsFile);
	if (settingsfile.exists())
		loadSettings();
	else
		saveSettings();

	createActions();	//create all actions needed
	createToolbar();	// create tool bar
	createMenus();		//create the top menu

	retranslateUi();
	ui.progressBar -> hide();
//	ui.pushButton_manage -> hide();
	this -> resize(mainWindowWidth, mainWindowHeight);

	// Connections ----------------------------------------------------------------------------------------------------------
	connect ( ui.pushButton_exit, SIGNAL( clicked() ), this, SLOT( exitApp() ) );		//pushbutton exit
	connect( ui.actionQuit, SIGNAL(triggered()), this, SLOT(exitApp()));			//menu action quit

	connect( ui.comboBox_profile, SIGNAL(currentIndexChanged (const int) ), this, SLOT(profileComboChanged()));

	connect( ui.actionNew, SIGNAL(triggered()), this, SLOT(NewProfile()));			//menu action newProfile
	connect( ui.actionRename, SIGNAL(triggered()), this, SLOT(renameCurrentProfile()));			//menu action renameProfile
	connect( ui.actionSave, SIGNAL(triggered()), this, SLOT(savePressed()));		//menu action saveProfile
	connect ( ui.actionSchedule, SIGNAL( triggered() ), this, SLOT( schedule() ) );		//action schedule
	connect ( ui.actionRefresh, SIGNAL( triggered() ), this, SLOT( refreshList() ) );	//action refresh (task list)
	connect ( ui.actionDelete, SIGNAL( triggered() ), this, SLOT( deleteCurrentProfile() ) );	//action delete (currentProfile)
	connect ( ui.actionDefault, SIGNAL( triggered() ), this, SLOT( setDefaultProfile() ) );	//action default (set currentProfile as default)
	connect ( ui.actionImport, SIGNAL( triggered() ), this, SLOT( importProfile() ) );	//action import profile
	connect ( ui.actionExport, SIGNAL( triggered() ), this, SLOT( exportProfile() ) );	//action export profile

	connect (ui.pushButton_add, SIGNAL (clicked() ), this, SLOT(add()));			//pushbutton create (task)
	connect (ui.pushButton_edit, SIGNAL (clicked() ), this, SLOT(modify()));		//pushbutton modify (task)
	connect ( ui.pushButton_remove, SIGNAL( clicked() ), this, SLOT( remove() ) );		//pushbutton remove (task)
	connect ( ui.pushButton_manage, SIGNAL( clicked() ), this, SLOT( manage() ) );		//pushbutton manage (task)
	connect ( ui.pushButton_up, SIGNAL( clicked() ), this, SLOT( moveUp() ) );		//pushbutton up
	connect ( ui.pushButton_down, SIGNAL( clicked() ), this, SLOT( moveDown() ) );		//pushbutton down
	connect ( ui.pushButton_start, SIGNAL( clicked() ), this, SLOT( start() ) );		//pushbutton start
	connect ( ui.listWidget_operations, SIGNAL( itemClicked ( QListWidgetItem*) ), this, SLOT( checkCurrentItem() ) );
	connect ( ui.listWidget_operations, SIGNAL( itemChanged ( QListWidgetItem* ) ), this, SLOT( taskStateChanged() ) );

	// Load the default profile as soon as the app starts
	setCurrentProfile(defaultProfile);	// set the defaultProfile as current
	int create = createCurrentProfile();	// if it does not exist create it
	loadCurrentProfile();			// load the current (default) profile
	createProfileCombo();	// update the profile combobox with all existing profiles and set index to currentProfile
	InfoData.append("<p align=\"center\"><font color=magenta><b>" + tr("Welcome to","full phrase: 'Welcome to luckyBackup'") +" "
	+ appName + " :-)</font></b></p>");
	if (create == 0)
		InfoData.append(loadData);
	else
		InfoData.append(createData);
	ui.textBrowser_info -> setText(InfoData);	//update the info window

	//variable that helps with translations
	QString currentLanguageIs = tr("English","Do not translate this. Enter here the language of the translation in native characters instead. This is how your translation will appear under the 'language' menu.");
}

//===================================================================================================================================================
//------------ SLOTS --------------------------------------------------------------------------------------------------------------------------------
//===================================================================================================================================================

// exit =============================================================================================================================================
// Exits luckybackup. Warn if the profile used is not saved
void luckyBackupWindow::exitApp()
{
	if (isProfileSaved() == 0)	//if the current profile is not saved, ask the user if he/she wants to
		return;
	mainWindowWidth = this -> width();
	mainWindowHeight = this -> height();
	IsVisibleProfileComboToolbar = profileComboToolbar -> isVisible();
	IsVisibleProfileToolbar = profileToolbar -> isVisible();
	saveSettings();
	exit(0);	//quit
}

//do the same if the main window close button (or alt+F4) is pressed
 void luckyBackupWindow::closeEvent(QCloseEvent *event)
 {
	exitApp();
	event->ignore();
 }

// profileComboChanged ================================================================================================================================
// SLOT to be executed when the profileCombo index changes
void luckyBackupWindow::profileComboChanged()
{
	if (GoBack)		//if this function is called but shouldn't be executed.
	{
		GoBack = FALSE;
		return;
	}
	InfoData = "";

	if (isProfileSaved() == 0)	//if the current profile is not saved, ask the user if he/she wants to
		return;

	//set the current profile according to the combobox reading
	QString ComboReadFullPath = profileDir + (ui.comboBox_profile -> currentText()) + ".profile";
	setCurrentProfile(ComboReadFullPath);
	loadCurrentProfile();			// load the current profile
	InfoData.append(saveData + "<br>");
	InfoData.append(loadData);
	ui.textBrowser_info -> setText(InfoData);	//update the info window
	loadData=""; saveData=""; createData="";
}

// lockToolbars ================================================================================================================================
// SLOT to lock the toolbars in their position
void luckyBackupWindow::setToolbarAttrs()
{
	profileComboToolbar -> setFloatable(!actionLockToolbars->isChecked());
	profileComboToolbar -> setMovable(!actionLockToolbars->isChecked());
	profileToolbar -> setFloatable(!actionLockToolbars->isChecked());
	profileToolbar -> setMovable(!actionLockToolbars->isChecked());

	AreToolbarsLocked = actionLockToolbars->isChecked();

	IsVisibleToolbarText = actionVisibleToolbarText -> isChecked();
	if (IsVisibleToolbarText)
		profileToolbar -> setToolButtonStyle(Qt::ToolButtonTextUnderIcon);
	else
		profileToolbar -> setToolButtonStyle(Qt::ToolButtonIconOnly);
}

// savePressed ===============================================================================================================================
// function to execute when save pressed
void luckyBackupWindow::savePressed()
{
	InfoData = "";
	saveCurrentProfile();
	InfoData.append(saveData);
	ui.textBrowser_info -> setText(InfoData);	//update the info window
	saveData="";
}

// NewProfile =====================================================================================================================================
// Create an new, blank profile
void luckyBackupWindow::NewProfile()
{
	InfoData = "";
	if (isProfileSaved() == 0)	//if the current profile is not saved, ask the user if he/she wants to
		return;

	textDialog textdialog ("NewProfile", "", this);
	textdialog.exec();
	if (textdialog.getGoOn() == 0)
		return;
	if (textdialog.getGoOn() == 2)
	{
		InfoData = tr("Failed to create new profile","full phrase: 'Failed to create new profile <PROFILENAME>'")+" <b>" +
			profileName + "</b><br><font color=red>"
			+ tr("The name you have entered contains only invalid characters") +"</font>";
		ui.textBrowser_info -> setText(InfoData);	//update the info window
		return;
	}
	
	QString newProfileName = textdialog.getUserInput();
	QString newProfile = profileDir + newProfileName + ".profile";

	//check if it exists and ask to replace
	QFile newprofile;
	newprofile.setFileName(newProfile);

	if (newprofile.exists())			//if the newprofile already exists ask to replace
	{
		message = tr("Profile","full phrase: 'Profile <PROFILENAME> already exists'")+" <b>" + newProfileName + "</b> " +
			tr("already exists","full phrase: 'Profile <PROFILENAME> already exists'") + "<br>" +
			tr("Would you like to replace it ?");
		int r = QMessageBox::question(this, appName,message,QMessageBox::Yes | 
						QMessageBox::Default,QMessageBox::No | QMessageBox::Escape);
		if ( r == QMessageBox::Yes )		//if user answers yes
		{
			if (!newprofile.remove())	// delete the old profile or display a message if you can't
			{
				InfoData = "<font color=red><b>" + tr("ERROR") + "</font></b><br>" +
					tr("Failed to replace profile","full phrase: 'Failed to replace profile <PROFILENAME>'")+
					" <b>" + newProfileName + "</b><br><font color=red>"
					+ profile.errorString() +"</font>";
				ui.textBrowser_info -> setText(InfoData);	//update the info window
				return;
			}
		}
		else
			return;
	}

	setCurrentProfile(newProfile);
	createCurrentProfile();	// try to create it (also calls createProfileCombo)
	loadCurrentProfile();	// load the current profile
	InfoData.append(createData);
	ui.textBrowser_info -> setText(InfoData);	//update the info window
	loadData=""; saveData=""; createData="";
}

// renameCurrentProfile =============================================================================================================================
// rename the current profile
void luckyBackupWindow::renameCurrentProfile()
{
	InfoData="";

	textDialog textdialog ("RenameProfile", profileName, this);
	textdialog.exec();
	if (textdialog.getGoOn() == 0)
		return;

	if (textdialog.getGoOn() == 2)
	{
		InfoData = tr("Failed to rename profile","full phrase: 'Failed to rename profile <PROFILENAME>'")+
			" <b>" + profileName + "</b><br><font color=red>"
			+ "The name you have entered contains only invalid characters" +"</font>";
		ui.textBrowser_info -> setText(InfoData);	//update the info window
		return;
	}

	if (textdialog.getGoOn() == 3)		//if the new profile name is the same with the old one do nothing
	{
		InfoData.append(tr("profile","full phrase: 'profile <PROFILENAME> renamed successfully to <NEWPROFILENAME>'")
			+ " <b>" + profileName + "</b><br>...<font color=green>"
			+ tr("renamed successfully to","full phrase: 'profile <PROFILENAME> renamed successfully to <NEWPROFILENAME>'") +
			"</font> <b>" + profileName +"</b>");
		ui.textBrowser_info -> setText(InfoData);	//update the info window
		return;
	}

	QString newProfileName = textdialog.getUserInput();
	QString newProfile = profileDir + newProfileName + ".profile";

	//check if it exists and ask to replace
	QFile newprofile;
	newprofile.setFileName(newProfile);

	if (newprofile.exists())			//if the newprofile already exists ask to replace
	{
		message = tr("Profile","full phrase: 'Profile <PROFILENAME> already exists'")+" <b>" + newProfileName + "</b> " +
			tr("already exists","full phrase: 'Profile <PROFILENAME> already exists'") + "<br>" +
			tr("Would you like to replace it ?");
		int r = QMessageBox::question(this, appName,message,QMessageBox::Yes | 
						QMessageBox::Default,QMessageBox::No | QMessageBox::Escape);
		if ( r == QMessageBox::Yes )		//if user answers yes
		{
			if (!newprofile.remove())	// delete the old profile or display a message if you can't
			{
				InfoData = "<font color=red><b>" + tr("ERROR") + "</font></b><br>" +
					tr("Failed to replace profile",
					"full phrase: 'Failed to replace profile <PROFILENAME>'")+" <b>" + newProfileName + "</b><br><font color=red>"
					+ profile.errorString() +"</font>";
				ui.textBrowser_info -> setText(InfoData);	//update the info window
				return;
			}
		}
		else
			return;
	}

	if (!profile.rename(newProfile))	//try to rename the current profile
	{
		InfoData = "<font color=red><b>" + tr("ERROR") + "</font></b><br>" +
			tr("Failed to rename profile",
			"full phrase: 'Failed to rename profile <PROFILENAME>'")+" <b>" + profileName + "</b><br><font color=red>"
			+ profile.errorString() +"</font>";
		ui.textBrowser_info -> setText(InfoData);	//update the info window
		return;
	}
	if (currentProfile == defaultProfile)
	{
		InfoData.append(tr("default","full phrase: 'default profile <PROFILENAME> renamed successfully to <NEWPROFILENAME>'") +" ");
		defaultProfile = newProfile;
		saveSettings();
	}
	InfoData.append(tr("profile",
		"full phrase: 'profile <PROFILENAME> renamed successfully to <NEWPROFILENAME>'") + " <b>" + profileName + "</b><br>...<font color=green>"
		+ tr("renamed successfully to",
		"full phrase: 'profile <PROFILENAME> renamed successfully to <NEWPROFILENAME>'") + "</font> <b>" + newProfileName +"</b>");
	setCurrentProfile(newProfile);
	createProfileCombo();	// update the profile combobox with all existing profiles and set index to currentProfile
	loadCurrentProfile();	// load the current profile
	ui.textBrowser_info -> setText(InfoData);	//update the info window
	loadData=""; saveData=""; createData="";

}
// deleteProfile ======================================================================================================================================
// deletes the current profile
void luckyBackupWindow::deleteCurrentProfile()
{
	InfoData="";
	if (currentProfile == defaultProfile)
		message = tr("Are you sure you want to delete the default profile",
			"full phrase: 'Are you sure you want to delete the default profile: <PROFILENAME> ? (a new blank one, with the same name, will be created)'")+
			": <b>" + profileName + "</b> ?<br>(" +
			tr("a new blank one, with the same name, will be created",
			"full phrase: 'Are you sure you want to delete the default profile: <PROFILENAME> ? (a new blank one, with the same name, will be created)'") +")";
	else
		message = tr("Are you sure you want to delete profile",
		"full phrase: 'Are you sure you want to delete profile: <PROFILENAME> ?'")+": <b>" + profileName + "</b> ?";

	int r = QMessageBox::question(this, appName,message,QMessageBox::Yes | 
					QMessageBox::Default,QMessageBox::No | QMessageBox::Escape);
	if ( r == QMessageBox::Yes )		//if user answers yes
	{
		savedProfile = TRUE;	//assume the profile is saved
		//try to delete the profile
		if (!profile.remove())
		{
			InfoData.append("<font color=red><b>" + tr("ERROR") + "</font></b><br>" +
				 tr("Failed to delete profile",
				 "full phrase: 'Failed to delete profile <PROFILENAME>'")+": <b>" + profileName + "</b><br><font color=red>"
				+ profile.errorString() +"</font><br>");
			ui.textBrowser_info -> setText(InfoData);	//update the info window
			return;
		}

		InfoData.append(tr("Profile","full phrase: 'Profile <PROFILENAME> deleted successfully'") +
				": <b>" + profileName + "</b><br>...<font color=green>" +
				tr("deleted successfully","full phrase: 'Profile <PROFILENAME> deleted successfully'") + " !!</font><br>");

		// Load the default profile
		setCurrentProfile(defaultProfile);	// set the defaultProfile as current
		int create = createCurrentProfile();	// if it does not exist create it
		loadCurrentProfile();			// load the current (default) profile
		createProfileCombo();	// update the profile combobox with all existing profiles and set index to currentProfile

		if (create == 0)
			InfoData.append(loadData);
		else
			InfoData.append(createData);
		ui.textBrowser_info -> setText(InfoData);	//update the info window
		loadData=""; saveData=""; createData="";
	}
}


// setDefaultProfile ================================================================================================================================
// sets the current profile as default
void luckyBackupWindow::setDefaultProfile()
{
	InfoData = "";

	defaultProfile = currentProfile;
	saveSettings();
	InfoData.append(tr("Profile","full phrase: 'Profile <PROFILENAME> set as default'") + ": <b>" + profileName + "</b><br>...<font color=green>" + tr("set as default","full phrase: 'Profile <PROFILENAME> set as default'") + " !!</font><br>");
	ui.textBrowser_info -> setText(InfoData);	//update the info window
}

// exportProfile ================================================================================================================================
// exports the current profile to another location
void luckyBackupWindow::exportProfile()
{
	InfoData = "";

	// Display a save dialog
	QString exportProfileName = QFileDialog::getSaveFileName(this,
			appName + " - " + tr("Export Profile","this is a dialog title"), "/"+profileName+".profile",
			appName + " " + tr("profile")+" (*.profile)");
	if ( exportProfileName == "")	//if cancel is pressed at the file dialog, do nothing
		return;

	// put the ".profile" extension if its not there
	if  (!exportProfileName.endsWith(".profile", Qt::CaseInsensitive))
		exportProfileName.append(".profile");

	if (!saveProfile(exportProfileName))	//try to save/export the new profile
	{
		InfoData = "<font color=red><b>" + tr("ERROR") + "</font></b><br>" +
			tr("Failed to export profile","full phrase: 'Failed to export profile <PROFILENAME> as <FILENAME>'")+
			" <b>" + profileName + "</b><br>"
			+tr("as","full phrase: 'Failed to export profile <PROFILENAME> as <FILENAME>'")+
			" <b>" + exportProfileName +"</b><br><font color=red>"
			+ profile.errorString() +"</font>";
		ui.textBrowser_info -> setText(InfoData);	//update the info window
		setCurrentProfile(currentProfile);		//reset all currentProfile variables
		return;
	}

	InfoData.append(tr("Profile","full phrase: 'Profile <PROFILENAME> exported successfully as <FILENAME>'") + ": <b>" +
		profileName + "</b><br>...<font color=green>"
		+ tr("exported successfully as","full phrase: 'Profile <PROFILENAME> exported successfully as <FILENAME>'") +
		"</font> <b>" + exportProfileName +"</b>");
	ui.textBrowser_info -> setText(InfoData);	//update the info window
	setCurrentProfile(currentProfile);		//reset all currentProfile variables
}

// importProfile ================================================================================================================================
// imports a profile to the profile directory
void luckyBackupWindow::importProfile()
{
	InfoData = "";
	QString importProfile =  QFileDialog::getOpenFileName(this,
			appName + " - " + tr("Import Profile","this is a dialog title"), "/",
			appName + " " + tr("profile")+" (*.profile)");

	if ( importProfile == "")	//if cancel is pressed at the file dialog, do nothing
		return;
	QFile importprofile;
	importprofile.setFileName(importProfile);

	QString newProfileName = importProfile;
	newProfileName = newProfileName.right(newProfileName.size() - newProfileName.lastIndexOf("/") - 1);
	newProfileName.chop(8);

	QString newProfile = profileDir + newProfileName + ".profile";
	QFile newprofile;
	newprofile.setFileName(newProfile);

	while (newprofile.exists())			//if the newprofile already exists ask to rename or cancel
	{
		textDialog textdialog ("ImportProfile", newProfileName,this);
		textdialog.exec();
		newProfileName = textdialog.getUserInput();

		if (textdialog.getGoOn() == 0)
			return;
		if (textdialog.getGoOn() == 2)
		{
			InfoData = tr("Failed to import profile","full phrase: 'Failed to import profile <PROFILENAME>'")+
				" <b>" + importProfile + "</b><br><font color=red>"
				+ tr("The name you have entered contains only invalid characters") +"</font>";
			ui.textBrowser_info -> setText(InfoData);	//update the info window
			return;
		}
		newProfile = profileDir + newProfileName + ".profile";
		newprofile.setFileName(newProfile);
	}

	if (importprofile.copy(newProfile))
	{
		createProfileCombo();
		InfoData.append(tr("Profile",
			"full phrase: 'Profile <PROFILENAME> imported successfully as <NEWPROFILENAME> and added at the profile list'") +
			": <b>" + importProfile + "</b><br>...<font color=green>"
			+ tr("imported successfully",
			"full phrase: 'Profile <PROFILENAME> imported successfully as <NEWPROFILENAME> and added at the profile list'") +
			"</font> " + tr("as","full phrase: 'Profile <PROFILENAME> imported successfully as <NEWPROFILENAME> and added at the profile list'")
			 + " <b>" + newProfileName + "</b><br><font color=green>"
			+ tr("and added at the profile list",
			"full phrase: 'Profile <PROFILENAME> imported successfully as <NEWPROFILENAME> and added at the profile list'") + "</font>");
	}
	else
		InfoData = tr("Failed to import profile",
			"full phrase: 'Failed to import profile <PROFILENAME>'")+" <b>" + importProfile + "</b><br><font color=red>"
			+ importprofile.errorString() +"</font>";
	ui.textBrowser_info -> setText(InfoData);	//update the info window
}

// setLanguage =====================================================================================================================================
// changes the language of the application
void luckyBackupWindow::setLanguage(QAction *action)
{
	QString currentLocale = action -> data().toString();
	transDir.setPath(relativeTransDir);
	if (transDir.exists())
		appTranslator.load(QString("luckybackup_") + currentLocale, relativeTransDir);
	else 
		appTranslator.load(QString("luckybackup_") + currentLocale, systemTransDir);
	defaultLanguage = currentLocale;
	retranslateUi();
}

// help =============================================================================================================================================
// Shows the "help" information window
void luckyBackupWindow::help()
{
	if (helpURL.toString() == "Does_not_exist")
	{
		InfoData = "<font color=red><b>" + tr("ERROR") + "</font></b><br>" +
			tr("Could not locate the file","full phrase: 'Could not locate the file <FILENAME>'")+": <b>" + relativeManual +"</b><br>	:-(";
		ui.textBrowser_info -> setText(InfoData);	//update the info window
		return;
	}
	helpBrowser helpbrowser (helpURL);
	helpbrowser.exec();
}

// about ============================================================================================================================================
// Shows the "about" information window
void luckyBackupWindow::about()
{
	aboutDialog aboutdialog(licenseURL);
	aboutdialog.exec();
}

//  add =============================================================================================================================================
// Creates a new operation by calling the modifyDialog
void luckyBackupWindow::add()
{
	modifyOK = FALSE;
	modifyConnected = FALSE;
	TotalOperations = ui.listWidget_operations -> count();				//Get the Operations list size
	currentOperation = TotalOperations;

	if ( (TotalOperations) >= (maxOperations-1) )					//if the maxOperations limit-1 is reached 
	{
		QMessageBox::warning(this, appName,
				tr("You have reached the maximum limit of supported tasks")+" <br><br>" +
				tr("You have to remove a task if you wish to add another !!"),
				QMessageBox::Ok | QMessageBox::Default);
		return;
	}

	//open the modify dialog and create "Operation array" item at position currentOperation
 	modifyDialog modifydialog(currentOperation); 
	modifydialog.exec();

	if (modifyOK)		//if all went ok - add new operation
	{
		//add a new item in the operations list
		ui.listWidget_operations -> addItem( Operation[currentOperation] -> GetName() );
		ui.listWidget_operations -> setCurrentRow(currentOperation);
		ui.listWidget_operations -> currentItem() -> setCheckState(Qt::Unchecked);
		checkCurrentItem();
		TotalOperations = ui.listWidget_operations -> count();		//set the TotalOperations to what it is now
		savedProfile = FALSE;			//change profile status to "unsaved"
	}

	//If also create Restore operation is checked - Create a new operation and modify it
	if ( (modifyOK) && (Operation[currentOperation] -> GetAlsoCreateRestore()) && (!Operation[currentOperation] -> GetTypeSync()) )
		AlsoCreateRestore();
}

//  modify ==========================================================================================================================================
// Modifies an existing operation by calling the modifyDialog
void luckyBackupWindow::modify()
{
	modifyOK = FALSE;
	currentOperation = ui.listWidget_operations -> currentRow();		//current operations list row number
	if (currentOperation < 0)						//if nothing is selected do nothing
	{
		InfoData = tr("Please select a task to modify first") + " !!";	
		ui.textBrowser_info -> setText(InfoData);
		return;
	}

	QString tempConnect = Operation[currentOperation] -> GetConnectRestore();
	if ( (tempConnect != "") && (!modifyConnected) )	//the operations is connected and is a not new RESTORE operation
		modifyConnected = TRUE;
	int connectPosition=0;
	if (tempConnect != "")	//if this op is already connected with another op, pop up a message & hold the connection's position
	{
		count=0; while (count<TotalOperations){					//get the connected operation position at connectPosition
			if  (Operation[count]->GetName()==tempConnect) connectPosition=count;
			count++;}

		int r=QMessageBox::warning(this, appName,tr("You have asked to modify a task that is connected to :",
			"full phrase: 'You have asked to modify a task that is connected to :<TASKNAME> which is not going to be modified'")+"<br>"+
			tempConnect + "<br>"+tr("which is not going to be modified",
			"full phrase: 'You have asked to modify a task that is connected to :<TASKNAME> which is not going to be modified'")+
			"<br><br>"+tr("Press 'Yes' to proceed"),
			QMessageBox::Yes | QMessageBox::Default,
			QMessageBox::Cancel | QMessageBox::Escape);
		if ( r == QMessageBox::Cancel ) return;
	}

	//open the modify dialog and modify "Operation array" item at position currentOperation
 	modifyDialog modifydialog(currentOperation); 
	modifydialog.exec();

	if (modifyOK)		//if all went ok - modify existing operation
	{
		Qt::CheckState isIncluded = ui.listWidget_operations -> currentItem() -> checkState(); //get the include checkbox state
		ui.listWidget_operations -> takeItem(currentOperation);	//take out the selected item from the list
									//insert a new one at the same position with the new name
		ui.listWidget_operations -> insertItem(currentOperation, Operation[currentOperation] -> GetName());	
		ui.listWidget_operations -> setCurrentRow(currentOperation);		//highlight this row
		ui.listWidget_operations -> currentItem() -> setCheckState(isIncluded); //set the include checkbox as it was
		Operation[currentOperation] -> SetConnectRestore(tempConnect);		//set the connection to what it was
		if (tempConnect != "") Operation[connectPosition]->SetConnectRestore(Operation[currentOperation] -> GetName());
		checkCurrentItem();
		savedProfile = FALSE;			//change profile status to "unsaved"
	}

	//If also create Restore operation is checked - Create a new operation and modify it
	if ( (modifyOK) && (!modifyConnected) && (Operation[currentOperation] -> GetAlsoCreateRestore()) 
		&& (!Operation[currentOperation] -> GetTypeSync()) )
		AlsoCreateRestore();
}

// remove ===========================================================================================================================================
// Removes highlighted operation from available list
void luckyBackupWindow::remove()
{
	currentOperation = ui.listWidget_operations -> currentRow();			//current Included list row number
	if (currentOperation < 0)							//if nothing is selected do nothing
	{
		InfoData = tr("Please select a task to remove first") + " !!";	
		ui.textBrowser_info -> setText(InfoData);
		return;
	}

	QString tempConnect = Operation[currentOperation] -> GetConnectRestore();

	int r = QMessageBox::warning(this, appName,tr("Are you sure you want to remove the following task ??")+" <br>" +
 				(ui.listWidget_operations -> currentItem()) -> text(),
				QMessageBox::Yes | QMessageBox::Default,
				QMessageBox::Cancel | QMessageBox::Escape);
	int r2=0;
	if ( r == QMessageBox::Yes )
	{
		if (tempConnect != "")	//if this op is already connected with another op ask to remove that too
			r2 = QMessageBox::warning(this, appName,tr("Would you like to also remove:")+" <br>" + tempConnect+"<br>??",
				QMessageBox::Yes | QMessageBox::Default,
				QMessageBox::Cancel | QMessageBox::Escape);
		//remove current operation
		ui.listWidget_operations -> takeItem(currentOperation);		//take out the selected item from the list
		TotalOperations = ui.listWidget_operations -> count();		//TotalOperations takes the new value (old - 1)
		while (currentOperation < TotalOperations)			//shift array "Operation" one position up starting at current row
		{
			Operation[currentOperation]=Operation[currentOperation+1]; //make the current array item the same with the next one
			currentOperation ++;
		}

		if (tempConnect != "")
		{
			currentOperation=0;
			while (currentOperation < TotalOperations)
			{	if (Operation[currentOperation]->GetName()==tempConnect)
				{
					if (r2==QMessageBox::Yes)		//remove current (conected) operation
					{
						ui.listWidget_operations -> takeItem(currentOperation);	
						TotalOperations = ui.listWidget_operations -> count();
						while (currentOperation < TotalOperations)
						{
							Operation[currentOperation]=Operation[currentOperation+1];
							currentOperation ++;
						}
					}
					else				//just remove the connection
						Operation[currentOperation]->SetConnectRestore("");
					break;
				}
				currentOperation ++;
			}
		}
		savedProfile = FALSE;			//change profile status to "unsaved"
	}
}

//  manage ==========================================================================================================================================
// Manage existing backups of an existing operation by calling the manageDialog
void luckyBackupWindow::manage()
{
	currentOperation = ui.listWidget_operations -> currentRow();			//current Included list row number
	if (currentOperation < 0)							//if nothing is selected do nothing
	{
		InfoData = tr("Please select a task to manage first") + " !!";	
		ui.textBrowser_info -> setText(InfoData);
		return;
	}

 	manageDialog managedialog;
	managedialog.exec();
}

// moveUp ===========================================================================================================================================
// Moves highlighted operation one position up in the available list
void luckyBackupWindow::moveUp()
{
	currentOperation = ui.listWidget_operations -> currentRow();		//current list row number
	TotalOperations = ui.listWidget_operations -> count();			//total number of list items
	if ( currentOperation < 0 )						//if nothing is selected
		return;

	if ( currentOperation > 0 )
	{
		operation *tempOp;	//temp operation to use for swaping
		tempOp = Operation[currentOperation];		//swap array items
		Operation[currentOperation] = Operation [currentOperation-1];
		Operation[currentOperation-1] = tempOp;

		QListWidgetItem * currentItem = ui.listWidget_operations -> takeItem(currentOperation);	//swap list items
		ui.listWidget_operations -> insertItem(currentOperation - 1, currentItem);
		ui.listWidget_operations -> setCurrentRow(currentOperation - 1);
		
		savedProfile = FALSE;			//change profile status to "unsaved"
	}
}

// moveDown =========================================================================================================================================
// Moves highlighted operation one position down in the available list
void luckyBackupWindow::moveDown()
{
	currentOperation = ui.listWidget_operations -> currentRow();		//current list row number
	TotalOperations = ui.listWidget_operations -> count();			//total number of list items
	if ( currentOperation < 0 )						//if nothing is selected
		return;

	if (currentOperation < (TotalOperations-1))
	{
		operation *tempOp;						//temp operation to use for swaping
		tempOp = Operation[currentOperation];				//swap array
		Operation[currentOperation] = Operation [currentOperation+1];
		Operation[currentOperation+1] = tempOp;

		QListWidgetItem * currentItem = ui.listWidget_operations -> takeItem(currentOperation);	//swap list items
		ui.listWidget_operations -> insertItem(currentOperation + 1, currentItem);
		ui.listWidget_operations -> setCurrentRow(currentOperation + 1);

		savedProfile = FALSE;			//change profile status to "unsaved"
	}
}


// start ============================================================================================================================================
//start the execution of operations
void luckyBackupWindow::start()
{
	TotalOperations = ui.listWidget_operations -> count();		//set TotalOperations to what it is now

	if (!checkOperationList())	//check the Operations list for "errors" (eg nothing is included or the list is empty)
		return;

	checkDeclared();		//Check if the declared data are ok by calling checkBackupDirs, checkSyncDirs accordingly

	if (NothingToDo == TRUE)	//if there is nothing to Do anyway then just display a message
	{
		InfoData = "<font color=red><b>" + tr("ERROR") + "</b></font><br>" +
				"~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~<br><b>" +
				tr("You have only incuded tasks that are going to be skipped") + "</b><br>" +
				"..." + tr("nothing to do") +" !!";
		
		ui.textBrowser_info -> setText(InfoData);
		return;
	}

	DryRun = ui.checkBox_DryRun -> isChecked();	//set bool Dryrun according to checkbox
	rsyncDialog rsyncdialog(this);	//rsync progress dialog
	rsyncdialog.exec();		//show the progress dialog
	InfoData = QDateTime::currentDateTime().toString(Qt::DefaultLocaleLongDate) + "<br>" +
		tr("Execution of profile","full phrase: 'Execution of profile <PROFILENAME> finished'") +
		" <b>" + profileName + "</b> " + tr("finished","full phrase: 'Execution of profile <PROFILENAME> finished'")+" !!<br>";
	if (DryRun)
		InfoData.append("(" + tr("simulation mode") +")");
	else
	{
		//save the profile to update last execution times
		if (!saveProfile(currentProfile))
		{
			InfoData.append(tr("Could not update last execution time of tasks") + "<br>" + profile.errorString());
			savedProfile = FALSE;
		}
		else
		{
			InfoData.append(tr("Last execution time of tasks updated"));
			savedProfile = TRUE;			//change profile status to "saved"
		}
	}
	ui.textBrowser_info -> setText(InfoData);
}

// schedule==========================================================================================================================================
// Displays a new dialog to make a cron job for included operations
void luckyBackupWindow::schedule()
{
	//open the modify dialog and create "Operation array" item at position currentOperation
 	scheduleDialog scheduledialog; 
	scheduledialog.exec();
}

// taskStateChanged =================================================================================================================================
// Sets the profile state to unsaved if a task checkbox is changed
void luckyBackupWindow::taskStateChanged()
{
	//theory (due to connections at luckybackwindow constructor):
	//when an item is just clicked functions called: checkCurrentItem -> taskStateChanged
	//when a checkbox state chenges functiond called: taskStateChanged -> checkCurrentItem -> taskStateChanged
	if (!taskChanged)
	{
		if (taskClicked)
			taskClicked = FALSE;
		else
			taskChanged = TRUE;
		return;
	}

	if ( (taskChanged) && (taskClicked) )
		savedProfile = FALSE;			//change profile status to "unsaved"

	taskClicked = FALSE;
	taskChanged = FALSE;
}

// checkCurrentItem =================================================================================================================================
// Checks the currently selected operation (if checked) for validity
void luckyBackupWindow::checkCurrentItem()
{
	taskClicked = TRUE;	//this is used at taskStateChanged() to determine if a task state is changed

	InfoData="";
	CheckedData = "";
	currentOperation = ui.listWidget_operations -> currentRow();

	//first set  variables source & dest
	QString source, dest;
	source = Operation[currentOperation] -> GetSource();
	dest = Operation[currentOperation] -> GetDestination();

	if (ui.listWidget_operations -> currentItem() -> checkState() == Qt::Unchecked)	//if the operation is " NOT included" do not check it
	{
		ui.listWidget_operations -> currentItem() -> setIcon(QIcon(":/luckyPrefix/about.png"));
		CheckedData.append(tr("status") + ": <font color=blue><b>"+tr("NOT INCLUDED")+"</font></b><br>");
		CheckedData.append(tr("Checks are disabled. This task will be <b>skipped</b>"));
		ui.textBrowser_info -> setText(CheckedData);
		Operation[currentOperation] -> SetIncluded(FALSE);	// also set its include state to FALSE
		return;
	}

	Operation[currentOperation] -> SetIncluded(TRUE);	// also set its include state to TRUE

	if ( (Operation[currentOperation] -> GetTypeDirContents()) || (Operation[currentOperation] -> GetTypeDirName()) )
		checkBackupDirs(source,dest);			//if the operation is of type "backup dir ...'
	if (Operation[currentOperation] -> GetTypeSync())	//if the operation is of type "sync dirs'
		checkSyncDirs(source,dest);

	if (Operation[currentOperation] -> GetOK())
		ui.listWidget_operations -> currentItem() -> setIcon(QIcon(":/luckyPrefix/okay.png"));
	if (Operation[currentOperation] -> GetWARNING())
		ui.listWidget_operations -> currentItem() -> setIcon(QIcon(":/luckyPrefix/cancel.png"));
	if (Operation[currentOperation] -> GetCRITICAL())
		ui.listWidget_operations -> currentItem() -> setIcon(QIcon(":/luckyPrefix/warning.png"));
	if (Operation[currentOperation] -> GetSourcePerms())
		ui.listWidget_operations -> currentItem() -> setIcon(QIcon(":/luckyPrefix/cancel.png"));
	if (Operation[currentOperation] -> GetDestPerms())
		ui.listWidget_operations -> currentItem() -> setIcon(QIcon(":/luckyPrefix/cancel.png"));
	ui.textBrowser_info -> setText(CheckedData);
}

// refreshList ======================================================================================================================================
// refreshes the task list and the profile combobox when button refresh clicked
void luckyBackupWindow::refreshList()
{
	TotalOperations = ui.listWidget_operations -> count();	//Get the Operations list size
	checkDeclared();					// Check tasks & set icons
	createProfileCombo();	// update the profile combobox with all existing profiles and set index to currentProfile
	InfoData = tr("Task list of profile","full phrase: 'Task list of profile <PROFILENAME> and 'profile' drop-down list refreshed'") +
			" <b>" + profileName + "</b><br>" +
			tr("and 'profile' drop-down list","full phrase: 'Task list of profile <PROFILENAME> and 'profile' drop-down list refreshed'") +
			"<br>...<font color=blue>" + tr("refreshed",
			"full phrase: 'Task list of profile <PROFILENAME> and 'profile' drop-down list refreshed'") + " !!</font>";
	ui.textBrowser_info -> setText(InfoData);
}

//===================================================================================================================================================
//end of luckybackupwindow.cpp
