'''
Defines the base class for all L{AEOutput} devices.

@author: Larry Weiss
@author: Peter Parente
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

class AEOutput(object):
  '''
  Defines the base class for all output devices. All output devices used by
  LSR should derive from this class and should implement the methods defined
  here.

  All methods defined here raise L{NotImplementedError} to ensure that
  derived classes create appropriate implementions.
  
  @cvar USE_THREAD: Should this device use a separate thread to queue output?
    Devices that block while doing output (e.g. serial Braille device) should 
    use a thread to keep the main thread unlocked and responsive. Defaults to 
    None so a subclass must override it with an explicit boolean value.
  @type USE_THREAD: boolean
  @cvar COMMAND_CHARS: String of characters that are treated as commands on the
    device implementing this interface. These characters are replaced with
    blanks in any string sent to the device.
  @type COMMAND_CHARS: string
  @ivar listeners: List of callables that should be notified when a marker 
    inserted with L{sendIndex} is encountered
  @type listeners: list
  '''
  USE_THREAD = None
  COMMAND_CHARS = ''
  
  def __init__(self):
    '''
    Initializes the empty listeners list.
    '''
    self.listeners = []
 
  def init(self):
    '''
    Called after the instance is created to initialize the device.

    If called when already initialized, this will restore the device to it's
    initialized state. May also be called to re-initialize the device after
    a call to L{close}.

    @raise NotImplementedError: When not overridden in a subclass
    @raise Error.InitError: When a communication or state problem exists for 
      the specific device
    '''
    raise NotImplementedError
  
  def createDistinctStyles(self, num_styles):
    '''
    Creates up to the given number of styles for this device. The device should 
    create L{AEOutput.Style}s having properties that reflect the capabilities of
    this device, leaving any unsupported fields set to the value of None. The
    device should also try to generate reasonable, maximally distinct styles. If
    the device cannot honor the request for all the styles it is asked to 
    generate, it may reuse styles it has already created. The device is B{not}
    expected to create distinct styles across invocations of this methods.
    
    @param num_styles: Number of styles that the requestor would like the
      device to create
    @type num_styles: integer
    @return: New styles
    @rtype: list of L{AEOutput.Style}
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError
  
  def getDefaultStyle(self):
    '''
    Gets the default style of the device.
    
    @return: Default style of the device
    @rtype: L{AEOutput.Style}
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError

  def close(self):
    '''
    Closes an initialized output device.

    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError
  
  def getProxy(self):
    '''
    Gets the object that the L{DeviceManager} will use to communicate with this
    device. The returned object may either be a proxy (e.g. a thread) or this
    device itself.
    
    @return: An output object that implements this class
    @rtype: L{AEOutput}
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError

  def getName(self):
    '''
    Gives the user displayable (localized) name for this output device.
    Relevant version and device status should be included.

    @return: The localized name for the device
    @rtype: string
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError

  def sendString(self, text, style):
    '''
    Sends a string of one or more characters to the device.

    Depending on the device and it's features, the character may first be
    buffered until either "enough" characters and commands have been
    received, or L{sendTalk} has been called. To guard against sending
    unintended commands, the class implementing this method should remove all
    L{COMMAND_CHARS} from the text.

    @param text: Text to send to the device
    @type text: string
    @param style: Style with which this text should be output
    @type style: L{AEOutput.Style}
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError

  def sendStop(self, style=None):
    '''
    Purges buffered text and styles, and interrupts current output.

    @param style: Style indicating which channel on which the stop should be 
      performed; None indicates stop on all channels
    @type style: L{AEOutput.Style}
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError

  def sendTalk(self, style=None):
    '''
    Indicates all text buffered by L{sendString} should now be output. 
    For devices that do the buffering in the driver, this action may mean simply 
    sending the command. For devices that do not buffer, this action means 
    sending text and styles buffered in the LSR device definition.

    @param style: Style indicating which channel on which the talk should be 
      performed; None indicates talk on all channels
    @type style: L{AEOutput.Style}
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError
    
  def sendIndex(self, marker, style):
    '''
    Sends an index marker to the device driver. The driver should notify the
    device when the marker has been reached during output. 

    @param style: Style indicating which channel on which the marker should be 
      inserted
    @type style: L{AEOutput.Style}
    @param marker: The value that should be returned when output has been 
      processed up to this point
    @type marker: integer
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError

  def sendStringSync(self, text, style=None, stop=True):
    '''
    Sends a complete string to a device followed by a talk command to output it
    immediately. Blocks until the text has finished being spoken.
    
    Do B{not} use this method in place of L{sendString} for normal device 
    operation. It is intended as a method for doing device output pre-runtime
    (e.g. when debugging).
    
    @param text: String to send to the device
    @type text: string
    @param style: Style on which this string should be output; None implies some
      reasonable default should be used
    @type style: L{AEOutput.Style}
    @param stop: Stop current output before doing this output?
    @type stop: boolean
    @raise NotImplementedError: When not overridden in a subclass
    '''
    raise NotImplementedError
    
  def addIndexListener(self, listener):
    '''
    Adds a listener that should be notified when speech has progressed to the 
    point where a marker was inserted with L{sendIndex}.
    
    @param listener: The method that should be called when markers received.
    @type listener: callable
    '''
    self.listeners.append(listener)

  def removeIndexListener(self, listener):
    '''
    Removes the specified listener.
    
    @param listener: The method that should no longer be called when markers 
      received.
    @type listener: callable
    @raise ValueError: When the given listener is not already registered
    '''
    self.listeners.remove(listener)
  
  def isActive(self):
    '''
    Indicates whether the device is active (giving output) or not.

    @return: True when content is buffered or the device is outputing
    @rtype: boolean
    @raise NotImplementedError: When not overriden in a subclass
    '''
    raise NotImplementedError
