/*
 *  Linux snipes, a text-based maze-oriented game for linux.
 *  Copyright (C) 1997 Jeremy Boulton.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Jeremy Boulton is reachable via electronic mail at
 *  boultonj@ugcs.caltech.edu.
 */


#include <stdlib.h>
#include <math.h>
#include <curses.h>
#include <ctype.h>
#include <unistd.h>
#include <signal.h>
#include <time.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/wait.h>

#include "snipes.h"
#include "player.h"
#include "enemies.h"
#include "weapons.h"
#include "hive.h"
#include "walls.h"
#include "collision.h"
#include "chars.h"
#include "kbd.h"
#include "sound.h"
#include "screen.h"

#ifndef VERSION
#define VERSION	"unknown"
#endif

/* Times are in microseconds (1e-6) */

#define	DISPLAY_TIME	20000

/* Counts are multiples of DISPLAY_TIME */

#define	MOVE_COUNT		3
#define	EYEFLASH_COUNT		30

/* Size of maze blocks in characters */

//#ifdef DOUBLEWIDE_FONT_HACK
//#define	BLKSZX			8
//#define	BLKSZY			7
//#else

#define	BLKSZX			12
#define	BLKSZY			6

/* Number of maze blocks in the horizontal and vertical directions */

#define BLOCKNUM_X		20
#define BLOCKNUM_Y		20

/* Scores */

#define DEAD_HIVE_SCORE		500
#define DEAD_ENEMY_SCORE	10


char gpl[6][80]= {
  "Linux snipes version %s, Copyright (C) 1997 Jeremy Boulton.\n",
  "Linux snipes comes with ABSOLUTELY NO WARRANTY; for details see\n",
  "the file named \"COPYING\", which you should have received with\n",
  "this program.\n",
  "This is free software, and you are welcome to redistribute it\n",
  "under certain conditions; see the file \"COPYING\" for details.\n",
};

#define	FEATURE_INDEX_DEADLY_WALLS		0
#define	FEATURE_INDEX_DESTRUCTIBLE_WALLS	1
#define	FEATURE_INDEX_ENABLE_GHOSTS		2
#define	FEATURE_INDEX_ENABLE_BULLET_BOUNCE	3

#ifdef ORIGINAL_DOS_VERSION
const char features[4][27] = {
/* ABCDEFGHIJKLMNOPQRSTUVWXYZ */
  "NNNNNNNNNNNNYYYYYYYYYYYYYY",		/* walls deadly */
  "NNNNNNYYYYYYNNNNYYYYYYYYYY",		/* walls can be destroyed */
  "NNNYYYNNNYYYNNYYNNYYYYYYYY",		/* ghosts */
  "NNNNNNNNNNNNNNNNYYYYYYYYYY",		/* bouncy shots? */
};
#else
const char features[4][27] = {
/* ABCDEFGHIJKLMNOPQRSTUVWXYZ */
  "NNNNYYYYNNNNYYYYNNNNYYYYYY",		/* walls deadly */
  "NNNNNNNNYYYYYYYYNNNNNNNNYY",		/* walls can be destroyed */
  "NYNYNYNYNYNYNYNYNYNYNYNYYY",		/* ghosts */
  "NNYYNNYYNNYYNNYYNNYYNNYYYY",		/* bouncy shots? */
};
#endif

const int max_enemies_at_level[9] = { 10, 25, 40, 55, 70, 90, 110, 130, 150 };
const int max_hives_at_level[9]   = { 3, 3, 3, 3, 4, 4, 4, 4, 5 };

/* Screen dimensions */
int maxx, maxy;

/* Mode information */
int force_non_raw_kbd;
int use_sound;
int jennymode;

enum DisplayType displaytype;

/* Time information */
time_t game_start_time;
int elapsed_time;

/* Level information */
char level_alpha;
int level_num;


/* ----------------------------------------------------------------- */
/*                  MISCELLANEOUS UTILITY FUNCTIONS                  */
/* ----------------------------------------------------------------- */

/* check_feature( int feature_index )
 * 
 * Returns true if the feature is enabled for this level, otherwise
 * false.
 */

int check_feature( int feature_index )
{
  return (features[feature_index][level_alpha-'A']=='Y')?1:0;
}


/* usage( char *programname )
 * 
 * Display usage message.
 */

void usage( char *programname )
{
  fprintf( stderr, "Usage: %s [OPTION]... [level]\n", programname );
#ifdef DOUBLEWIDE_FONT_HACK
  fprintf( stderr, "\t-d: Use doublewide font hack for display\n" );
#endif
#ifdef USE_XWIN
  fprintf( stderr, "\t-x: Use X for display\n" );
#endif
#ifdef USE_SVGALIB
  fprintf( stderr, "\t-s: Use SVGALIB for display\n" );
#endif
#ifdef USE_CURSES
  fprintf( stderr, "\t-c: Use CURSES for display\n" );
#endif
  fprintf( stderr, "\n" );
  fprintf( stderr, "\t-j: Use Jenny scrolling mode\n" );
  fprintf( stderr, "\t-k: Force keyboard to safe (\"cooked\") mode.\n" );
  fprintf( stderr, "\t-q: Quiet mode (no sound)\n" );
  fprintf( stderr, "\t-v: Display version information." );
  fprintf( stderr, "\n" );
  fprintf( stderr, "\tlevel is of the form [A-Z][1-9] (e.g. A1)\n" );
  exit(1);
}


/* err_sys( char *str )
 * 
 * Abort the program with an error message.
 */

void err_sys( char *str )
{
  screen_clear();
  screen_refresh();
  screen_close();
  close_kbd();
  fprintf( stderr, "LinuxSnipes: ERROR: %s\n\n", str );
  exit(1);
}


/* do_level( char *str )
 * 
 * Parse the level string from the command line.  If the string
 * is correctly formatted, sets the level variables and returns
 * true, else returns false.
 */

int do_level( char *str )
{
  if( strlen( str ) != 2 )
    return 0;
  
  if( toupper( str[0] ) >= 'A' && toupper( str[0] ) <= 'Z' &&
      str[1] >= '1' && str[1] <= '9' ) {
    level_alpha = toupper( str[0] );
    level_num   = str[1] - '0';
    return 1;
  }
  else
    return 0;
}


/* sig_segv( int signo )
 * 
 * Signal handler for SEGV and other signals so that the keyboard
 * can be restored to cooked mode.
 */

void sig_segv( int signo )
{
  emergency_close_kbd();
  abort();
}


void install_signal_handlers( void )
{
  if( signal( SIGSEGV, sig_segv ) == SIG_ERR )
    err_sys( "can't catch SIGSEGV" );
  if( signal( SIGBUS,  sig_segv ) == SIG_ERR )
    err_sys( "can't catch SIGBUS" );
  if( signal( SIGFPE,  sig_segv ) == SIG_ERR )
    err_sys( "can't catch SIGFPE" );
  if( signal( SIGILL,  sig_segv ) == SIG_ERR )
    err_sys( "can't catch SIGILL" );
  if( signal( SIGQUIT,  sig_segv ) == SIG_ERR )
    err_sys( "can't catch SIGQUIT" );
}


void remove_signal_handlers( void )
{
  signal( SIGSEGV, SIG_DFL );
  signal( SIGBUS,  SIG_DFL );
  signal( SIGFPE,  SIG_DFL );
  signal( SIGILL,  SIG_DFL );
}


/* ----------------------------------------------------------------- */
/*                       COORDINATE FUNCTIONS                        */
/* ----------------------------------------------------------------- */

void init_maze_coordinates( screen_coords *scr_coords )
{
  scr_coords->square_count_x = BLOCKNUM_X;
  scr_coords->square_count_y = BLOCKNUM_Y;
  scr_coords->square_width   = BLKSZX;
  scr_coords->square_height  = BLKSZY;

  scr_coords->max_x=BLOCKNUM_X*(BLKSZX-1)-1;
  scr_coords->max_y=BLOCKNUM_Y*(BLKSZY-1)-1;
  
  scr_coords->screen_x1 = 1;
  scr_coords->screen_x2 = maxx - 2;
  scr_coords->screen_y1 = 4;
  scr_coords->screen_y2 = maxy - 2;

  scr_coords->screen_max_col = maxx - 3;
  scr_coords->screen_max_row = maxy - 6;

  scr_coords->player_pos_x = (maxx - 3)/2 - 1;
  scr_coords->player_pos_y = (maxy - 6)/2 - 1;
}


void recalculate_maze_coordinates( int my_player, screen_coords *sc,
				   wall_info *w )
{
  coordinate tc, td;
  
  tc = get_player_pos( my_player );
    
  if( jennymode == 1 ) {
    coords_to_screen_pos( sc, tc, &td );
    td.x -= sc->screen_x1;
    td.y -= sc->screen_y1;

    if( ( td.x < sc->screen_max_col/6 || td.x > (5*sc->screen_max_col)/6 ) ||
	( td.y < sc->screen_max_row/6 || td.y > (5*sc->screen_max_row)/6 ) ) {

      /* Re-center screen on player. */
      sc->maze_x1 = normalize_x_coord( w, tc.x - sc->player_pos_x );
      sc->maze_y1 = normalize_y_coord( w, tc.y - sc->player_pos_y );
    }
  }
  else {
    /* Re-center screen on player. */
    sc->maze_x1 = normalize_x_coord( w, tc.x - sc->player_pos_x );
    sc->maze_y1 = normalize_y_coord( w, tc.y - sc->player_pos_y );
  }

  sc->maze_x2 = normalize_x_coord( w, sc->maze_x1+sc->screen_max_col );
  sc->maze_y2 = normalize_y_coord( w, sc->maze_y1+sc->screen_max_row );
}

/* ----------------------------------------------------------------- */
/*                         DISPLAY FUNCTIONS                         */
/* ----------------------------------------------------------------- */

void display_game_header( int players_left )
{
#if 0
  time_t elapsed_time = time(NULL) - game_start_time;
#endif
  int game_score = DEAD_HIVE_SCORE * get_num_dead_hives() +
    DEAD_ENEMY_SCORE * get_num_dead_enemies(0) +
    DEAD_ENEMY_SCORE * get_num_dead_enemies(1);
  char tmp_str[100];

  screen_setcolorpair(3);

  screen_move(0,0);
  screen_addch( HIVE_TOPLEFT );
  screen_addch( HIVE_TOPRIGHT );
  sprintf( tmp_str, "  dead = %3d ", get_num_dead_hives() );
  screen_addstr( tmp_str );
  screen_addch( VERTICAL_BAR );
  screen_addstr("   ");
  screen_addch( EVILFACE );
  screen_addch( WEAPON_RIGHT );
  sprintf( tmp_str, " dead = %5d ", get_num_dead_enemies(0) );
  screen_addstr( tmp_str );
  screen_addch( VERTICAL_BAR );
  screen_addstr("   ");
  screen_addch( GHOSTFACE );
  sprintf( tmp_str, " dead = %5d ", get_num_dead_enemies(1) );
  screen_addstr( tmp_str );
  screen_addch( VERTICAL_BAR );
  sprintf( tmp_str, "   Skill Level  =     %c%d", level_alpha, level_num );
  screen_addstr( tmp_str );

  screen_move(1,0);
  screen_addch( HIVE_BOTTOMLEFT );
  screen_addch( HIVE_BOTTOMRIGHT );
  sprintf( tmp_str, "  left = %3d ", get_num_live_hives() );
  screen_addstr( tmp_str );
  screen_addch( VERTICAL_BAR );
  screen_addstr("   ");
  screen_addch( EVILFACE );
  screen_addch( WEAPON_RIGHT );
  sprintf( tmp_str, " left = %5d ", get_num_live_enemies(0) );
  screen_addstr( tmp_str );
  screen_addch( VERTICAL_BAR );
  screen_addstr("   ");
  screen_addch( GHOSTFACE );
  sprintf( tmp_str, " left = %5d ", get_num_live_enemies(1) );
  screen_addstr( tmp_str );
  screen_addch( VERTICAL_BAR );
  sprintf( tmp_str, "   Elapsed Time = %6d", elapsed_time>>2 );
  screen_addstr( tmp_str );

  sprintf( tmp_str, "Men Left = %3d", players_left );
  screen_mvaddstr( 2, 0, tmp_str );
  sprintf( tmp_str, "Game Score   = %06d", game_score );
  screen_mvaddstr( 2, 58, tmp_str );

  screen_setnormalattr();
}


void draw_border( screen_coords scr_coords )
{
  int i;

  screen_setcolorpair(1);

  screen_mvaddch(scr_coords.screen_y1-1,scr_coords.screen_x1-1,'+');
  screen_mvaddch(scr_coords.screen_y1-1,scr_coords.screen_x2+1,'+');
  screen_mvaddch(scr_coords.screen_y2+1,scr_coords.screen_x1-1,'+');
  screen_mvaddch(scr_coords.screen_y2+1,scr_coords.screen_x2+1,'+');

  for( i=scr_coords.screen_y1; i<=scr_coords.screen_y2; i++ )
    screen_mvaddch(i,scr_coords.screen_x1-1,'|');
  for( i=scr_coords.screen_y1; i<=scr_coords.screen_y2; i++ )
    screen_mvaddch(i,scr_coords.screen_x2+1,'|');

  for( i=scr_coords.screen_x1; i<=scr_coords.screen_x2; i++ )
    screen_mvaddch(scr_coords.screen_y1-1,i,'-');
  for( i=scr_coords.screen_x1; i<=scr_coords.screen_x2; i++ )
    screen_mvaddch(scr_coords.screen_y2+1,i,'-');
}


void display_maze( int my_player, screen_coords *sc, wall_info *w )
{
  screen_clear();

  screen_setcolorpair(1);
  screen_setnormalattr();

  draw_border( *sc );

  if( maxx >= 80 )
    display_game_header( get_player_lives(my_player) );

  draw_walls( w, sc );
  show_hives( sc );
  show_enemies( sc );
  show_weapons( sc );

  show_players( sc );
  
  /* In an xterm, the cursor doesn't seem to disappear so it's nice to
   * move it out of the way when the screen update has finished.
   */
  screen_move(0,0);

  screen_refresh();
}


/* ----------------------------------------------------------------- */
/*                        CORE GAME FUNCTIONS                        */
/* ----------------------------------------------------------------- */

/* alarm_occurred is set to one by the SIGALRM interrupt handler.
 * SIGALRMs are generated by the real (wall-clock time) interval
 * timer and are normally blocked.  They are unblocked by
 * sigsuspend() when the program is done with a display cycle.
 */

int alarm_occurred;


/* sig_alrm( int signo )
 * 
 * Signal handler for ALRM (interval timer).
 */

void sig_alrm( int signo )
{
	alarm_occurred = 1;
}


/* start_timer( void )
 *
 * Set sig_alrm() as the signal handler for alarms.  Block reception
 * of SIGALRM.  Start the real interval timer to generate SIGALRM
 * periodically.  SIGALRMs are unblocked by sigsuspend() when
 * the program is done with a display cycle.
 */

void start_timer( void )
{
	int ret;
	char errstr[200];
	struct itimerval value;
	sigset_t sigset;

	alarm_occurred = 0;

	if( signal( SIGALRM, sig_alrm ) == SIG_ERR ) {
		err_sys( "can't catch SIGALRM" );
	}

	sigemptyset( &sigset );
	sigaddset( &sigset, SIGALRM );
	sigprocmask( SIG_BLOCK, &sigset, NULL );

	value.it_interval.tv_sec  = 0;
	value.it_interval.tv_usec = DISPLAY_TIME;
	value.it_value.tv_sec     = 0;
	value.it_value.tv_usec    = DISPLAY_TIME;
	ret = setitimer( ITIMER_REAL, &value, NULL );

	if( ret ) {
		strcpy( errstr, "Unable to start interval timer: " );
		strncat( errstr, strerror(errno), 100 );
		err_sys( errstr );
	}
}


/* wait_timer( void )
 *
 * Use sigsuspend() to temporarily unblock SIGALRM.  Wait until
 * a SIGALRM occurs.
 */

void wait_timer( void )
{
	sigset_t sigset;

	sigemptyset(&sigset);
	while( !alarm_occurred )
		sigsuspend(&sigset);
	alarm_occurred = 0;
}


/* stop_timer( void )
 *
 * Disable the real interval timer.
 */

void stop_timer( void )
{
	struct itimerval value;

	value.it_interval.tv_sec  = 0;
	value.it_interval.tv_usec = 0;
	value.it_value.tv_sec     = 0;
	value.it_value.tv_usec    = 0;
	setitimer( ITIMER_REAL, &value, NULL );

	/* return value is ignored, assume it will work okay */
}


int process_local_player_actions( int my_player, screen_coords *sc, int move_count  )
{
  struct kbd_action ka;
  coordinate tc;

  if( get_kbd_action(&ka) ) {
    switch( ka.action ) {
      case SUSPEND_ACTION:
	screen_clear();
#if 0
	screen_mvaddstr(1,1,"The game is suspended.  <Enter> to resume.");
#else
	screen_mvaddstr(1,1,"The keyboard has been restored, but the game");
	screen_mvaddstr(2,1,"goes on.  <Enter> to rejoin the game.");
#endif
	screen_cursor_on();
	screen_refresh();
	wait_kbd_action();

	screen_cursor_off();
	break;
      case QUIT_ACTION:
	return 1;			/* Quit */
	break;
      case MOVE_ACTION:
	if( !ka.accel && move_count != 0 )
	  return 0;

	tc.x = ka.mx;
	tc.y = ka.my;
	player_move( sc, my_player, tc );

	/* Now try to fire. */

	tc.x = ka.fx;
	tc.y = ka.fy;
	player_fire( sc, my_player, tc, elapsed_time );
	break;
    }
  }
  return 0;
}


/* run_snipes( void )
 * 
 * The game's main loop.
 */

void run_snipes( void )
{
  screen_coords scr_coords;
  wall_info w;

  int quit=0, my_player=0;
  int player_attribs[10] = { 1, 2, 3, 4, 1, 2, 3, 4, 1, 2 };

  int move_count = 0;
  int eyeflash_count = 0;
  
  game_start_time = time(NULL);
  elapsed_time = 0;
  
  start_timer();

  sound_init( use_sound );

  init_maze_coordinates( &scr_coords );
  init_collision_detector( BLOCKNUM_X*(BLKSZX-1), BLOCKNUM_Y*(BLKSZY-1) );
  init_walls( &w, BLOCKNUM_X, BLOCKNUM_Y, BLKSZX, BLKSZY );
  init_hives( 4, &scr_coords, max_hives_at_level[level_num-1] );
  init_enemies( 4, 13-level_num, check_feature(FEATURE_INDEX_ENABLE_GHOSTS),
		max_enemies_at_level[level_num-1] );
  init_weapons( 3, 4, check_feature(FEATURE_INDEX_ENABLE_BULLET_BOUNCE),
		check_feature(FEATURE_INDEX_DESTRUCTIBLE_WALLS)  );
  init_players( &scr_coords, 1, player_attribs,
		check_feature(FEATURE_INDEX_DEADLY_WALLS) );
  
  screen_cursor_off();

  while( !quit )
  {
	wait_timer();
    
    /* Recalculate maze border coordinates. */
    
    recalculate_maze_coordinates( my_player, &scr_coords, &w );
    
    /* Draw the screen. */

    display_maze( my_player, &scr_coords, &w );
    
    /* Perform movements and other actions. */
    
    elapsed_time++;

    move_count     = (move_count + 1)%MOVE_COUNT;
    eyeflash_count = (eyeflash_count + 1)%EYEFLASH_COUNT;

    if( eyeflash_count == 0 )
      eyechange();

    do_hive_stuff( &scr_coords );
    move_enemies( &scr_coords );
    all_player_collision_check( &scr_coords );
    
    quit = process_local_player_actions( my_player, &scr_coords, move_count );

    move_weapons( &scr_coords, &w );
    
    /* Check for end of game. */

    if( (get_num_live_enemies(1) == 0 &&
	 get_num_live_enemies(0) == 0 &&
	 get_num_live_hives()    == 0) || get_player_lives(my_player) == 0 ) {
      quit = 1;
    }
  }

  screen_cursor_on();

  stop_timer();

  close_walls( &w );

  sound_stop();

  screen_setnormalattr();
  if( get_num_live_enemies(1)     == 0 &&
      get_num_live_enemies(0)     == 0 &&
      get_num_live_hives()        == 0 &&
      get_player_lives(my_player) != 0 ) {
    screen_mvaddstr( maxy-2, 0, "Congratulations --- YOU ACTUALLY WON!!!" );
  }
  else {
    if( maxx >= 64 )
      screen_mvaddstr( maxy-2, 0, "The SNIPES have triumphed!!!  Long live the SNIPES of the Maze." );
    else
      screen_mvaddstr( maxy-2, 0, "The SNIPES have triumphed!!!" );
  }
}


int main( int argc, char *argv[] )
{
  int i, j;
  int quit=0;
  int ch, my_y, my_x;

  force_non_raw_kbd = 0;
  use_sound = 1;
  jennymode = 0;

  displaytype = DT_CURSES;
  
  level_alpha = 'A';
  level_num   = 1;
  
  if( argc != 1 ) {
    for( i=1; i<argc; i++ ) {
      if( argv[i][0] != '-' ) {
	if( i == argc-1 ) {
	  if( !do_level( argv[i] ) )
	    usage( argv[0] );
	}
	else
	  usage( argv[0] );
      }
      else {
	for( j=1; j<strlen(argv[i]); j++ )
	  switch( argv[i][j] ) {
#ifdef DOUBLEWIDE_FONT_HACK
	    case 'd':
	      displaytype = DT_CURSES_DOUBLEWIDE;
	      break;
#endif
#ifdef USE_XWIN
	    case 'x':
	      displaytype = DT_XWIN;
	      break;
#endif
#ifdef USE_SVGALIB
	    case 's':
	      displaytype = DT_SVGALIB;
	      break;
#endif
#ifdef USE_CURSES
	    case 'c':
	      displaytype = DT_CURSES;
	      break;
#endif

	    case 'k':
	      force_non_raw_kbd = 1;
	      break;
	    case 'j':
	      jennymode = 1;
	      break;
	    case 'q':
	      use_sound = 0;
	      break;
	    case 'v':
	      printf( "Version %s\n", VERSION );
	      exit(0);
	      break;
	    default:
	      usage( argv[0] );
	  }
      }
    }
  }
  
#if 0
  printf("\n");
  for( i=0; i<6; i++ )
    printf( gpl[i], VERSION );
  
  printf("\nPress <Enter> to play...");
#endif
  
  srandom(time(NULL));

  install_signal_handlers();

  if( !screen_init( displaytype ) ) {
    fprintf( stderr, "Unable to initialize screen.\n" );
    remove_signal_handlers();
    exit( 1 );
  }

  // Keyboard init needs to be after curses init because curses
  // has its own ideas about the keyboard.
  if( !init_kbd( displaytype, !force_non_raw_kbd ) ) {
    fprintf( stderr, "Unable to initialize keyboard.\n" );
    screen_close();
    remove_signal_handlers();
    exit( 1 );
  }

  screen_getmaxyx( &maxy, &maxx );

  while( !quit ) {
    kbd_setmode( KMODE_ACTION );
    run_snipes();
    kbd_setmode( KMODE_NORMAL );

    quit=2;
    while( quit == 2 ) {
      screen_setnormalattr();
      screen_addstr("\nPlay another game? (Y or N) ");
      screen_refresh();
      switch( kbd_getch() ) {
	case 'y':
	case 'Y':
	  quit=0;
	  break;
	case 'n':
	case 'N':
	  quit=1;
	  break;
      }
    }

    if( quit == 0 ) {

      screen_scroll_on();
      screen_setnormalattr();
      screen_addstr("\nEnter new skill level (A-Z)(1-9): ");
      screen_getyx( &my_y, &my_x );
      screen_addch( level_alpha );
      screen_addch( level_num+'0' );
      screen_refresh();
      screen_scroll_off();

      while( (ch = kbd_getch()) != '\r' ) {
	if( isalpha(ch) ) {
	  level_alpha = toupper(ch);
	  screen_mvaddch( my_y, my_x, level_alpha );
	  screen_refresh();
	}
	else if( isdigit(ch) && ch != '0' ) {
	  level_num = ch - '0';
	  screen_mvaddch( my_y, my_x+1, ch );
	  screen_refresh();
	}
      }
    }
  }

  /* We opened the keyboard after the screen.  Modes get restored
   * better if we close them in LIFO order.
   */
  close_kbd();
  screen_close();
  remove_signal_handlers();

  return 0;
}
