/*
 * dlock.c - SunOS lslk lock functions
 *
 * Vic Abell
 * Purdue University Computing Center
 */


/*
 * Copyright 1996 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell.
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */
#ifndef lint
static char copyright[] =
"@(#) Copyright 1996 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dlock.c,v 1.7 99/11/10 15:01:43 abe Exp $";
#endif


#include "lslk.h"
#include "kernelbase.h"

/*
 * Local function prototypes
 */

_PROTOTYPE(static void savelock,(struct inode *i, struct data_lock *dp));
_PROTOTYPE(static void process_lock_list,(struct inode *i, struct vnode *v, KA_T ll));


/*
 * gather_lock_info() -- gather lock information
 */

void
gather_lock_info()
{
	int hx;
	struct data_lock dl;
	struct inode i, *ip;
	union ihead ih, *ihp;
	struct lock_list ll;
	struct vnode *va;
/*
 * Loop through the inode hash buckets.
 */
	for (hx = 0, ihp = (union ihead *)Nl[X_IHEAD].n_value;
	     hx < INOHSZ;
	     hx++, ihp++)
	{
	    if (kread((KA_T)ihp, (char *)&ih, sizeof(ih))) {
		if (Owarn)
		    (void) fprintf(stderr, "%s: can't read ihead from %#x\n",
			Pn, ihp);
		return;
	    }
	/*
	 * Loop through the inodes of a hash bucket.
	 */
	    for (ip = ih.ih_chain[0];
	         ip && ip != (struct inode *)ihp;
	         ip = i.i_forw)
	    {
		if (kread((KA_T)ip, (char *)&i, sizeof(i))) {
		    if (Owarn)
			(void) fprintf(stderr,
			    "%s: premature exit from inode chain\n", Pn);
		    return;
		}
		if (!i.i_vnode.v_filocks)
		    continue;
	    /*
	     * If the vnode, contained in the inode, has a non-NULL v_filocks
	     * pointer and a non-zero reference count, read the associated
	     * lock_list structure and process the exclusive and shared lists.
	     */
		if (kread((KA_T)i.i_vnode.v_filocks, (char *)&ll, sizeof(ll)))
		    continue;
		if (i.i_vnode.v_count) {
		    va = (struct vnode *)((char *)ip
		       + (2 * sizeof(struct inode *)));
		    process_lock_list(&i, va, (KA_T)ll.exclusive);
		    process_lock_list(&i, va, (KA_T)ll.shared);
		}
	    }
	}
}


/*
 * get_cmdnm() -- get command name
 */

char *
get_cmdnm(lp)
	struct llock_info *lp;		/* local lock structure */
{
	struct proc *p;
	struct user *u;

/*
 * Get the proc structure for the lock info structure and return the
 * command name of its user structure.
 */
	if (lp->pid
	&&  (p = kvm_getproc(Kd, lp->pid))
	&&  (u = kvm_getu(Kd, p)))
	    return(u->u_comm);
	return("(unknown)");
}


/*
 * get_kernel_access() -- get access to kernel information
 */

void
get_kernel_access()
{
	int err, i;
/*
 * Convert kernel variable names to addresses and check the result.
 */
	if (Nmlst && !is_readable(Nmlst, 1))
	    Exit(1);
	if (nlist(Nmlst ? Nmlst : N_UNIX, Nl) < 0) {
	    (void) fprintf(stderr, "%s: nlist(%s) failed\n",
		Pn, Nmlst ? Nmlst : N_UNIX);
	    Exit(1);
	}
	for (err = i = 0; Nl[i].n_name[0]; i++) {
	    if (!Nl[i].n_value) {
		(void) fprintf(stderr, "%s: no nlist address for %s\n",
		    Pn, Nl[i].n_name);
		err++;
	    }
	}
	if (err)
	    Exit(1);
/*
 * Open access to kernel memory.
 */
	if (!(Kd = kvm_open(NULL, NULL, NULL, O_RDONLY, Pn)))
	    Exit(1);
/*
 * Drop setgid permission.
 */
	(void) dropgid();
}


/*
 * initialize() -- initialize
 */

void
initialize()
{
	(void) get_kernel_access();
}


/*
 * kread() -- read kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel address */
	char *buf;			/* local receiving buffer address */
	int len	;			/* length to read */
{
	int br;

	if ((unsigned long)addr < (unsigned long)KERNELBASE)
	    return(1);
	br = kvm_read(Kd, (unsigned long)addr, buf, len);
	return((br == len) ? 0 : 1);
}


/*
 * print_dev() -- print device number
 */

char *
print_dev(lp)
	struct llock_info *lp;		/* local lock structure */
{
	static char buf[128];

	(void) sprintf(buf, "%d,%d", major(lp->dev), minor(lp->dev));
	return(buf);
}


/*
 * process_lock_list() -- process lock list
 */

static void
process_lock_list(ip, va, ll)
	struct inode *ip;		/* local inode */
	struct vnode *va;		/* kernel vnode address */
	KA_T ll;			/* lock list address */
{
	struct data_lock dl;
	KA_T dla;

	if (!(dla = ll))
	    return;
	do {

	/*
	 * Read and process the chain of data_lock structures.
	 */
	    if (kread(dla, (char *)&dl, sizeof(dl)))
		return;
	    if (dl.vp == va) {
		(void) savelock(ip, &dl);
		if (is_lock_sel()) {
		    NLockU++;
		    Lp = (struct llock_info *)NULL;
		}
	    }
	    dla = (KA_T)dl.Next;
	} while (dla && dla != ll);
}


/*
 * savelock() -- save lock information
 */

static void
savelock(ip, dp)
	struct inode *ip;		/* inode structure pointer */
	struct data_lock *dp;		/* data_lock structure pointer */
{

/*
 * Allocate a local lock structure.
 */
	(void) alloc_llock();
/*
 * Save: inode number; device; process ID; and size.
 */
	Lp->inum = ip->i_number;
	Lp->dev = ip->i_dev;
	if (dp->rsys)
	    Lp->pid = (unsigned long)dp->rpid;
	else
	    Lp->pid = (unsigned long)dp->pid;
	Lp->sz = (unsigned long)ip->i_size;
	Lp->szs = 1;
/*
 * Save the lock description: mandatory status; type; start; and
 * length or end.
 */
	if ((ip->i_vnode.v_type == VREG)
	&&  ((ip->i_mode & (ISGID|(IEXEC>>3))) == ISGID))
	    Lp->mand = 1;
	else
	    Lp->mand = 0;
	if (dp->type == EXCLUSIVE)
	    Lp->type = 2;
	else
	    Lp->type = 1;
	Lp->ls = Lp->ss = 1;
	Lp->start = (long)dp->base;
	Lp->len = dp->length;
/*
 * If the rsys element of the data lock is non-zero, this is a remote lock.
 */
	if (dp->rsys) {
	    Lp->src = 1;
	    Lp->hn = (char *)NULL;
	    Lp->iap = alloc_in_addr();
	    Lp->iap->s_addr = (unsigned long)dp->rsys;
	} else
	    Lp->src = 0;
}
