/*
 * dlock.c - Digital Unix lslk lock functions
 *
 * Vic Abell
 * Purdue University Computing Center
 *
 * Chris Eleveld <chris@sector7.com>
 */


/*
 * Copyright 1996 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell.
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */
#ifndef lint
static char copyright[] =
"@(#) Copyright 1996 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dlock.c,v 1.11 2000/10/22 13:16:46 abe Exp $";
#endif


#include "lslk.h"


/*
 * Local definitions
 */

#if	DUV>=50000 && (!defined(PID_MAX) || PID_MAX==INT_MAX)
#undef	PID_MAX
#define	PID_MAX	65536
#endif	/* DUV>=50000 && (!defined(PID_MAX) || PID_MAX==INT_MAX) */

#if	DUV>=50000
typedef struct cnode {			/* CFS node structure definition for
					 * Tru64 UNIX 5.0 and above */
        udecl_simple_lock_data(, d1)
	unsigned int d2;
	time_t d3;
	unsigned long d4[3];

# if	DUV<50100
	int d5[2];
	off_t d6;
# endif	/* DUV<50100 */

	unsigned long d7;
        vattr_t          c_attr;        /* 96:Cached vnode attributes */
} cnode_t;
#endif	/* DUV>=50000 */


/*
 * Local static variables
 */

static int Maxprivsz = 0;	/* size of maximum private vnode area */
struct pidnmc {			/* process ID name cache */

    char *cmd;			/* pointer to command name for PID */
    unsigned long pid;		/* process ID number */
    struct pidnmc *next;	/* pointer to next hashed entry */
} **Pnc = (struct pidnmc **)NULL;
int PncMask = 0;		/* Pnc[] mask */


/*
 * Local function prototypes
 */

_PROTOTYPE(static char *addpnc,(unsigned long lockpid, char *cmd));
_PROTOTYPE(static void get_kernel_access,(void));
_PROTOTYPE(static char *get_proc_cmd,(unsigned long lockpid));
_PROTOTYPE(static char *is_hashed,(unsigned long lockpid));
_PROTOTYPE(static char *lkup_cmd,(unsigned long lockpid));
_PROTOTYPE(static void savelock,(struct filock *flp, struct vnode *vp, char *cmdp));


/*
 * addpnc() - add a process name to the hash/cache
 */

static char *
addpnc(lockpid, cmd)
	unsigned long lockpid;
	char *cmd;
{
	char *cp;
	MALLOC_S len;
	struct pidnmc *p;
/*
 * Allocate space for the command name.
 */
	if ((len = strlen(cmd)) == 0)
	    return((char *)NULL);
	if ((cp = (char *)malloc(len + 1)) == (char *)NULL) {
	    (void) fprintf(stderr,
		"%s: PID %ld; no space for command name: %s\n",
		Pn, lockpid, cmd);
	    Exit(1);
	}
	(void) strcpy(cp, cmd);
/*
 * Allocate hash table space and fill the entry.  Put it at the
 * head of the chain of entries that hash to the same value.
 */
	if ((p = (struct pidnmc *)malloc(sizeof(struct pidnmc)))
	== (struct pidnmc *)NULL) {
	    (void) fprintf(stderr,
		"%s: PID %ld; no process name cache space; command: %s\n",
		Pn, lockpid, cmd);
	    Exit(1);
	}
	p->pid = lockpid;
	p->cmd = cp;
	p->next = Pnc[lockpid & PncMask];
	Pnc[lockpid & PncMask] = p;
	return cp;
}

/*
 * gather_lock_info() -- gather lock information
 */

void
gather_lock_info()
{
	struct flino fi;
	struct filock fl;
	KA_T fla, flf, fiha, fia;
	struct vnode *vp;
	char *cmd;
/*
 * Get the maximum size of a vnode private data area.
 */
	if (!Nl[X_VNMAXPRIV].n_type
	||  kread((KA_T)Nl[X_VNMAXPRIV].n_value, (char *)&Maxprivsz,
		  sizeof(Maxprivsz)))
	{
	    (void) fprintf(stderr,
		"%s: can't get max vnode private area size: %#x\n",
		Pn, Nl[X_VNMAXPRIV].n_value);
	    Exit(1);
	}
/*
 * Allocate vnode data area.
 */
	if ((vp = (struct vnode *)malloc(sizeof(struct vnode) + Maxprivsz))
	== (struct vnode *)NULL)
	{
	    (void) fprintf(stderr, "%s: can't allocate vnode space: %d bytes\n",
		Pn, sizeof(struct vnode) + Maxprivsz);
	    Exit(1);
	}
/*
 * Read pointer to lock<->vnode map list head.
 */
	if (kread((KA_T)Nl[X_FIDS].n_value, (char *)&fia, sizeof(fia))) {
	    (void) fprintf(stderr,
		"%s: can't read vnode <-> lock map pointer: %#x\n",
		Pn, Nl[X_FIDS].n_value);
	    Exit(1);
	}
	if (!(fiha = fia))
	    return;
/*
 * Read and process the chain of flino structures.
 */
	do {
	    if (kread(fia, (char *)&fi, sizeof(fi)))
		return;
	/*
	 * Read the associated vnode.
	 */
	    if (kread((KA_T)fi.vp, (char *)vp, sizeof(*vp) + Maxprivsz))
		return;
	/*
	 * Read through the lock list.
	 */
	    if (!(flf = (KA_T)fi.fl_flck))
		continue;
	    fla = flf;
	    do {

	    /*
	     * Get the current lock entry.
	     */
		if (kread(fla, (char *)&fl, sizeof(fl)))
		    break;	/* other vnodes may have locks */
	    /*
	     * If it's not a remote lock. get the command name.
	     */
		if (!fl.set.l_rsys)
		    cmd = lkup_cmd((unsigned long)fl.set.l_pid);
		else
		    cmd = (char *)0;
	    /*
	     * Save lock information.
	     */
		(void) savelock(&fl, vp, cmd);
		if (is_lock_sel()) {
		    NLockU++;
		    Lp = (struct llock_info *)NULL;
		}
	    } while ((fla = (KA_T)fl.next) && fla != flf);
	} while ((fia = (KA_T)fi.next) && fia != fiha);
}


/*
 * get_cmdnm() -- get command name for printing
 */

char *
get_cmdnm(lp)
	struct llock_info *lp;		/* local lock structure */
{
	if (lp->cmd)
	    return(lp->cmd);
	return("(unknown)");
}


/*
 * get_kernel_access() -- get access to kernel information
 */

void
get_kernel_access()
{
	int i;
/*
 * Get kernel name list path.
 */
	if (!Nmlst) {
	    if (!(Nmlst = get_nlist_path(1))) {
		(void) fprintf(stderr, "%s: can't get kernel name list path\n",
		    Pn);
		Exit(1);
	    }
	}
/*
 * Open kernel memory access.
 */
	if ((Kd = open(KMEM, O_RDONLY, 0)) < 0) {
	    (void) fprintf(stderr, "%s: can't open %s: %s\n",
		Pn, KMEM, strerror(errno));
	    Exit(1);
	}
/*
 * Drop setgid permission.  If not dropping it, see if the name list path
 * is readable.
 */

#if	defined(WILLDROPGID)
	(void) dropgid();
#else	/* !defined(WILLDROPGID) */
	if (!isreadable(Nmlst, 1))
	    Exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Access kernel symbols.
 */
	if ((i = nlist(Nmlst, Nl)) < 0) {
	    (void) fprintf(stderr,
		"%s: can't read kernel name list from %s: %s\n",
		Pn, Nmlst, strerror(errno));
	    Exit(1);
	}
/*
 * Check for unidentified kernel symbols.
 */
	if (i) {
	    for (i = 0; Nl[i].n_name; i++) {
		if (!Nl[i].n_value) {
		    (void) fprintf(stderr,
			"%s: can't get kernel address: %s\n",
			Pn, Nl[i].n_name);
		}
	    }
	    Exit(1);
	}
/*
 * Allocate space for kernel name cache.
 */
	for (i = 512; i < (PID_MAX / 2); i = i << 1)
	    ;
	PncMask = i - 1;
	if ((Pnc = (struct pidnmc **)calloc(i, sizeof(struct pidnmc *)))
	== (struct pidnmc **)NULL) {
	    (void) fprintf(stderr,
		"%s: can't allocate %d bytes for process ID name cache\n",
		Pn, (i * sizeof(struct pidnmc *)));
	    Exit(1);
	}
}


/*
 * get_nlist_path() - get kernel name list path
 */

char *
get_nlist_path(ap)
	int ap;				/* on success, return an allocated path
					 * string pointer if 1; return a
					 * constant character pointer if 0;
					 * return NULL if failure */
{
	char *ba, buf[MAXPATHLEN+2], *ps;
	int len, rv;
/*
 * Get bootfile name.
 */
	len = 0;
	if ((rv = getsysinfo(GSI_BOOTEDFILE, &buf[1], sizeof(buf) - 1, &len,
			     (char *)NULL))
	!= 1)
	{
	    if (rv < 0) {
		(void) fprintf(stderr, "%s: can't get booted file name: %s\n",
		    Pn, strerror(errno));
		Exit(1);
	    }
	    return((char *)NULL);
	}
/*
 * Check for a non-NULL path.
 */
	buf[sizeof(buf) - 2] = '\0';
	len = strlen(&buf[1]);
	if (len < 1)
	    return((char *)NULL);
/*
 * If no path return is requested by the value of ap, return a NULL string
 * pointer.
 */
	if (!ap)
	    return("");
/*
 * Make sure the path has a leading '/'.
 */
	if (buf[1] != '/') {
	    buf[0] = '/';
	    ba = buf;
	    len++;
	} else
	   ba = &buf[1];
/*
 * Allocate permanent space for the path, copy it to the space, and return
 * a pointer to the space.
 */
	len++;
	if (!(ps = (char *)malloc(len))) {
	    (void) fprintf(stderr,
		"%s: can't allocate %d bytes for boot file path: %s\n",
		Pn, len, ba);
	    Exit(1);
	}
	(void) strcpy(ps, ba);
	return(ps);
}


/*
 * get_proc_cmd() - get command name from kernel's proc table
 */

static char *
get_proc_cmd(lockpid)
	unsigned long lockpid;		/* process ID */
{
	struct proc p;
	struct utask u;

#if	DUV<30000
	int i;
	static KA_T kp = (KA_T)0;
	static int np = 0;
	static int px = 0;
#else	/* DUV>=30000 */
	struct pid_entry pe;
	static KA_T pi;
# if	DUV<30200 || DUV>=50000
	int i;
	static int np = 0;
	static int px = 0;
# else	/* DUV>=30200 && DUV<50000 */
	static int pid_entry_mask = 0;	/* for PID_INDEX() macro */
# endif	/* DUV<32000 || DUV>=50000 */
#endif	/* DUV>=30000 */

#if	DUV<30000
/*
 * Get kernel process table information for Digital UNIX versions below 3.2.
 */
	if (np == 0) {

	/*
	 * If first time, read process count and process table address.
	 */
	    if (!Nl[X_NPROCS].n_value
	    ||  kread((KA_T)Nl[X_NPROCS].n_value, (char *)&np, sizeof(np))
	    ||  np == 0)
	    {
		(void) fprintf(stderr,
		    "%s: can't read kernel process count (%d): %#x\n",
		    Pn, np, Nl[X_NPROCS].n_value);
		Exit(1);
	    }
	    if (!Nl[X_PROCS].n_value
	    ||  kread((KA_T)Nl[X_PROCS].n_value, (char *)&kp, sizeof(kp)))
	    {
		(void) fprintf(stderr,
		    "%s: can't read kernel process table address: %#x\n",
		    Pn, Nl[X_PROCS].n_value);
		Exit(1);
	    }
	}
/*
 * Resume reading the kernel process table.  If the process is not located,
 * reread the process table from its beginning.  Once the target proc
 * structure is located, read its user task structure, add its command name
 * to the process name cache, and return the name's address.
 */
	for (i = px ? 0 : 1; i < 2; i++) {
	    for (; px < np; px++) {
		if (kread((KA_T)(kp+(px * sizeof(p))), (char *)&p, sizeof(p)))
		    return (char *)NULL;
		if (!p.p_stat || p.p_stat == SZOMB || p.p_pid != lockpid)
		    continue;
		if (!p.utask
		||  kread((KA_T)p.utask, (char *)&u, sizeof(u)))
		    continue;
		return(addpnc(lockpid, u.uu_comm));
	    }
	    px = 0;
	}
#else	/* DUV>=30000 */
/*
 * Get kernel process table information for Digital UNIX versions above 3.0.
 */

# if	DUV<30200 || DUV>=50000
	if (np == 0)
# else	/* DUV>=30200 && DUV<50000 */
	if (pid_entry_mask == 0)
# endif	/* DUV<30200 || DUV>=50000 */

	{

	/*
	 * If this is the first time, get the kernel PID table address.
	 */
	    if (!Nl[X_PROCS].n_value
	    ||  kread((KA_T)Nl[X_PROCS].n_value, (char *)&pi, sizeof(pi))
	    ||  !pi)
	    {
		(void) fprintf(stderr,
		    "%s: can't read proc table base address (%d) from %#x\n",
		    Pn, pi, Nl[X_PROCS].n_value);
		Exit(1);
	    }

# if	DUV<30200 || DUV>=50000
	/*
	 * Read the PID table count for versions below 3.2 and 5.0 and above.
	 */
	    if (!Nl[X_NPID].n_value
	    ||  kread((KA_T)Nl[X_NPID].n_value, (char *)&np, sizeof(np))
	    || !np)
	    {
		(void) fprintf(stderr,
		    "%s: can't read process ID entry count (%d) from %#x\n",
		    Pn, np, Nl[X_NPID].n_value);
		Exit(1);
	    }
# else	/* DUV>=30200 && DUV<50000 */
	/*
	 * Read the pid_entry_mask for versions 3.2 and above PID_INDEX()
	 * macro.
	 */
	    if (!Nl[X_PIDENTRYMASK].n_value
	    ||  kread((KA_T)Nl[X_PIDENTRYMASK].n_value, (char *)&pid_entry_mask,
		      sizeof(pid_entry_mask))
	    || !pid_entry_mask)
	    {
		(void) fprintf(stderr,
		    "%s: can't read process ID entry mask (%d) from %#x\n",
		    Pn, pid_entry_mask, Nl[X_PIDENTRYMASK].n_value);
		Exit(1);
	    }
# endif	/* DUV<30200 || DUV>=50000 */
	}
# if	DUV<30200 || DUV>=50000
/*
 * Resume reading the kernel PID table for versions below 3.2 and above 5.0.
 *  If the process ID is not located, reread the table from its beginning.
 * Once the target PID is located, read its user task structure, add its
 * command name to the process name cache, and return the name's address.
 */
	for (i = px ? 0 : 1; i < 2; i++) {
	    for (; px < np; px++) {
		if (kread((KA_T)pi+(px * sizeof(pe)), (char *)&pe, sizeof(pe)))
		    return (char *)NULL;
		if (pe.pe_pid != lockpid)
		    continue;
		if (!pe.pe_proc
		||  kread((KA_T)pe.pe_proc, (char *)&p, sizeof(p))
		||  p.p_pid != lockpid
		||  kread((KA_T)pe.pe_proc + sizeof(struct proc), (char *)&u,
			  sizeof(u)))
		{
		    continue;
		}
		return(addpnc(lockpid, u.uu_comm));
	    }
	    px = 0;
	}
# else	/* DUV>=30200 && DUV<50000 */
/*
 * Read the version 3.2 and below 5.0 PID structure for the given process ID;
 * then read its user task structure.  Add the command name to the process
 * name cache and return a pointer to it.
 */
	if (kread(pi + (PID_INDEX(lockpid) * sizeof(pe)), (char *)&pe,
		  sizeof(pe))
	||  pe.pe_pid != lockpid)
	    return (char *)NULL;
	if (!pe.pe_proc
	||  kread((KA_T)pe.pe_proc, (char *)&p, sizeof(struct proc))
	||  p.p_pid != lockpid
	||  kread((KA_T)pe.pe_proc + (KA_T)sizeof(struct proc), (char *)&u,
		  sizeof(u)))
	{
	    return (char *)NULL;
	}
	return(addpnc(lockpid, u.uu_comm));
# endif	/* DUV<30200 || DUV>=50000 */
#endif	/* DUV<30000 */

}


/*
 * initialize() -- initialize
 */

void
initialize()
{
	(void) get_kernel_access();
}


/*
 * is_hashed() - is the process name hashed?
 */

static char *
is_hashed(lockpid)
	unsigned long lockpid;		/* process ID */
{
	struct pidnmc *p;

	for (p = Pnc[lockpid & PncMask]; p; p = p->next) {
	    if (p->pid == lockpid)
		return(p->cmd);
	}
	return((char *)NULL);
}


/*
 * kread() -- read kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel address */
	char *buf;			/* local receiving buffer address */
	int len	;			/* length to read */
{
	int br;

	if (lseek(Kd, addr, SEEK_SET) == (off_t)-1L)
		return(-1);
	br = read(Kd, buf, len);
	return((br == len) ? 0 : 1);
}


/*
 * lkup_cmd() -- look up command name
 */

static char *
lkup_cmd(lockpid)
	unsigned long lockpid;		/* process ID */
{
	char *cmd;

	if ((cmd = is_hashed(lockpid)))
	    return(cmd);
	return(get_proc_cmd(lockpid));
}


/*
 * print_dev() -- print device number
 */

char *
print_dev(lp)
	struct llock_info *lp;		/* local lock structure */
{
	static char buf[128];

	(void) sprintf(buf, "%d,%d", major(lp->dev), minor(lp->dev));
	return(buf);
}



/*
 * savelock() -- save lock information
 */

static void
savelock(flp, vp, cmd)
	struct filock *flp;		/* file_lock structure pointer */
	struct vnode *vp;		/* inode structure pointer */
	char *cmd;			/* command name pointer */
{
	MALLOC_S len;
	struct vnops *vnopptr;
/*
 * Allocate a local lock structure.
 */
	(void) alloc_llock();
/*
 * Save: inode number; device; process ID; and size.
 */
	switch (vp->v_tag) {
	    case VT_NON: 	/* no File system type */
	    case VT_MFS:	/* Memory file system */
	    case VT_DFS: 	/* DCE DFS ???? */
	    case VT_EFS:	/* DCE Episode FS ???? */
	    case VT_FFM:	/* File on File mount */
	    case VT_FDFS:	/* File Descriptor File System ???? */
	    case VT_ADDON:	/* Add on file system ???? */
		if (Owarn)
		    (void) fprintf(stderr,
			"%s: WARNING: can't process node for FSTYPE: %d\n",
			Pn, vp->v_tag);
		return;

#if	DUV>=50000
	    case VT_CFS:	/* clustered file system */
		{
		    cnode_t *c = (cnode_t *)vp->v_data;

		    Lp->dev = 0;
		    if (vp->v_type == VCHR) {
			if (c->c_attr.va_mask & AT_FSID)
			    Lp->dev = c->c_attr.va_fsid;
		    } else  {
			if (c->c_attr.va_mask & AT_RDEV)
			    Lp->dev = c->c_attr.va_rdev;
		    }
		    if (c->c_attr.va_mask & AT_NODEID)
			Lp->inum = c->c_attr.va_fileid;
		    else
			Lp->inum = 0;
		    if (c->c_attr.va_mask & AT_SIZE) {
			Lp->sz = c->c_attr.va_qsize;
			Lp->szs = 1;
		    }
		}
		break;
#endif	/* DUV>=50000 */

	    case VT_UFS: 	/* unix file system */
		{
		    struct inode *ip = (struct inode *)vp->v_data;

		    Lp->inum = ip->i_number;
		    Lp->dev =  (vp->v_type == VCHR) ? ip->i_din.di_db[0]
						    : ip->i_dev;
		    Lp->sz = ip->i_din.di_qsize;
		    Lp->szs = 1;
		}
		break;
	    case VT_NFS:	/* network file system */
		{
		    struct rnode *rp = (struct rnode *)vp->v_data;

		    Lp->inum =  rp->r_attr.va_fileid;
		    Lp->dev =  rp->r_attr.va_fsid;
		    Lp->sz =  rp->r_size;
		    Lp->szs = 1;
		}
		break;
	    case VT_S5FS:	/* System V file system */
		{
		    struct s5inode *s5p = (struct s5inode *)vp->v_data;

		    Lp->inum = (ino_t)s5p->i_number;
		    Lp->dev = s5p->i_dev;
		    Lp->sz = s5p->i_size;
		    Lp->szs = 1;
		}
		break;
	    case VT_CDFS:	/* CD ROM file system (ISO 9660, etc.) */
		{
		    struct cdnode *cdp = (struct cdnode *)vp->v_data;

		    Lp->inum = cdp->cd_number;
		    Lp->dev = cdp->cd_dev;
		    Lp->sz = cdp->cd_size;
		    Lp->szs = 1;
		}
	    case VT_PRFS:	/* Proc file system (/proc) */
		if (Owarn)
		    (void) fprintf(stderr,
			"%s: WARNING: Proc fs can't have locks.\n", Pn);
		return;
	    case VT_MSFS:	/* ADVFS */
		{
		    struct advfsnode *advfsp = (struct advfsnode *)vp->v_data;

		    Lp->inum = advfsp->a_number;
		    Lp->dev = (vp->v_type == VCHR) ? advfsp->a_rdev
						  :  getmntdev(vp);
		    Lp->sz = advfsp->a_size;
		    Lp->szs = 1;
		}
		break;
	    default:
		if (Owarn)
		    (void) fprintf(stderr,
			"%s: WARNING: unknown FSTYPE: %d\n", Pn, vp->v_tag);
		return;
	}
/*
 * Save the lock description: mandatory status; type; start; and
 * length or end.  End under Digital UNIX is the len field,
 * and the value of the len field is the first unlocked byte
 */
	Lp->mand = (flp->set.l_type & VMANFLCK) ? 1 : 0;
	Lp->type = 0;
	if ((flp->set.l_type & 0xf) == F_RDLCK)
	    Lp->type = 1;
	else if ((flp->set.l_type & 0xf) == F_WRLCK)
	    Lp->type = 2;
	Lp->ss = Lp->es = Lp->ws = 1;
	Lp->start = (unsigned long)flp->set.l_start;
	Lp->whence = (unsigned long)flp->set.l_whence;
	Lp->end = (unsigned long)flp->set.l_len;
	Lp->pid = (unsigned long)flp->set.l_pid;
/*
 * Record the lock type -- local or remote -- and the remote address, if
 * applicable.
 */
	if (!flp->set.l_rsys) {
	    Lp->src = 0;
	} else {
	    Lp->src = 1;
	    Lp->hn = (char *)NULL;
	    Lp->iap = alloc_in_addr();
	    Lp->iap->s_addr = (unsigned long)flp->set.l_rsys;
	}
/*
 * Save the command name and set the don't-free-the-command flag if there
 * is a non-NULL command.
 */
	if ((Lp->cmd = cmd) != (char *)NULL)
	    Lp->cs = 0;
}
