###################################################################################
# LAVA QA api - This class represents a LAVA test object.
# Copyright (C) 2015 Collabora Ltd.

# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.

# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.

# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  US
###################################################################################


class Test(object):
    """This class represents a LAVA test"""

    def __init__(self, test_run):
        self._test_run = test_run
        self._name = test_run['test_id']
        self._uuid = test_run['analyzer_assigned_uuid']
        self._results = test_run['test_results']
        self._attributes = test_run.get('attributes', None)
        self._attachments = test_run.get('attachments', None)
        self._software_ctx = test_run.get('software_context', None)
        self._hardware_ctx = test_run.get('hardware_context', None)
        self._testdef_metadata = test_run.get('testdef_metadata', None)

    def __str__(self):
        return "{}: {}".format(self._name, self._uuid)

    @property
    def name(self):
        return self._name

    @property
    def uuid(self):
        return self._uuid

    @property
    def results(self):
        return self._results

    @property
    def attributes(self):
        return self._attributes

    @property
    def attachments(self):
        return self._attachments

    @property
    def params(self):
        if self._software_ctx:
            return self._software_ctx['sources'][0]['test_params']

    @property
    def project(self):
        if self._software_ctx:
            return self._software_ctx['sources'][0]['project_name']

    @property
    def image(self):
        if self._software_ctx:
            return self._software_ctx['image']['name']

    @property
    def packages(self):
        if self._software_ctx:
            return self._software_ctx['packages']

    @property
    def branch_url(self):
        if self._software_ctx:
            return self._software_ctx['sources'][0]['branch_url']

    @property
    def branch_rev(self):
        if self._software_ctx:
            return self._software_ctx['sources'][0]['branch_revision']

    @property
    def branch_vcs(self):
        if self._software_ctx:
            return self._software_ctx['sources'][0]['branch_vcs']

    @property
    def metadata(self):
        return self._testdef_metadata
    
    @property
    def version(self):
        if self._testdef_metadata:
            return self._testdef_metadata['version']

    @property
    def description(self):
        if self._testdef_metadata:
            return self._testdef_metadata['description']

    @property
    def format(self):
        if self._testdef_metadata:
            return self._testdef_metadata['format']

    @property
    def location(self):
        if self._testdef_metadata:
            return self._testdef_metadata['location']

    @property
    def devices(self):
        if self._hardware_ctx:
            return self._hardware_ctx['devices']

    # Convenient functions to filter tests results.
    @property
    def results_passed(self):
        return self._filter_result('pass')

    @property
    def results_failed(self):
        return self._filter_result('fail')

    @property
    def results_skipped(self):
        return self._filter_result('skip')

    @property
    def results_unknown(self):
        return self._filter_result('unknown')

    def _filter_result(self, result):
        return filter(lambda tc: tc['result'] == result, self.results)
