#!/usr/bin/perl -w

#$Header: /home2/cvsroot/LogTrend/Visu/Web/SessionManager.pm,v 1.11 2001/12/27 17:10:25 slhullier Exp $
##******************************************************************************
## Class SessionManager
##  Description  : class to manage the session for the web
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Author       : Sylvain Lhullier s.lhullier@atrid.fr
##******************************************************************************
#$Log: SessionManager.pm,v $
#Revision 1.11  2001/12/27 17:10:25  slhullier
#Plugins 1st step
#
#Revision 1.10  2001/12/05 16:32:36  slhullier
#Error management with try/otherwise ...
#
#Revision 1.9  2001/11/28 17:29:05  slhullier
#No more session ...
#
#Revision 1.8  2001/11/19 12:55:50  slhullier
#Accent bug fixed at all ...
#
#Revision 1.7  2001/11/19 10:56:54  slhullier
#little bit stable, before big modification
#
#Revision 1.6  2001/11/15 16:08:57  slhullier
#UserNode managing first version (still bugs)
#
#Revision 1.5  2001/11/15 14:42:55  slhullier
#DataBaseManager adding
#
#Revision 1.4  2001/11/09 16:12:53  slhullier
#Form for report generation
#
#Revision 1.3  2001/10/29 15:38:22  slhullier
#Accent bug fixed?
#
#Revision 1.2  2001/10/29 14:06:03  slhullier
#Stable but non-realy working report via the web
#
#Revision 1.1  2001/10/18 15:22:58  slhullier
#Inheritance for report/web
#

package LogTrend::Visu::Web::SessionManager;

use strict;
use Error qw(:try);
use POSIX ":sys_wait_h";
use XML::DOM;

use LogTrend::Common::LogDie;
use LogTrend::Visu::Constants;
use LogTrend::Visu::DataBaseManager;
use LogTrend::Visu::Request;
use LogTrend::Visu::Utils;
use LogTrend::Visu::Web::SystemsDeclaration;

##******************************************************************************
## Constructor  public
##  Parameters   : the XML file-name, the session port
##                 number, the wanted content
##******************************************************************************
sub new
{
   my ($classname,$user,$sessionSocket,$wantedContent) = @_;
   my $self = {};
   bless($self, $classname);
   $self->{USER} = $user;
   $self->{ERROR} = "";

   try
   {
      $self->{SYSTEMSDECLARATION} = new LogTrend::Visu::Web::SystemsDeclaration($user);
   }
   otherwise
   {
      $self->{ERROR} = "File $Utils_CurrentFile<BR>\n$_[0]<BR>\n$Utils_WarningMessage";
   }
   finally
   {
      $Utils_WarningMessage = "";
   };

   $self->{SESSIONSOCKET} = $sessionSocket;
   $self->{WANTEDCONTENT} = $wantedContent;

   return $self;
}

##******************************************************************************
## Method run  public
##  Description  : run the SessionManager
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub run
{
   my ($self) = @_;
   my $maxChild = $CstChildrenNb;

   $SIG{TERM} = sub { $self->quit(); };
   $SIG{CHLD} = sub { $self->childSignal(); };
   $SIG{USR1} = sub { ; };

   my @child = ();
   $self->{CHILD} = \@child;
   $maxChild = 1  if( $self->{ERROR} ne "" );
   for( my $i=0; $i<$maxChild; $i++ )
   {
      unshift @{$self->{CHILD}}, $self->childCreate();
   }

   my $error = 4; # = Interrupted system call  (here: USR1/CHLD signal catch)
   while( $error == 4 )
   {
      $! = 0;
      sleep( $CstInactiveUserTimeMax );
      $error = int($!);
   }

   $self->quit();

}

##******************************************************************************
## Method childSignal  protected
##  Description  : manage the reception of a SIGCHLD
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub childSignal
{
   my ($self) = @_;
   my $pid;

   ##===========================================================================
   ## For each dead processes :
   ##===========================================================================
   do
   {
      $pid = waitpid(-1,&WNOHANG);

      if( $pid !=  0 and
          $pid != -1 )
      {
         ##---------------------------------------------------------------------
         ## Quit ?
         ##---------------------------------------------------------------------
         if( WIFEXITED($?) &&
             WEXITSTATUS($?) == $CstQuitStatus )
         {
            $self->quit();
         }
         ##---------------------------------------------------------------------
         ## Probleme with a child : re-creation
         ##---------------------------------------------------------------------
         else
         {
            @{$self->{CHILD}} = grep !/^$pid$/, @{$self->{CHILD}};
            unshift @{$self->{CHILD}}, $self->childCreate();
         }
      }

   } until( $pid ==  0 or
            $pid == -1);


}

##******************************************************************************
## Method quit  public
##  Description  : quit the session
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub quit
{
   my ($self) = @_;
   $SIG{CHLD} = 'IGNORE';
   kill 'KILL', @{$self->{CHILD}};
   Message("----> Quit for user='$self->{USER}' $$");
   exit(0);
}

##******************************************************************************
## Method childCreate  protected
##  Description  : the method for child creation and run
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub childCreate
{
   my ($self) = @_;
   my $line;
   my $request;

   ##===========================================================================
   ## Fork
   ##===========================================================================
   my $pid = fork();
   Die("fork : $!") if( !defined $pid );
   return $pid      if( $pid != 0 ); # parent process

   $SIG{TERM} = 'DEFAULT';
   $SIG{CHLD} = 'IGNORE';
   $SIG{USR1} = 'DEFAULT';
   $self->{CHILD} = ();

   ##===========================================================================
   ## Connection to the database
   ##===========================================================================
   if( $self->{ERROR} eq "" )
   {
      try
      {
         LogTrend::Visu::DataBaseManager->reConnectDatabases();
      }
      otherwise
      {
         $self->{ERROR} = "$_[0]<BR>\n$Utils_WarningMessage";
      }
      finally
      {
         $Utils_WarningMessage = "";
      };

   }

   ##===========================================================================
   ## For each connection
   ##===========================================================================
   ACCEPT: while( my $new_sock = $self->{SESSIONSOCKET}->accept() )
   {
      Message("SessionManager user='$self->{USER}' $$");
      kill 'USR1', getppid();

      ##-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
      ## Parsing the request
      ##-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
      $request = new LogTrend::Visu::Request($new_sock);
      if( ref($request) ne "LogTrend::Visu::Request" )
      {
         print $new_sock $self->htmlAnswer("Bad Request","$request");
         close $new_sock;
         next ACCEPT;
      }

      $request->wantedContent( $self->{WANTEDCONTENT} );

      ##-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
      ## If XML parsing error in xml/user/* files
      ##-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
      if( $self->{ERROR} ne "" )
      {
         print $new_sock $self->htmlAnswer("Server Error</B><BR><BR>$self->{ERROR}<B>");
         close $new_sock;
         close( $self->{SESSIONSOCKET} );
         exit( $CstQuitStatus );
      }

      ##-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
      ## Passing the request to the SystemsDeclaration
      ##-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
      my $response = $self->{SYSTEMSDECLARATION}->treat( $request );

      ##-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
      ## Responding
      ##-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
      print $new_sock $self->header( 200, $response->contentType(),
                                     $response->fileName(), $response->quit() );
      if( $response->contentType() eq "text/html" )
      {
         print $new_sock AccentsText2HTML( $response->getContent() );
      }
      else
      {
         print $new_sock $response->getContent();
      }
      close $new_sock;

      if( $response->quit() )
      {
         close( $self->{SESSIONSOCKET} );
         exit( $CstQuitStatus );
      }

   }

   exit(0); # Should never hapend

}

##******************************************************************************
## Method header  public
##  Description  : return XML standard en-tete
##******************************************************************************
sub header
{
   my ($self,$status,$contentType,$fileName,$quit) = @_;
   return "<Response Status=\"$status\" Content-Type=\"$contentType\" FileName=\"$fileName\" Quit=\"$quit\"/>\n";
}

##******************************************************************************
## Method htmlAnswer  public
##  Description  : return an HTML answer
##******************************************************************************
sub htmlAnswer
{
   my ($self,$message,$errno) = @_;
   my $result = $self->header( "200", "text/html", "", 1 );
   $result .= "<HTML>\n";
   $result .= "<HEAD><TITLE>Error</TITLE></HEAD>\n";
   $result .= "<META HTTP-EQUIV=\"pragma\" CONTENT=\"no-cache\">\n";
   $result .= "<BODY>\n";
   $result .= "<CENTER>\n";
   $result .= "<BR><HR WIDTH=\"50%\">\n";
   $result .= "<B>$message</B><BR>\n";
   $result .= "<FONT SIZE=\"-1\">($errno)</FONT><BR>\n" if( defined($errno) );
   $result .= "<HR WIDTH=\"50%\"><BR>\n";
   $result .= "<H2>-&gt; <A HREF=\"/$CstVisuName/\" TARGET=\"_top\">New connection</A> &lt;-</H2><BR>\n";
   $result .= "</CENTER>\n";
   $result .= "</BODY>\n";
   $result .= "</HTML>\n";
   return $result;

}

##******************************************************************************
1;
