#!/usr/bin/perl -w

#$Header: /home2/cvsroot/LogTrend/Visu/Web/ManagersCreator.pm,v 1.9 2002/01/08 12:42:56 lsimonneau Exp $
##******************************************************************************
## Class ManagersCreator
##  Description  : class to speak with apache and to creat SessionManager's
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Author       : Sylvain Lhullier s.lhullier@atrid.fr
##******************************************************************************
#$Log: ManagersCreator.pm,v $
#Revision 1.9  2002/01/08 12:42:56  lsimonneau
#Major bugfixes : Correct signature crash in daemon mode and use /usr/lib/sendmail -t -i
#to send mail insetead of MIME::Entity->smtpsend
#
#Revision 1.8  2001/12/05 16:32:36  slhullier
#Error management with try/otherwise ...
#
#Revision 1.7  2001/11/28 17:29:05  slhullier
#No more session ...
#
#Revision 1.6  2001/11/26 15:52:31  slhullier
#User inclusion for admin user seems to be ok
#
#Revision 1.5  2001/11/16 16:45:01  slhullier
#Text color format
#
#Revision 1.4  2001/11/15 16:08:57  slhullier
#UserNode managing first version (still bugs)
#
#Revision 1.3  2001/11/12 10:03:00  slhullier
#no more tab
#
#Revision 1.2  2001/10/29 14:06:03  slhullier
#Stable but non-realy working report via the web
#
#Revision 1.1  2001/10/18 15:22:58  slhullier
#Inheritance for report/web
#

package LogTrend::Visu::Web::ManagersCreator;

use strict;
use XML::DOM;
use Errno;
use Sys::Hostname;
use IO::Socket;
use POSIX qw(:sys_wait_h setsid);
use LogTrend::Common::LogDie;
use LogTrend::Common::Duration;
use LogTrend::Visu::Constants;
use LogTrend::Visu::Utils;
use LogTrend::Visu::Web::SessionManager;

my $g_name = $CstVisuName;

##******************************************************************************
## Constructor  public
##  Parameters   : the TCP port to run
##******************************************************************************
sub new
{
   my ($classname,$file) = @_;
   my $self = {};
   bless($self, $classname);

   ##===========================================================================
   ## Hash tables for socket management
   ##===========================================================================
   my %without_manager = ();  # user   => socket (""=>main_sock)
   my %with_manager = ();     # user   => socket
   my %fileno_socket = ();    # fileno => socket
   my %fileno_user = ();      # fileno => user
   my %pid_user = ();         # pid    => user
   $self->{WITHOUT_MANAGER} = \%without_manager;
   $self->{WITH_MANAGER}    = \%with_manager;
   $self->{FILENO_SOCKET}   = \%fileno_socket;
   $self->{FILENO_USER}     = \%fileno_user;
   $self->{PID_USER}        = \%pid_user;

   ##===========================================================================
   ## Tag 'Configuration'
   ##===========================================================================
   my ($list,$node,$attributes,$attrnode);
   my $parser = new XML::DOM::Parser() || Die($!);
   open( FILE, "$file" ) || Die("$file: $!");
   close(FILE);
   my $doc = $parser->parsefile( $file ) || Die("$file: $!");

   my $rootlist = $doc->getElementsByTagName("VisuConf") ||
                                         Die("$file: No \"VisuConf\" tag.");
   my $rootnode = $rootlist->item(0) || Die("$file: No \"VisuConf\" tag.");

   ##===========================================================================
   ## Tag 'Port'
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("Port") || Die("$file: No \"Port\" tag.");
   $node = $list->item(0) || Die("No \"Port\" tag.");
   $attributes = $node->getAttributes() || Die("$file: Error in \"Port\" tag.");

   ##---------------------------------------------------------------------------
   $attrnode = $attributes->getNamedItem("Nb") ||
                   Die("$file: No 'Nb' field in \"Port\" tag.");
   $self->{PORT} = $attrnode->getValue();

   ##===========================================================================
   ## Tag 'Images'
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("Images") || Die("$file: No \"Images\" tag.");
   $node = $list->item(0) || Die("No \"Images\" tag.");
   $attributes = $node->getAttributes() || Die("$file: Error in \"Images\" tag.");

   ##---------------------------------------------------------------------------
   $attrnode = $attributes->getNamedItem("Path") ||
                   Die("$file: No 'Path' field in \"Images\" tag.");
   $CstImagesPath = $attrnode->getValue();

   ##===========================================================================
   ## Tag 'UserConf'
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("UserConf") || Die("$file: No \"UserConf\" tag.");
   $node = $list->item(0);
   if( defined( $node ) )
   {
      $attributes = $node->getAttributes() || Die("$file: Error in \"UserConf\" tag.");
      $attrnode = $attributes->getNamedItem("Path") ||
                      Die("$file: No 'Path' field in \"UserConf\" tag.");
      $CstUsersPath = $attrnode->getValue();
   }

   ##===========================================================================
   ## Tag 'RGBFile'
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("RGBFile") || Die("$file: No \"RGBFile\" tag.");
   $node = $list->item(0);
   if( defined( $node ) )
   {
      $attributes = $node->getAttributes() || Die("$file: Error in \"RGBFile\" tag.");
      $attrnode = $attributes->getNamedItem("Path") ||
                      Die("$file: No 'Path' field in \"RGBFile\" tag.");
      $CstRGBPath = $attrnode->getValue();
   }

   ##===========================================================================
   ## Tag 'InactiveUser'
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("InactiveUser") || Die("$file: No \"InactiveUser\" tag.");
   $node = $list->item(0) || Die("No \"InactiveUser\" tag.");
   $attributes = $node->getAttributes() || Die("$file: Error in \"InactiveUser\" tag.");

   ##---------------------------------------------------------------------------
   $attrnode = $attributes->getNamedItem("TimeMax") ||
                   Die("$file: No 'TimeMax' field in \"InactiveUser\" tag.");
   $CstInactiveUserTimeMax = Duration( $attrnode->getValue() );

   ##===========================================================================
   ## Tag 'ProcessByUser'
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("ProcessByUser") || Die("$file: No \"ProcessByUser\" tag.");
   $node = $list->item(0) || Die("No \"ProcessByUser\" tag.");
   $attributes = $node->getAttributes() || Die("$file: Error in \"ProcessByUser\" tag.");

   ##---------------------------------------------------------------------------
   $attrnode = $attributes->getNamedItem("Nb") ||
                   Die("$file: No 'Nb' field in \"ProcessByUser\" tag.");
   $CstChildrenNb = $attrnode->getValue();

   ##===========================================================================
   ## Tag 'DefaultColors'
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("DefaultColors") || Die("$file: No \"DefaultColors\" tag.");
   $node = $list->item(0) || Die("No \"DefaultColors\" tag.");
   $attributes = $node->getAttributes() || Die("$file: Error in \"DefaultColors\" tag.");

   ##---------------------------------------------------------------------------
   foreach my $att ("BackGround", "CellGround", "Text", "Links", "VisitedLinks")
   {
      $attrnode = $attributes->getNamedItem("$att") ||
                      Die("$file: No '$att' field in \"DefaultColors\" tag.");
      $CstHTMLColors{"$att"} = TextToRGB( $attrnode->getValue() );
   }


   $doc = undef;
   $parser = undef;

   return $self;
}

##******************************************************************************
## Method daemon_mode  public
##  Description  : make the visu go to daemon mode : chdir to /, close files, fork ...
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub daemon_mode
{
   OpenLog( $g_name, "daemon" );
   chdir '/' || Die( "chdir / :$!" );
   open( STDIN, "</dev/null");
   open( STDOUT, ">/dev/null");
   open( STDERR, ">/dev/null");
   if( ( my $x = fork() ) > 0 )
   { exit(0); }
   elsif( $x == -1 )
   { Die("fork: $!"); }

   Die("setsid: $!") if( setsid() == -1 );

   open( FILE, ">/var/run/$g_name.pid" ) || return;
   printf FILE "$$\n";
   close FILE;
}

##******************************************************************************
## Method run  public
##  Description  : run the ManagersCreator
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub run
{
   my ($self) = @_;

   my @child = ();
   $self->{CHILD} = \@child;
   $SIG{TERM} = sub { $self->quit(); };
   $SIG{INT}  = sub { $self->quit(); };

##    $SIG{__DIE__}  = sub { LogTrend::Common::LogDie::Die($_[0]) }; ##

   ##===========================================================================
   my $main_sock = new IO::Socket::INET( LocalHost => hostname(),
                          LocalPort => $self->{PORT}, Listen => 5,
                          Proto => 'tcp', Reuse => 1) || Die("IO::Socket::INET: $!");
   Message("Ok, waiting connections on port $self->{PORT}");
   $self->{WITHOUT_MANAGER}->{ "" } = $main_sock;
   $self->{FILENO_SOCKET}->{ $main_sock->fileno() } = $main_sock;
   $self->{FILENO_USER}->{ $main_sock->fileno() } = "";
   
   ##===========================================================================
   ## For ever ...
   ##===========================================================================
   $SIG{CHLD} = sub {$self->childSignal();};
   EVER: while( 1 )
   {
      ##------------------------------------------------------------------------
      ## Preparing select
      ##------------------------------------------------------------------------
      my $bits = "";
      my $maxFileno = 0;
      foreach my $s (values %{$self->{WITHOUT_MANAGER}})
      {
         my $fn = $s->fileno();
         $maxFileno = $fn if( $fn > $maxFileno );
         vec( $bits, $fn, 1 ) = 1;
      }
      ##------------------------------------------------------------------------
      ## Selecting socket with no manager & main_socket
      ##------------------------------------------------------------------------
      my ($rout,$wout,$eout);
      $! = 0;
      my ($nfound,$timeleft) = select( $rout=$bits, $wout=$bits, $eout=$bits, undef );

      # If sigChild : re-prepare select
      next EVER if( $!{EINTR} ); # Interrupted function call

      ##------------------------------------------------------------------------
      ## Which socket ?
      ##------------------------------------------------------------------------
      foreach my $n (0..$maxFileno)
      {
         if( vec( $rout, $n, 1 ) )
         {
            my $user = $self->{FILENO_USER}->{$n};
            if( $user eq "" )  # $main_sock
            {
               $self->Initial_Connection( $self->{FILENO_SOCKET}->{$n} );

            }
            else  # a socket for an user whithout manager
            {
               $self->CreateSessionManager( $user );
            }
         }
      }
   }

}

##******************************************************************************
## Method Initial_Connection  protected
##  Description  : Init the connection from main socket
##  Parameters   : 
##  Return value : 
##******************************************************************************
sub Initial_Connection
{
   my ($self,$main_sock) = @_;
   my $actual_port = $self->{PORT} + 1;
   my $userSocket;

   ##===========================================================================
   ## new connection (must be ok, because selected) :
   ##===========================================================================
   my $new_sock = $main_sock->accept();

   ##--------------------------------------------------------------------------
   my $line = <$new_sock> || next ACCEPT;
   my ($user, $remoteHost);
   if( $line =~ /<NewConnection User="(.*)" RemoteHost="(.*)"\/>/ )
   {
      $user = $1;
      $remoteHost = $2;
   }
   else
   {
      print $new_sock "<Error Text=\"bad format\"/>\n";
      close $new_sock;
      next ACCEPT;
   }

   ##--------------------------------------------------------------------------
   Message("NewConnection from '$remoteHost' for user='$user'");

   my $dir = "$CstUsersPath/$user";
   if( ! -d "$dir" )
   {
      print $new_sock "<Error Text=\"No such user '$user'\"/>\n";
      close $new_sock;
      next ACCEPT;
   }

   ##------------------------------------------------------------------------
   while( !defined( $userSocket = new IO::Socket::INET( LocalHost => hostname(),
                               LocalPort => $actual_port,  Listen    => 10,
                               Proto     =>'tcp',          Reuse     => 1 ) ) &&
          ( $!{EADDRINUSE} || $!{EBADF} ) ) # Address already in use
                                            # EBADF perl5.005  EADDRINUSE perl5.6
   {
      $actual_port++;
      $actual_port = 1025 if( $actual_port == 65536);
   }

   if( !defined( $userSocket ) )
   {
      print $new_sock "<Error Text=\"\$userSocket not defined $actual_port : $!\"/>\n";
      close $new_sock;
      next ACCEPT;
   }

   $self->{WITHOUT_MANAGER}->{$user} = $userSocket;
   $self->{FILENO_SOCKET}->{$userSocket->fileno()} = $userSocket;
   $self->{FILENO_USER}->{$userSocket->fileno()} = $user;

   print $new_sock "<Manager Port=\"$actual_port\"/>\n";
   close $new_sock;

   ##------------------------------------------------------------------------
   $self->CreateSessionManager( $user );

}

##******************************************************************************
## Method CreateSessionManager  public
##  Description  : Create a SessionManager
##  Parameters   : the user
##  Return value : 
##******************************************************************************
sub CreateSessionManager
{
   my ($self,$user) = @_;
   my $socket = $self->{WITHOUT_MANAGER}->{$user};

   my $pid = fork(); 
   if( !defined $pid )
   {
      Die("fork : $!");
   }
   if( $pid != 0 ) # parent process
   {
      $self->{PID_USER}->{$pid} = $user;
      $self->{WITH_MANAGER}->{$user} = $self->{WITHOUT_MANAGER}->{$user};
      delete $self->{WITHOUT_MANAGER}->{$user};

      unshift @{$self->{CHILD}}, $pid;
   }
   else   # child process
   {
      $SIG{INT}  = 'DEFAULT';
      foreach my $s (values %{$self->{WITHOUT_MANAGER}})
      {
         $s->close() if( $s ne $socket );
      }
      foreach my $s (values %{$self->{WITH_MANAGER}})
      {
         $s->close();
      }

      my $sm = new LogTrend::Visu::Web::SessionManager( $user, $socket );
      $sm->run();
      exit(0); #should never happend
   }

}

##******************************************************************************
## Method childSignal  protected
##  Description  : manage the reception of a SIGCHLD
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub childSignal
{
   my ($self) = @_;
   my $pid;

   do
   {
      $pid = waitpid(-1,&WNOHANG);
      if( $pid !=  0 and
          $pid != -1 )
      {
         my $user = $self->{PID_USER}->{$pid};
         $self->{WITHOUT_MANAGER}->{$user} = $self->{WITH_MANAGER}->{$user};
         delete $self->{WITH_MANAGER}->{$user};
         delete $self->{PID_USER}->{$pid};

         @{$self->{CHILD}} = grep !/^$pid$/, @{$self->{CHILD}};
      }
   } until( $pid ==  0 or
            $pid == -1);

}

##******************************************************************************
## Method quit  public
##******************************************************************************
sub quit
{
   my ($self) = @_;
   $SIG{CHLD} = 'IGNORE';
   kill 'TERM', @{$self->{CHILD}};
   Message("Bye ...");
   exit(0);
}

##******************************************************************************
1;
