#!/usr/bin/perl -w

#$Header: /home2/cvsroot/LogTrend/Agent.pm,v 1.25.2.1 2002/02/14 15:43:30 lsimonneau Exp $
##******************************************************************************
## Class Agent  virtual
##  Description  : class of generic agent
##                 'Real' agents must inherit from this class and implements
##                 CreateAgentDescription and CollectData methods
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Author       : Laurent Simonneau l.simonneau@atrid.fr
##  Author       : Sylvain Lhullier s.lhullier@atrid.fr
##  Author       : David Mascle d.mascle@atrid.fr
##******************************************************************************
#$Log: Agent.pm,v $
#Revision 1.25.2.1  2002/02/14 15:43:30  lsimonneau
#Minor features enhancement
#Add a Sender attribut in the Mail tag of agents config file.
#Use SMTP Server when available or /usr/lib/sendmail otherwise.
#Add the -s <smtp_server> option to MailBridge.pl.
#
#Revision 1.25  2002/02/06 13:06:39  lsimonneau
#Minor bugfixes.
#
#Revision 1.24  2002/02/06 08:14:01  lsimonneau
#Error report messages.
#
#Revision 1.23  2002/01/16 14:42:45  slhullier
#sendmail -f$self->{ADMINMAIL}
#
#Revision 1.22  2002/01/08 12:42:56  lsimonneau
#Major bugfixes : Correct signature crash in daemon mode and use /usr/lib/sendmail -t -i
#to send mail insetead of MIME::Entity->smtpsend
#
#Revision 1.21  2001/12/10 17:37:28  lsimonneau
#Minor bugfixes.
#
#Revision 1.20  2001/12/10 16:58:34  lsimonneau
#Use GnuPG::Interface instead of Crypt::RSA for perl 5.005 compatibility.
#
#Revision 1.19  2001/11/20 16:28:30  lsimonneau
#Change use Getopt::Long for more compatibility with standard debian potato packages.
#
#Revision 1.18  2001/11/16 10:32:28  lsimonneau
#Major feature enhancement : Implementation of authentification with RSA signatu
#re.
#
#IMPORTANT : Agents are incompatible with older version.
#
#Revision 1.17  2001/11/02 13:55:42  lsimonneau
#Add Action support for agents' alarms.
#Add action support for LinuxAgent Process alarms.
#
#Revision 1.16  2001/10/31 17:33:31  lsimonneau
#*** empty log message ***
#
#Revision 1.15  2001/10/31 16:59:07  lsimonneau
#Before_Exit_In_Time_Interval became Before_Warn_In_Time_Interval.
#Before_Exit is kept for backward compatibility, but is deprecated.
#
#Revision 1.14  2001/10/31 15:04:05  lsimonneau
#Minor bugfixes.
#
#Revision 1.13  2001/10/15 14:41:24  lsimonneau
#Implementation of a not sent data queue.
#When StorageServer is down or smtp server not respond, data is queued.
#The queue is emptied when connection can be established.
#If the agent is stopped (term int die ...) data is sent.
#If data can't be sent, a mail with queue file is sent to agent admin.
#
#Revision 1.12  2001/10/05 11:55:08  slhullier
#in the other case -> otherwise
#
#Revision 1.11  2001/10/02 16:00:45  slhullier
#Documentation updated
#
#Revision 1.10  2001/09/27 16:15:23  slhullier
#Config files in /etc/LogTrend
#
#Revision 1.9  2001/09/20 09:40:55  slhullier
#POD
#
#Revision 1.8  2001/09/19 12:50:14  slhullier
#Before_Exit_If_Serveur_Not_Responding -> Before_Exit_If_Server_Not_Responding
#
#Revision 1.7  2001/09/18 15:12:15  slhullier
#POD creation/modification
#
#Revision 1.6  2001/09/18 11:10:51  slhullier
#_TIME_BETWEEN_COLLECTIONS : 2 seconds min
#
#Revision 1.5  2001/08/31 16:01:54  lsimonneau
#Envoie les donnes au serveur de stockage sur SIGURS1.
#
#Revision 1.4  2001/08/03 13:46:02  fdesar
#Predicate mathod IsRunning() added.
#
#Revision 1.3  2001/07/24 14:38:41  slhullier
#
#Scripting for Visu
#
#Revision 1.2  2001/07/23 13:28:34  fdesar
#- Command line options handled by Getopt::Long,
#  ``a la POSIX''
#- New methode invocation ParseAgentOptions which
#  by default exits but can be overloaded to enable
#  agents to get extra-options ie:
#
#  ShadokAgent -r -Daemon -- --pump up
#
#Revision 1.1  2001/07/19 16:40:27  fdesar
#
#Moved module files to the right directories
#Updated package names and uses to reflect those changes
#Corrected bug in SnortAgent.pm for negating first value in SID parsing
#
#Revision 1.34  2001/07/10 12:54:27  slhullier
#
#Correction bug fichier tmp
#
#Revision 1.33  2001/06/21 09:10:59  slhullier
#
#Encore du GMT (cette fois juste pour l'ActivationDate)
#
#Revision 1.32  2001/06/08 07:42:12  slhullier
#
#Maj __WARN__ et __DIE__
#
#Revision 1.31  2001/06/08 07:39:12  slhullier
#
#Ajout de SendDie
#
#Revision 1.30  2001/06/07 14:08:54  slhullier
#
#Passage de  unshift @INC,'..';  aux packages Logtrend::....
#
#Revision 1.29  2001/06/01 12:17:59  fdesar
#
#Corrected writing in a xml file from bags
#
#Revision 1.28  2001/06/01 08:19:05  fdesar
#
#No collection if empty collection. No Send() if no collection.
#
#Revision 1.27  2001/05/29 12:43:30  fdesar
#
#Correction typo : deamon -> daemon
#
#Revision 1.26  2001/04/20 07:13:31  slhullier
#
#Premier essais (concluant) de stockage des messages sur disque
#
#Revision 1.25  2001/04/10 15:43:00  slhullier
#
#Repository pour MailBridge en cours
#
#Revision 1.24  2001/03/27 16:31:33  slhullier
#
#Mail envoye avant arret de l'agent
#
#Revision 1.23  2001/03/27 12:17:56  slhullier
#
#LinuxAgent : description par type de FS
#
#Revision 1.22  2001/03/26 09:00:10  slhullier
#
#MailBridge : XML et gzip opperationnel, mis en place spa/serveur
#
#Revision 1.21  2001/03/23 17:14:34  slhullier
#
#MailBridge: todo en XML, donnees gzip
#Pas fini : gunzip description
#
#Revision 1.20  2001/03/22 13:32:24  slhullier
#
#Petit bug du serveur du au parseur
#
#Revision 1.19  2001/03/19 10:57:58  slhullier
#
#Agents : fichier de conf par defaut dans /etc   +  mise en commun
#des routines de gestion des arguments dans Agent.pm
#
#Revision 1.18  2001/03/14 09:10:46  slhullier
#
#Die moins stupides
#
#Revision 1.17  2001/03/13 09:10:23  slhullier
#
#LogDie sur SimpleAgent + version&number dans Configuration.xml
#
#Revision 1.16  2001/03/12 13:35:37  slhullier
#
#Ajout du champ lastconnectiondate dans la table agents
#
#Revision 1.15  2001/03/12 10:22:25  slhullier
#
#Modification de la methode Run pour une meilleur gestion des temps
#
#Revision 1.14  2001/03/09 16:16:23  slhullier
#
#Les seuils ne sont plus dans linuxagentdescription.xml
#mais relus dans Configuration.xml
#Gestion de la non reponse du serveur
#
#Revision 1.13  2001/03/08 17:33:54  slhullier
#
#L'agent ne meurt plus des que le serveur ne repond plus.
#
#Revision 1.12  2001/03/07 10:34:03  slhullier
#
#Mise en place de LogDie dans l'agent Linux
#
#Revision 1.11  2001/02/23 15:55:37  slhullier
#
#Ajout de tout pleins de fonctionnalites pour le LinuxAgent :
#filesystems, services (HTTP,FTP)
#Mise en place sur spa, tests en grandeur nature en cours.
#
#Revision 1.10  2001/02/22 17:02:23  slhullier
#
#Gestion des systemes de fichiers par LinuxAgent.
#Premiers tests concluants, reste a ajouter des alarmes et a faire
#des tests en situation reelle.
#
#Revision 1.9  2001/02/21 08:53:24  slhullier
#
#Reformatage
#
#Revision 1.8  2001/02/20 17:01:26  slhullier
#
#Mise en commentaire (+tab etc)
#
#Revision 1.7  2001/02/20 15:50:20  slhullier
#
#Apres modification du nom de certaines classes
#
#Revision 1.6  2001/02/20 10:21:55  slhullier
#
#Passage de la fonction de parsage de config XML dans Agent.pm
#Creation des tags Generic et Specific
#Fichiers agentdescription.xml plus lisibles (avec \n)
#Supression de methodes pas appellee du fichier Agent.pm
#
#Revision 1.5  2001/02/08 15:21:52  slhullier
#
#
#Mail avec attachement : cote Agent fait.
#
#Revision 1.4  2001/02/07 14:36:33  slhullier
#
#
#Le systeme MailBridge fonctionne.
#Mise en service sur spa.
#
#Revision 1.3  2001/02/07 10:46:25  slhullier
#
#
#Premier essai concluant pour l'envoi via TCP en utilisant
#une classe pour cela.
#
#Revision 1.2  2001/02/06 17:16:10  slhullier
#
#
#Premiers essais de mail pour l'envoi de donnees
#
#Revision 1.1.1.1  2001/02/06 09:48:27  fdubuy
#
#First CVS version : all work done by D.Mascle
#

package LogTrend::Agent;

use strict;
use IO::Socket;
use POSIX qw(setsid);
use XML::DOM;
use MIME::Entity;
use Net::SMTP;
use Getopt::Long;
use File::stat;

use LogTrend::Agent::DataAlarmsBag;
use LogTrend::Agent::DataAlarmsSet;
use LogTrend::Agent::DataDescriptionSet;
use LogTrend::Agent::AlarmDescriptionSet;
use LogTrend::Agent::TcpDataSender;
use LogTrend::Agent::MailDataSender;
use LogTrend::Common::LogDie;
use LogTrend::Common::Duration;
use LogTrend::Crypto::Signature qw(sign_message);

my $g_name;

##******************************************************************************
## Method new  public
##  Description  : creat a new Agent
##  Parameters   : the name & version of the agent
##******************************************************************************
sub new
{
   my ($classname,$name,$version) = @_;
   my $self = {};
   my($opts) = {};
   bless($self, $classname);

   $g_name = $name;
   my $agentinformation = "$name version $version";
   my $agentconfigurationfile;
   my $agentstate = 0;
   my $daemon = 0;


   ##===========================================================================
   ## Arguments
   ##===========================================================================
   Getopt::Long::Configure("bundling", "no_ignore_case");
 
       GetOptions($opts, 'run|r',
                         'Daemon|D',
                         'description|d',
                         'version|v',
                         'help|h',
                         'configuration|c=s')
   or  $self->display_help_and_exit(1);

       keys(%$opts) > 1
   and (    exists($opts->{help})
        or  exists($opts->{version}))
   and $self->display_help_and_exit(1);

       exists($opts->{help})
   and $self->display_help_and_exit(0);

       exists($opts->{version})
   and do {
       print $agentinformation, "\n";
       exit( 0 );
   };

       exists($opts->{configuration})
   and $agentconfigurationfile = $opts->{configuration};

       exists($opts->{description})
   and $agentstate = 1;

       exists($opts->{run})
   and do {
           $agentstate != 0
       and $self->display_help_and_exit(1);
       $agentstate = 2;
   };

       exists($opts->{Daemon})
   and do {
           $agentstate == 2
       or  $self->display_help_and_exit(1);
       ++$daemon;
   };

       $agentstate
   or  $self->display_help_and_exit(1);

       $agentconfigurationfile
   or  $agentconfigurationfile = "/etc/LogTrend/$g_name.conf";

       @ARGV
   and $self->ParseAgentOptions();

       $daemon
   and do {
      $self->daemon_mode();
      $SIG{__WARN__} = sub { LogTrend::Common::LogDie::SysLog($_[0]) };
      $SIG{__DIE__}  = sub { $self->SendDie(@_)};
   };

   ##===========================================================================
   ## Settings
   ##===========================================================================
   $ENV{PATH} = "/bin:/sbin:/usr/bin:/usr/sbin:/usr/local/bin:/usr/local/sbin";

   $self->{_AGENTSTATE} = $agentstate;
   $self->{_INFORMATION} = $agentinformation;
   $self->{_DATADESCRIPTIONSET} = LogTrend::Agent::DataDescriptionSet->new();
   $self->{_ALARMDESCRIPTIONSET} = LogTrend::Agent::AlarmDescriptionSet->new();
   $self->{_DATAALARMSBAG} = LogTrend::Agent::DataAlarmsBag->new();

   $self->ParseXMLConfigFile( $agentconfigurationfile, $agentstate );

   if( $self->{_SENDTOSTORAGESERVER} == 1 )
   {
      $self->{_DATASENDER} =
         LogTrend::Agent::TcpDataSender->new( $self->{_HOST}, $self->{_PORT});
   }
   elsif( $self->{_SENDTOSTORAGESERVER} == 2 )
   {
      $self->{_DATASENDER} =
         LogTrend::Agent::MailDataSender->new( $self->{_HOST}, $self->{_PORT}, $self->{_MAILFORBRIDGE},
                              $self->{_MAIL_SMTP}, $self->{_MAIL_ADMIN}, $self->{_MAIL_SENDER} );
   }

   ##===========================================================================
   ## Connect Send to SIGUSR1
   ##===========================================================================
   $SIG{TERM} = sub {$self->SendDie("Explicit Term signal. Sent remaining data to server.");};
   $SIG{INT} = sub {$self->SendDie("Explicit Int signal. Sent remaining data to server.");};
	
   ##===========================================================================
   ## Let's run
   ##===========================================================================
   $self->Run();
   Die( $! );   # Should never append !

}

##******************************************************************************
## Method display_help_and_exit  protected
##  Description  : display the help message and exit
##  Parameters   : the exit status
##******************************************************************************
sub display_help_and_exit
{
   my $status = shift;

   print "\n",
         "Usage: $g_name --run|-r [--Daemon|-D] [--configuration|c filename] [-- arguments specific to agent ]\n",
         "       $g_name --description|-d [--configuration|c filename] [-- arguments specific to agent ] \n",
         "       $g_name --help|h\n",
         "       $g_name --version\n",
         "Options :\n",
         "   -r : normal mode, optional agent's configuration filename.\n",
         "   -D : run as daemon (detach form terminal, syslog...).\n",
         "   -d : description generation mode, optional agent's configuration filename.\n",
         "Default configuration file is /etc/LogTrend/$g_name.conf\n",
         "\n";
   exit( $status );
}

##******************************************************************************
## Method daemon_mode  protected
##  Description  : make agent go to daemon mode : chdir to /, close files, fork ...
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub daemon_mode
{
   OpenLog( $g_name, "daemon" );
   chdir '/' || Die( "chdir / :$!" );
   open( STDIN, "</dev/null");
   open( STDOUT, ">/dev/null");
   open( STDERR, ">/dev/null");
   if( ( my $x = fork() ) > 0 )
   { exit(0); }
   elsif( $x == -1 )
   { Die("fork: $!"); }

   Die("setsid: $!") if( setsid() == -1 );

   open( FILE, ">/var/run/$g_name.pid" ) || return;
   printf FILE "$$\n";
   close FILE;
}

##******************************************************************************
## Method ParseAgentOptions  protected
##  Description  : parses command line switches specific to Agent
##  Parameters   : none
##  Return value : exit with error status 1 (must be overloaded)
##******************************************************************************
sub ParseAgentOptions {
    print "\n";
    for my $opt (@ARGV) {
        print "$g_name: unknown option: $opt \n";
    }
    print "\n";
    exit(1);
}

##******************************************************************************
## Method ParseXMLConfigFile  protected
##  Description  : parses the XML config file into 'Generic' tag
##  Parameters   : the file name to parse, the state of the agent
##  Return value : none
##******************************************************************************
sub ParseXMLConfigFile
{
   my $self = shift;
   my $file = shift;
   my $agentstate = shift;
   my ($list,$node,$valueref,$attributes,$attrnode);

   ##===========================================================================
   ## Tag 'Configuration'
   ##===========================================================================
   my $parser = new XML::DOM::Parser() || Die($!);
   open( FILE, "$file" ) || Die("$file: $!");
   close(FILE);
   my $doc = $parser->parsefile( $file ) || Die("$file: $!");

   my $rootlist = $doc->getElementsByTagName("Configuration") ||
                                         Die("$file: No \"Configuration\" tag.");
   my $rootnode = $rootlist->item(0) || Die("$file: No \"Configuration\" tag.");

   ##===========================================================================
   ## Tag 'Generic'
   ##===========================================================================
   $rootlist = $rootnode->getElementsByTagName("Generic") ||
                                        Die("$file: No \"Generic\" tag.");
   $rootnode = $rootlist->item(0) || Die("$file: No \"Generic\" tag.");

   ##===========================================================================
   ## Tag 'AgentDescriptionFile'
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("AgentDescriptionFile") ||
                                        Die("$file: No \"AgentDescriptionFile\" tag.");
   $node = $list->item(0) || Die("$file: No \"AgentDescriptionFile\" tag.");
   $valueref = $node->getFirstChild() || Die("$file: No \"AgentDescriptionFile\" tag.");

   $self->{_AGENTDESCRIPTIONFILE} = $valueref->getNodeValue();

   ##===========================================================================
   ## Tag 'Source'
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("Source") || Die("$file: No \"Source\" tag.");
   $node = $list->item(0) || Die("$file: No \"Source\" tag.");
   $valueref = $node->getFirstChild() || Die("$file: No \"Source\" tag.");

   $self->{_LOGIN} = $self->{_SOURCE} = $valueref->getNodeValue();

   ##===========================================================================
   ## Tag 'Agent'
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("Agent") || Die("$file: No \"Agent\" tag.");
   $node = $list->item(0) || Die("No \"Agent\" tag.");
   $attributes = $node->getAttributes() || Die("$file: Error in \"Agent\" tag.");

   ##---------------------------------------------------------------------------
   $attrnode = $attributes->getNamedItem("Number") ||
                   Die("$file: No 'Number' field in \"Agent\" tag.");
   $self->{_NUMBER} = $attrnode->getValue();

   ##---------------------------------------------------------------------------
   $attrnode = $attributes->getNamedItem("Version") ||
             Die("$file: No 'Version' field in \"Agent\" tag.");
   $self->{_VERSION} = $attrnode->getValue();

   ##===========================================================================
   ## Tag 'Time'
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("Time") || Die("$file: No \"Time\" tag.");
   $node = $list->item(0) || Die("No \"Time\" tag.");
   $attributes = $node->getAttributes() || Die("$file: Error in \"Time\" tag.");

   ##---------------------------------------------------------------------------
   $attrnode = $attributes->getNamedItem("Between_Collections") ||
                   Die("$file: No 'Between_Collections' field in \"Time\" tag.");
   $self->{_TIME_BETWEEN_COLLECTIONS} = Duration( $attrnode->getValue() );

   ##---------------------------------------------------------------------------
   $attrnode = $attributes->getNamedItem("Between_Deliveries") ||
             Die("$file: No 'Between_Deliveries' field in \"Time\" tag.");
   $self->{_TIME_BETWEEN_DELIVERIES} = Duration( $attrnode->getValue() );

   ##---------------------------------------------------------------------------
   $attrnode = $attributes->getNamedItem("Before_Warn_If_Server_Not_Responding") or
       ($attrnode = $attributes->getNamedItem("Before_Exit_If_Server_Not_Responding") 
	and warn "Before_Exit_If_Server_Not_Responding is deprecated, use Before_Warn_If_Server_Not_Responding instead.") or
	    Die("$file: No 'Before_Warn_If_Server_Not_Responding' field in \"Time\" tag.");
   $self->{_TIME_SERVEUR_NOT_RESPONDING} = Duration( $attrnode->getValue() );
   $self->{_TIME_SERVEUR_NOT_RESPONDING_S} = $attrnode->getValue();

   ##===========================================================================
   ## Tag 'Mail'
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("Mail") || Die("$file: No \"Mail\" tag.");
   $node = $list->item(0) || Die("No \"Mail\" tag.");
   $attributes = $node->getAttributes() || Die("$file: Error in \"Mail\" tag.");

   ##---------------------------------------------------------------------------
   $attrnode = $attributes->getNamedItem("SMTP") ||
                   Die("$file: No 'SMTP' field in \"Mail\" tag.");
   $self->{_MAIL_SMTP} = $attrnode->getValue();

   ##---------------------------------------------------------------------------
   $attrnode = $attributes->getNamedItem("Admin") ||
                   Die("$file: No 'Admin' field in \"Mail\" tag.");
   $self->{_MAIL_ADMIN} = $attrnode->getValue();

   ##---------------------------------------------------------------------------
   if($attrnode = $attributes->getNamedItem("Sender")) {
       $self->{_MAIL_SENDER} = $attrnode->getValue();
   }
   else {
       $self->{_MAIL_SENDER} = $self->{_MAIL_ADMIN};
   }

   ##===========================================================================
   ## Tag 'DataFuture'
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("DataFuture") ||
                                        Die("$file: No \"DataFuture\" tag.");
   $rootnode = $list->item(0) || Die("$file: No \"DataFuture\" tag.");

   ##---------------------------------------------------------------------------
   ## Tag Save
   ##---------------------------------------------------------------------------
   $list = $rootnode->getElementsByTagName("Save") || Die("$file: $!");
   $node = $list->item(0);
   if( defined($node) )
   {
      $self->{_SENDTOSTORAGESERVER} = 0; # eg Save
      $self->{_HOST} = $self->{_PORT} = $self->{_PASSWORD} =
                                                    $self->{_MAILFORBRIDGE} = "";
      $attributes = $node->getAttributes() || Die("$file: Error in \"Save\" tag.");

      ##........................................................................
      $attrnode = $attributes->getNamedItem("FileName") ||
                                    Die("$file: No 'FileName' field in \"Save\" tag.");
      $self->{_FILENAME} = $attrnode->getValue();

      ##........................................................................
      $list = $rootnode->getElementsByTagName("Send")->item(0) &&
                        Die("$file: 'Save' and 'Send' tags in same 'DataFuture' tag.");

      $parser = undef;
      return;
   }

   ##---------------------------------------------------------------------------
   ## Tag Send
   ##---------------------------------------------------------------------------
   $list = $rootnode->getElementsByTagName("Send") || Die("$file: $!");
   $node = $list->item(0);
   if( defined($node) )
   {
      $self->{_SENDTOSTORAGESERVER} = 1;
      $self->{_FILENAME} = "/var/cache/LogTrend/logtrend-agent-cache-$$.xml.gz";
      $attributes = $node->getAttributes() || Die("$file: Error in \"Send\" tag.");

      ##........................................................................
      $attrnode = $attributes->getNamedItem("Host") ||
                                        Die("$file: No 'Host' field in \"Send\" tag.");
      $self->{_HOST} = $attrnode->getValue();

      $attrnode = $attributes->getNamedItem("Port") ||
                                        Die("$file: No 'Port' field in \"Send\" tag.");
      $self->{_PORT} = $attrnode->getValue();

      $attributes->getNamedItem("Password") and
	  warn ("Password attributes for Send tag is deprecated.");

      if($attrnode = $attributes->getNamedItem("GPGHome")) {
          $self->{_GPG_HOME} = $attrnode->getValue();
      }
      else {
          $self->{_GPG_HOME} = "/etc/LogTrend/.gnupg";
      }

      if(-e $self->{_GPG_HOME}) {
	  my $mode = stat($self->{_GPG_HOME})->mode;

	  if($mode & 0004) {
	      warn "Warning !!!! The private key is world readable !!!";
	  }
	  if($mode & 0002) {
	      warn "Warning !!!! The private key is world writable !!!";
	  }

	  if(! ($mode & 0400)) {
	      Die("Can't access to private key file : $self->{_GPG_HOME}");
	  }
      }
      else {
	  Die("Can't found private key file : $self->{_GPG_HOME}");
      }
      
      ##........................................................................
      $attrnode = $attributes->getNamedItem("MailForBridge");
      if( defined( $attrnode ) )
      {
         $self->{_MAILFORBRIDGE} = $attrnode->getValue();
         $self->{_SENDTOSTORAGESERVER} = 2;
      }
      else
      {
         $self->{_MAILFORBRIDGE} = "";
      }

      $parser = undef;
      return;
   }

   ##---------------------------------------------------------------------------
   Die("$file: None of 'Save' or 'Send' tag in same 'DataFuture' tag.");

}


##******************************************************************************
## Method CreateAgentDescription  virtual public
##  Description  : creates an agent's description
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub CreateAgentDescription
{
}

##******************************************************************************
## Method CollectData  virtual public
##  Description  : collects data and alarms
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub CollectData
{
}

##******************************************************************************
## Method Run  public
##  Description  : suites for all the work that an agent must do
##  Parameters   : none
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
sub Run
{
   my $self = shift;
   my $result = 0;

   ##===========================================================================
   ## agent's description file generation state
   ##===========================================================================
   if( $self->{_AGENTSTATE} == 1 )
   {
      if ($self->{_AGENTDESCRIPTIONFILE} eq "")
      { Die("invalid agent's description filename"); }

      $self->CreateAgentDescription();
      $self->{_ACTIVATIONDATE} = gmtime(time())." GMT";

      my $agentdescriptionfile = $self->{_AGENTDESCRIPTIONFILE};
      open(ADF, ">$agentdescriptionfile" ) || Die("open $agentdescriptionfile: $!");
      print ADF $self->GetXmlDescription();
      close(ADF);

      exit(0); # All is correct !
   }
   ##===========================================================================
   ## agent's collect state
   ##===========================================================================
   elsif( $self->{_AGENTSTATE} == 2 )
   {
      $self->ParseAgentDescriptionFile() || Die("reading agent description file");

      my $nb_collections = 0;
      my $last_delivery_date = time();
      my $delay = 0;
      my $sleep_time;

      $self->{_DATAALARMSSET} = LogTrend::Agent::DataAlarmsSet->new( $self->{_SOURCE}, $self->{_NUMBER},
                                 $self->{_VERSION}, $self->{_ACTIVATIONDATE},
                                 $self->{_DATADESCRIPTIONSET},
                                 $self->{_ALARMDESCRIPTIONSET} );

      while( 1 )
      {
         ##---------------------------------------------------------------------
         ## Collect
         ##---------------------------------------------------------------------

         $self->{_DATAALARMSSET}->Empty();
         $self->{_DATAALARMSSET}->NewDate();
         $self->CollectData();
         if(not $self->{_DATAALARMSSET}->IsEmpty())
         {
            $self->{_DATAALARMSBAG}->AddDataAlarmsSet( $self->{_DATAALARMSSET} );
            $nb_collections++;
         }

         ##---------------------------------------------------------------------
         ## Send ?
         ##---------------------------------------------------------------------
         my $now = time();
         $delay = 0;
         if( $last_delivery_date + $self->{_TIME_BETWEEN_DELIVERIES} <= $now )
         {
            $last_delivery_date = $now;
            if( $nb_collections > 0 and $self->Send() == 1 )
            { $nb_collections = 0; }
            $delay = time() - $now;
         }

         ##---------------------------------------------------------------------
         ## Wait
         ##---------------------------------------------------------------------
         $sleep_time = $self->{_TIME_BETWEEN_COLLECTIONS} - $delay;
         if( $sleep_time < 2 ) { $sleep_time = 2; }
         sleep( $sleep_time );
      }
   }
   ##===========================================================================
   ## unknown state
   ##===========================================================================
   else
   {
      Die("agent unknown state");
   }

   return $result;
}

##******************************************************************************
## Method IsRunning  public
##  Description  : In collection mode predicate method
##  Parameters   : none
##  Return value : 1 if in collection state, 0 otherwise
##******************************************************************************
sub IsRunning
{
    my($self) = shift;

    $self->{_AGENTSTATE} == 2
}

##******************************************************************************
## Method Send  private
##  Description  : Send data to serveur or save them into file
##  Parameters   : none
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
my $last_serveur_error_date = undef;
sub Send
{
   my $self = shift;

   if ($self->{_SENDTOSTORAGESERVER} == 0)
   {
       $self->AppendToFile() || Die($self->{_FILENAME}." $!");
       $self->{_DATAALARMSSETLIST} = undef;

       return 1;
   }

   my $xmldata = $self->{_DATAALARMSBAG}->ConvertToXml();
   
   # Signing data 

   my $signature = sign_message($self->{_GPG_HOME}, $xmldata);

   my $response;
   if($xmldata ne "") {
       $response = $self->{_DATASENDER}->SendData( $xmldata, $signature );
   }
   else {
       $response = "OK";
   }

   if( $response eq "OK" )
   {			
       $self->{_DATAALARMSSETLIST} = undef;
       $last_serveur_error_date = undef;
       
       ## Try to send cache file.
       if(-e $self->{_FILENAME} and
	  open(CFD, "gunzip -c $self->{_FILENAME}|")) {
	   $xmldata = "";
	   while(<CFD>) { $xmldata .= $_ }

	   if($xmldata ne "") {               
	       $xmldata = "<ADAT>$xmldata</ADAT>";
               $signature = sign_message($self->{_GPG_HOME}, $xmldata);
	       $response = $self->{_DATASENDER}->SendData($xmldata, $signature);
	   }
	   
	   close(CFD);
	   unlink($self->{_FILENAME});
       }
   }
   
   if( $response eq "OK" )
   {			
       return 1;
   }
   
   if($response !~ /^ERROR/) {
       $response = "ERROR PROTOCOL Unknown response \"$response\"";
   }
   
   
   my ($error_id, $error_message) = $response =~ /^ERROR\s+(\w+)(\s+.+)?$/sm;
   $error_message =~ s/^\s*// if  defined $error_message;
   
   ###
   ### Storage Server conection error (not critical)
   ### Save data in a cache file 
   ###
   if( $error_id eq "CONNECT" or 
       $error_id eq "IO" or 
       $error_id eq "DB_REQUEST" or
       $error_id eq "DB_CONNECT" or
       $error_id eq "INTERNAL")
   {
       SysLog "$error_id, $error_message\n";

       my $now = time();
       if( !defined( $last_serveur_error_date ) )
       {
	   $last_serveur_error_date = $now;
       }
       if( $last_serveur_error_date + $self->{_TIME_SERVEUR_NOT_RESPONDING} <= $now )
       {
	   if($self->{_SENDTOSTORAGESERVER} == 1) {
	       $self->SendMailToAdmin("The storage server on host $self->{_HOST} port $self->{_PORT} hasn't replied for $self->{_TIME_SERVEUR_NOT_RESPONDING_S}.\n".
				      "The reason is :\n\t$error_id : $error_message\n\n".
				      "NB : The agent is running.\n".
				      "     The collected data are currently stored in a file which will be automatically sent to the server when the connection will be restored.\n");
	   }
	   else {
	       $self->SendMailToAdmin("Can't send mail to $self->{_MAILFORBRIDGE}. It's probably due to a problem with cache file (/var is full) or smtp server.\n\n".
				      "NB : The agent is running.\n".
				      "     The collected data are currently stored in a file which will be automatically sent to the server when the connection will be restored.\n"); 
	   }
       }
       
       ### Add data to cache
       open(CFD, "| gzip -c >> $self->{_FILENAME}") or die "Can't open cache file ($self->{_FILENAME}) : $!";
       $xmldata =~ s/\<ADAT\>(.*)\<\/ADAT\>/$1/;
       print CFD $xmldata;
       close(CFD);
   }
   
   ###
   ### Authentication problem. The source is not recognized by the Storage Server.
   ### Critical error : The agent die and send not sent data to the agent admin.
   ###
   elsif( $error_id eq "AUTHENTICATION" )
   {
       SysLog "$error_id\n";

       ### Add data to cache
       open(CFD, "| gzip -c >> $self->{_FILENAME}") or die "Can't open cache file ($self->{_FILENAME}) : $!";
       $xmldata =~ s/\<ADAT\>(.*)\<\/ADAT\>/$1/;
       print CFD $xmldata;
       close(CFD);
       
       ### Can't connect to storage server : bad login/password
       $self->SendMailToAdmin("Authentication error on storage server ($self->{_HOST}:$self->{_PORT}).\n".
			      "If you have change your GnuPG key pair, you must modify the public key in the database.\n\n".
			      "NB : The agent is dead, you must restart it manually.\n".
			      "     The current cache file (which contains collected but not sent data) is joined to this message.",
			      $self->{_FILENAME});
       
       Die("Can't connect to storage server : Authentication error.");
   }
   
   ###
   ### Protocol, XML and NOT_A_SIGNATURE error, Critical Error probably due to
   ### an incompatible version of the StorageServer an internal bug.
   ### 
   elsif( $error_id eq "PROTOCOL" or
	  $error_id eq "NOT_A_SIGNATURE" or 
	  $error_id eq "XML") {
       my $message = "$error_id";
       $message .= "$error_message" if defined $error_message;
       $message .= "\n";

       SysLog $message;

       ### Add data to cache
       open(CFD, "| gzip -c >> $self->{_FILENAME}") or die "Can't open cache file ($self->{_FILENAME}) : $!";
       $xmldata =~ s/\<ADAT\>(.*)\<\/ADAT\>/$1/;
       print CFD $xmldata;
       close(CFD);
       
       ### Invalid XML data.
       $message = "An error occured while sending data to Storage Server. This one has replied : $error_id";
       $message .= " : $error_message" if defined $error_message;
       $message .= ".\nPlease, check if your Storage Server is compatible with this agent.\n".
  	           "If yes, it's probably due to a bug in the agent. Please, forward this mail to the agent's developer.\n\n".
		   "NB : The agent is dead, you must restart it manually.\n".
		   "     The current cache file (which contains collected but not sent data) is joined to this message.";
	  
       $self->SendMailToAdmin($message, $self->{_FILENAME});
       
       $message = "$error_id error";
       $message .= " : $error_message" if defined $error_message;
       Die($message);
   }

   ###
   ### Agent description present in database is incoherent with sent data.
   ### Critical Error : perhaps the agent admin forgot to update the agent 
   ### description in database.
   ###
   elsif( $error_id eq "DB_AGENTDESCRIPTION"  ) {
       SysLog "$error_id, $error_message\n";

       ### Add data to cache
       open(CFD, "| gzip -c >> $self->{_FILENAME}") or die "Can't open cache file ($self->{_FILENAME}) : $!";
       $xmldata =~ s/\<ADAT\>(.*)\<\/ADAT\>/$1/;
       print CFD $xmldata;
       close(CFD);
       
              ### Can't connect to storage server : bad login/password
       $self->SendMailToAdmin("The agent description present in database is incoherent with sent data.\n".
			      "Perhaps the agent admin has changed the agent description and forgot to update the agent description with AgentDescriptionToDB\n\n".
			      "NB : The agent is dead, you must restart it manually.\n".
			      "     The current cache file (which contains collected but not sent data) is joined to this message.",
			      $self->{_FILENAME});

       
       Die("Agent Description error : $error_message");
   }

   ###
   ### Permission error, the agent try to add data for an another source.
   ### Critical Error.
   ###
   elsif( $error_id eq "PERMISSION"  ) {
       SysLog "$error_id, $error_message\n";

       ### Add data to cache
       open(CFD, "| gzip -c >> $self->{_FILENAME}") or die "Can't open cache file ($self->{_FILENAME}) : $!";
       $xmldata =~ s/\<ADAT\>(.*)\<\/ADAT\>/$1/;
       print CFD $xmldata;
       close(CFD);
        
       ### Can't connect to storage server : bad login/password
       $self->SendMailToAdmin("Unauthorized access : $error_message.\n\n".
			      "NB : The agent is dead, you must restart it manually.\n".
			      "     The current cache file (which contains collected but not sent data) is joined to this message.",
			      $self->{_FILENAME});

       
       Die("Permission error : $error_message");
   }   

   else {
       SysLog "PROTOCOL, Unknown response from the server\n";

       ### Add data to cache
       open(CFD, "| gzip -c >> $self->{_FILENAME}") or die "Can't open cache file ($self->{_FILENAME}) : $!";
       $xmldata =~ s/\<ADAT\>(.*)\<\/ADAT\>/$1/;
       print CFD $xmldata;
       close(CFD);

       $self->SendMailToAdmin("The agent has received an unknown response from the storage server ($response).\n".
			      "Please, check if your Storage Server is compatible with this agent.\n".
			      "If yes, it's probably due to a bug in the agent. Please, forward this mail to the agent's developer.\n\n".
			      "NB : The agent is dead, you must restart it manually.\n".
			      "     The current cache file (which contains collected but not sent data) is joined to this message.", 
			      $self->{_FILENAME});
       
       Die("PROTOCOL, Unknown restart from the server ($response)");
   }
   
   return 1;
}

##******************************************************************************
## Method SendDie  public
##  Description  : Send data to serveur or save them into file  and die
##                 Could be called into  CollectData
##  Parameters   : the die message
##******************************************************************************
sub SendDie
{
   my ($self,$message) = @_;

   $self->Send();
	
	$self->{_DATAALARMSBAG}->termSignal;

	if(-f $self->{_FILENAME}) {
	    $self->SendMailToAdmin("The agent is dead, you must restart it manually.\n".
				   "The current cache file (which contains collected but not sent data) is joined to this message.",
				   $self->{_FILENAME});
	    unlink($self->{_FILENAME});
	}
	
   Die( $message );
}

##******************************************************************************
## Method SendMailToAdmin  private
##  Description  : send a mail to the admin for inform him/her for agent's stop
##  Parameters   : the time that the agent has wait before killing itself
##  Return value : none
##******************************************************************************
sub SendMailToAdmin
{
    my ($self, $message, $attachement) = @_;
    
    my $entity = MIME::Entity->build(Type     => "multipart/mixed",
				     From     => "LogTrend Agent <$self->{_MAIL_SENDER}>",
				     To       => "LogTrend Admin <$self->{_MAIL_ADMIN}>",
				     Subject  => "[LogTrend] $self->{_INFORMATION} number $self->{_NUMBER} on source $self->{_SOURCE} error report");
    
    $entity->attach(Type   => "text/plain",
		    Data   => $message);
    
    if(defined $attachement) {
	$entity->attach(Type   => "application/logtrend",
			Path   => $attachement);
    }


    if($self->{_MAIL_SMTP} eq 'localhost') {
        open(MAILER, "| /usr/lib/sendmail -t  -i -f$self->{_MAIL_SENDER}");
        $entity->print(\*MAILER);
        close(MAILER);
    }
    else {
        my $smtp = Net::SMTP->new($self->{_MAIL_SMTP});
        $smtp->mail($self->{_MAIL_SENDER});
        $smtp->to($self->{_MAIL_ADMIN});
        $smtp->data([ $entity->stringify ]);
        $smtp->quit;
    }
}

##******************************************************************************
## Method AppendToFile  protected
##  Description  : send the agent's data and alarms to a cache file
##  Parameters   : none
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
sub AppendToFile
{
   my $self = shift;
   my $result = 0;

   if( open(CFD, ">> ".$self->{_FILENAME}) )
   {
      print CFD $self->{_DATAALARMSBAG}->ConvertToXml();
      $result = 1;
      close(CFD);
   }

   return $result;
}

##******************************************************************************
## Method GetXmlDataAndAlarms  protected
##  Description  : returns the data and the alarms collected by the agent
##                 formatted in xml
##  Parameters   : none
##  Return value : the data and the alarms collected by the agent formatted in xml
##******************************************************************************
sub GetXmlDataAndAlarms
{
   my $self = shift;
   my $result = "";

   $result = $result."<ADAT>";
   my $dataalarmsset = undef;
   foreach $dataalarmsset (@{$self->{_DATAALARMSSETLIST}})
   {
     $result = $result.${dataalarmsset}->ConvertToXml();
   }

   $result = $result."</ADAT>";

   return $result;
}

##******************************************************************************
## Method GetXmlDescription  protected
##  Description  : returns the xml description of the agent
##  Parameters   : none
##  Return value : a string formatted in xml of the agent's description
##******************************************************************************
sub GetXmlDescription
{
   my $self = shift;
   my $result = "<AgentDescription>\n".
                "<Source>".$self->{_SOURCE}."</Source>\n".
                "<Number>".$self->{_NUMBER}."</Number>\n".
                "<Version>".$self->{_VERSION}."</Version>\n".
                "<ActivationDate>".$self->{_ACTIVATIONDATE}."</ActivationDate>\n".
                "<Information>".$self->{_INFORMATION}."</Information>\n";

   # adds the data and alarms' descriptions
   $result = $result.$self->{_DATADESCRIPTIONSET}->ConvertToXml().
                     $self->{_ALARMDESCRIPTIONSET}->ConvertToXml();

   $result = $result."</AgentDescription>\n";

   return $result;
}

##******************************************************************************
## Method AddADataDescription  protected
##  Description  : adds a data description to the list of data descriptions
##  Parameters   : a data number, a data type, a data unit,
##                 an information message describing the data, an internal-info
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
sub AddADataDescription
{
   my ($self, $number, $type, $unit, $description, $internal) = @_;
   return $self->{_DATADESCRIPTIONSET}->AddDataDescription($number, $type, $unit,
                                                           $description, $internal);
}

##******************************************************************************
## Method AddAnAlarmDescription  protected
##  Description  : adds an alarm description to the list of alarms descriptions
##  Parameters   : an alarm number, an alarm level, an information message,
##                 an internal-info
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
sub AddAnAlarmDescription
{
   my ($self, $number, $level, $message, $internal) = @_;
   return $self->{_ALARMDESCRIPTIONSET}->AddAlarmDescription($number, $level,
                                                             $message, $internal);
}

##******************************************************************************
## Method AddAlarm  protected
##  Description  : adds an alarm to the current set
##  Parameters   : an alarm number
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
sub AddAlarm
{
   my ($self, $number) = @_;
   
   return $self->{_DATAALARMSSET}->AddAlarm($number);
}

##******************************************************************************
## Method AddDataText  protected
##  Description  : adds a Text data to the current set
##  Parameters   : a data number, a data value
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
sub AddDataText
{
   my ($self, $number, $value) = @_;
   return $self->{_DATAALARMSSET}->AddDataText($number, $value);
}

##******************************************************************************
## Method AddDataInteger  protected
##  Description  : adds an Integer data to the current set
##  Parameters   : a data number, a data value
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
sub AddDataInteger
{
   my ($self, $number, $value) = @_;
   return $self->{_DATAALARMSSET}->AddDataInteger($number, $value);
}

##******************************************************************************
## Method AddDataReal  protected
##  Description  : adds an Real data to the current set
##  Parameters   : a data number, a data value
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
sub AddDataReal
{
   my ($self, $number, $value) = @_;
   return $self->{_DATAALARMSSET}->AddDataReal($number, $value);
}

##******************************************************************************
## Method AddDataDate  protected
##  Description  : adds an Date data to the current set
##  Parameters   : a data number, a data value
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
sub AddDataDate
{
   my ($self, $number, $value) = @_;
   return $self->{_DATAALARMSSET}->AddDataDate($number, $value);
}

##******************************************************************************
## Method AddDataTime  protected
##  Description  : adds an Time data to the current set
##  Parameters   : a data number, a data value
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
sub AddDataTime
{
   my ($self, $number, $value) = @_;
   return $self->{_DATAALARMSSET}->AddDataTime($number, $value);
}

##******************************************************************************
## Method AddDataDateTime  protected
##  Description  : adds an Date/Time data to the current set
##  Parameters   : a data number, a data value
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
sub AddDataDateTime
{
   my ($self, $number, $value) = @_;
   return $self->{_DATAALARMSSET}->AddDataDateTime($number, $value);
}


##   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
##  @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
## @@@
## @@                  PRIVATE METHODS
## @@@
##  @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
##   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

##******************************************************************************
## Method ParseAgentDescriptionFile  private
##  Description  : parses the agent's description file associated to the agent
##  Parameters   : none
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
sub ParseAgentDescriptionFile
{
   my $self = shift;
   my $result = 0;

   # creates the DOM parser
   my $parser = new XML::DOM::Parser;

   # parse of the agent's description file
   my $doc = undef;
   eval
   {
      # the eval is used to prevent from savage exits from the DOM object
      $doc = $parser->parsefile($self->{_AGENTDESCRIPTIONFILE});
   };

   if (defined $doc)
   {
      # get the root node of the document
      my $rootnodelist = $doc->getElementsByTagName("AgentDescription");
      if (defined $rootnodelist)
      {
         my $rootnode = $rootnodelist->item(0);
         if (defined $rootnode)
         {
            $self->{_SOURCE} = $self->GetAgentSource($rootnode);
            $self->{_NUMBER} = $self->GetAgentNumber($rootnode);
            $self->{_VERSION} = $self->GetAgentVersion($rootnode);
            $self->{_ACTIVATIONDATE} = $self->GetAgentActivationDate($rootnode);
            $self->{_INFORMATION} = $self->GetAgentInformation($rootnode);
            my $agentdatadescriptionsetnodelist =
                               $self->GetAgentDataDescriptionSetNode($rootnode);
            my $agentalarmdescriptionsetnodelist =
                              $self->GetAgentAlarmDescriptionSetNode($rootnode);

            if( (defined $self->{_SOURCE}) && (defined $self->{_NUMBER}) &&
                (defined $self->{_VERSION}) && ($self->{_ACTIVATIONDATE}) &&
                ($self->{_INFORMATION}) && (defined $agentdatadescriptionsetnodelist) &&
                (defined $agentalarmdescriptionsetnodelist) )
            {
               my $agentdatadescriptionsetnode =
                  $agentdatadescriptionsetnodelist->getElementsByTagName("Data");
               my $agentalarmdescriptionsetnode =
                  $agentalarmdescriptionsetnodelist->getElementsByTagName("Alarm");

               my $datadescriptionsetadding = 1;
               my $alarmdescriptionsetadding = 1;
               # adding of the data's descriptions
               my $nbdatadescriptionset = $agentdatadescriptionsetnode->getLength();
               for (my $ind = 0; $ind < $nbdatadescriptionset; $ind++)
               {
                  my $datanode = $agentdatadescriptionsetnode->item($ind);
                  my ($number, $type, $unit, $description, $internal) =
                                           $self->GetDataDescription($datanode);
                  if( (defined $number) && (defined $type) &&
                      (defined $unit) && (defined $description) )
                  {
                     if (!$self->AddADataDescription( $number, $type,
                                                      $unit, $description, $internal ))
                     {
                        $datadescriptionsetadding = 0;
                     }
                  }
               }
               # adding of the alarm's descriptions
               my $nbalarmdescriptionset = $agentalarmdescriptionsetnode->getLength();
               for (my $ind = 0; $ind < $nbalarmdescriptionset; $ind++)
               {
                  my $alarmnode = $agentalarmdescriptionsetnode->item($ind);
                  my ($number, $level, $message, $internal) =
                                         $self->GetAlarmDescription($alarmnode);
                  if ((defined $number) && (defined $level) && (defined $message))
                  {
                     if (!$self->AddAnAlarmDescription($number, $level, $message, $internal))
                     {
                        $alarmdescriptionsetadding = 0;
                     }
                  }
               }

               if ($datadescriptionsetadding && $alarmdescriptionsetadding)
               {
                  $result = 1;
               }
            }
         }
      }
   }

   return $result;
}



##******************************************************************************
## Method GetANode  private
##  Description  : gets the first subnode of a node
##  Parameters   : a node and a subnode name
##  Return value : a subnode or nothing
##******************************************************************************
sub GetANode
{
   my ($self, $node, $nodename) = @_;
   my $result = undef;

   my $nodelist = $node->getElementsByTagName($nodename);
   if (defined $nodelist)
   {
      my $thenode = $nodelist->item(0); # we suppose that it's the good one
      if (defined $thenode)
      {
         $result = $self->GetValue($thenode);
      }
   }

   return $result;
}

##******************************************************************************
## Method GetANodeRef  private
##  Description  : gets the first subnode reference of a node
##  Parameters   : a node and a subnode name
##  Return value : a subnode reference or nothing
##******************************************************************************
sub GetANodeRef
{
   my ($self, $node, $nodename) = @_;
   my $result = undef;

   my $nodelist = $node->getElementsByTagName($nodename);
   if (defined $nodelist)
   {
      $result = $nodelist->item(0); # we suppose that it's the good one
   }

   return $result;
}

##******************************************************************************
## Method GetAgentSource  private
##  Description  : gets the source of an agent
##  Parameters   : the parent's node of the source node
##  Return value : the source of an agent
##******************************************************************************
sub GetAgentSource
{
   my ($self, $node) = @_;
   my $result = $self->GetANode($node, "Source");
   return $result;
}

##******************************************************************************
## Method GetAgentNumber  private
##  Description  : gets the number of an agent
##  Parameters   : the parent's node of the source node
##  Return value : the number of an agent
##******************************************************************************
sub GetAgentNumber
{
   my ($self, $node) = @_;
   my $result = $self->GetANode($node, "Number");
   return $result;
}

##******************************************************************************
## Method GetAgentVersion  private
##  Description  : gets the version of an agent
##  Parameters   : the parent's node of the source node
##  Return value : the version of an agent
##******************************************************************************
sub GetAgentVersion
{
   my ($self, $node) = @_;
   my $result = $self->GetANode($node, "Version");
   return $result;
}

##******************************************************************************
## Method GetAgentActivationDate  private
##  Description  : gets the activationdate of an agent
##  Parameters   : the parent's node of the source node
##  Return value : the activationdate of an agent
##******************************************************************************
sub GetAgentActivationDate
{
   my ($self, $node) = @_;
   my $result = $self->GetANode($node, "ActivationDate");
   return $result;
}

##******************************************************************************
## Method GetAgentInformation  private
##  Description  : gets the information of an agent
##  Parameters   : the parent's node of the source node
##  Return value : the information of an agent
##******************************************************************************
sub GetAgentInformation
{
   my ($self, $node) = @_;
   my $result = $self->GetANode($node, "Information");
   return $result;
}

##******************************************************************************
## Method GetValue  private
##  Description  : gets the value associated to a node
##  Parameters   : a node
##  Return value : a node value
##******************************************************************************
sub GetValue
{
   my ($self, $node) = @_;
   my $result = undef;

   my $nodevalueref = $node->getFirstChild();
   if (defined $nodevalueref)
   {
      $result = $nodevalueref->getNodeValue();
   }

   return $result;
}

##******************************************************************************
## Method GetAgentDataDescriptionSetNode  private
##  Description  : gets an agent's data description node
##  Parameters   : the parent's node of an agent's data description node
##  Return value : the agent's data description node
##******************************************************************************
sub GetAgentDataDescriptionSetNode
{
   my ($self, $node) =@_;
   my $result = $self->GetANodeRef($node, "DataDescriptionSet");
   return $result;
}

##******************************************************************************
## Method GetAgentAlarmDescriptionSetNode  private
##  Description  : gets the agent's alarm description node
##  Parameters   : the parent's node of an agent's alarm description node
##  Return value : the agent's alarm description node
##******************************************************************************
sub GetAgentAlarmDescriptionSetNode
{
   my ($self, $node) =@_;
   my $result = $self->GetANodeRef($node, "AlarmDescriptionSet");
   return $result;
}

##******************************************************************************
## Method GetDataType  private
##  Description  : gets the type of a type's node
##  Parameters   : a type's node
##  Return value : the type of a type's node
##******************************************************************************
sub GetDataType
{
   my ($self, $node) = @_;
   my $result = undef;

   my $typenodelist = $node->getElementsByTagName("Type");
   if (defined $typenodelist)
   {
      my $typenode = $typenodelist->item(0);
      if (defined $typenode)
      {
         if ($typenode->getElementsByTagName("Text")->getLength() > 0)
         { $result = "Text"; }
         elsif ($typenode->getElementsByTagName("Integer")->getLength() > 0)
         { $result = "Integer"; }
         elsif ($typenode->getElementsByTagName("Real")->getLength() > 0)
         { $result = "Real"; }
         elsif ($typenode->getElementsByTagName("Date")->getLength() > 0)
         { $result = "Date"; }
         elsif ($typenode->getElementsByTagName("Time")->getLength() > 0)
         { $result = "Time"; }
         elsif ($typenode->getElementsByTagName("DateTime")->getLength() > 0)
         { $result = "DateTime"; }
      }
   }

   return $result;
}

##******************************************************************************
## Method GetDataDescription  private
##  Description  : gets the data's description of a data's description node
##  Parameters   : a data's description node
##  Return value : a list with data's number, type, unit, description, internal
##******************************************************************************
sub GetDataDescription
{
   my ($self, $node) = @_;

   my $number = $self->GetANode($node, "Number");
   my $type = $self->GetDataType($node);
   my $unit = $self->GetANode($node, "Unit");
   my $description = $self->GetANode($node, "Description");
   my $internal = $self->GetANode($node, "Internal");

   if( defined($internal) )
   {
      $self->{$internal} = $number;
   }

   return ($number, $type, $unit, $description, $internal);
}

##******************************************************************************
## Method GetAlarmLevel  private
##  Description  : gets an alarm level associated to an alarm's description node
##  Parameters   : an alarm's description node
##  Return value : an alarm level
##******************************************************************************
sub GetAlarmLevel
{
   my ($self, $node) = @_;
   my $result = undef;

   my $levelnodelist = $node->getElementsByTagName("Level");
   if (defined $levelnodelist)
   {
      my $levelnode = $levelnodelist->item(0);
      if (defined $levelnode)
      {
         if ($levelnode->getElementsByTagName("Info")->getLength() > 0)
         { $result = "Info"; }
         elsif ($levelnode->getElementsByTagName("Warning")->getLength() > 0)
         { $result = "Warning"; }
         elsif ($levelnode->getElementsByTagName("Error")->getLength() > 0)
         { $result = "Error"; }
      }
   }

   return $result;
}

##******************************************************************************
## Method GetAlarmDescription  private
##  Description  : gets the alarm's description of an alarm's description node
##  Parameters   : an alarm's description node
##  Return value : a list with alarm's number, level, message, internal
##******************************************************************************
sub GetAlarmDescription
{
   my ($self, $node) = @_;

   my $number = $self->GetANode($node, "Number");
   my $level = $self->GetAlarmLevel($node);
   my $message = $self->GetANode($node, "Message");
   my $internal = $self->GetANode($node, "Internal");

   if( defined($internal) )
   {
      $self->{$internal."_".$level} = $number;
   }

   return ($number, $level, $message, $internal);
}

##******************************************************************************
1;

__END__

=head1 NAME

Agent.pm - Perl Extension for LogTrend : Agent

=head1 SYNOPSIS

  use LogTrend::Agent;

  package WidgetAgent;

  @ISA = ("LogTrend::Agent");

=head1 DESCRIPTION

LogTrend::Agent is a Perl module implementing an Agent for LogTrend.

This module is not intended for direct use, but to be inherited
by any WidgetAgent ('Widget' could be Linux, SNMP ...).

The Agent is a template for writing LogTrend Agent.
Any LogTrend Agent must inherit from Agent and implement
CreateAgentDescription and CollectData methods
(see How to create a new agent - the developper guide).

An agent could be in two modes: the description generation mode
and the running mode.

The description generation mode is used at the configuration of
the agent, to generate the file which contains the description of data
that will be collected by the agent.

The running mode is used after the description generation mode
and is the normal mode, in the sens that this is the mode for
collecting data and sending them to the StorageServer.

=head1 PRE-REQUISITES

The following Perl modules are definitly needed for this agent to
work:

    Getopt::Long
    XML::DOM
    MIME::Entity

The first one is needed for parsing in-line arguments.
The second one is needed for XML analysis.
The three others are for sending mails (to administrator
and/or StorageServer).

=head1 CONFIGURATION

The Agent configuration is done using an XML file.

See documentation:
/usr/share/doc/LogTrend/Agent/install-guide/agent-install-guide.ps

=head1 AUTHOR

Sylvain Lhullier -- Atrid Systmes (s.lhullier@atrid.fr)

=head1 COPYRIGHT

Copyright 2001, Atrid Systme http://www.atrid.fr/

Project home page: http://www.logtrend.org/

Licensed under the same terms as LogTrend project is.

=head1 WARRANTY

THIS SOFTWARE COMES WITH ABSOLUTLY NO WARRANTY OF ANY KIND.
IT IS PROVIDED "AS IS" FOR THE SOLE PURPOSE OF EVENTUALLY
BEEING USEFUL FOR SOME PEOPLE, BUT ONLY AT THEIR OWN RISK.

=cut

