#!/usr/bin/perl -w

#$Header: /home2/cvsroot/LogTrend/Agent/LinuxAgent.pm,v 1.13 2002/02/06 13:06:39 lsimonneau Exp $
##******************************************************************************
## Class LinuxAgent  isa  Agent
##  Description  : class for Linux supervision
##                 For *_FS methods, see LinuxAgent_FS.pm file
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Author       : Sylvain Lhullier s.lhullier@atrid.fr
##  Author       : David Mascle d.mascle@atrid.fr
##******************************************************************************
#$Log: LinuxAgent.pm,v $
#Revision 1.13  2002/02/06 13:06:39  lsimonneau
#Minor bugfixes.
#
#Revision 1.12  2002/01/15 10:04:18  lsimonneau
#Correct a bug with df -i and devfs long device name.
#
#Revision 1.11  2002/01/07 13:12:27  lsimonneau
#Apply a patch correcting a bug in the number of different users calculation.
#
#Revision 1.10  2001/12/24 08:02:08  slhullier
#df non-posix output bug
#
#Revision 1.9  2001/11/02 13:55:42  lsimonneau
#Add Action support for agents' alarms.
#Add action support for LinuxAgent Process alarms.
#
#Revision 1.8  2001/10/31 15:03:52  lsimonneau
#Add Process alarms.
#
#Revision 1.7  2001/10/02 16:00:45  slhullier
#Documentation updated
#
#Revision 1.6  2001/09/28 15:19:02  lsimonneau
#*** empty log message ***
#
#Revision 1.5  2001/09/24 13:35:42  slhullier
#CPU collection modification
#
#Revision 1.4  2001/09/18 15:12:15  slhullier
#POD creation/modification
#
#Revision 1.3  2001/09/18 11:20:02  slhullier
#POD doc
#
#Revision 1.2  2001/07/20 12:26:39  slhullier
#
#One uniq file for LinuxAgent
#
#Revision 1.1  2001/07/19 16:40:27  fdesar
#
#Moved module files to the right directories
#Updated package names and uses to reflect those changes
#Corrected bug in SnortAgent.pm for negating first value in SID parsing
#
#Revision 1.25  2001/07/10 12:55:16  slhullier
#*** empty log message ***
#
#Revision 1.24  2001/07/09 11:44:45  slhullier
#*** empty log message ***
#
#Revision 1.23  2001/07/09 11:39:37  slhullier
#
#Deletion of glog calls -> readdir
#
#Revision 1.22  2001/06/20 11:49:28  fdesar
#
#Traitement du SIGCHLD (zombies)
#
#Revision 1.21  2001/06/07 14:08:54  slhullier
#
#Passage de  unshift @INC,'..';  aux packages Logtrend::....
#
#Revision 1.20  2001/03/30 12:58:07  slhullier
#
#Le serveur ne fait plus de messages sur la console  +  gzip : qq bugs
#Mise en place d'un nv StorageServer sur serveur
#
#Revision 1.19  2001/03/19 10:57:58  slhullier
#
#Agents : fichier de conf par defaut dans /etc   +  mise en commun
#des routines de gestion des arguments dans Agent.pm
#
#Revision 1.18  2001/03/14 09:10:46  slhullier
#
#Die moins stupides
#
#Revision 1.17  2001/03/13 09:44:10  slhullier
#
#SimpleAgent mis au gout du jour
#
#Revision 1.16  2001/03/13 09:10:23  slhullier
#
#LogDie sur SimpleAgent + version&number dans Configuration.xml
#
#Revision 1.15  2001/03/09 17:01:03  slhullier
#
#.
#
#Revision 1.14  2001/03/09 16:16:23  slhullier
#
#Les seuils ne sont plus dans linuxagentdescription.xml
#mais relus dans Configuration.xml
#Gestion de la non reponse du serveur
#
#Revision 1.13  2001/03/08 17:33:54  slhullier
#
#L'agent ne meurt plus des que le serveur ne repond plus.
#
#Revision 1.12  2001/03/07 10:34:03  slhullier
#
#Mise en place de LogDie dans l'agent Linux
#
#Revision 1.11  2001/03/02 17:17:06  slhullier
#
#Fautes
#
#Revision 1.10  2001/02/27 17:10:03  slhullier
#
#Ajout des quotas : nombre d'users/groups qui depassent.
#Semble fonctionner, a tester en grandeur reelle.
#
#Revision 1.9  2001/02/23 15:55:38  slhullier
#
#Ajout de tout pleins de fonctionnalites pour le LinuxAgent :
#filesystems, services (HTTP,FTP)
#Mise en place sur spa, tests en grandeur nature en cours.
#
#Revision 1.8  2001/02/22 17:02:23  slhullier
#
#Gestion des systemes de fichiers par LinuxAgent.
#Premiers tests concluants, reste a ajouter des alarmes et a faire
#des tests en situation reelle.
#
#Revision 1.7  2001/02/21 11:00:22  slhullier
#
#Mise a plat des methodes de recolte de donnees
#
#Revision 1.6  2001/02/20 17:01:27  slhullier
#
#Mise en commentaire (+tab etc)
#
#Revision 1.5  2001/02/20 15:50:20  slhullier
#
#Apres modification du nom de certaines classes
#
#Revision 1.4  2001/02/20 10:21:55  slhullier
#
#Passage de la fonction de parsage de config XML dans Agent.pm
#Creation des tags Generic et Specific
#Fichiers agentdescription.xml plus lisibles (avec \n)
#Supression de methodes pas appellee du fichier Agent.pm
#
#Revision 1.3  2001/02/07 14:36:33  slhullier
#
#
#Le systeme MailBridge fonctionne.
#Mise en service sur spa.
#
#Revision 1.2  2001/02/06 15:24:10  slhullier
#
#
#Modifications de la variable INC
#
#Revision 1.1.1.1  2001/02/06 09:48:28  fdubuy
#
#First CVS version : all work done by D.Mascle
#
#

package LogTrend::Agent::LinuxAgent;

use strict;

use vars qw( @ISA );

use HTTP::Status;
use LWP::UserAgent;
use Net::FTP;
use strict;
use Time::HiRes qw(gettimeofday);
use XML::DOM;

use LogTrend::Agent;
use LogTrend::Common::LogDie;
use LogTrend::Action::ActionSet;

@ISA = ("LogTrend::Agent");

my $name = "LinuxAgent";
my $version = "1.0.0.0";

##******************************************************************************
## Constructor  public > Agent
##  Description  : creat a new LinuxAgent
##  Parameters   : none
##******************************************************************************
sub new
{
   my ( $classname ) = @_;

   $SIG{CHLD} = 'IGNORE';

   my $self = $classname->SUPER::new( $name, $version );
   bless($self, $classname);

   return $self;
}

##   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
##  @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
## @@@
## @@                  ParseXMLConfigFile methods
## @@@
##  @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
##   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

##******************************************************************************
## Method ParseXMLConfigFile  public  (>Agent)
##  Description  : parses the XML config file
##  Parameters   : the file name to parse, the agent state
##  Return value : none
##******************************************************************************
sub ParseXMLConfigFile
{
   my ($self,$file,$agentstate) = @_;
   $self->SUPER::ParseXMLConfigFile( $file, $agentstate );

   ##===========================================================================
   ## Agent-specific configuration parameters :
   ## use $self->{FOO} to stock information ( all $self->{_FOO} are reserved )
   ##===========================================================================
   ## Tag 'Configuration'
   my $parser = new XML::DOM::Parser() || Die("XML::DOM::Parser: $!");
   my $doc = $parser->parsefile( $file ) || Die("$file :$!");

   my $rootlist = $doc->getElementsByTagName("Configuration") ||
                                        Die("$file :No \"Configuration\" tag.");
   my $rootnode = $rootlist->item(0) || Die("$file :No \"Configuration\" tag.");

   ## Tag 'Specific'
   $rootlist = $rootnode->getElementsByTagName("Specific") ||
                                        Die("$file :No \"Specific\" tag.");
   $rootnode = $rootlist->item(0) || Die("$file :No \"Specific\" tag.");


   ##===========================================================================
   ## Parsing for standard features
   ##===========================================================================
   my ($list,$node,$valueref,$attributes,$attrnode);
   FOO: foreach my $foo ("LoadAverage","TooMuchProcesses","TooMuchZombies")
   {
      $list = $rootnode->getElementsByTagName("$foo") || Die($!);
      $node = $list->item(0) || next FOO;
      $attributes = $node->getAttributes() || Die("Error in \"$foo\" tag.");
      foreach my $foo2 ("Warning", "Error")
      {
         $attrnode = $attributes->getNamedItem("$foo2");
         if( defined($attrnode) )
         { $self->{"Cfg_".$foo."_".$foo2} = $attrnode->getValue(); }
      }
   }


   ##===========================================================================
   ## Parsing the Processes Entries
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("Process");
   foreach $node (@$list) {
       $attributes = $node->getAttributes() || Die("Error in \"Process\" tag.");
       $attrnode = $attributes->getNamedItem("command") || 
	   Die "Error, no \"command\" attribute in \"Process\" tag.";
              
       my $cmdline = $attrnode->getValue;

       my $actionset = new LogTrend::Action::ActionSet($node);

       push @{$self->{PROCESSES}}, [$cmdline, $actionset];
   }

   ##===========================================================================
   ## Specific parsing :
   ##===========================================================================
   $self->ParseXMLConfigFile_FS( $rootnode );
   $self->ParseXMLConfigFile_Services( $rootnode );

}

##******************************************************************************
## Method ParseXMLConfigFile_FS  protected
##  Description  : parses the XML config file for FS info
##  Parameters   : the node for 'Specific' tag
##  Return value : none
##******************************************************************************
sub ParseXMLConfigFile_FS
{
   my $self = shift;
   my $SpecificNode = shift;

   my ($fs,$node,$attributes,$attrnode);

   ##***************************************************************************
   ## Tag 'FS'
   ##***************************************************************************
   my $list = $SpecificNode->getElementsByTagName("FS") || Die($!);
   my $n = $list->getLength();
   for( my $i=0; $i<$n; $i++ )
   {
      $node = $list->item($i) || Die("Error in \"FS\" tag.");
      $attributes = $node->getAttributes() || Die("Error in \"FS\" tag.");
      $attrnode = $attributes->getNamedItem("Dev");

      if( defined($attrnode) )    ## Specific device info by name
      {
         my $v = $attrnode->getValue();
         $fs = "Cfg_FS_".$v;
      }
      else
      {
         $attrnode = $attributes->getNamedItem("Type");
         if( defined($attrnode) ) ## Specific device info by type
         {
            my $v = $attrnode->getValue();
            $fs = "Cfg_FS_".$v;
         }
         else                     ## Info for all other devices
         {
            if( defined( $self->{"FS_*"} ) )
            { Die("Several default infos for filesystems in XML file"); }
            $fs = "Cfg_FS_*";
         }
      }

      $self->{$fs} = 1;

      ##========================================================================
      ## Space / Inodes
      ##========================================================================
      foreach my $foo ("Space","Inodes")
      {
         my $list2 = $node->getElementsByTagName("$foo") || Die($!);
         my $node2 = $list2->item(0);
         if( defined($node2) )
         {
            my $prefix = "$fs"."_$foo";
            ##------------------------------------------------------------------
            ## PercentUsed/NumberFree
            ##------------------------------------------------------------------
            foreach my $foo2 ("PercentUsed","NumberFree")
            {
               my $list3 = $node2->getElementsByTagName("$foo2") || Die($!);
               my $node3 = $list3->item(0);
               if( defined($node3) )
               {
                  my $attributes = $node3->getAttributes() ||
                                                  Die("Error in \"$foo2\" tag.");
                  $self->{$prefix."_$foo2"} = 1;
                  ##::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
                  ## Warning/Error
                  ##::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
                  foreach my $foo3 ("Warning", "Error")
                  {
                     my $attrnode = $attributes->getNamedItem("$foo3");
                     if( defined($attrnode) )
                     { $self->{$prefix."_$foo2"."_"."$foo3"} = $attrnode->getValue(); }
                  }
               }
            }
            ##------------------------------------------------------------------
            ## Quota
            ##------------------------------------------------------------------
            my $list3 = $node2->getElementsByTagName("Quota") || Die($!);
            my $node3 = $list3->item(0);
            if( defined($node3) )
            {
               my $attributes = $node3->getAttributes() ||
                                               Die("Error in \"Quota\" tag.");
               ##::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
               ## Users/Groups
               ##::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
               foreach my $foo3 ("Users", "Groups")
               {
                  my $attrnode = $attributes->getNamedItem("$foo3");
                  if( defined($attrnode) &&
                      $attrnode->getValue() eq "yes" )
                  { $self->{$prefix."_Quota"."_"."$foo3"} = 1; }
               }
            }
         }
      }
   }

}

##******************************************************************************
## sub Print_FS ##
## { ##
##    my $self = shift; ##
##  ##
##    foreach my $k (sort (keys %$self)) ##
##    { ##
##       if( $k =~ /FS_/ && ($k=~/Error/ || $k=~/Warning/) ) ##
##       {   ##
##          printf "%s : %s\n", $k, $self->{$k}; ##
##       }   ##
##    } ##
## } ##

##******************************************************************************
## Method ParseXMLConfigFile_Services  protected
##  Description  : parses the XML config file for services info
##  Parameters   : the node for 'Specific' tag
##  Return value : none
##******************************************************************************
sub ParseXMLConfigFile_Services
{
   my $self = shift;
   my $SpecificNode = shift;
   my ($node,$attributes,$attrnode);

   for my $foo ("HTTP","FTP")
   {
      my $list = $SpecificNode->getElementsByTagName("$foo") || Die($!);
      $node = $list->item(0);
      if( defined( $node ) )
      {
         $self->{"Cfg_$foo"} = 1;

         for my $foo2 ("Warning", "Error")
         {
            $attributes = $node->getAttributes() || Die("Error in \"$foo\" tag.");
            $attrnode = $attributes->getNamedItem($foo2);
            if( defined($attrnode) )
            {
               $self->{"Cfg_$foo"."_"."$foo2"} = $attrnode->getValue();
            }
         }
      }
   }
}


##   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
##  @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
## @@@
## @@                  CreateAgentDescription methods
## @@@
##  @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
##   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

##******************************************************************************
## Method CreateAgentDescription  public  (>Agent)
##  Description  : creates an agent's description
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub CreateAgentDescription
{
   my $self = shift;
   my ($d,$a) = (1,1);

   # CPU
   $self->AddADataDescription($d++, "Real", "percentage", "CPU user", "");
   $self->AddADataDescription($d++, "Real", "percentage", "CPU system", "");
   $self->AddADataDescription($d++, "Real", "percentage", "CPU nice", "");
   $self->AddADataDescription($d++, "Real", "percentage", "CPU idle", "");

   # Memory
   $self->AddADataDescription($d++, "Integer", "bytes",   "Memory total", "");
   $self->AddADataDescription($d++, "Integer", "bytes",   "Memory used", "");
   $self->AddADataDescription($d++, "Integer", "bytes",   "Memory free", "");
   $self->AddADataDescription($d++, "Integer", "bytes",   "Memory shared", "");
   $self->AddADataDescription($d++, "Integer", "bytes",   "Memory buffers", "");
   $self->AddADataDescription($d++, "Integer", "bytes",   "Memory cached", "");

   # Swap
   $self->AddADataDescription($d++, "Integer", "bytes",   "Swap total", "");
   $self->AddADataDescription($d++, "Integer", "bytes",   "Swap free", "");
   $self->AddADataDescription($d++, "Integer", "bytes",   "Swap used", "");

   # context changing
   $self->AddADataDescription($d++, "Integer", "none",    "Context changing", "");

   # load average
   $self->AddADataDescription($d++, "Real", "none", "System load average for 1 minute", "");
   $self->AddADataDescription($d++, "Real", "none", "System load average for 5 minutes", "");
   $self->AddADataDescription($d++, "Real", "none", "System load average for 15 minutes", "");
   if( defined( $self->{"Cfg_LoadAverage_Warning"} ) )
   { $self->AddAnAlarmDescription($a++, "Warning", "System overloaded", "LoadAverage" ); }
   if( defined( $self->{"Cfg_LoadAverage_Error"} ) )
   { $self->AddAnAlarmDescription($a++, "Error", "System overloaded", "LoadAverage" ); }

   # processes
   $self->AddADataDescription($d++, "Integer", "none", "Total number of processes", "");
   $self->AddADataDescription($d++, "Integer", "none", "Number of sleeping processes", "");
   $self->AddADataDescription($d++, "Integer", "none", "Number of running processes", "");
   $self->AddADataDescription($d++, "Integer", "none", "Number of zombie processes", "");
   $self->AddADataDescription($d++, "Integer", "none", "Number of stopped processes", "");
   if( defined( $self->{"Cfg_TooMuchProcesses_Warning"} ) )
   { $self->AddAnAlarmDescription($a++, "Warning", "Too much processes", "TooMuchProcesses" ); }
   if( defined( $self->{"Cfg_TooMuchProcesses_Error"} ) )
   { $self->AddAnAlarmDescription($a++, "Error", "Too much processes", "TooMuchProcesses" ); }
   if( defined( $self->{"Cfg_TooMuchZombies_Warning"} ) )
   { $self->AddAnAlarmDescription($a++, "Warning", "Too much zombies", "TooMuchZombies" ); }
   if( defined( $self->{"Cfg_TooMuchZombies_Error"} ) )
   { $self->AddAnAlarmDescription($a++, "Error", "Too much zombies", "TooMuchZombies" ); }

   # processes alarms
   foreach my $proc_entry (@{$self->{PROCESSES}}) {
       my $proc = $proc_entry->[0];
       $self->AddAnAlarmDescription($a++, "Error", "Process $proc is not running", "Process_$proc");
   }

   # users
   $self->AddADataDescription($d++, "Integer", "none", "Number of users", "");
   $self->AddADataDescription($d++, "Integer", "none", "Number of different users", "");

   # filesystems
   ($d,$a) = $self->CreateAgentDescription_FS( $d, $a );

   # eth interfaces
   ($d,$a) = $self->CreateAgentDescription_Eth( $d, $a );

   # services
   ($d,$a) = $self->CreateAgentDescription_Services( $d, $a );

}

##******************************************************************************
## Method CreateAgentDescription_FS  protected
##  Description  : creates an agent's description for filesystems
##  Parameters   : the first free data number, the first free alarm number
##  Return value : the first free data number, the first free alarm number
##******************************************************************************
sub CreateAgentDescription_FS
{
   my ($self, $freedatanbr, $freealarmnbr) = @_;

   ##***************************************************************************
   ## For each fs in /etc/mtab
   ##***************************************************************************
   my %mounted = ();
   open(MTAB,"/etc/mtab") || Die("/etc/mtab: $!");
   while( my $line=<MTAB> )
   {
      my ($device) = split( /[ \t]+/, $line );
      $mounted{$device} = 1;
   }
   close(MTAB);

   ##***************************************************************************
   ## For each fs in /etc/fstab
   ##***************************************************************************
   open(FSTAB,"/etc/fstab") || Die("/etc/fstab: $!");

   LINE: while( my $line=<FSTAB> )
   {
      ##========================================================================
      ## Eliminate non supervised fs
      ##========================================================================
      next LINE if( $line =~ /^#/ );
      next LINE if( $line =~ /^[ \t]*$/ );
      my ($device,$mountpoint,$type,$options) = split( /[ \t]+/, $line );

      next LINE if( $device eq "none" );
      next LINE if( $mountpoint !~ /^\// );
      next LINE if( $type eq "auto" );
      next LINE if( $type eq "proc" );
      next LINE if( $type eq "nfs" );
      next LINE if( $type eq "swap" );
      next LINE if( $type eq "devpts" );
      next LINE if( $type eq "iso9660" );
      next LINE if( $type eq "supermount" );
      next LINE if( $options =~ /noauto/ );

      next LINE if( !defined($mounted{$device}) );

      ##========================================================================
      ## Searching witch config to apply
      ##========================================================================
      my $config;
      if( defined( $self->{"Cfg_FS_".$device} ) ) # Specific fs description by name
      { $config = "Cfg_FS_".$device; }
      elsif( defined( $self->{"Cfg_FS_".$type} ) ) # Specific fs description by type
      { $config = "Cfg_FS_".$type; }
      elsif( defined( $self->{"Cfg_FS_*"} ) ) # Generic fs description
      { $config = "Cfg_FS_*"; }        # Generic fs description
      else
      { next LINE; }           # Neither specific nor generic fs description

      ##========================================================================
      ## Space / Inodes
      ##========================================================================
      FOO: for my $foo ("Space","Inodes")
      {
         my $minFoo = $foo;
         $minFoo =~ s/(.*)/\L$1/;
         my ($unit,$type);

         ##---------------------------------------------------------------------
         ## Partition with no inode
         ##---------------------------------------------------------------------
         if( $foo eq "Inodes" )
         {
            open( DF, "df -iP $device|" ) || Die("df -iP $device : $!");
            my $df_line = <DF> || Die("df -iP $device : $!");
            $df_line = <DF> || Die("df -iP $device : $!");
            close( DF );
            my ($d,$v) = split( /[ \t]+/, $df_line );
            next FOO if( $v == 0 );
            $unit = "none";
         }
         else
         { $unit = "kilobytes"; }
         $type = "Real";

         ##---------------------------------------------------------------------
         ## PercentUsed/NumberFree
         ##---------------------------------------------------------------------
         for my $foo2 ("NumberFree","PercentUsed")
         {
            if( defined($self->{$config."_"."$foo"."_"."$foo2"}) )
            {
               my $minFoo2 = $foo2;
               $minFoo2 =~ s/[A-Z][a-z]+([A-Z][a-z]+)/$1/;

               if( $foo2 eq "PercentUsed" )
               { $unit = "percentage"; }

               $self->AddADataDescription(
                  $freedatanbr, "$type", $unit, "$minFoo2 $minFoo on $mountpoint",
                  "FS_".$device."_"."$foo"."_"."$foo2" );
               $freedatanbr++;

               ##:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
               ## Warning/Error
               ##:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
               foreach my $foo3 ("Warning", "Error")
               {
                  if( defined($self->{$config."_"."$foo"."_"."$foo2"."_"."$foo3"}) )
                  {
                     $self->AddAnAlarmDescription(
                        $freealarmnbr, "$foo3", "$mountpoint getting full for $minFoo",
                        "FS_".$device."_"."$foo"."_"."$foo2" );
                     $freealarmnbr++;
                  }
               }
            }
         }
         ##---------------------------------------------------------------------
         ## Quota
         ##---------------------------------------------------------------------
         for my $foo2 ("Users", "Groups")
         {
            my $minFoo2 = $foo2;
            $minFoo2 =~ s/(.*)/\L$1/;
            if( defined($self->{$config."_".$foo."_Quota_"."$foo2"}) )
            {
               $self->AddADataDescription( $freedatanbr++, "Integer", "none",
                  "Number of $minFoo2 out of quota for $minFoo on $mountpoint",
                  "FS_".$device."_"."$foo"."_Quota_"."$foo2" );
            }
         }
      }
   }

   close(FSTAB);

   return ($freedatanbr, $freealarmnbr);

}

##******************************************************************************
## Method CreateAgentDescription_Eth  protected
##  Description  : creates an agent's description for filesystems
##  Parameters   : the first free data number, the first free alarm number
##  Return value : the first free data number, the first free alarm number
##******************************************************************************
sub CreateAgentDescription_Eth
{
   my ($self, $freedatanbr, $freealarmnbr) = @_;

   ##***************************************************************************
   ## For each device in /proc/net/dev
   ##***************************************************************************
   open(FILE,"/proc/net/dev") || Die("/proc/net/dev: $!");

   LINE: while( my $line=<FILE> )
   {
      ##========================================================================
      ## Eliminate non supervised interfaces
      ##========================================================================
      next LINE if( $line =~ /^#/ );
      next LINE if( $line =~ /^Inter/ );
      next LINE if( $line =~ /^ *face/ );
      my $interface = $line;
      $interface =~ s/^[ \t]*(.*):.*\n/$1/;

      next LINE if( $interface eq "lo" );

      ##========================================================================
      ## Ip adress
      ##========================================================================
      open( IFCONFIG, "ifconfig $interface|" ) || Die("ifconfig $interface : $!");
      my $l = <IFCONFIG> || Die("ifconfig $interface : $!");
      $l = <IFCONFIG> || Die("ifconfig $interface : $!");
      chomp( $l );
      my ($ip_adress) = ( $l =~ /.*inet adr: *([0-9.]*) .*/ );
      if( !defined($ip_adress) )
      { $ip_adress = ""; }
      else
      { $ip_adress = " ($ip_adress)"; }
      close( IFCONFIG );

      ##========================================================================
      ## Data
      ##========================================================================
      # Reception
      $self->AddADataDescription( $freedatanbr, "Real", "bytes",
                                  "Received bytes on $interface$ip_adress",
                                  "Eth_".$interface."_"."Received_bytes" );
      $freedatanbr++;
      $self->AddADataDescription( $freedatanbr, "Real", "none",
                                  "Received packets on $interface$ip_adress",
                                  "Eth_".$interface."_"."Received_packets" );
      $freedatanbr++;
      $self->AddADataDescription( $freedatanbr, "Real", "none",
                                  "Errors in reception on $interface$ip_adress",
                                  "Eth_".$interface."_"."Received_errs" );

      # Transmission
      $freedatanbr++;
      $self->AddADataDescription( $freedatanbr, "Real", "bytes",
                                  "Transmited bytes on $interface$ip_adress",
                                  "Eth_".$interface."_"."Transmited_bytes" );
      $freedatanbr++;
      $self->AddADataDescription( $freedatanbr, "Real", "none",
                                  "Transmited packets on $interface$ip_adress",
                                  "Eth_".$interface."_"."Transmited_packets" );
      $freedatanbr++;
      $self->AddADataDescription( $freedatanbr, "Real", "none",
                                  "Errors in transmission on $interface$ip_adress",
                                  "Eth_".$interface."_"."Transmited_errs" );
      $freedatanbr++;
      $self->AddADataDescription( $freedatanbr, "Real", "none",
                                  "Collisions in transmission on $interface$ip_adress",
                                  "Eth_".$interface."_"."Transmited_colls" );
      $freedatanbr++;

   }

   close(FILE);

   return ($freedatanbr, $freealarmnbr);

}

##******************************************************************************
## Method CreateAgentDescription_Services  protected
##  Description  : creates an agent's description for services
##  Parameters   : the first free data number, the first free alarm number
##  Return value : the first free data number, the first free alarm number
##******************************************************************************
sub CreateAgentDescription_Services
{
   my ($self, $freedatanbr, $freealarmnbr) = @_;

   for my $foo ("HTTP","FTP")
   {
      if( defined( $self->{"Cfg_$foo"} ) )
      {
         $self->AddADataDescription( $freedatanbr, "Real", "none",
                                     "Connection time for $foo", "$foo" );
         $freedatanbr++;

         for my $foo2 ("Warning", "Error")
         {
            if( defined($self->{"Cfg_$foo"."_"."$foo2"}) )
            {
               $self->AddAnAlarmDescription(
                        $freealarmnbr, "$foo2", "Connection time for $foo",
                        "$foo" );
               $freealarmnbr++;
            }
         }
      }
   }

   return ($freedatanbr, $freealarmnbr);
}


##   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
##  @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
## @@@
## @@                  CollectData methods
## @@@
##  @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
##   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

##******************************************************************************
## Method CollectData  public  (>Agent)
##  Description  : collects data and alarms
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub CollectData
{
   my $self = shift;

   ##===========================================================================
   ## Get values
   ##===========================================================================
   my ($cpuuser, $cpusyst, $cpunice, $cpuidle, $contextchanging)=
       $self->GetStatValues();

   my ($memorytotal, $memoryused, $memoryfree, $memoryshared,
       $memorybuffers, $memorycached, $swaptotal, $swapused,
       $swapfree) = $self->GetMemoryAndSwapValues();

   my ($la1,$la5,$la15) = $self->GetLoadAveragesValues();

   my ($proc_total, $proc_sleeping, $proc_running, $proc_zombie, $proc_stopped, $cmdlines) =
      $self->GetProcessesValues();

   my ($nb_users,$nb_diff_users) = $self->GetUsersValues();

   ##===========================================================================
   ## Set values
   ##===========================================================================
   $self->AddDataReal   ($self->{"CPU user"}, $cpuuser );
   $self->AddDataReal   ($self->{"CPU system"}, $cpusyst );
   $self->AddDataReal   ($self->{"CPU nice"}, $cpunice );
   $self->AddDataReal   ($self->{"CPU idle"}, $cpuidle );
   $self->AddDataInteger($self->{"Memory total"}, $memorytotal );
   $self->AddDataInteger($self->{"Memory used"}, $memoryused );
   $self->AddDataInteger($self->{"Memory free"}, $memoryfree );
   $self->AddDataInteger($self->{"Memory shared"}, $memoryshared );
   $self->AddDataInteger($self->{"Memory buffers"}, $memorybuffers );
   $self->AddDataInteger($self->{"Memory cached"}, $memorycached );
   $self->AddDataInteger($self->{"Swap total"}, $swaptotal );
   $self->AddDataInteger($self->{"Swap free"}, $swapfree );
   $self->AddDataInteger($self->{"Swap used"}, $swapused );
   $self->AddDataInteger($self->{"Context changing"}, $contextchanging );
   $self->AddDataReal   ($self->{"System load average for 1 minute"}, $la1 );
   $self->AddDataReal   ($self->{"System load average for 5 minutes"}, $la5 );
   $self->AddDataReal   ($self->{"System load average for 15 minutes"}, $la15 );
   $self->AddDataInteger($self->{"Total number of processes"}, $proc_total );
   $self->AddDataInteger($self->{"Number of sleeping processes"}, $proc_sleeping );
   $self->AddDataInteger($self->{"Number of running processes"}, $proc_running );
   $self->AddDataInteger($self->{"Number of zombie processes"}, $proc_zombie );
   $self->AddDataInteger($self->{"Number of stopped processes"}, $proc_stopped );
   $self->AddDataInteger($self->{"Number of users"}, $nb_users);
   $self->AddDataInteger($self->{"Number of different users"}, $nb_diff_users);

   ##===========================================================================
   ## Set alarms
   ##===========================================================================
   ## TooMuchProcesses
   if(       defined( $self->{"Cfg_TooMuchProcesses_Error"} ) &&
        $proc_total > $self->{"Cfg_TooMuchProcesses_Error"} &&
             defined( $self->{"TooMuchProcesses_Error"} ) )
   { $self->AddAlarm( $self->{"TooMuchProcesses_Error"} ); }
   elsif(    defined( $self->{"Cfg_TooMuchProcesses_Warning"} ) &&
        $proc_total > $self->{"Cfg_TooMuchProcesses_Warning"} &&
             defined( $self->{"TooMuchProcesses_Warning"} ) )
   { $self->AddAlarm( $self->{"TooMuchProcesses_Warning"} ); }

   ## TooMuchZombies
   if(       defined( $self->{"Cfg_TooMuchZombies_Error"} ) &&
       $proc_zombie > $self->{"Cfg_TooMuchZombies_Error"} &&
             defined( $self->{"TooMuchZombies_Error"} ) )
   { $self->AddAlarm( $self->{"TooMuchZombies_Error"} ); }
   elsif(    defined( $self->{"Cfg_TooMuchZombies_Warning"} ) &&
       $proc_zombie > $self->{"Cfg_TooMuchZombies_Warning"} &&
             defined( $self->{"TooMuchZombies_Warning"} ) )
   { $self->AddAlarm( $self->{"TooMuchZombies_Warning"} ); }

   ## LoadAverage
   if(       defined( $self->{"Cfg_LoadAverage_Error"} ) &&
               $la5 > $self->{"Cfg_LoadAverage_Error"} &&
             defined( $self->{"LoadAverage_Error"} ) )
   { $self->AddAlarm( $self->{"LoadAverage_Error"} ); }
   elsif(    defined( $self->{"Cfg_LoadAverage_Warning"} ) &&
               $la5 > $self->{"Cfg_LoadAverage_Warning"} &&
             defined( $self->{"LoadAverage_Warning"} ) )
   { $self->AddAlarm( $self->{"LoadAverage_Warning"} ); }

   ## Check processes alarms
   foreach my $proc_entry (@{$self->{PROCESSES}}) {
       my ($cmd, $actionset) = @$proc_entry;
       my $ok = 0;
       foreach my $run_proc (@$cmdlines) {
	   if($cmd eq $run_proc) {
	       $ok = 1;
	       last;
	   }
       }

       if(! $ok) {
	   $self->AddAlarm( $self->{"Process_${cmd}_Error"} );
	   $actionset->Run("Error");
       }
   }
   
   ##===========================================================================
   ## Modules
   ##===========================================================================
   # filesystems
   $self->CollectData_FS( );

   # eth interfaces
   $self->CollectData_Eth( );

   # services
   $self->CollectData_Services( );

}

##******************************************************************************
## Method GetMemoryAndSwapValues  private
##  Description  : gets the memory and swap values from the /proc/meminfo file
##  Parameters   : none
##  Return value : memory total, memory used, memory free, memory shared, memory
#                  buffers, memory cached, swap total, swap free, swap used
##******************************************************************************
sub GetMemoryAndSwapValues
{
   my $self = shift;
   my ($memorytotal, $memoryused, $memoryfree,
       $memoryshared, $memorybuffers, $memorycached,
       $swaptotal, $swapused, $swapfree) = (-1,-1,-1,-1,-1,-1,-1,-1,-1);

   open(MFD, "/proc/meminfo") || Die("/proc/meminfo: $!");

   while( my $l=<MFD> )
   {
      if( $l =~ /^Mem:/ )
      {
         $l =~ s/^Mem: *(.*)/$1/;
         ($memorytotal, $memoryused, $memoryfree, $memoryshared, $memorybuffers,
          $memorycached) = split( /[ \t\n]+/, $l );

      }
      elsif( $l =~ /^Swap:/ )
      {
         ($swaptotal, $swapused, $swapfree) =
         ( $l =~ /^Swap: *([0-9]*) *([0-9]*) *([0-9]*)/ );
      }
   }

   close(MFD);

   return ($memorytotal, $memoryused, $memoryfree,
           $memoryshared, $memorybuffers, $memorycached,
           $swaptotal, $swapused, $swapfree);
}

##******************************************************************************
## Method GetStatValues  private
##  Description  : gets the values from the /proc/stat file
##  Parameters   : none
##  Return value : cpuuser value, cpusystem value, cpunice value, cpuidle value
##                 the context changing value, the processes value
##******************************************************************************
sub GetStatValues
{
   my $self = shift;

   my ($cpuuser, $cpusyst, $cpunice, $cpuidle, $contextchanging) = (-1,-1,-1,-1,-1);

   open( CFD, "/proc/stat" ) || Die("/proc/stat: $!");

   while( my $l=<CFD> )
   {
      if( $l =~ /^cpu / )
      {
         my ($cpuuser_read, $cpusyst_read, $cpunice_read, $cpuidle_read) =
         ( $l =~ /^cpu *([0-9]*) *([0-9]*) *([0-9]*) *([0-9]*)/ );

	 
	 if(!defined $self->{"CPU_USER"}) {
	     $self->{"CPU_USER"} = 0;
	     $self->{"CPU_SYST"} = 0;
	     $self->{"CPU_NICE"} = 0;
	     $self->{"CPU_IDLE"} = 0;
	 }
	 
         $cpuuser = $cpuuser_read - $self->{"CPU_USER"};
         $cpusyst = $cpusyst_read - $self->{"CPU_SYST"};
         $cpunice = $cpunice_read - $self->{"CPU_NICE"};
         $cpuidle = $cpuidle_read - $self->{"CPU_IDLE"};

         my $cputotal = $cpuuser + $cpusyst + $cpunice + $cpuidle;

         $cpuuser = ( $cpuuser * 100 )/$cputotal;
         $cpusyst = ( $cpusyst * 100 )/$cputotal;
         $cpunice = ( $cpunice * 100 )/$cputotal;
         $cpuidle = ( $cpuidle * 100 )/$cputotal;

         $self->{"CPU_USER"} = $cpuuser_read;
         $self->{"CPU_SYST"} = $cpusyst_read;
         $self->{"CPU_NICE"} = $cpunice_read;
         $self->{"CPU_IDLE"} = $cpuidle_read;
      }
      elsif( $l =~ /^ctxt / )
      {
         ($contextchanging) = ( $l =~ /^ctxt *([0-9]*)/ );
      }
   }

   close(CFD);

   return ($cpuuser, $cpusyst, $cpunice, $cpuidle, $contextchanging);
}

##******************************************************************************
## Method GetLoadAveragesValues private
##  Description  : gets the values from the /proc/loadavg file
##  Parameters   : none
##  Return value : the 3 averages
##******************************************************************************
sub GetLoadAveragesValues
{
   my $self = shift;
   my ($la1,$la5,$la15) = (-1,-1,-1);
   open( FILE, "/proc/loadavg" ) || Die("/proc/loadavg: $!");
   my $l=<FILE> || Die("/proc/loadavg: $!");
   ($la1,$la5,$la15) = split( / /, $l );
   close(FILE);
   return ($la1,$la5,$la15);
}

##******************************************************************************
## Method GetProcessesValues private
##  Description  : gets the values from the /proc/[0-9]*/status files, 
##                 and create the commande lines list
##  Parameters   : none
##  Return value : number of total, sleeping, running, zombie, stopped processes
##******************************************************************************
sub GetProcessesValues
{
   my ($total, $sleeping, $running, $zombie, $stopped) = (0,0,0,0,0);

   my @cmdline_list;

   #foreach my $proc (glob "/proc/[0-9]*/status")
   #--> glob failed (child exited with status -1, core dumped)

   if( opendir(PROC, "/proc") )
   {
      foreach my $proc (grep(/^[0-9]+$/, readdir PROC))
      {
         open( FILE, "/proc/$proc/status" ) || next;
         my $line = <FILE> || next;
         $line = <FILE>;
         my $status = (split( /[ \t]+/, $line ))[1] || next;

         $total ++;
         if( $status eq "S" || $status eq "D" )
         { $sleeping++; }
         elsif( $status eq "T" )
         { $stopped++; }
         elsif( $status eq "Z" )
         { $zombie++; }
         elsif( $status eq "R" )
         { $running++; }

         close( FILE );

	 open(F_CMDLINE, "/proc/$proc/cmdline") || next;
	 $line = join (" ",split(/\x00/, <F_CMDLINE>));
	 push @cmdline_list, $line;
	 close(F_CMDLINE);
      }
   }

   closedir PROC;

   return ($total, $sleeping, $running, $zombie, $stopped, \@cmdline_list);

}

##******************************************************************************
## Method GetUsersValues  private
##  Description  : gets the numbers of users via who command
##  Parameters   : none
##  Return value : the numbers of users, th number of different users
##******************************************************************************
sub GetUsersValues
{
   my ($nb_users,$nb_diff_users) = (-1,-1);

   open( PIPE, "who | cut -d' ' -f 1 | wc -l |" ) || Die($!);
   $nb_users = int(<PIPE>);
   close PIPE;

   open( PIPE, "who | cut -d' ' -f 1 | sort | uniq | wc -l |" ) || Die($!);
   $nb_diff_users = int(<PIPE>);
   close PIPE;

   return ($nb_users,$nb_diff_users);
}

##******************************************************************************
## Method FS_Create_Alarms_Values
##  Description  : call one time in -r mode, creates alarms values from
##                 config file
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub FS_Create_Alarms_Values
{
   my $self = shift;

   ##===========================================================================
   ## Parsing /etc/fstab for types
   ##===========================================================================
   my %types = ();
   open( FSTAB, "/etc/fstab" ) || Die("/etc/fstab : $!");
   while(my $l=<FSTAB>)
   {
      next if( $l =~ /^#/ );
      next if( $l =~ /^[ \t]*$/ );
      my ($device,$mountpoint,$type,$options) = split( /[ \t]+/, $l );
      $types{$device} = $type;
   }
   close(FSTAB);

   ##===========================================================================
   ## For each alarm :
   ##===========================================================================
   foreach my $alarmdescription (@{$self->{_ALARMDESCRIPTIONSET}->{ALARMDESCRIPTIONLIST}})
   {
      my $str = ${alarmdescription}->GetInternal()."_".${alarmdescription}->GetLevel();
      next if( $str !~ /^FS_/ );

      if( defined( $self->{"Cfg_".$str} ) )          # by name
      { $self->{$str."_Limit"} = $self->{"Cfg_".$str}; }
      else
      {
         my $str2;
         ($str2 = $str) =~ s/^FS_([^_]*)_(.*)$/FS_$types{$1}_$2/;
         if( defined( $self->{"Cfg_".$str2} ) )      # by type
         { $self->{$str."_Limit"} = $self->{"Cfg_".$str2}; }
         else
         {
            ($str2 = $str) =~ s/^FS_[^_]*_(.*)$/FS_*_$1/;
            if( defined( $self->{"Cfg_".$str2} ) )   # generic
            { $self->{$str."_Limit"} = $self->{"Cfg_".$str2}; }
         }
      }

   }
}

my $bool_FS_Create_Alarms_Values_already_called=0;

##******************************************************************************
## Method CollectData_FS  protected
##  Description  : collects data and alarms for FS
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub CollectData_FS
{
   my $self = shift;

   if( !$bool_FS_Create_Alarms_Values_already_called )
   {
      $bool_FS_Create_Alarms_Values_already_called =1;
      $self->FS_Create_Alarms_Values();
   }

   ##===========================================================================
   ## Space/Inodes
   ##===========================================================================
   foreach my $foo ("Space","Inodes")
   {
      ##========================================================================
      ## df
      ##========================================================================
      my $df_line;
      if( $foo eq "Space" )
      { $df_line = "df -klP"; }
      else
      { $df_line = "df -ilP"; }
      open(DF,$df_line."|") || Die("$df_line : $!");

      my $line = <DF> || Die("$df_line : $!");
      LINE: while($line = <DF>)
      {
         my ($device,$z,$used,$available) = split( /[\s]+/, $line );
         next LINE if( !defined($device) || !defined($used) || !defined($available) );
         next LINE if( ($used == 0) || ($available == 0) );

         if( $foo eq "Space" )
         { $self->CollectData_FS_Quota( $device ); }
         ##---------------------------------------------------------------------
         ## NumberFree
         ##---------------------------------------------------------------------
         if( defined( $self->{"FS_".$device."_".$foo."_NumberFree"} ) )
         {
            $self->AddDataReal( $self->{"FS_".$device."_".$foo."_NumberFree"},
                                   $available );
            ##::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
            ## Alarms
            ##::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
            if(      defined( $self->{"FS_".$device."_".$foo."_NumberFree_Error"} ) &&
                     defined( $self->{"FS_".$device."_".$foo."_NumberFree_Error_Limit"} ) &&
                $available <= $self->{"FS_".$device."_".$foo."_NumberFree_Error_Limit"} )
            { $self->AddAlarm($self->{"FS_".$device."_".$foo."_NumberFree_Error"}); }
            elsif(   defined( $self->{"FS_".$device."_".$foo."_NumberFree_Warning"} ) &&
                     defined( $self->{"FS_".$device."_".$foo."_NumberFree_Warning_Limit"} ) &&
                $available <= $self->{"FS_".$device."_".$foo."_NumberFree_Error_Limit"} )
            { $self->AddAlarm($self->{"FS_".$device."_".$foo."_NumberFree_Error"}); }
            elsif(   defined( $self->{"FS_".$device."_".$foo."_NumberFree_Warning"} ) &&
                     defined( $self->{"FS_".$device."_".$foo."_NumberFree_Warning_Limit"} ) &&
                $available <= $self->{"FS_".$device."_".$foo."_NumberFree_Warning_Limit"} )
            { $self->AddAlarm($self->{"FS_".$device."_".$foo."_NumberFree_Warning"}); }
         }

         ##---------------------------------------------------------------------
         ## PercentUsed
         ##---------------------------------------------------------------------
         if( defined( $self->{"FS_".$device."_".$foo."_PercentUsed"} ) )
         {
            my $percent = ( $used * 100 ) / ( $used + $available );
            $percent = int(100*($percent+0.005)) / 100;

            $self->AddDataReal( $self->{"FS_".$device."_".$foo."_PercentUsed"},
                                $percent );
            ##::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
            ## Alarms
            ##::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
            if(      defined( $self->{"FS_".$device."_".$foo."_PercentUsed_Error"} ) &&
                     defined( $self->{"FS_".$device."_".$foo."_PercentUsed_Error_Limit"} ) &&
                $percent   >= $self->{"FS_".$device."_".$foo."_PercentUsed_Error_Limit"} )
            { $self->AddAlarm($self->{"FS_".$device."_".$foo."_PercentUsed_Error"}); }
            elsif(   defined( $self->{"FS_".$device."_".$foo."_PercentUsed_Warning"} ) &&
                     defined( $self->{"FS_".$device."_".$foo."_PercentUsed_Warning_Limit"} ) &&
                $percent   >= $self->{"FS_".$device."_".$foo."_PercentUsed_Warning_Limit"} )
            { $self->AddAlarm($self->{"FS_".$device."_".$foo."_PercentUsed_Warning"}); }
         }
      }
      close( DF );

   }

}

##******************************************************************************
## Method CollectData_FS_Quota  private
##  Description  : collects quota data and alarms for FS
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub CollectData_FS_Quota
{
   my ($self,$device) = @_;

   foreach my $foo ("Users", "Groups")
   {
      if( defined( $self->{"FS_".$device."_Inodes_Quota_"."$foo"} ) ||
          defined( $self->{"FS_".$device. "_Space_Quota_"."$foo"} ) )
      {
         local %SIG;
         $SIG{__WARN__} = sub { };

         if( $foo eq "Users" )
         { open(REPQUOTA,"repquota -u $device|") || return; }
         else
         { open(REPQUOTA,"repquota -g $device|") || return; }

         my $line = <REPQUOTA>;
         if( !defined($line) ) { close( REPQUOTA ); return; }
         $line = <REPQUOTA>;
         if( !defined($line) ) { close( REPQUOTA ); return; }

         my ($inodes,$space) = (0,0);
         while($line = <REPQUOTA>)
         {
            my ($s,$i) = ( $line =~ /.*([+-])([+-]).*/ );
            if( defined($s) && ($s eq "+") )
            { $space++; }
            if( defined($i) && ($i eq "+") )
            { $inodes++; }
         }
         close( REPQUOTA );

         if( defined( $self->{"FS_".$device."_Inodes_Quota_"."$foo"} ) )
         {
            $self->AddDataInteger( $self->{"FS_".$device."_Inodes_Quota_"."$foo"},
                                   $inodes );
         }
         if( defined( $self->{"FS_".$device."_Space_Quota_"."$foo"} ) )
         {
            $self->AddDataInteger( $self->{"FS_".$device."_Space_Quota_"."$foo"},
                                   $space );
         }
      }
   }
}

##******************************************************************************
## Method CollectData_Eth  protected
##  Description  : collects data and alarms for eth interfaces
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub CollectData_Eth
{
   my $self = shift;

   open(FILE,"/proc/net/dev") || Die("/proc/net/dev: $!");

   LINE: while( my $line=<FILE> )
   {
      next LINE if( $line =~ /^Inter/ );
      next LINE if( $line =~ /^ *face/ );
      $line =~ s/[ \t]*(.*)/$1/;
      my ($interface, $r_bytes, $r_packets, $r_errs, $r_drop, $r_fifo, $r_frame,
          $r_compressed, $r_multicast, $t_bytes, $t_packets, $t_errs, $t_drop,
          $t_fifo, $t_colls, $t_carrier, $t_compressed) =
         split( /[ \t\n:]+/, $line );
      $interface =~ s/(.*):/$1/;

      next LINE if( $interface eq "lo" );

      if( defined( $self->{"Eth_".$interface."_"."Received_bytes"} ) )
      {
         $self->AddDataReal( $self->{"Eth_".$interface."_"."Received_bytes"},
                                $r_bytes );
      }
      if( defined( $self->{"Eth_".$interface."_"."Received_packets" } ) )
      {
         $self->AddDataReal( $self->{"Eth_".$interface."_"."Received_packets"},
                                $r_packets );
      }
      if( defined( $self->{"Eth_".$interface."_"."Received_errs" } ) )
      {
         $self->AddDataReal( $self->{"Eth_".$interface."_"."Received_errs"},
                                $r_errs );
      }
      if( defined( $self->{"Eth_".$interface."_"."Transmited_bytes" } ) )
      {
         $self->AddDataReal( $self->{"Eth_".$interface."_"."Transmited_bytes"},
                                $t_bytes );
      }
      if( defined( $self->{"Eth_".$interface."_"."Transmited_packets" } ) )
      {
         $self->AddDataReal( $self->{"Eth_".$interface."_"."Transmited_packets"},
                                $t_packets );
      }
      if( defined( $self->{"Eth_".$interface."_"."Transmited_errs" } ) )
      {
         $self->AddDataReal( $self->{"Eth_".$interface."_"."Transmited_errs"},
                                $t_errs );
      }
      if( defined( $self->{"Eth_".$interface."_"."Transmited_colls" } ) )
      {
         $self->AddDataReal( $self->{"Eth_".$interface."_"."Transmited_colls"},
                                $t_colls );
      }
   }

   close(FILE);

}

##******************************************************************************
## Method CollectData_Services  protected
##  Description  : collects data and alarms for services
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub CollectData_Services
{
   my $self = shift;
   my ($time,$error);

   if( defined( $self->{"HTTP"} ) &&
       defined( $self->{"Cfg_HTTP"} ) )
   {
      ($time,$error) = $self->CollectData_Services_HTTP();
      $self->CollectData_Services_Generic( "HTTP", $time, $error );
   }

   if( defined( $self->{"FTP"} ) &&
       defined( $self->{"Cfg_FTP"} ) )
   {
      ($time,$error) = $self->CollectData_Services_FTP();
   }
}

##******************************************************************************
## Method CollectData_Services_Generic  private
##  Description  : collects data and alarms for generic services
##  Parameters   :
##  Return value : none
##******************************************************************************
sub CollectData_Services_Generic
{
   my ($self,$str,$time,$error) = @_;

   if( $error != 1 )
   { $self->AddDataReal( $self->{$str}, $time ); }

   if(       defined( $self->{"Cfg_".$str."_Error"} ) &&
             ($time > $self->{"Cfg_".$str."_Error"} || $error == 1 ) &&
             defined( $self->{$str."_Error"} ) )
   { $self->AddAlarm( $self->{$str."_Error"} ); }
   elsif(    defined( $self->{"Cfg_".$str."_Warning"} ) &&
              $time > $self->{"Cfg_".$str."_Warning"} &&
             defined( $self->{$str."_Warning"} ) )
   { $self->AddAlarm( $self->{$str."_Warning"} ); }

}

##******************************************************************************
## Method CollectData_Services_HTTP  private
##  Description  : collects data and alarms for HTTP service
##  Parameters   : none
##  Return value : the time, the error
##******************************************************************************
sub CollectData_Services_HTTP
{
   my $self = shift;

   my $timeout = 180;
   if( defined($self->{"Cfg_HTTP_Error"}) )
   { $timeout = $self->{"Cfg_HTTP_Error"} + 2; }

   ##===========================================================================
   my $ua       = LWP::UserAgent->new;
   $ua->timeout($timeout);
   my $request  = HTTP::Request->new(GET => 'http://localhost/');
   my $beforetime = gettimeofday();
   my $response = $ua->simple_request($request);
   my $aftertime = gettimeofday();

   ##===========================================================================
   my $time = $aftertime - $beforetime;
   $time = (int($time*10000))/10000;

   my $error = 0;
   if( $response->is_error() )
   { $error = 1; }

   return ($time,$error);

}

##******************************************************************************
## Method CollectData_Services_FTP  private
##  Description  : collects data and alarms for FTP service
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub CollectData_Services_FTP
{
   my $self = shift;

   my $timeout = 180;
   if( defined($self->{"Cfg_FTP_Error"}) )
   { $timeout = $self->{"Cfg_FTP_Error"} + 2; }

   ##===========================================================================
   my $error = 0;
   my $beforetime = gettimeofday();

   my $ftp = Net::FTP->new("localhost", Debug => 0, Timeout=>30);
   if( !defined( $ftp ) || !$ftp->quit() )
   { $error = 1; }

   my $aftertime = gettimeofday();

   ##===========================================================================
   my $time = $aftertime - $beforetime;
   $time = (int($time*10000))/10000;

   return ($time,$error);
}

##******************************************************************************
1;

__END__

=head1 NAME

LinuxAgent.pm - Perl Extension for LogTrend : LinuxAgent Agent

=head1 SYNOPSIS

  use LogTrend::Agent::LinuxAgent;

  LogTrend::Agent::LinuxAgent->new();

=head1 DESCRIPTION

LogTrend::Agent::LinuxAgent is a Perl extention implementing a
Linux Agent for LogTrend.

This module is not intended for direct use, but to be called
through its intertface utility called LinuxAgent.

As it inherits from LogTrend::Agent, the various Agent command
line switches apply to it.

The Linux Agent is aimed at collecting information about a
Linux system : CPU, memory, swap, system load avg, number of
processes, free/used space/inode on disc devices, network devices
statistics, HTTP and FTP response time. Some alarms could be thrown
when simple conditions are reached.

=head1 PRE-REQUISITES

The following Perl modules are definitly needed for this agent to
work:

    HTTP::Status
    LWP::UserAgent
    Net::FTP
    Time::HiRes
    XML::DOM

The first and the second ones are needed for HTTP statistics.
The third one for FTP statistics. The fourth one for time
answers form such servers. The last one for XML analysis.

=head1 CONFIGURATION

The Linux Agent configuration is done using an XML file.

See documentation:
/usr/share/doc/LogTrend/Agent/install-guide/agent-install-guide.ps

=head1 AUTHOR

Sylvain Lhullier -- Atrid Systmes (s.lhullier@atrid.fr)

=head1 COPYRIGHT

Copyright 2001, Atrid Systme http://www.atrid.fr/

Project home page: http://www.logtrend.org/

Licensed under the same terms as LogTrend project is.

=head1 WARRANTY

THIS SOFTWARE COMES WITH ABSOLUTLY NO WARRANTY OF ANY KIND.
IT IS PROVIDED "AS IS" FOR THE SOLE PURPOSE OF EVENTUALLY
BEEING USEFUL FOR SOME PEOPLE, BUT ONLY AT THEIR OWN RISK.

=cut

