#!/usr/bin/perl -w 

#$Header: /home2/cvsroot/LogTrend/Agent/FtpAgent.pm,v 1.8 2001/11/07 16:08:07 lsimonneau Exp $
##*****************************************************************************
##  FtpAgent
##  Description  : 
##
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Author       : Laurent Simonneau (l.simonneau@atrid.fr)
##*****************************************************************************
#$Log: FtpAgent.pm,v $
#Revision 1.8  2001/11/07 16:08:07  lsimonneau
#*** empty log message ***
#
#Revision 1.7  2001/09/19 08:29:16  lsimonneau
#Modify POD doc.
#
#Revision 1.6  2001/09/14 15:22:27  lsimonneau
#Add documentation in pod.
#
#Revision 1.5  2001/08/31 07:26:08  lsimonneau
#Minor bugfixes.
#
#Revision 1.4  2001/08/29 16:11:03  lsimonneau
#Add Proxy entry in Config file.
#
#Revision 1.3  2001/08/22 08:22:43  lsimonneau
#Major bugfixes : remove 'ping' because of proxy problems.
#                 'Can not reach host' alarm => Can't connect to ftp port.
#		 ' FTP Server Down' no longer exists.
#		 LoginFailureAlarm and NbUserAlarm became optional.
#Add a lot of comments.
#
#Revision 1.2  2001/08/21 16:34:25  lsimonneau
#Minor bugfixes
#
#Revision 1.1  2001/07/19 16:40:27  fdesar
#
#Moved module files to the right directories
#Updated package names and uses to reflect those changes
#Corrected bug in SnortAgent.pm for negating first value in SID parsing
#
#Revision 1.8  2001/06/27 12:37:35  lsimonneau
#Rien qui ne vaille un commentaire.
#
#Revision 1.7  2001/06/26 15:22:04  lsimonneau
#Quelques amliorations mineur du code.
#
#Revision 1.6  2001/06/26 08:46:06  lsimonneau
#Add too much login failure alarm.
#
#Revision 1.5  2001/06/25 12:32:23  lsimonneau
#Amlioration du code. Ne reparcours plus tout le fichier de log a chaque fois.
#
#Revision 1.4  2001/06/21 15:38:17  lsimonneau
#die -> Die
#
#Revision 1.3  2001/06/21 15:21:41  lsimonneau
#Minor bugfixes
#
#Revision 1.2  2001/06/21 15:20:38  lsimonneau
#Minor bugfixes
#
#Revision 1.1  2001/06/15 08:50:09  lsimonneau
##Premiere release du FtpAgent.
##
##5 Data : file transfert time
##         connected user
##         bytes sent
##         bytes sent daily
##         bytes sent for /path/
##
##4 Alarms : Can not reach host
##           More than /nbr/ users  connected
##           Too much login failure
#

package LogTrend::Agent::FtpAgent;

use strict;

use vars qw( @ISA );

use POSIX qw(tmpnam);
use XML::DOM;
use LogTrend::Agent;

use Net::FTP;
use Net::Domain;
use URI;

@ISA = qw( LogTrend::Agent );

my $name = "FtpAgent";
my $version = "1.0.0";

##******************************************************************************
## Constructor  public > Agent
##  Description  : creat a new FtpAgent
##  Parameters   : none
##******************************************************************************
sub new
{
    my ( $classname ) = @_;
    
    my $self = $classname->SUPER::new( $name, $version );
    bless($self, $classname);

    return $self;
}



##******************************************************************************
## Method ParseXMLConfigFile  public  (>Agent)
##  Description  : parses the XML config file
##  Parameters   : the file name to parse, the agent state
##  Return value : none
##******************************************************************************
sub ParseXMLConfigFile
{       
    my ($self,$file,$agentstate) = @_;
    $self->SUPER::ParseXMLConfigFile( $file, $agentstate );
  

    ##===========================================================================
    ## Agent-specific configuration parameters :
    ## use $self->{FOO} to stock information ( all $self->{_FOO} are reserved )
    ##===========================================================================

    my $parser = new XML::DOM::Parser() || die("XML::DOM::Parser: $!");
    my $doc = $parser->parsefile( $file ) || die("$file :$!");
    
    my $rootlist = $doc->getElementsByTagName("Configuration") ||
	die("$file :No \"Configuration\" tag.");
    my $rootnode = $rootlist->item(0) || die("$file :No \"Configuration\" tag.");
    
    ##
    ## Tag 'Specific'
    ## 
    $rootlist = $rootnode->getElementsByTagName("Specific") ||
	die("$file :No \"Specific\" tag.");
    $rootnode = $rootlist->item(0) || die("$file :No \"Specific\" tag.");

    ##
    ## Tag 'FtpServer' (required, all attributes are required)
    ## 
    $rootlist = $rootnode->getElementsByTagName("FtpServer") ||
	die("$file :No \"FtpServer\" tag.");
    my $ftpnode = $rootlist->item(0) || die("$file :No \"FtpServer\" tag.");
    
    $self->{HOST} = $ftpnode->getAttribute ("host") 
	|| die("Error in \"FtpServer\" tag, can't found 'host' attribute.");

    $self->{LOGIN} = $ftpnode->getAttribute ("login") 
	|| die("Error in \"FtpServer\" tag, can't found 'login' attribute.");

    $self->{PASSWORD} = $ftpnode->getAttribute ("password") 
	|| die("Error in \"FtpServer\" tag, can't found 'password' attribute.");
    
    ##
    ## Tag 'Proxy' (optional)
    ##
    my $proxylist = $rootnode->getElementsByTagName("Proxy");
    if(my $proxynode =  $proxylist->item(0)) {	
	my $cdatanode = $proxynode->getFirstChild
	    || die("Error in \"Proxy\" tag.");
	$self->{PROXY} = $cdatanode->getNodeValue();
    }

    ##
    ## Tag 'FileToDownload' (required)
    ## 
    $rootlist = $rootnode->getElementsByTagName("FileToDownload")
	|| die("Error : no \"FileToDownload\" tag.");

    my $filenode = $rootlist->item(0) 
	|| die("Error no \"FileToDownload\" tag.");

    $filenode = $filenode->getFirstChild 
	|| die("Error, \"FileToDownload\" tag should not be empty.");

    $self->{FILENAME} = $filenode->getNodeValue();

    ##
    ## Tag 'LoginFailureAlarm' (optional)
    ## 
    $rootlist = $rootnode->getElementsByTagName("LoginFailureAlarm");
    if(my $loginnode = $rootlist->item(0)){
	$self->{LOGIN_LOG_FILE} = $loginnode->getAttribute ("log_path") 
	    || die("Error in \"LoginFailureAlarm\" tag, can't found 'log_path' attribute.");
	
	$self->{LOGIN_NB_FAILURE} = $loginnode->getAttribute ("nb_failure") 
	    || die("Error in \"LoginFailureAlarm\" tag, can't found 'nb_failure' attribute.");
	
	$self->{LOGIN_TIME_INTERVAL} = $loginnode->getAttribute ("time_interval") 
	    || die("Error in \"LoginFailureAlarm\" tag, can't found 'time_interval' attribute.");
    }

    ##
    ## Tags 'Report' (optional)
    ## 
    if($rootlist = $rootnode->getElementsByTagName("Report")) {
	my $n = $rootlist->getLength;
	
	for (my $i = 0; $i < $n; $i++)
	{
	    my $node = $rootlist->item ($i);
	    my $path = $node->getAttribute ("path") 
		|| die("Error in \"Report\" tag, can't found 'path' attribute.");
	    
	    push @{$self->{PATH_TO_REPORT}}, $path;
	}
    }

    ##
    ## Tags 'NbUserAlarm' (optional)
    ## 
    if($rootlist = $rootnode->getElementsByTagName("NbUserAlarm") and
       $rootnode = $rootlist->item(0)) {

	my $limit = $rootnode->getAttribute ("limit") 
	    || die("Error in \"NbUserAlarm\" tag, can't found 'limit' attribute.");
	
	$self->{NB_USER_MAX} = $limit;
    }   
}


##******************************************************************************
## Method CreateAgentDescription  public  (>Agent)
##  Description  : creates an agent's description
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub CreateAgentDescription
{
   my $self = shift;
   my ($d,$a) = (1,1);

   ## Must use AddADataDescription and AddAnAlarmDescription methods
   ## Data
   $self->AddADataDescription($d++, "Integer", "none", "file transfer time",  "");

   ## Alarm
   $self->AddAnAlarmDescription($a++, "Error", "Can not reach host", "HostDown" );
   $self->AddAnAlarmDescription($a++, "Error", "FTP service not available", "FTPServerDown");

   if($self->{HOST} ne Net::Domain::hostfqdn and
      $self->{HOST} ne Net::Domain::hostname) {
       return;
   }

   $self->AddADataDescription($d++, "Integer", "none", "connected user",  "");
   $self->AddADataDescription($d++, "Integer", "bytes", "bytes sent",  "");
   $self->AddADataDescription($d++, "Real", "percentage", "bytes sent daily", "");

   if(defined $self->{LOGIN_NB_FAILURE}) {
       $self->AddAnAlarmDescription($a++, "Warning", "Too much login failure", "LoginFailure");
   }

   foreach my $path (@{$self->{PATH_TO_REPORT}}) {
       $self->AddADataDescription($d++, "Real", "percentage", "bytes sent for $path", "");
   }

   if (defined $self->{NB_USER_MAX}) {
       $self->AddAnAlarmDescription($a++, "Error", 
				    "More than $self->{NB_USER_MAX} users connected", "TooMuchUsers");
   }
}


##******************************************************************************
## Method initVariable private 
##  Description  : Initialize private variables
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub initVariable {
    my $self = shift;

    $self->{CUR_LINE} = 1;
    $self->{CUR_DAY} = "";
    $self->{NB_DAYS} = 0;
    $self->{TOTAL_BYTES_SENT} = 0;
    foreach my $path (@{$self->{PATH_TO_REPORT}}) {
	$self->{PATH_BYTES_SENT}->{$path} = 0;
    }
    
    ##
    ## Don't analyze previous data
    ## Start to the end of the log file and wait for new data
    ## Use wc to count lines in the log file
    ##
    if(defined $self->{LOGIN_LOG_FILE}) {
	my $wc_result = `wc -l $self->{LOGIN_LOG_FILE}`; 
	$wc_result =~ s/^\s+//;
	$self->{LOGIN_CUR_LINE} = (split /\s+/, $wc_result)[0] + 1;
	$self->{LOGIN_HASH} = {};
    }
}


##******************************************************************************
## Method CollectData  public  (>Agent)
##  Description  : collects data and alarms
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub CollectData
{
    my $self = shift;
    
    if(defined $self->{PROXY}) {
	$ENV{ftp_proxy} = $self->{PROXY};
    }    
    
    ## 
    ## The first time, initialize variables
    ##
    $self->initVariable() unless (defined $self->{CUR_LINE});
    
    ##
    ## Try to reach the ftp service on host
    ##   if the server is up, download the file and measure download time
    ##   else, add an alarm
    ##
    my $ftp = Net::FTP->new($self->{HOST});
   
    if (! defined $ftp) {
	$self->AddAlarm ($self->{"HostDown_Error"});
    }
    else {
	# login to the ftp server
	$ftp->login($self->{LOGIN}, $self->{PASSWORD}) || die("Login failed");
	
	# measure file transfert time
	my $tmpfilename = POSIX::tmpnam();
	my $now = time;	
	$ftp->get($self->{FILENAME}, $tmpfilename) || die("Invalid file name $self->{FILENAME}");
	my $file_transfer_time = time - $now;
	unlink($tmpfilename);
	$ftp->close();

	$self->AddDataInteger($self->{"file transfer time"}, $file_transfer_time);
    }

    ##
    ## Stop the Data collection here if the agent is not running on the server
    ##
    if($self->{HOST} ne Net::Domain::hostfqdn and
       $self->{HOST} ne Net::Domain::hostname) {
	return;
    }

    ##
    ## Get the number of connected users
    ##
    my $ftpcount_result = `ftpcount`;
    
    my $nb_user_connected = 0;
    if ($ftpcount_result =~ /(\d+) user(s?)$/m) {
	$nb_user_connected = $1;
    }
    
    $self->AddDataInteger($self->{"connected user"}, $nb_user_connected);
    
    if(defined $self->{NB_USER_MAX} and $nb_user_connected > $self->{NB_USER_MAX}) {
	$self->AddAlarm($self->{"TooMuchUsers_Error"});
    }
    
    ##
    ## Retrieve data from logs
    ##
    $self->getStatFromLog;
    $self->lookForLoginFailure if defined $self->{LOGIN_LOG_FILE};
}



##******************************************************************************
## Method lookForLoginFailure private 
##  Description  : Look for login failure in the ExtendedLog file
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub lookForLoginFailure {
    my $self = shift;

    ##
    ## Get new lines in the log file
    ##
    open(LOGIN_LOG, "tail -n +$self->{LOGIN_CUR_LINE} $self->{LOGIN_LOG_FILE} | ") 
	|| die("Can't found login log file");
    
    ##
    ## Look for loggin failed report
    ##
    while (<LOGIN_LOG>) {
	$self->{LOGIN_CUR_LINE}++;

	my @line = split;

	# is it a "wrong password" report ?
	next if $line[4] ne "\"PASS";
	next if $line[6] != 530;
	
	# sort failure date by client
	push @{$self->{LOGIN_HASH}->{$line[0]}}, $line[3];	
    }
    close(LOGIN_LOG);
    
    ##
    ## for each client, count the loggin failure
    ##
    foreach my $key (keys %{$self->{LOGIN_HASH}}) {       
	my $failure_list = \@{$self->{LOGIN_HASH}->{$key}};
	
	# check there are more than LOGIN_NB_FAILURE in LOGIN_TIME_INTERVAL seconds
	for(my $i=0;  defined $failure_list->[$i+$self->{LOGIN_NB_FAILURE}-1]; $i++){

	    if($failure_list->[$i+$self->{LOGIN_NB_FAILURE}-1] - $failure_list->[$i] <= $self->{LOGIN_TIME_INTERVAL}) {
		$self->AddAlarm($self->{"LoginFailure_Warning"});
		print "Alarm $key\n";
	    }
	}
	
	# remove old failure date in lists for each clients
	my $now = time;
	for(my $i=0; defined $failure_list->[$i]; $i++){
	    if($failure_list->[$i] + $self->{LOGIN_TIME_INTERVAL} < $now){
		shift @$failure_list;
	    }
	}
	
	# if there is no more date in list, remove the entry in hash
	if($#$failure_list == -1) {
	    print "Undefine $key\n";
	    delete $self->{LOGIN_HASH}->{$key};
	}
    }
}

##******************************************************************************
## Method getStatFromLog private
##  Description  : Retrieve data from xferlog file.
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub getStatFromLog {
    my $self = shift;

    return if  $#{$self->{PATH_TO_REPORT}} == -1;

    ##
    ## get new lines in log file
    ##
    open(LOG, "tail -n +$self->{CUR_LINE} /var/log/xferlog | ") 
	|| die("Can't found log file");
    

    ##
    ## This is copied from ftpstats(8)
    ##
    while (<LOG>) {
	$self->{CUR_LINE}++;
	
	my @line = split;
	
	##
	## is the first entry week day abbreviation ?
	##
	next if (length("$line[0]") != 3);
	
	my $offset;
	## check whether there is a valid 'username'
	if ($line[$#line-7] eq "a" or $line[$#line-7] eq "b") {
	    # yes, there is
	    # offset points to the first element just behind the filename
	    $offset = $#line - 7;
	} elsif ($line[$#line-6] eq "a" or $line[$#line-6] eq "b") {
	    $offset = $#line - 6;
	} else {
	    next;
	}
	
	# don't use upload info
	next if ($line[$offset+2] eq "i");
	
	# check if it's a new day
	my $new_day = substr($_, 0, 10) . substr($_, 19, 5);
	$self->{NB_DAYS}++ if $new_day ne $self->{CUR_DAY};
	$self->{CUR_DAY} = $new_day;
	
	# update bytes sent by path informations
	foreach my $path (@{$self->{PATH_TO_REPORT}}) {
	    if ((substr($line[8],0,length($path)) eq $path)) {
		$self->{PATH_BYTES_SENT}->{$path} += $line[7];
	    }
	}
	
	$self->{TOTAL_BYTES_SENT} += $line[7];
    }
    close LOG;
    
    # Add data
    foreach my $path (@{$self->{PATH_TO_REPORT}}) {
	$self->AddDataReal($self->{"bytes sent for $path"}, 
			   ($self->{PATH_BYTES_SENT}->{$path}*100)/$self->{TOTAL_BYTES_SENT});
    }
    
    $self->AddDataReal($self->{"bytes sent daily"}, $self->{TOTAL_BYTES_SENT}/$self->{NB_DAYS});
    
    $self->AddDataInteger($self->{"bytes sent"}, $self->{TOTAL_BYTES_SENT});
}


1;

__END__

=head1 NAME

FtpAgent.pm - Perl Extension for LogTrend : FtpAgent Agent

=head1 SYNOPSIS 

    use LogTrend::Agent::FtpAgent

    LogTrend::Agent::FtpAgent->new();

=head1 DESCRIPTION

LogTrend::Agent::FtpAgent is a Perl extention implementing a
Ftp server Agent for LogTrend.

This module is not intended for direct use, but to be called
through its intertface utility called FtpAgent.

As it inherits from LogTrend::Agent, the various Agent command
line switches apply to it.

The FtpAgent can work in two modes : local and remote.

In local mode (when the agent is running on the server), this agent collect data from log files.
In remote mode (when the agent is running on a remote machine), the agent just collects file transfert time.

=head2 Data and alarms collected in remote mode :

=over 2

=item *

Data: 
   - File transfert time : time needed to download a file from the server.

=item *

Alarms :
   - Can not reach host : the server is not available.

=back

=head2 Data and alarms collected in local mode :

=over 2

=item *

Data :
   - File transfert time : time needed to download a file from the server.
   - Connected user : number of user connected to the server.
   - Bytes sent : number of bytes sent by the server.
   - Bytes sent daily : number of bytes sent by the server each day.
   - Bytes sent for /path/ : percentage of data transfert dedicated to /path/.

=item *

Alarms :
   - Can not reach host : the server is not available.
   - More than /nbr/ users connected.
   - Too Login Failure.

=back
  
=head1 PRE-REQUISITES

The following Perl modules are definitly needed for this
agent to work:

    Net::FTP
    Net::Domain
    URI

=head1 CONFIGURATION

The Ftp Agent configuration is done using an XML file.

See documentation:
/usr/share/doc/LogTrend/Agent/install-guide/agent-install-guide.ps

=head1 AUTHOR

Laurent Simonneau -- Atrid Systmes (l.simonneau@atrid.fr)

=head1 COPYRIGHT

Copyright 2001, Atrid Systme http://www.atrid.fr/

Project home page: http://www.logtrend.org/

Licensed under the same terms as LogTrend project is.

=head1 WARRANTY

THIS SOFTWARE COMES WITH ABSOLUTLY NO WARRANTY OF ANY KIND.
IT IS PROVIDED "AS IS" FOR THE SOLE PURPOSE OF EVENTUALLY
BEEING USEFUL FOR SOME PEOPLE, BUT ONLY AT THEIR OWN RISK.

=cut

