#!/usr/bin/perl

#$Header: /var/lib/cvs/LogTrend/Consolidation.pm,v 1.15 2002/07/12 08:36:58 lsimonneau Exp $
##*****************************************************************************
##  ConsolidationSet
##  Description  : Parse XML File and create Consolidation object. 
##
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Author       : Laurent Simonneau l.simonneau@atrid.fr
##*****************************************************************************
#$Log: Consolidation.pm,v $
#Revision 1.15  2002/07/12 08:36:58  lsimonneau
#Minor bugfixes.
#
#Revision 1.14  2002/05/02 14:47:08  lsimonneau
#A lot of bugfixes.
#Critical bugfixes in consolidation.
#
#Revision 1.13  2002/04/24 07:52:10  lsimonneau
#Remove debug print.
#
#Revision 1.12  2001/11/06 16:01:18  lsimonneau
#Major bugfixes : can remove data correctly.
#
#Revision 1.11  2001/11/05 14:09:08  lsimonneau
#Major bugfixes.
#
#Revision 1.10  2001/10/29 10:33:38  lsimonneau
#Minor bugfixes, add debug outputs.
#
#Revision 1.9  2001/10/29 10:01:06  lsimonneau
#Minor bugfixes.
#
#Revision 1.8  2001/10/23 15:32:58  lsimonneau
#Major bugfixes : dtd, example and code are coherent for Default rules.
#
#Revision 1.7  2001/09/27 14:30:05  lsimonneau
#DIsable autocommit for DataBase and add Commit methods to Consolidation/ConsolidationPostgreSQLDataBase.pm to prevent data lost on die or sigtrem, sigint ...
#
#Revision 1.6  2001/09/24 15:29:10  lsimonneau
#Reimplementation of consolidation.
#
#Revision 1.3  2001/09/04 13:46:58  lsimonneau
#Major feature enhancements. Add --offset option to Consolidation.pl.
#
#Revision 1.2  2001/08/27 14:38:06  lsimonneau
#GetSourcesList (DataBaseAccess) returned values have changed.
#
#Revision 1.1  2001/08/23 15:48:37  lsimonneau
#Parser rewriting.
#
#Revision 1.2  2001/08/22 09:51:27  lsimonneau
#First usable version of Consolidation.
#
#Revision 1.1  2001/08/17 15:35:14  lsimonneau
#Ajout de Consolidation au CVS, parser XML OK.
#

package LogTrend::Consolidation;

use strict;
use LogTrend::Consolidation::ConsolidationPostgreSQLDataBase;
use LogTrend::Consolidation::DataBaseAnalyzer;
use LogTrend::Consolidation::Rules::RulesStdLib;
use XML::DOM;
use Data::Dumper;

##*****************************************************************************
## Constructor  public
##  Description  : creat a new Consolidation
##  Parameters   : the XML Root node of the consolidation and defaults rules.
##*****************************************************************************

sub new {
    my ($classname, $filename) = @_;
    my $self = {};

    bless($self, $classname);
    
    # parse the consolidation node
    $self->parseXML($filename);
    
    #load rules packages
    $self->loadRulesPackages;
    
    # Create the db analyzer
    $self->{DB_ANALYZER} = new LogTrend::Consolidation::DataBaseAnalyzer($self->{DB_HANDLE});

    return $self;
}


##*****************************************************************************
##  parseXML private
##  Description  : parse the XML file
##  Parameters   : the XML file name
##*****************************************************************************

sub parseXML {
    my ($self, $filename) = @_;

    my $parser = new XML::DOM::Parser;

    my $doc = $parser->parsefile($filename) 
	or die("$filename: $!");

    my $rootnode = $doc->getDocumentElement
	or die("$filename: $!");
    
    my $attrnode;

    ##
    ## Parse the DataBase Element
    ## 
    my @dbnodelist = $rootnode->getElementsByTagName("DataBase");
    
    die "$filename: Can't find 'DataBase' element." if $#dbnodelist < 0;
    die "$filename: Too many 'DataBase' elements." if $#dbnodelist > 0;
    
    my $dbattributes = $dbnodelist[0]->getAttributes;
    
    # Parse required attributes (Name, User and Password)
    my ($db_name, $db_host, $db_port, $db_user, $db_password);
    foreach my $attrname ("Name", "User", "Password") {
	$attrnode = $dbattributes->getNamedItem($attrname)
	    or die("$filename: Can't find $attrname attribute in 'DataBase'");
	
	eval '$db_'.lc($attrname).' = $attrnode->getValue'; 
    }

    # Parse optionnal attributes (Host and Port)
    # defaults values are localhost:5432
    if($attrnode = $dbattributes->getNamedItem("Host")) {
	$db_host = $attrnode->getValue;
    }
    else {
	$db_host = "localhost";
    }

    if($attrnode = $dbattributes->getNamedItem("Port")) {
	$db_port = $attrnode->getValue;
    }
    else {
	$db_port = "5432";
    }

    ##
    ## Connection to the DataBase
    ## 
    $self->{DB_HANDLE} = new LogTrend::Consolidation::ConsolidationPostgreSQLDataBase($db_name, $db_host, $db_port, $db_user, $db_password);

    ##
    ## Parse the Default Element
    ## 
    my @defaultnodelist = $rootnode->getElementsByTagName("DefaultRules");

    die "$filename: Too many 'Default' elements." if $#defaultnodelist > 0;
    if($#defaultnodelist == 0) {
	
	##
	## Parse the Rule Elements
	## 
	my $rulenodelist = $defaultnodelist[0]->getElementsByTagName("Rule");
	$self->{DEFAULT_RULES_LIST} = $self->parseRules($rulenodelist);
	
    }

    ##
    ## Parse the Consolidation Elements
    ## 
    my @consolidation_list;

    my @consolidationnodelist = $rootnode->getElementsByTagName("Consolidation");
    
    foreach my $consolidationnode (@consolidationnodelist) {
	push @consolidation_list, $self->parseConsolidation($consolidationnode);
    }

    $self->{CONSOLIDATION_LIST} = \@consolidation_list;
}


##*****************************************************************************
##  parseConsolidation private
##  Description  : parse a Consolidation node
##  Parameters   : the Consolidation node
##*****************************************************************************

sub parseConsolidation {
    my ($self, $consolidationnode) = @_;
    
    my ($sourcenumber, @agenttypenodelist, @agentnodelist, $attrnode, $cdatachild);

    ##
    ## Parse the 'Active' Attribute
    ## 
    my $consolidationattributes = $consolidationnode->getAttributes;
    my $active = "Yes";

    if($attrnode = $consolidationattributes->getNamedItem("Active")) {
	$active = $attrnode->getValue;
    }
    
    # if the consolidation is not active, don't parse the rest of the Element.
    return [] if $active eq "No";

    ##
    ## Parse the Source Elements
    ## 
    my @sourcenodelist = $consolidationnode->getElementsByTagName("Source");
    
    my @agent_list;
    foreach my $sourcenode (@sourcenodelist) {
	## Parse the Number attribute (required)
	my $sourceattributes = $sourcenode->getAttributes;
	
	$attrnode = $sourceattributes->getNamedItem("Number")
	    or die "Can't find 'Number' attribute in 'Source'"; 
	$sourcenumber = $attrnode->getValue;
	
	## Look for AgentType child
	if(my @agenttypenodelist = $sourcenode->getElementsByTagName("AgentType")) {
	    die "Too many 'AgentType' elements in 'Source'." if $#agenttypenodelist > 0;
	    
	    $cdatachild = $agenttypenodelist[0]->getFirstChild
		or die "'AgentType' should not be empty in 'Source'";
	    
	    my $agent_type = $cdatachild->getNodeValue;

	    my $agent_type_list = $self->{DB_HANDLE}->GetListOfAgentTypeOnSource($sourcenumber, $agent_type);
	    if($agent_type_list) {
		push @agent_list, @$agent_type_list;
	    }
	    else {
		warn "No $agent_type on source $sourcenumber";
	    } 	    
	}
	elsif(my @agentnodelist = $sourcenode->getElementsByTagName("Agent")) {
	    foreach my $agentnode (@agentnodelist) {
		$cdatachild = $agentnode->getFirstChild
		    or die "'Agent' should not be empty in 'Source'";
		push @agent_list, [$sourcenumber, $cdatachild->getNodeValue];
	    }
	}
	else {
	    die "Can't find 'AgentType' or 'Agent' elements in 'Source'";
	}
    }
    
    ##
    ## Parse the Data Elements
    ## 
    my @data_list;
    my $data_name_list;
    my $data_name;
    if(my @alldatanodelist = $consolidationnode->getElementsByTagName("AllData")) {
	die "Too many 'AllData' elements in 'Consolidation'." if $#alldatanodelist > 0;
	
	foreach my $agent (@agent_list) {
	    $data_name_list = $self->{DB_HANDLE}->GetListOfDataNameOnAgent(@$agent);
	    foreach $data_name (@$data_name_list) {
		push @data_list, [@$agent, $data_name];
	    }
	}
    }
    elsif(my @datanodelist = $consolidationnode->getElementsByTagName("Data")) {
	$data_name_list = [];
	foreach my $datanode (@datanodelist) {
	    $cdatachild = $datanode->getFirstChild
		or die "'Data' should not be empty in 'Consolidation'";

	    $data_name = $cdatachild->getNodeValue;	    
	    
	    foreach my $agent (@agent_list) {
		push @data_list, [@$agent, $data_name];
	    }
	}
    }

    ##
    ## Parse the Rule Elements
    ## 
    my $rulenodelist = $consolidationnode->getElementsByTagName("Rule");
    my $rulelist = $self->parseRules($rulenodelist);    
    
    ##
    ## Create the Consolidation Object
    ##
    return [\@data_list, $rulelist];
}


##*****************************************************************************
##  parseRules private
##  Description  : parse a Rule node list
##  Parameters   : the Rule node list
##*****************************************************************************

sub parseRules {
    my ($self, $rulenodelist) = @_;

    my @rulelist;
    my $cdatachild;
    my $attrnode;
    foreach my $rulenode (@$rulenodelist) {
        my $rule = [];

	##
        ## look for Method attribute, default is "Mean"
	##
        my $ruleattributes = $rulenode->getAttributes;
        if($attrnode = $ruleattributes->getNamedItem("Method")) {
            $rule->[0] = $attrnode->getValue;
        }
        else {
            $rule->[0] = "Mean";
        }
        
	##
        ## Parse Offset and Factor elements
        ## 
	## Offset
        my @ruleoffsetnodelist = $rulenode->getElementsByTagName("Offset");
        
        die "Can't find 'Offset' element in 'Rule'." if $#ruleoffsetnodelist < 0;
        die "Too many 'Offset' elements in 'Rule'." if $#ruleoffsetnodelist > 0;
        
        $cdatachild = $ruleoffsetnodelist[0]->getFirstChild
            or die "'Offset' should not be empty in 'Rule'.";
        $rule->[1] = $cdatachild->getNodeValue;
        
        ## Factor
        my @rulefactornodelist = $rulenode->getElementsByTagName("Factor");
        
        die "Can't find 'Factor' element in 'Rule'." if $#rulefactornodelist < 0;
        die "Too many 'Factor' elements in 'Rule'." if $#rulefactornodelist > 0;
        
        $cdatachild = $rulefactornodelist[0]->getFirstChild
            or die "'Factor' should not be empty in 'Rule'.";
        $rule->[2] = $cdatachild->getNodeValue;
        
        push @rulelist, $rule;
    }

    # sort the rules list ascending by offset
    my @temp_list = sort {$a->[1] <=> $b->[1]} @rulelist;

    return \@temp_list;
}



##*****************************************************************************
##  applyDefaultRule private
##  Description  : Create a new consolidation with all non consolided data 
##                 and defaults rules
##*****************************************************************************

sub applyDefaultRule {
    my ($self) = @_;
    
    my @consolidated_data;
    my @non_cons_data_list;
    my $i=0;
    my $db_handle = $self->{DB_HANDLE};

    my $list;
    foreach my $cons (@{$self->{CONSOLIDATION_LIST}}) {
	$list = $cons->[0];
	push @consolidated_data, @$list;
    }
    
    # Create a list of non consolidated data.
    my $sources_list = $db_handle->GetSourcesList;
    
    foreach my $source (@$sources_list) {
	my $agents_list = $db_handle->GetListOfAgentsOnSource($source->[0]);
	
	foreach my $agent (@$agents_list) {
	    my $data_list = $db_handle->GetListOfDataNameOnAgent($source->[0], $agent->[0]);
 	    foreach my $data (@$data_list) {
	  	my $not_yet_cons = 1;
  		foreach my $cons_data (@consolidated_data) {
  		    if($cons_data->[0] == $source->[0] and
  		       $cons_data->[1] == $agent->[0] and 
  		       $cons_data->[2] eq $data) {
  			$not_yet_cons=0;
  			last;
  		    }
  		}

  		push @non_cons_data_list, [$source->[0], $agent->[0], $data] if $not_yet_cons;
	    }
	}
    }

    $self->consolidate(\@non_cons_data_list, $self->{DEFAULT_RULES_LIST});
}

##*****************************************************************************
##  Run public
##  Description  : Run each consolidation
##  Parameters   : the offset that rules must have to be run (optional)
##*****************************************************************************
sub Run {
    my ($self) = @_;

    ## Run consolidation
    foreach my $cons (@{$self->{CONSOLIDATION_LIST}}) {
	$self->consolidate(@$cons) if $#$cons != -1;
    }
    
    if(defined $self->{DEFAULT_RULES_LIST}) {
	$self->applyDefaultRule;
    }
}

##*****************************************************************************
##  consolidate private
##  Description  : Consolidate Data
##  Parameters   : the offset that rules must have to be run (optional)
##*****************************************************************************
sub consolidate {
    my ($self, $data_list, $rule_list) = @_;
    my $db_handle = $self->{DB_HANDLE};
    my $now = time;

    my $nb_rule = @$rule_list;

    # foreach data, apply rules
    foreach my $data (@$data_list) {
	my $cur_rule = 0;

#	print "@$data\n";

	# look for a previous consolidation in db
	my $zone_list = $self->{DB_ANALYZER}->GetZoneList(@$data);
        my $nb_zone = @$zone_list;

	my $conso_list = $self->{DB_ANALYZER}->GetConsolidationList($zone_list);	
        my $nb_conso = @$conso_list;

	next if $nb_zone == 0;

	# look for the lower periode (faster frequency) before the first consolidation. 
	# Take this value as reference.
	my $ref_freq = $zone_list->[0]->[0];
	for(  my $i = 0; 
	      
	      $i < $nb_zone and
	      ( $nb_conso == 0 or
		$zone_list->[$i]->[1] > $conso_list->[0]->[1]); 
	      
	      $i++) {

	    $ref_freq = $zone_list->[$i]->[0] if $zone_list->[$i]->[0] < $ref_freq;
	}

	# look for the first rule applicable to data
	my $ref_freq_div = 1;
	my $cur_time = $now;
	for(my $i = 0; $i < $nb_rule; $i++) {
	    
	    if($cur_time - $rule_list->[$i]->[1] * 86400 <= $zone_list->[0]->[2]) {
		$cur_rule = $i;
		last;
	    }
	    
	    $cur_time -= $rule_list->[$i]->[1] * 86400;
	    $ref_freq_div = $rule_list->[$i]->[2];
	}

	
	# Now, analyze data zone and create a todo list
	my ($method, $offset, $factor);
	my $cur_conso = 0;
	my $cur_zone = 0;
	my $next_rule_step;
	my $cur_rule_step = $now;
	my $cur_factor;
	my @todo_list;


	# The first rule appliable to data is a delete rule. Can't find reference freq.
	# Just apply the delete rule
	if( $ref_freq_div == 0 ) {
            push @todo_list, ["", 0, 
			      ($cur_rule < $#$rule_list ? $now - $rule_list->[$cur_rule+1]->[1]*86400 : 0),
			      $now];
	}
	# Skip this data, if there are no recent data and only one applicable rule
	elsif($ref_freq_div != 1 and
              $cur_rule < $nb_rule-1 and
              $cur_time - $rule_list->[$cur_rule+1]->[1] * 86400 < $zone_list->[$nb_zone-1]->[1]){
	    next;
	}
	elsif($cur_rule == $nb_rule-1 and $rule_list->[$cur_rule]->[2] != 0) {
	    next;
	}
	else {
	    $ref_freq /= $ref_freq_div if $ref_freq_div;
	    
	    for(; $cur_rule < $nb_rule; $cur_rule++) {
		($method, $offset, $factor) = @{$rule_list->[$cur_rule]};
		$cur_rule_step = $now - $offset * 86400;
		if($cur_rule < $nb_rule-1) {
		    $next_rule_step = $now - $rule_list->[$cur_rule+1]->[1] * 86400;
		}
		else {
		    $next_rule_step = 0;
		}
		
		if($factor == 0) {
		    push @todo_list, ["", 0, 
				      $next_rule_step,
				      $zone_list->[$cur_zone]->[2] >= $cur_rule_step ? $cur_rule_step : $zone_list->[$cur_zone]->[2]];
		    next;
		}
		
		# look for the first zone where the rule can be applied
		while($cur_zone < $nb_zone and 
		      $zone_list->[$cur_zone]->[1] >= $cur_rule_step) {
		    $cur_zone++;
		}
		
		last if $cur_zone >= $nb_zone;
		
		while($cur_zone < $nb_zone and 
		      $zone_list->[$cur_zone]->[2] > $next_rule_step) {
		    
		    $cur_factor = $zone_list->[$cur_zone]->[0]/$ref_freq;
		    $cur_factor = sprintf "%.0f", $cur_factor;

		    if(($cur_factor < $factor or $factor == 0) and ($zone_list->[$cur_zone]->[2] - $zone_list->[$cur_zone]->[1]) >= ($ref_freq*$factor/$cur_factor)) {
			$cur_factor = $factor/$cur_factor;
			$cur_factor = sprintf "%.0f", $cur_factor;
			
			push @todo_list, [$method, $cur_factor, 
					  $zone_list->[$cur_zone]->[1] <= $next_rule_step ? $next_rule_step : $zone_list->[$cur_zone]->[1],
					  $zone_list->[$cur_zone]->[2] >= $cur_rule_step ? $cur_rule_step : $zone_list->[$cur_zone]->[2]];
		    }
		    
		    
		    if($zone_list->[$cur_zone]->[1] < $next_rule_step) {
			last;
		    }
		    
		    $cur_zone++;
		}
	    }

	}

	
#	print "\tTODO : \n";
	foreach my $todo (@todo_list) {
            # security to prevent all data deletion
            next if($todo->[1] == 0 and $todo->[2] == 0);

#	    print "\t\t@$todo\n";
            my $data_values = $db_handle->GetDataInTimeInterval(300000, @$data, 
								$todo->[2], $todo->[3]);


	    next if $data_values == 0;
	    next if $#$data_values < $todo->[1];

	    if($todo->[1]) {				
		my $consolidated_data = eval("LogTrend::Consolidation::Rules::$todo->[0]".'::Run($data_values, $todo->[1])');
		if($@){
		    warn $@;
		    next;
		}
		
		$db_handle->RemoveDataInTimeInterval(@$data, $todo->[2], $todo->[3]);

		$db_handle->AddData(@$data, $consolidated_data);

		$db_handle->Commit;
	    }
	    else {
		$db_handle->RemoveDataInTimeInterval(@$data, $todo->[2], $todo->[3]);
		$db_handle->Commit;
	    }
	}
    }
}



##*****************************************************************************
##  loadRulesPackages private
##  Description  : Load Rules Package if needed
##
##  Parameters   : none
##*****************************************************************************
sub loadRulesPackages {
    my $self = shift;
    
    my $rule_name;
    foreach my $rule (@{$self->{RULE_LIST}}) {
	$rule_name = $rule->[0];

	# if the packages is not loaded
	if( ! defined &{"LogTrend::Consolidation::Rules::${rule_name}::Run"}) {
	    # try to load this package
	    require "LogTrend/Consolidation/Rules/${rule_name}.pm";

	    if (! defined &{"LogTrend::Consolidation::Rules::${rule_name}::Run"}) {
		die "LogTrend/Consolidation/Rules/${rule_name}.pm is not a valid Consolidation rule package";
	    }
	}
    }
}

1;
