# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""this module contains some utilities to navigate in the tree or to
extract information from it

:version:   $Revision: 1.14 $  
:author:    Sylvain Thenault
:copyright: 2003-2005 LOGILAB S.A. (Paris, FRANCE)
:contact:   http://www.logilab.fr/ -- mailto:python-projects@logilab.org
:copyright: 2003-2005 Sylvain Thenault
:contact:   mailto:thenault@gmail.com
"""

__revision__ = "$Id: utils.py,v 1.14 2005/06/27 21:02:52 syt Exp $"
__doctype__ = "restructuredtext en"

from logilab.astng._exceptions import IgnoreChild

def extend_class(original, addons):
    """add methods and attribute defined in the addons class to the original
    class
    """
    for special_key in ('__doc__', '__module__'):
        if special_key in addons.__dict__:
            del addons.__dict__[special_key]
    original.__dict__.update(addons.__dict__)
        
class ASTWalker:
    """a walker visiting a tree in preorder, calling on the handler:
    
    * visit_<class name> on entering a node, where class name is the class of
    the node in lower case
    
    * leave_<class name> on leaving a node, where class name is the class of
    the node in lower case
    """
    def __init__(self, handler):
        self.handler = handler
        self._cache = {}
        
    def walk(self, node):
        """walk on the tree from <node>, getting callbacks from handler
        """
        try:            
            self.visit(node)
        except IgnoreChild:
            pass
        else:
            for child_node in node.getChildNodes():
                self.walk(child_node)
        self.leave(node)

    def get_callbacks(self, node):
        """get callbacks from handler for the visited node
        """
        klass = node.__class__
        methods = self._cache.get(klass)
        if methods is None:
            handler = self.handler
            kid = klass.__name__.lower()
            e_method = getattr(handler, 'visit_%s' % kid,
                               getattr(handler, 'visit_default', None))
            l_method = getattr(handler, 'leave_%s' % kid, 
                               getattr(handler, 'leave_default', None))
            self._cache[klass] = (e_method, l_method)
        else:
            e_method, l_method = methods
        return e_method, l_method
    
    def visit(self, node):
        """walk on the tree from <node>, getting callbacks from handler"""
        method = self.get_callbacks(node)[0]
        if method is not None:
            method(node)
            
    def leave(self, node):
        """walk on the tree from <node>, getting callbacks from handler"""
        method = self.get_callbacks(node)[1]
        if method is not None:
            method(node)


class LocalsVisitor(ASTWalker):
    """visit a project by traversing the locals dictionnary"""
    def __init__(self):
        ASTWalker.__init__(self, self)
        self._visited = {}
        
    def visit(self, astng):
        """launch the visit starting from the given node"""
        if self._visited.has_key(astng):
            return
        self._visited[astng] = 1
        methods = self.get_callbacks(astng)
        recurse = 1
        if methods[0] is not None:
            try:
                methods[0](astng)
            except IgnoreChild:
                recurse = 0
        if recurse:
            if hasattr(astng, 'locals'):
                for node in astng.values():
                    self.visit(node)
        if methods[1] is not None:
            return methods[1](astng)
