# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""this module contains a set of functions to create astng trees from scratch
(build_* functions) or from living object (object_build_* functions)

:version:   $Revision: 1.7 $  
:author:    Sylvain Thenault
:copyright: 2003-2005 LOGILAB S.A. (Paris, FRANCE)
:contact:   http://www.logilab.fr/ -- mailto:python-projects@logilab.org
:copyright: 2003-2005 Sylvain Thenault
:contact:   mailto:thenault@gmail.com
"""

__revision__ = "$Id: raw_building.py,v 1.7 2005/10/21 13:37:24 syt Exp $"
__doctype__ = "restructuredtext en"

import sys
from inspect import getargspec

from logilab.astng import nodes

def build_module(name, doc=None):
    """create and initialize a astng Module node"""
    node = nodes.Module(doc, nodes.Stmt([]))
    node.node.parent = node
    node.name = name
    node.pure_python = False
    node.package = False
    node.parent = None
    node.globals = node.locals = {}
    return node

def build_class(name, basenames=None, doc=None):
    """create and initialize a astng Class node"""
    klass = nodes.Class(name, [], doc, nodes.Stmt([]))
    bases = [nodes.Name(base) for base in basenames]
    for base in bases:
        base.parent = klass
    klass.basenames = basenames
    klass.bases = bases
    klass.code.parent = klass
    klass.locals = {}
    klass.instance_attrs = {}
    return klass

# introduction of decorators has changed the Function initializer arguments
if sys.version_info >= (2, 4):    
    def build_function(name, args=None, defaults=None, flag=0, doc=None):
        """create and initialize a astng Function node"""
        args, defaults = args or [], defaults or []
        # first argument is now a list of decorators
        func = nodes.Function([], name, args, defaults, flag, doc,
                              nodes.Stmt([]))
        func.code.parent = func
        func.locals = {}
        if args:
            register_arguments(func, args)
        return func
    
else:    
    def build_function(name, args=None, defaults=None, flag=0, doc=None):
        """create and initialize a astng Function node"""
        args, defaults = args or [], defaults or []
        func = nodes.Function(name, args, defaults, flag, doc, nodes.Stmt([]))
        func.code.parent = func
        func.locals = {}
        if args:
            register_arguments(func, args)
        return func


def build_name_assign(name, value):
    """create and initialize an astng Assign for a name assignment"""
    return nodes.Assign([nodes.AssName(name, 'OP_ASSIGN')], nodes.Const(value))

def build_attr_assign(name, value, attr='self'):
    """create and initialize an astng Assign for an attribute assignment"""
    return nodes.Assign([nodes.AssAttr(nodes.Name(attr), name, 'OP_ASSIGN')],
                        nodes.Const(value))

def build_from_import(fromname, names):
    """create and intialize an astng From import statement"""
    return nodes.From(fromname, [(name, None) for name in names])

def register_arguments(node, args):
    """add given arguments to local
    
    args is a list that may contains nested lists
    (i.e. def func(a, (b, c, d)): ...)
    """
    for arg in args:
        if type(arg) is type(''):
            node.set_local(arg, node)
        else:
            register_arguments(node, arg)


def object_build_class(node, member):
    """create astng for a living class object"""
    basenames = [base.__name__ for base in member.__bases__]
    return _base_class_object_build(node, member, basenames)

def object_build_function(node, member):
    """create astng for a living function object"""
    args, varargs, varkw, defaults = getargspec(member)
    if varargs is not None:
        args.append(varargs)
    if varkw is not None:
        args.append(varkw)
    func = build_function(member.__name__, args, defaults,
                          member.func_code.co_flags, member.__doc__)
    node.add_local_node(func)

def object_build_datadescriptor(node, member):
    """create astng for a living data descriptor object"""
    return _base_class_object_build(node, member, [])

def object_build_methoddescriptor(node, member):
    """create astng for a living method descriptor object"""
    # FIXME get arguments ?
    func = build_function(member.__name__, doc=member.__doc__)
    # set argnames to None to notice that we have no information, not
    # and empty argument list
    func.argnames = None 
    node.add_local_node(func)
    
def _base_class_object_build(node, member, basenames):
    """create astng for a living class object, with a given set of base names
    (e.g. ancestors)
    """
    klass = build_class(member.__name__, basenames, member.__doc__)
    node.add_local_node(klass)
    return klass

