/* APPLE LOCAL begin LLVM (ENTIRE FILE!)  */
/* Processor ABI customization hooks
Copyright (C) 2005 Free Software Foundation, Inc.
Contributed by Chris Lattner (sabre@nondot.org)

This file is part of GCC.

GCC is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

GCC is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA
02111-1307, USA.  */

//===----------------------------------------------------------------------===//
// This is a C++ header file that specifies how argument values are passed and
// returned from function calls.  This allows the target to specialize handling
// of things like how structures are passed by-value.
//===----------------------------------------------------------------------===//

#ifndef LLVM_ABI_H
#define LLVM_ABI_H

#include "llvm-internal.h"
#include "llvm/Constants.h"
#include "llvm/DerivedTypes.h"

namespace llvm {
  class BasicBlock;
}
extern "C" {
#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "tm.h"
#include "tree.h"
}  

/// DefaultClient - This is a simple implementation of the ABI client interface
/// that can be subclassed.
struct DefaultClient {
  /// HandleScalarArgument - This is the primary callback that specifies an LLVM
  /// argument to pass.
  void HandleScalarArgument(const llvm::Type *LLVMTy) {}
  
  /// EnterField - Called when we're about the enter the field of a struct
  /// or union.  FieldNo is the number of the element we are entering in the
  /// LLVM Struct, StructTy is the LLVM type of the struct we are entering.
  void EnterField(unsigned FieldNo, const llvm::Type *StructTy) {}
  void ExitField() {}
};

/// isSingleElementStructOrArray - Return true if this is a structure with one
/// field or an array with one element.
static bool isSingleElementStructOrArray(tree type) {
  if (TREE_CODE(type) == RECORD_TYPE) {
    bool FoundField = false;
    for (tree Field = TYPE_FIELDS(type); Field; Field = TREE_CHAIN(Field))
      if (TREE_CODE(Field) == FIELD_DECL) {
        if (!FoundField)
          FoundField = true;
        else
          return false;   // More than one field.
      }
    return FoundField;
  } else if (TREE_CODE(type) == ARRAY_TYPE) {
    tree Domain = TYPE_DOMAIN(type);
    if (!Domain || !TYPE_MIN_VALUE(Domain) || !TYPE_MAX_VALUE(Domain))
      return false;
    if (TREE_CODE(TYPE_MIN_VALUE(Domain)) != INTEGER_CST ||
        TREE_CODE(TYPE_MAX_VALUE(Domain)) != INTEGER_CST)
      return false;
    return TREE_INT_CST_LOW(TYPE_MAX_VALUE(Domain)) ==
           TREE_INT_CST_LOW(TYPE_MIN_VALUE(Domain));
  }
  return false;
}

// LLVM_SHOULD_PASS_AGGREGATE_IN_INTEGER_REGS - Return true if this aggregate
// value should be passed in integer registers.
static bool LLVM_SHOULD_PASS_AGGREGATE_IN_INTEGER_REGS(tree type) {
  // GCC passes single-element structs and arrays in registers.
  if (isSingleElementStructOrArray(type)) return false;
  return true;
}


/// DefaultABI - This class implements the default LLVM ABI where structures are
/// passed by decimating them into individual components and unions are passed
/// by passing the largest member of the union.
///
template<typename Client>
class DefaultABI {
protected:
  Client &C;
public:
  DefaultABI(Client &c) : C(c) {}
  
  void HandleArgument(tree type) {
    const Type *Ty = ConvertType(type);

    if (TREE_ADDRESSABLE(type)) {    // Constructible object, pass by-ref
      C.HandleScalarArgument(PointerType::get(Ty));
    } else if (Ty->isFirstClassType()) {
      C.HandleScalarArgument(Ty);
    } else if (LLVM_SHOULD_PASS_AGGREGATE_IN_INTEGER_REGS(type)) {
      PassInIntegerRegisters(type, Ty);
    } else if (TREE_CODE(type) == RECORD_TYPE) {
      for (tree Field = TYPE_FIELDS(type); Field; Field = TREE_CHAIN(Field))
        if (TREE_CODE(Field) == FIELD_DECL) {
          unsigned FNo = cast<ConstantUInt>(DECL_LLVM(Field))->getValue();
          assert(FNo != ~0U && "Case not handled yet!");
          
          C.EnterField(FNo, Ty);
          HandleArgument(TREE_TYPE(Field));
          C.ExitField();
        }
    } else if (TREE_CODE(type) == COMPLEX_TYPE) {
      C.EnterField(0, Ty);
      HandleArgument(TREE_TYPE(type));
      C.ExitField();
      C.EnterField(1, Ty);
      HandleArgument(TREE_TYPE(type));
      C.ExitField();
    } else if (TREE_CODE(type) == UNION_TYPE) {
      HandleUnion(type);
    } else if (TREE_CODE(type) == ARRAY_TYPE) {
      const ArrayType *ATy = cast<ArrayType>(Ty);
      for (unsigned i = 0, e = ATy->getNumElements(); i != e; ++i) {
        C.EnterField(i, Ty);
        HandleArgument(TREE_TYPE(type));
        C.ExitField();
      }
    } else {
      assert(0 && "unknown aggregate type!");
      abort();
    }
  }

  /// HandleUnion - Handle a UNION_TYPE tree.
  ///
  void HandleUnion(tree type) {
    if (TYPE_TRANSPARENT_UNION(type)) {
      tree Field = TYPE_FIELDS(type);
      assert(Field && "Transparent union must have some elements!");
      while (TREE_CODE(Field) != FIELD_DECL) {
        Field = TREE_CHAIN(Field);
        assert(Field && "Transparent union must have some elements!");
      }
      
      HandleArgument(TREE_TYPE(Field));
    } else {
      // Unions pass the largest element.
      unsigned MaxSize = 0;
      tree MaxElt = 0;
      for (tree Field = TYPE_FIELDS(type); Field; Field = TREE_CHAIN(Field)) {
        if (TREE_CODE(Field) == FIELD_DECL) {
          tree SizeTree = TYPE_SIZE(TREE_TYPE(Field));
          unsigned Size = ((unsigned)TREE_INT_CST_LOW(SizeTree)+7)/8;
          if (Size > MaxSize) {
            MaxSize = Size;
            MaxElt = Field;
          }
        }
      }
      
      if (MaxElt)
        HandleArgument(TREE_TYPE(MaxElt));
    }
  }
    
  /// PassInIntegerRegisters - Given an aggregate value that should be passed in
  /// integer registers, convert it to a structure containing ints and pass all
  /// of the struct elements in.
  void PassInIntegerRegisters(tree type, const Type *Ty) {
    unsigned Size = TREE_INT_CST_LOW(TYPE_SIZE(type))/8;
    
    // FIXME: In cases where we can, we should use the original struct.
    // Consider cases like { int, int } and {int, short} for example!  This will
    // produce far better LLVM code!
    std::vector<const Type*> Elts;
    for (; Size >= 8; Size -= 8)
      Elts.push_back(Type::LongTy);
    if (Size >= 4) {
      Elts.push_back(Type::IntTy);
      Size -= 4;
    }
    if (Size >= 2) {
      Elts.push_back(Type::IntTy);
      Size -= 2;
    }
    if (Size >= 1) {
      Elts.push_back(Type::IntTy);
      Size -= 1;
    }
    assert(Size == 0 && "Didn't cover value?");
    const StructType *STy = StructType::get(Elts);

    for (unsigned i = 0, e = Elts.size(); i != e; ++i) {
      C.EnterField(i, STy);
      C.HandleScalarArgument(Elts[i]);
      C.ExitField();
    }
  } 
};

#define TheLLVMABI DefaultABI

#endif
/* APPLE LOCAL end LLVM (ENTIRE FILE!)  */
