package tests::XMLSpecI18NTest;

use strict;

use base qw/ Lire::Test::TestCase tests::TestStoreFixture/;

use Lire::XMLSpecI18N;
use Lire::ReportConfig;
use Lire::FilterSpec;
use Lire::ReportSpec;
use Lire::DlfSchema;
use Lire::Config::SpecParser;
use IO::Scalar;
use Carp;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();

    return $self;
}

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->set_up_test_schema();
    $self->{'cfg'}{'lr_reports_path'} = [ $self->{'testdir'} . "/reports" ];
    $self->{'cfg'}{'lr_filters_path'} = [ $self->{'testdir'} . "/filters" ];

    my $parser = new Lire::Config::SpecParser();
    $parser->merge_specification( $self->{'testdir'} . "/data/test-cfg_spec.xml" );
    $self->{'test_cfg_spec'} = $parser->configspec();
    $self->{'test_cfg_spec'}{'xml_file'} =
      $self->{'testdir'} . "/data/test-cfg_spec.xml";
    $self->{'report_cfg'} =
      new_from_file Lire::ReportConfig( 'test', $self->{'testdir'} .
                                        "/data/test.cfg" );
    $self->{'test_schema'} = Lire::DlfSchema::load_schema( 'test' );
    $self->{'ext_schema'} = Lire::DlfSchema::load_schema( 'test-extended' );
    $self->{'top_files'} = Lire::ReportSpec->load( 'test','top-files' );
    $self->{'select_file'} = Lire::FilterSpec->load( 'test','select-file' );

    $self->{'specs'} = [ $self->{'report_cfg'}, $self->{'test_schema'},
                         $self->{'top_files'}, $self->{'select_file'},
                         $self->{'test_cfg_spec'} ];
    $self->{'extractor'} = new Lire::XMLSpecI18N( @{$self->{'specs'}} );
    $self->{'mock_extractor'} =
      tests::XMLSpecI18NTest::MockExtractor->new( @{$self->{'specs'}} );
    return;
}

sub tear_down {
    my $self = shift->SUPER::tear_down();

    return;
}

sub test_new {
    my $self = $_[0];

    $self->assert_died( sub { new Lire::XMLSpecI18N() },
                        qr/missing one or more \'specs\' parameter/ );

    $self->assert_died( sub { new Lire::XMLSpecI18N( undef ) },
                        qr/missing \'spec\' parameter/ );
    $self->assert_died( sub { new Lire::XMLSpecI18N( {} ) },
                        qr/\'spec\' parameter should be a \'Lire::ReportConfig\', \'Lire::DlfSchema\', \'Lire::XMLSpecContainer\' or \'Lire::Config::ConfigSpec\' instance, not \'HASH/ );

    my $specs = $self->{'specs'};
    my $extractor = $self->{'extractor'};
    $self->assert_not_null( $extractor, 'new() returned undef' );
    $self->assert( UNIVERSAL::isa( $extractor, 'Lire::XMLSpecI18N' ),
                   "new() didn't return a Lire::XMLSpecI18N instance" );
    $self->assert_deep_equals( $specs, $extractor->{'_specs'} );
}

sub test_strip_para {
    my $self = $_[0];

    my $ext = $self->{'extractor'};

    $self->assert_str_equals( 'wawa',
                              $ext->_strip_para( '<para>wawa</para>' ) );
    $self->assert_str_equals( 'wawa wowo wawao',
                              $ext->_strip_para( '    <para>wawa wowo wawao</para>   ' ) );
    $self->assert_str_equals( "\nwawa\n ",
                              $ext->_strip_para( "\n<para>\nwawa\n </para>\n" ) );
}

sub test_generate_ref {
    my $self = $_[0];

    my $ext = $self->{'extractor'};

    $self->assert_died( sub{ $ext->_generate_ref() },
                        qr/'_cur_spec' attribute is undef/ );

    $ext->{'_cur_spec'} = {};
    $self->assert_died( sub{ $ext->_generate_ref() },
                        qr/unknown spec reference: HASH/ );

    my $buf;;
    local *STDOUT = $buf = new IO::Scalar();

    $ext->{'_cur_spec'} = $self->{'report_cfg'};
    $ext->_generate_ref();
    $self->assert_equals( "# REFERENCE: test.cfg\n",
                          ${$buf->sref()} );

    local *STDOUT = $buf = new IO::Scalar();
    $ext->{'_cur_spec'} = $self->{'top_files'};
    $ext->_generate_ref();
    $self->assert_equals( "# REFERENCE: report:test:top-files\n",
                          ${$buf->sref()} );

    local *STDOUT = $buf = new IO::Scalar();
    $ext->{'_cur_spec'} = $self->{'select_file'};
    $ext->_generate_ref();
    $self->assert_equals( "# REFERENCE: filter:test:select-file\n",
                          ${$buf->sref()} );

    local *STDOUT = $buf = new IO::Scalar();
    $ext->{'_cur_spec'} = $self->{'test_schema'};
    $ext->_generate_ref();
    $self->assert_equals( "# REFERENCE: schema:test\n",
                          ${$buf->sref()} );

    local *STDOUT = $buf = new IO::Scalar();
    $ext->{'_cur_spec'} = $self->{'test_cfg_spec'};
    $ext->_generate_ref();
    $self->assert_equals( "# REFERENCE: test-cfg_spec.xml\n",
                          ${$buf->sref()} );

    delete $ext->{'_cur_spec'}{'xml_file'};
    $self->assert_died( sub { $ext->_generate_ref() },
                        qr/no 'xml_file' attribute, was Lire;:Config::ConfigSpec->xml_file\(\) called\?/ );
}

sub test_generate_msgid {
    my $self = $_[0];

    my $ext = $self->{'extractor'};
    $ext->{'_cur_spec'} = $self->{'report_cfg'};

    my $buf;
    local *STDOUT = $buf = new IO::Scalar();
    $ext->_generate_msgid( 'Test' );
    $self->assert_equals( "# REFERENCE: test.cfg\n__( q{Test} );\n",
                          ${$buf->sref()} );

    local *STDOUT = $buf = new IO::Scalar();
    $ext->_generate_msgid( "Test\n    " );
    $self->assert_equals( "# REFERENCE: test.cfg\n__( q{Test\n    } );\n",
                          ${$buf->sref()} );
}

sub test_report_config_i18n {
    my $self = $_[0];

    my $ext = $self->{'mock_extractor'};
    $ext->{'_cur_spec'} = {};
    $self->assert_died( sub { $ext->_report_config_i18n() },
                        qr/\'_cur_spec\' parameter should be a \'Lire::ReportConfig\' instance, not \'HASH/ );

    $ext->{'_cur_spec'} = $self->{'report_cfg'};
    $ext->_report_config_i18n();
    $self->assert_deep_equals( [ 'All Files',
                                 'JPEG Files',
                                 'Sessions',
                                 'Download Reports'
                               ],
                               $ext->{'msgid'} );
}

sub test_schema_i18n {
    my $self = $_[0];

    my $ext = $self->{'mock_extractor'};
    $ext->{'_cur_spec'} = {};
    $self->assert_died( sub { $ext->_schema_i18n() },
                        qr/\'_cur_spec\' parameter should be a \'Lire::DlfSchema\' instance, not \'HASH/ );

    $ext->{'_cur_spec'} = $self->{'test_schema'};
    $ext->_schema_i18n();
    $self->assert_deep_equals( [ 'Simple Schema Definition for Test',
                                 'Simple schema definition for testing purpose',
                                 'Start Time',
                                 'Start time of the file transfer.',
                                 'End Time',
                                 'The time at which the transfer ended.',
                                 'Duration',
                                 'Time taken by the transfer.',
                                 'Connection ID',
                                 "Connection identifier.\n   ",
                                 'Client Host',
                                 'The host to which the file was transferred.',
                                 'Username',
                                 "Name of the authenticated user that did the file transfer.\n   ",
                                 'File',
                                 "The file that was transferred.\n   ",
                                 'Size',
                                 'The number of bytes transferred.',
                                 'Complete?',
                                 'Was the file transferred completely?',
                               ],
                               $ext->{'msgid'} );
}

sub test_schema_i18n_extended {
    my $self = $_[0];

    my $ext = $self->{'mock_extractor'};
    $ext->{'_cur_spec'} = {};
    $self->assert_died( sub { $ext->_schema_i18n() },
                        qr/\'_cur_spec\' parameter should be a \'Lire::DlfSchema\' instance, not \'HASH/ );

    $ext->{'_cur_spec'} = $self->{'ext_schema'};
    $ext->_schema_i18n();
    $self->assert_deep_equals( [ 'Simple Extended Schema for Tests',
                                 'Simple extended schema for testing purpose.',
                                 'Directory',
                                 'Contains the file\'s directory.',
                               ],
                               $ext->{'msgid'} );
}

sub test_xml_spec_i18n {
    my $self = $_[0];

    my $ext = $self->{'mock_extractor'};
    $ext->{'_cur_spec'} = {};
    $self->assert_died( sub { $ext->_xml_spec_i18n() },
                        qr/\'_cur_spec\' parameter should be a \'Lire::XMLSpecContainer\' instance, not \'HASH/ );

    $self->{'select_file'}->display_description( '<para>Display Description</para>' );
    $ext->{'_cur_spec'} = $self->{'select_file'};
    $ext->_xml_spec_i18n();
    $self->assert_deep_equals( [ 'Select File Filter Specification',
                                 "Test global filter specification.\n  ",
                                 'Test parameter.',
                                 'selected files matching $file_match',
                                 'Display Description',
                               ],
                               $ext->{'msgid'} );
}

sub test_report_spec_i18n {
    my $self = $_[0];

    my $ext = $self->{'mock_extractor'};
    $ext->{'_cur_spec'} = {};
    $self->assert_died( sub { $ext->_report_spec_i18n() },
                        qr/\'_cur_spec\' parameter should be a \'Lire::ReportSpec\' instance, not \'HASH/ );

    $ext->{'_cur_spec'} = $self->{'top_files'};
    $ext->_report_spec_i18n();
    $self->assert_deep_equals( [ 'Top Files Report Specification',
                                 'Test report specification.',
                                 'Downloads by File, Top $files_to_show',
                                 'Downloads',
                                 '% Total',
                               ],
                               $ext->{'msgid'} );
}

sub test_config_spec_i18n {
    my $self = $_[0];

    my $ext = $self->{'mock_extractor'};
    $ext->{'_cur_spec'} = {};
    $self->assert_died( sub { $ext->_config_spec_i18n() },
                        qr/\'_cur_spec\' parameter should be a \'Lire::Config::ConfigSpec\' instance, not \'HASH/ );

    $ext->{'_cur_spec'} = $self->{'test_cfg_spec'};
    $ext->_config_spec_i18n();
    $self->assert_deep_equals( [ 'Select2 summary',
                                 "Select2\n    description",
                                 'Explanation on option1',
                                 'option2 summary',
                                 "option2\n     description",
                                 "Var1 summary",
                                 "\n    Var1 description\n"
                                 . "    with many lines.\n   ",
                               ],
                               $ext->{'msgid'} );
}

package tests::XMLSpecI18NTest::MockExtractor;

use base qw/Lire::XMLSpecI18N/;

sub _generate_msgid {
    my ( $self, $msgid ) = @_;

    push @{$self->{'msgid'}}, $msgid;
}

1;
