package MyDlfStream;

use Carp;

%streams = ();

sub new {
    my ( $class, $name ) = @_;
    croak "undefined" unless defined $name;

    return $streams{$name} if exists $streams{ $name };

    my $self = bless { 'stime' => 1072933200 }, $class;
    $streams{$name} = $self;

    return $self;
}

sub start_time {
    my $stime = $_[0]->{'stime'};
    croak "bad time format: '$stime'"
      unless ( $stime =~ m/^\d+$/ );
    return $stime;
}

sub end_time {
    return 1104555600; # 2005-01-01 EST
}

sub nrecords {
    return 1;
}

sub clean {
    my $stime = $_[1];
    croak "bad time format: '$stime'"
      unless ( $stime =~ m/^\d+$/ );
    $_[0]->{'stime'} = $stime;
}

sub close {
}

package tests::UIStoreWindowTest;

use strict;

use base qw/ Lire::Test::TestCase tests::TestStoreFixture /;

use Curses::UI;
use Cwd qw/ realpath /;
use File::Basename qw/ dirname /;

use Lire::DlfStore;
use Lire::Test::CursesUIDriver;
use Lire::Utils qw/ tempdir /;
use Lire::UI::Utils qw/ text_for_width /;
use Lire::UI::StoreWindow;
use Lire::Test::HashConfig;

sub new {
    my $self = shift->SUPER::new( @_ );

    $self->{'driver'} = new Lire::Test::CursesUIDriver();
    $self->{'tempdir'} = tempdir( 'StoreTest_XXXXXX', CLEANUP => 1 );

    $self->init();
    $self->{'test_cfg'} = "$self->{'testdir'}/data/test.cfg";

    return $self;
}

sub set_up {
    my $self = $_[0];

    $self->SUPER::set_up();

    $self->set_up_tz( 'EST' );

    my $spec = $self->lire_default_config_spec();
    $self->{'spec'} = $spec;

    $self->{'cfg'}{'lr_schemas_path'} = [ $self->{'schemasdir'} ];
    $self->{'cfg'}{'_lr_config_spec'} = $spec;

    my $store = Lire::DlfStore->open( "$self->{'tempdir'}", 1 );
    $self->{'store'} = $store;

    $self->set_up_import_jobs();
    $self->set_up_report_jobs();

    $self->{'driver'}->setup_curses_ui();
    $self->{'ui'} = new Curses::UI();
    $self->{'driver'}->set_curses_ui( $self->{'ui'} );

    return;
}

sub _create_simple_dict_list {
    my ( $self, $list_name, $dict_name, %params ) = @_;

    my $list_spec = $self->{'spec'}->get( $list_name );
    my $dict_spec = $list_spec->get( $dict_name );
    my $dict = $dict_spec->instance();
    foreach my $key ( keys %params ) {
        my $scalar = $dict_spec->get( $key )->instance('value'=>$params{$key});
        $dict->set( $scalar );
    }

    return $dict;
}

sub set_up_import_jobs {
    my $self = $_[0];

    my $cfg = $self->{'store'}->{'_config'};
    my $import_jobs = $cfg->get( 'import_jobs' );

    my $jobspec = $self->{'spec'}->get( 'import_jobs' )->get( 'import_job' );
    my $job = $jobspec->instance();
    $job->get( 'name' )->set( 'test job 1' );
    $job->get( 'filter' )->set( '' );
    $job->get( 'log_encoding' )->set( 'UTF-8' );
    $job->get( 'log_file' )->set( 'a file' );
    $job->get( 'period' )->set( 'hourly' );
    $job->get( 'service' )->set_plugin( 'www' );
    $import_jobs->append( $job );

    $job = $jobspec->instance();
    $job->get( 'name' )->set( 'job_2' );
    $job->get( 'filter' )->set( '' );
    $job->get( 'log_encoding' )->set( 'iso-8859-15' );
    $job->get( 'log_file' )->set( 'another file' );
    $job->get( 'period' )->set( 'weekly' );
    $job->get( 'service' )->set_plugin( 'exim' );
    $import_jobs->append( $job );

    $job = $jobspec->instance();
    $job->get( 'filter' )->set( '' );
    $job->get( 'log_encoding' )->set( 'iso-8859-1' );
    $job->get( 'log_file' )->set( 'unnamed job file' );
    $job->get( 'period' )->set( 'weekly' );
    $job->get( 'service' )->set_plugin( 'sendmail' );
    $import_jobs->append( $job );

    return;
}

sub set_up_report_jobs {
    my $self = $_[0];

    my $cfg = $self->{'store'}->{'_config'};
    my $report_jobs = $cfg->get( 'report_jobs' );

    my $jobspec = $self->{'spec'}->get( 'report_jobs' )->get( 'report_job' );
    my $job = $jobspec->instance();
    $job->get( 'name' )->set( 'myReport' );
    $job->get( 'superservice' )->set( 'test' );

    my $schedules = $job->get( 'schedules' );
    my $schedspec = $jobspec->get( 'schedules' )->get( 'schedule' );
    my $sched = $schedspec->instance();
    $sched->get( 'period' )->set( 'daily' );
    $sched->get( 'report_cfg' )->set( $self->{'test_cfg'} );
    $schedules->append( $sched );

    $sched = $schedspec->instance();
    $sched->get( 'period' )->set( 'hourly' );
    $sched->get( 'report_cfg' )->set( $self->{'test_cfg'} );
    $schedules->append( $sched );

    $report_jobs->append( $job );

    return;
}

sub tear_down {
    my $self = $_[0];

    $self->{'driver'}->teardown_curses_ui();
    $self->{'store'}->close();

    $self->SUPER::tear_down();

    return;
}

sub test_new {
    my $self = $_[0];

    my $ui = $self->{'ui'};
    $self->assert_died( sub { $ui->add( 'store_win', 'Lire::UI::StoreWindow')},
                        qr/missing 'store' parameter/ );
    $self->assert_died( sub { $ui->add( 'store_win', 'Lire::UI::StoreWindow',
                                        'store' => {} ) },
                        qr/'store' parameter should be a 'Lire::DlfStore' instance, not 'HASH/ );

    my $win = $ui->add( 'store_win', 'Lire::UI::StoreWindow',
                        'store' => $self->{'store'} );

    my $e_object_names = [ 'import_jobs', 'import_jobs_label',
                           'nrecords_label', 'nrecords_value',
                           'report_jobs', 'report_jobs_label',
                           'store_label',
                           'stream_clean_btn',
                           'stream_end_label', 'stream_end_value',
                           'stream_start_label', 'stream_start_value',
                           'streams', 'streams_label' ];
    my @object_names = sort keys %{$win->{'-id2object'}};
    $self->assert_deep_equals( $e_object_names, \@object_names );

    $self->_test_new_lists( $win );
    $self->_test_new_stream_clean( $win );

    return;
}

sub _test_new_lists {
    my ( $self, $win ) = @_;

    my $label = $win->getobj( 'store_label' );
    $self->assert_isa( 'Curses::UI::Label', $label );
    my $label_text = text_for_width( $self->{'tempdir'}, 80 - 14 );
    $self->assert_str_equals( "Store: $label_text",
                              $label->text() );
    my $import_jobs_label = $win->getobj( 'import_jobs_label' );
    $self->assert_str_equals( 'Import jobs',
                              $import_jobs_label->text() );
    my $report_jobs_label = $win->getobj( 'report_jobs_label' );
    $self->assert_str_equals( 'Report jobs',
                              $report_jobs_label->text() );

    my $cfg = $self->{'store'}->{'_config'};
    my $e_import_jobs_value = $cfg->get( 'import_jobs' );
    my $e_report_jobs_value = $cfg->get( 'report_jobs' );

    my $import_jobs = $win->getobj( 'import_jobs' );
    $self->assert_isa( 'Lire::UI::ListWidget', $import_jobs );
    $self->assert_str_equals( $e_import_jobs_value, $import_jobs->{'value'} );

    my $report_jobs = $win->getobj( 'report_jobs' );
    $self->assert_isa( 'Lire::UI::ListWidget', $report_jobs );
    $self->assert_str_equals( $e_report_jobs_value, $report_jobs->{'value'} );

    return;
}

sub _test_new_stream_clean {
    my ( $self, $win ) = @_;

    my $streams = $win->getobj( 'streams' );
    $self->assert_isa( 'Curses::UI::Listbox', $streams );

    my $stream_clean_btn = $win->getobj( 'stream_clean_btn' );
    $self->assert_isa( 'Curses::UI::Buttonbox', $stream_clean_btn );
    $self->assert_num_equals( 1, scalar @{$stream_clean_btn->{'-buttons'}} );
    $self->assert_str_equals( '[Clean]',
                              $stream_clean_btn->{'-buttons'}[0]{'-label'} );

    my $stream_start_label = $win->getobj( 'stream_start_label' );
    $self->assert_isa( 'Curses::UI::Label', $stream_start_label );
    $self->assert_str_equals( 'Starts:', $stream_start_label->{'-text'} );
    my $stream_end_label = $win->getobj( 'stream_end_label' );
    $self->assert_isa( 'Curses::UI::Label', $stream_end_label );
    $self->assert_str_equals( 'Ends:', $stream_end_label->{'-text'} );

    my $stream_start_value = $win->getobj( 'stream_start_value' );
    $self->assert_isa( 'Curses::UI::Label', $stream_start_value );
    my $stream_end_value = $win->getobj( 'stream_end_value' );
    $self->assert_isa( 'Curses::UI::Label', $stream_end_value );

    my $nrecords_label = $win->getobj( 'nrecords_label' );
    $self->assert_isa( 'Curses::UI::Label', $nrecords_label );
    my $nrecords_value = $win->getobj( 'nrecords_value' );
    $self->assert_isa( 'Curses::UI::Label', $nrecords_value );

    return;
}

sub test_new_streams_none {
    my $self = $_[0];

    no warnings 'redefine';
    local *Lire::DlfStore::dlf_streams = sub { return qw// };

    my $ui = $self->{'ui'};
    my $win = $ui->add( 'store_win', 'Lire::UI::StoreWindow',
                        'store' => $self->{'store'} );
    my $streams = $win->getobj( 'streams' );
    $self->assert_deep_equals( [ '-- no stream --' ],
                                $streams->{'-values'} );
    $self->assert_null( $streams->{'-selected'} );
    $self->assert_num_equals( 0, $streams->{'-ypos'} );
    $self->assert_num_equals( 0,  $streams->{'-focusable'} );

    my $stream_start_value = $win->getobj( 'stream_start_value' );
    $self->assert_str_equals( '', $stream_start_value->{'-text'} );
    my $stream_end_value = $win->getobj( 'stream_end_value' );
    $self->assert_str_equals( '', $stream_end_value->{'-text'} );

    local *Lire::DlfStore::open_dlf_stream = sub {
        $self->fail( 'this should not have been called' )
    };
    my $stream_clean_btn = $win->getobj( 'stream_clean_btn' );
    $stream_clean_btn->{'-buttons'}[0]{'-onpress'}->();

    return;
}

sub test_new_streams_one {
    my $self = $_[0];

    no warnings 'redefine';
    local *Lire::DlfStore::dlf_streams = sub { return ( 'mock_stream' ) };
    local *Lire::DlfStore::open_dlf_stream = sub { return MyDlfStream->new( $_[1] ) };

    my $dlf_stream = MyDlfStream->new( 'mock_stream' );
    $dlf_stream->{'stime'} = '1072933200';

    my $ui = $self->{'ui'};
    my $win = $ui->add( 'store_win', 'Lire::UI::StoreWindow',
                        'store' => $self->{'store'} );
    my $streams = $win->getobj( 'streams' );
    $self->assert_num_equals( 0, $streams->{'-selected'} );
    $self->assert_num_equals( 0, $streams->{'-ypos'} );
    $self->assert_num_equals( 1,  $streams->{'-focusable'} );

    $self->assert_deep_equals ( [ 'mock_stream' ], $streams->{'-values'} );

    my $stream_start_value = $win->getobj( 'stream_start_value' );
    $self->assert_str_equals( '2004-01-01 00:00',
                              $stream_start_value->{'-text'} );
    my $stream_end_value = $win->getobj( 'stream_end_value' );
    $self->assert_str_equals( '2005-01-01 00:00',
                              $stream_end_value->{'-text'} );

    return;
}

sub test_stream_clean_cb {
    my $self = $_[0];

    no warnings 'redefine';
    local *Lire::DlfStore::dlf_streams = sub { return ( 'mock_stream' ) };
    local *Lire::DlfStore::open_dlf_stream = sub { return new MyDlfStream( $_[1] ) };

    my $ui = $self->{'ui'};
    my $win = $ui->add( 'store_win', 'Lire::UI::StoreWindow',
                        'store' => $self->{'store'} );
    my $streams = $win->getobj( 'streams' );
    $streams->{'-selected'} = 0;
    $streams->{'-ypos'} = 0;

    my $dlf_stream = 'undef';
    my $ok_handler = sub {
        my $dialog = $_[1];
        $dlf_stream = $dialog->parent->getobj( 'streams' )->get();
        my $field = $dialog->getobj( 'time_field' );
        $field->text( '2004-03-31' );
        my $buttons = $dialog->getobj( 'buttons' );
        $buttons->{'-buttons'}[1]{'-onpress'}->();
    };

    my $clean_button = $win->getobj( 'stream_clean_btn' );
    $self->{'driver'}->add_event_loop_handler( $ok_handler );

    my $stream_start_value = $win->getobj( 'stream_start_value' );
    $self->assert_str_equals( '2004-01-01 00:00',
                              $stream_start_value->text() );
    $clean_button->{'-buttons'}[0]{'-onpress'}->();
    $self->assert_str_equals( '2004-03-31 00:00',
                              $stream_start_value->text() );

    $self->assert_null( $win->getobj( 'stream_clean' ) );

    my $cancel_handler = sub {
        my $dialog = $_[1];
        my $field = $dialog->getobj( 'time_field' );
        $field->text( '2004-01-01' );
        my $buttons = $dialog->getobj( 'buttons' );
        $buttons->{'-buttons'}[0]{'-onpress'}->();
    };
    $self->{'driver'}->add_event_loop_handler( $cancel_handler );
    $self->assert_str_equals( '2004-03-31 00:00', $stream_start_value->text() );
    $clean_button->{'-buttons'}[0]{'-onpress'}->();
    $self->assert_str_equals( '2004-03-31 00:00', $stream_start_value->text() );

    return;
}

sub test_test_date {
    my $self = $_[0];

    foreach my $date ( qw/ 2003-10-04 03-10-04 0000-01-13 0000-01-31 / ) {
        $self->assert( Lire::UI::StoreWindow::_test_date( $date ),
                       $date );
    }

    foreach my $date (qw/1988 aaaaa 3-10-4 0000-00-00 0000-13-13 2004-01-53/) {
        $self->assert( !Lire::UI::StoreWindow::_test_date( $date ),
                       $date );
    }

    return;
}

sub test_date_to_epoch {
    my $self = $_[0];

    my %dates = ( '2004-01-01' => 1072933200,
                  '04-01-01' => 1072933200,
                  '70-01-01' => 18000 );
    foreach my $date ( keys %dates ) {
        $self->assert_num_equals( $dates{$date},
                                  Lire::UI::StoreWindow::_date_to_epoch( $date ) );
    }
}

sub test_epoch_to_date {
    my $self = $_[0];

    my %epochs = ( 1072933200 => '2004-01-01',
                   18000 => '1970-01-01' );
    foreach my $epoch ( keys %epochs ) {
        $self->assert_num_equals( $epochs{$epoch},
                                  Lire::UI::StoreWindow::_epoch_to_date( $epoch ) );
    }

    $self->assert_str_equals( '-- no date --',
                              Lire::UI::StoreWindow::_epoch_to_date() );
    $self->assert_str_equals( '-- no date --',
                              Lire::UI::StoreWindow::_epoch_to_date( '' ) );
}

1;
