package tests::ReportConfigTest;

use strict;

use File::Temp qw/tempfile/;

use Lire::ReportSpec;
use Lire::FilterSpec;
use Lire::ReportSection;
use Lire::ReportConfig;
use Lire::I18N;

use base qw/Lire::Test::TestCase tests::TestStoreFixture/;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();

    return $self;
}

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->set_up_test_schema();

    $self->{'report_cfg'} = $self->{'testdir'} . "/data/test.cfg";
    $self->{'cfg'}{'lr_reports_path'} = [ $self->{'testdir'} . "/reports" ];
    $self->{'cfg'}{'lr_filters_path'} = [ $self->{'testdir'} . "/filters" ];
    ( $self->{'tmp_fh'}, $self->{'tmp_filename'} ) = tempfile();

    return $self;
}

sub tear_down {
    my $self = shift->SUPER::tear_down();

    unlink ($self->{'tmp_filename'});

    return $self;
}

sub add_to_section {
    my ( $self, $section, $specs ) = @_;

    # Specs: [ [ 'id', { param1 => value, ... } ], ... ]
    foreach my $s ( @$specs ) {
        my ( $id, $params ) = @$s;

        my $spec;
        if ( substr( $id, 0, 1) eq '|' ) {
           $spec = Lire::FilterSpec->load( 'test', substr( $id, 1 ) );
           $section->add_filter( $spec );
        } else {
           $spec = Lire::ReportSpec->load( 'test', $id );
           $section->add_report( $spec );
        }
        for my $param ( keys %$params ) {
            $spec->param( $param )->value( $params->{$param} );
        }
    }
}

sub build_expected_cfg {
    my $self = $_[0];

    my $cfg = new Lire::ReportConfig( "test" );
    $cfg->{'_filename'} = $self->{'report_cfg'};

    my $all_section = new Lire::ReportSection( "test", "All Files" );
    $cfg->add_section( $all_section );
    $self->add_to_section( $all_section, 
                           [ [ 'top-files', { 'files_to_show' => 5 } ],
                             [ 'top-dirs', { 'dirs_to_show' => 5 } ],
                             [ 'downloads-by-period',
                               { 'period' => '15m' } ] ] );

    my $jpeg_section = new Lire::ReportSection( "test", "JPEG Files" );
    $cfg->add_section( $jpeg_section );
    $self->add_to_section( $jpeg_section, 
                           [ [ '|select-file', { 'file_match' => '.*.jpg$' }],
                             [ 'top-files', { 'files_to_show' => 5, } ],
                             [ 'top-dirs',  { 'dirs_to_show' => 5 } ] ] );

    my $sess_section = new Lire::ReportSection( "test", "Sessions" );
    $cfg->add_section( $sess_section );
    $self->add_to_section( $sess_section, 
                           [[ 'sessions-by-length', {} ],
                            [ 'sessions-by-user_class',  {} ] ] );

    my $dl_section = new Lire::ReportSection( "test", "Download Reports" );
    $cfg->add_section( $dl_section );
    $self->add_to_section( $dl_section, 
                           [ ['avg-file_size-by-timeslot', { 'unit' => '1h' }],
                             [ 'user_downloads-report', {} ] ] );

    return $cfg;
}

sub test_load_from_file {
    my $self = $_[0];


    my $e_cfg = $self->build_expected_cfg();

    my $cfg = Lire::ReportConfig->new_from_file( "test", $self->{'report_cfg'} );
    $self->assert_deep_equals( $e_cfg, $cfg );
}

sub test_parse_encoding_line {
    my $self = $_[0];

    return unless $Lire::I18N::USE_ENCODING;

    my $fh = $self->{'tmp_fh'};
    print $fh "This is an eacute: \xc3\xa9\n";
    close $fh;
    open $fh, $self->{'tmp_filename'};
    my $cfg = new Lire::ReportConfig( 'test' );
    $cfg->{'_fh'} = $fh;

    $cfg->{'_curr_section'} = '1';
    local $. = 2; # prevents warning
    $self->assert_died( sub { $cfg->_parse_encoding_line( '=encoding test' ); },
                        qr/'encoding' directive must be the first directive/ );

    $cfg->{'_curr_section'} = undef;

    $self->assert_died( sub { $cfg->_parse_encoding_line ( '=encoding' ); },
                        qr/invalid 'encoding' directive, at line/ );
    $self->assert_died(sub { $cfg->_parse_encoding_line( '=encoding wa wa' );},
                       qr/invalid 'encoding' directive, at line/ );

    $cfg->_parse_encoding_line( '=encoding utf-8' );
    $self->assert_equals( $cfg->{'_encoding'}, 'utf-8' );
    my $line = <$fh>;
    $self->assert_str_equals( "This is an eacute: \x{e9}\n", $line );

    $self->assert_died( sub { $cfg->_parse_encoding_line ( '=encoding test' ); },
                        qr/only one 'encoding' directive allowed, at line/ );
}

sub test_parse_encoding_line_no_support {
    my $self = $_[0];

    local $Lire::I18N::USE_ENCODING = 0;
    local $. = 2; # prevents warning

    my $cfg = new Lire::ReportConfig( 'test' );
    $self->assert_died( sub { $cfg->_parse_encoding_line ( '=encoding UTF-8' ) },
                        qr/'encoding' directive requires perl >= 5.8.0, at line/ );
}

sub test_print {
    my $self = $_[0];

    return unless $Lire::I18N::USE_ENCODING;

    my $fh = $self->{'tmp_fh'};
    my $cfg = new Lire::ReportConfig( 'test' );
    $cfg->{'_encoding'} = 'utf-8';
    my $sect = new Lire::ReportSection( 'test', "Eacute: \x{00e9}" );
    $cfg->add_section( $sect );
    $self->add_to_section( $sect, 
                           [ [ '|select-file', { 'file_match' => '.*.jpg$' }],
                             [ 'top-files', { 'files_to_show' => 5, } ] ] );

    $cfg->print( $fh );
    close $fh;
    open $fh, $self->{'tmp_filename'};
    local $/ = undef;
    my $file =  <$fh>;
    $self->assert_equals( <<EOC, $file );
=encoding utf-8

=section Eacute: \xc3\xa9
|select-file file_match=".*.jpg\$"
top-files files_to_show="5"

EOC
}
1;
