package tests::DlfStreamTest;

use strict;

use base qw/ Lire::Test::TestCase /;

use File::Spec;
use Lire::DlfStore;
use Lire::Config::TypeSpec;
use Lire::DlfStream;
use Lire::I18N;
use Lire::Utils qw/tempdir/;

use Time::Local;

my $schema_v1 = <<EOF;
<lire:dlf-schema superservice="test" timestamp="time_start"
 xmlns:lire="http://www.logreport.org/LDSML/">
<lire:field name="time_start" type="timestamp"/>
<lire:field name="field-1" type="string"/>
<lire:field name="field_2" type="string"/>
<lire:field name="int_3" type="int"/>
</lire:dlf-schema>
EOF

my $time_start = 900_000;
my $time_end = 1_000_000;
my @v1_fields = qw/time_start field-1 field_2 int_3/;
my @v1_dlf =
(
 { 'time_start' => $time_start + 100,
   'field-1' => "Field one",
   'field_2' => "This is field 2",
   'int_3' => 4,
 },
 { 'time_start' => $time_start,
   'field-1' => "field-1",
   'field_2' => undef,
   'int_3' => undef,
 },
 { 'time_start' => $time_start + 1000,
   'field-1' => "Day two",
   'field_2' => "Another test",
   'int_3' => undef,
 },
 { 'time_start' => $time_end,
   'field-1' => undef,
   'field_2' => undef,
   'int_3' => 1,
 },
);

# Remove the field-1 and add field_4, swap time_start and field_2
my @v2_fields = qw/field_2 time_start int_3 field_4/;
my $schema_v2 = <<EOF;
<lire:dlf-schema superservice="test" timestamp="time_start"
 xmlns:lire="http://www.logreport.org/LDSML/">
<lire:field name="field_2" type="string"/>
<lire:field name="time_start" type="timestamp"/>
<lire:field name="int_3" type="int"/>
<lire:field name="field_4" type="string"/>
</lire:dlf-schema>
EOF

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->{'tmpdir'} = tempdir( __PACKAGE__ . "XXXXXX",
                               'CLEANUP' => 1, TMPDIR => 1) ;

    $self->{'v1_dlf'} = \@v1_dlf;
    $self->{'v1_dlf_aref'} = [];
    $self->create_aref( \@v1_fields, \@v1_dlf, $self->{'v1_dlf_aref'} );

    # Create the DLF v2 fields
    $self->{'v2_dlf'} = [];
    my $i=0;
    foreach my $dlf ( @v1_dlf ) {
        my $v2_dlf = { map { $_ => undef } @v2_fields};
        foreach my $k ( keys %$dlf ) {
            $v2_dlf->{$k} = $dlf->{$k} if grep { $_ eq $k } @v2_fields;
        }
        push @{$self->{'v2_dlf'}}, $v2_dlf;
    }

    $self->{'v2_dlf_aref'} = [];
    $self->create_aref( \@v2_fields, $self->{'v2_dlf'}, $self->{'v2_dlf_aref'} );

    $self;
}

sub create_aref {
    my ( $self, $fields, $dlf, $dlf_aref ) = @_;

    foreach my $r ( @$dlf ) {
        my $n = [];
        for (my $i=0; $i<@$fields; $i++) {
            $n->[$i] = $r->{$fields->[$i]};
        }
        push @$dlf_aref, $n;
    }
}

sub set_up {
    my $self = shift->SUPER::set_up();

    # Some tests might have changed the schema.
    $self->write_schema( "test", $schema_v1 );
    $self->{'cfg'}{'lr_schemas_path'} = [ $self->{'tmpdir'} ];
    $self->{'cfg'}{'_lr_config_spec'} = new Lire::Config::ConfigSpec();

    $self->{'store'} = Lire::DlfStore->open( "$self->{'tmpdir'}/store", 1 )
      unless defined $self->{'store'};

    $self->{'mock_stream'} = bless { '_name' => "test",
                                   '_store' => $self->{'store'},
                                   '_mode' => "r",
                                   '_sort_spec' => undef
                                 }, "Lire::DlfStream";
    $self->{'mock_stream'}->_init_schema_infos;
}

sub tear_down {
    my $self = shift->SUPER::tear_down();

    $self->{'store'}{'_dbh'}->rollback;
}

sub write_schema {
    my ( $self, $name, $schema ) = @_;

    open my $fh, "> $self->{'tmpdir'}/$name.xml"
      or die "can't create temporary schema: $!\n";
    print $fh $schema;
    close $fh;
}

sub test_new_stream {
    my $self = $_[0];

    # Create
    my $s = new Lire::DlfStream( $self->{'store'}, "test", "w" );
    $self->assert_not_null( $s, "creating the stream failed" );
    $self->assert_equals( 0, $s->nrecords,
                          "nrecords() should returns 0 on new stream" );
    $self->assert_null( $s->start_time,
                        "start_time() should returns undef on new stream" );
    $self->assert_null( $s->end_time,
                        "start_time() should returns undef on new stream" );
    $s->close;

    # Open now in read mode
    $s = new Lire::DlfStream( $self->{'store'}, "test", "r" );
    $self->assert_not_null( $s, "failed to reopen new stream" );
    $self->assert_equals( 0, $s->nrecords );
}

sub test_read_write_dlf {
    my $self = $_[0];

    my $s = new Lire::DlfStream( $self->{'store'}, "test", "w" );
    $self->assert_not_null( $s, "creating the stream failed" );
    foreach my $dlf ( @{$self->{'v1_dlf'}} ) {
        $s->write_dlf( $dlf );
    }

    $self->assert_dies( qr/read_dlf\(\) can't be called.*'w' mode/,
                        sub { $s->read_dlf() } );
    $s->close();

    # Reopen
    $s = new Lire::DlfStream( $self->{'store'}, "test", "r" );
    $self->assert_dies( qr/write_dlf\(\) can't be called.*'r' mode/,
                        sub { $s->write_dlf( $self->{'v1_dlf'}[0] ) } );

    my @dlf = ();
    while ( defined( $_ = $s->read_dlf ) ) {
        push @dlf, $_;
    }
    $self->assert_deep_equals( $self->{'v1_dlf'}, \@dlf );
    $s->close;
}

sub test_read_dlf_aref {
    my $self = $_[0];

    my $s = new Lire::DlfStream( $self->{'store'}, "test", "w" );
    $self->assert_not_null( $s, "creating the stream failed" );

    $self->assert_died( sub { $s->read_dlf_aref() },
                        qr/read_dlf_aref\(\) can't be called.*'w' mode /
                      );
    foreach my $dlf ( @{$self->{'v1_dlf'}} ) {
        $s->write_dlf( $dlf );
    }

    # Reopen
    $s = new Lire::DlfStream( $self->{'store'}, "test", "r" );
    eval { $s->write_dlf( $self->{'v1_dlf'}[0] ) };
    $self->assert_not_null( $@, "writing to 'r' stream should fail" );

    my @dlf = ();
    while ( defined( $_ = $s->read_dlf_aref ) ) {
        push @dlf, $_;
    }
    $self->assert_deep_equals( $self->{'v1_dlf_aref'}, \@dlf );
    $s->close;
}

sub test_dlf_stats {
    my $self = $_[0];

    my $s = new Lire::DlfStream( $self->{'store'}, "test", "w" );
    $self->assert_not_null( $s, "creating the stream failed" );
    my $i=0;
    foreach my $dlf ( @{$self->{'v1_dlf'}} ) {
        $s->write_dlf( $dlf );
        $i++;
        $self->assert_equals( $i, $s->nrecords );
    }
    $self->assert_equals( $time_start, $s->start_time,
                          "start_time() differs" );
    $self->assert_equals( $time_end, $s->end_time,
                          "end_time() differs" );
    $s->close;
}

sub create_test_stream {
    my $self = $_[0];

    my $s = new Lire::DlfStream( $self->{'store'}, "test", "w" );
    $self->assert_not_null( $s, "creating the stream failed" );
    foreach my $dlf ( @{$self->{'v1_dlf'}} ) {
        $s->write_dlf( $dlf );
    }
    $s->close;
}

sub migrate_schema_v2 {
    my $self = $_[0];

    $self->write_schema( "test", $schema_v2 );
    # Make sure that the changes are noticed
    my $now = time + 5;
    utime $now, $now, "$self->{'tmpdir'}/test.xml";
}

sub test_read_dlf_migrate {
    my $self = $_[0];

    $self->create_test_stream;

    $self->migrate_schema_v2;

    my $s = new Lire::DlfStream( $self->{'store'}, "test", "r" );

    my @dlf = ();
    while (defined($_ = $s->read_dlf)) {
        push @dlf, $_;
    }
    $self->assert_deep_equals( $self->{'v2_dlf'}, \@dlf);
}

sub test_read_dlf_aref_migrate {
    my $self = $_[0];

    $self->create_test_stream;
    $self->migrate_schema_v2;

    my $s = new Lire::DlfStream( $self->{'store'}, "test", "r" );

    my @dlf = ();
    while (defined($_ = $s->read_dlf_aref)) {
        push @dlf, $_;
    }
    $self->assert_deep_equals( $self->{'v2_dlf_aref'}, \@dlf);
}

sub test_write_dlf_migrate {
    my $self = $_[0];

    $self->create_test_stream;
    $self->migrate_schema_v2;

    # Migration should be done transparently
    my $s = new Lire::DlfStream( $self->{'store'}, "test", "w" );
    # Write another one just to make sure
    my @v2_cmp = ( @{$self->{'v2_dlf'}}, $self->{'v2_dlf'}[0] );
    $s->write_dlf( $self->{'v2_dlf'}[0] );
    $self->assert_equals( @{$self->{'v2_dlf'}} + 1, $s->nrecords );
    $s->close;

    $s = new Lire::DlfStream( $self->{'store'}, "test", "r" );
    my @dlf = ();
    while (defined($_ = $s->read_dlf)) {
        push @dlf, $_;
    }

    $self->assert_deep_equals( \@v2_cmp, \@dlf);
}

sub test_sorted_time_start {
    my $self = $_[0];

    $self->create_test_stream;

    # Sort on time_start
    my $s = new Lire::DlfStream( $self->{'store'}, "test", "r", "time_start" );
    $self->assert_not_null( $s, "new() returned undef" );
    my @dlf = ();
    while (defined($_ = $s->read_dlf)) {
        push @dlf, $_;
    }
    $s->close;
}

sub test_rsorted_time_start {
    my $self = $_[0];

    $self->create_test_stream;

    # Sort on time_start
    my $s = new Lire::DlfStream( $self->{'store'}, "test", "r", "-time_start" );
    $self->assert_not_null( $s, "new() returned undef" );
    my @dlf = ();
    while (defined($_ = $s->read_dlf)) {
        push @dlf, $_;
    }
    $s->close;

    my @sorted_dlf = sort { $b->{'time_start'} <=> $a->{'time_start'} } @{$self->{'v1_dlf'}};
    $self->assert_deep_equals( \@sorted_dlf, \@dlf);
}

sub test_sorted_mkeys {
    my $self = $_[0];

    $self->create_test_stream;

    # Sort on time_start
    my $s = new Lire::DlfStream( $self->{'store'}, "test", "r", "field-1 -time_start" );
    $self->assert_not_null( $s, "new() returned undef" );
    my @dlf = ();
    while (defined($_ = $s->read_dlf)) {
        push @dlf, $_;
    }
    $s->close;

    # Some field-1 are undef ->
    no warnings 'uninitialized';

    my @sorted_dlf = sort { $a->{'field-1'} cmp $b->{'field-1'} ||
                            $b->{'time_start'} <=> $a->{'time_start'} } @{$self->{'v1_dlf'}};
    $self->assert_deep_equals( \@sorted_dlf, \@dlf);
}

sub test_sorted_migrate {
    my $self = $_[0];

    $self->create_test_stream;
    $self->migrate_schema_v2;

    # Sort on time_start
    my $s = new Lire::DlfStream( $self->{'store'}, "test", "r", "time_start" );
    $self->assert_not_null( $s, "new() returned undef" );
    my @dlf = ();
    while (defined($_ = $s->read_dlf)) {
        push @dlf, $_;
    }
    $s->close;

    my @sorted_dlf = sort { $a->{'time_start'} <=> $b->{'time_start'} } @{$self->{'v2_dlf'}};
    $self->assert_deep_equals( \@sorted_dlf, \@dlf);
}

sub test_select_query {
    my $self = $_[0];

    $self->create_test_stream;

    my $stream = bless { '_store' => $self->{'store'},
                         '_mode' => 'r',
                         '_name' => 'test',
                         '_sort_spec' => "-time_start field-1" },
                           'Lire::DlfStream';
    $stream->_init_schema_infos;
    my $query = $stream->_select_query();
    $self->assert_not_null( $query, '_select_query() returned undef' );
    $self->assert( UNIVERSAL::isa( $query, 'Lire::DlfQuery' ),
                   "_select_query() should return a Lire::DlfQuery instance: $query" );
    $self->assert_equals( 'test', $query->stream_name() );
    $self->assert_deep_equals( [], $query->aggr_fields() );
    $self->assert_deep_equals( [], $query->group_fields() );
    $self->assert_deep_equals( [ 'time_start', 'field-1', 'field_2',
                                 'int_3'  ],
                               $query->fields() );
    $self->assert_equals( "time_start DESC, \"field-1\"",
                          $query->order_by_clause() );
}

sub test_insert_query {
    my $self = $_[0];

    my $e_sql = "INSERT INTO dlf_test (time_start, \"field-1\", field_2, int_3) VALUES (?,?,?,?)";
    $self->{'mock_stream'}{'_mode'} = "w";
    $self->assert_equals( $e_sql, $self->{'mock_stream'}->_insert_query() );
}

sub test__migration_insert_query {
    my $self = $_[0];

    my $e_sql = "INSERT INTO dlf_test (time_start, \"field-1\", field_2, int_3) SELECT time_start, \"field-1\", field_2, int_3 FROM temp_dlf_test";

    my $cfields = [ "time_start", "field-1", "field_2", "int_3"];
    $self->assert_equals( $e_sql,
                          $self->{'mock_stream'}->_migration_insert_query( $cfields) );
}

sub test_migrate_empty_schema {
    my $self = $_[0];

    # Create empty schema
    my $s = new Lire::DlfStream( $self->{'store'}, "test", "w" );
    $s->close;
    $self->migrate_schema_v2;

    $s = new Lire::DlfStream( $self->{'store'}, "test", "w" );
    foreach my $dlf ( @{$self->{'v2_dlf'}} ) {
        $s->write_dlf( $dlf );
    }
    $s->close;

    $s = new Lire::DlfStream( $self->{'store'}, "test", "r" );
    my @dlf = ();
    while ( defined( $_ = $s->read_dlf ) ) {
        push @dlf, $_;
    }
    $self->assert_deep_equals( $self->{'v2_dlf'}, \@dlf );
    $s->close;
}

sub test__sql_table {
    my $self = $_[0];

    my $stream = $self->{'mock_stream'};
    $self->assert_equals( 'dlf_test', $stream->_sql_table() );
    $self->assert_equals( 'temp_dlf_test', $stream->_sql_table( 'temp_' ) );
    $self->assert_equals( 'temp_dlf_test_idx',
                          $stream->_sql_table( 'temp_', '_idx' ) );
    $self->assert_equals( 'dlf_test_idx', $stream->_sql_table( undef, '_idx'));


    $self->assert_equals( '"temp-dlf_test"', $stream->_sql_table( "temp-" ) );
    $self->assert_equals( '"dlf_test:idx"',
                          $stream->_sql_table( undef, ":idx" ) );
    $stream->{'_name'} = 'test-extended';
    $self->assert_equals( '"dlf_test-extended"', $stream->_sql_table() );
}

sub test_create_dlf_schema {
    my $self = $_[0];

    $self->{'mock_stream'}->_create_dlf_schema;
    my $sql_def = Lire::DlfSchema::load_schema( "test" )->sql_fields_def;
    chomp $sql_def; # Trailing newline removed by SQLite

    my $table = $self->{'store'}{'_dbh'}->selectrow_hashref( "SELECT * FROM sqlite_master WHERE name = 'dlf_test'" );
    $self->assert_not_null( $table, "table dlf_test wasn't created" );
    $self->assert_matches( qr/\Q$sql_def\E/, $table->{'sql'} );

    my $index = $self->{'store'}{'_dbh'}->selectrow_hashref( "SELECT * FROM sqlite_master WHERE name = 'dlf_test_time_start_idx'" );
    $self->assert_not_null( $index, "index dlf_test_time_start_idx wasn't created"  );
    $self->assert_equals( "index", $index->{'type'} );
    $self->assert_equals( "CREATE INDEX dlf_test_time_start_idx ON dlf_test ( time_start )", $index->{'sql'} );
}

sub test_close {
    my $self = $_[0];

    my $stream_w = new Lire::DlfStream( $self->{'store'}, "test", "w" );
    $stream_w->close;
    $self->assert_null( $stream_w->{'_sth'}, 
                        "'_sth' attribute isn't undef after calling close()" );

    my $stream_r = new Lire::DlfStream( $self->{'store'}, "test", "r" );
    $stream_r->close;
    $self->assert_null( $stream_r->{'_dlf_query'},
                        "'_dlf_query' attribute isn't undef after calling close()" );
}

# Make sure that data is stored in UTF8 in the SQLite DB.
sub test_write_dlf_utf8 {
    my $self = $_[0];

    return unless $Lire::I18N::USE_ENCODING;

    require Encode;
    Encode->import( 'is_utf8' );

    my $stream_w = new Lire::DlfStream( $self->{'store'}, "test", "w" );
    $stream_w->write_dlf( { "field_2" => "ISO eacute: \xe9" } );
    $stream_w->close();

    my $row = $self->{'store'}{'_dbh'}->selectrow_hashref( "SELECT field_2 FROM dlf_test" );
    $self->assert( ! is_utf8( $row->{'field_2'} ), 
                   "SQLite driver returned unicode string where a byte-encoded string was expected" );
    $self->assert_equals( "ISO eacute: \x{c3}\x{a9}", $row->{'field_2'} );
}

sub test_write_dlf_utf8_no_support {
    my $self = $_[0];

    local $Lire::I18N::USE_ENCODING = 0;

    my $stream_w = new Lire::DlfStream( $self->{'store'}, "test", "w" );
    $stream_w->write_dlf( { "field_2" => "ISO eacute: \xe9" } );
    $stream_w->close();

    my $row = $self->{'store'}{'_dbh'}->selectrow_hashref( "SELECT field_2 FROM dlf_test" );
    $self->assert_str_equals( "ISO eacute: ?", $row->{'field_2'} );
}

sub test_write_dlf_escape {
    my $self = $_[0];

    my $stream_w = new Lire::DlfStream( $self->{'store'}, "test", "w" );
    $stream_w->write_dlf( { 'field-1' => "caract\x{100}re",
                            "field_2" => "c\015r\017" } );
    $stream_w->close();

    my $stream_r = new Lire::DlfStream( $self->{'store'}, "test", "r" );
    my $dlf = $stream_r->read_dlf();
    $stream_r->close();

    my $value = $Lire::I18N::USE_ENCODING ? "caract\x{100}re" : "caract?re";
    $self->assert_deep_equals( { 'time_start' => undef,
                                 'field-1' => $value,
                                 'field_2' => 'c?r?',
                                 'int_3' => undef }, $dlf );
}

sub test_clean {
    my $self = $_[0];

    $self->assert_dies( qr/clean\(\) can't be called.*'r' mode/,
                        sub { $self->{'mock_stream'}->clean() } );

    my $stream = new Lire::DlfStream( $self->{'store'}, 'test', 'w' );
    my $mar11_2004 = timelocal( 0, 0, 12, 11, 2, 2004 );
    $self->{'store'}->_dbh()->do( "INSERT INTO dlf_test ( time_start ) VALUES ( ? )", {}, $mar11_2004 );
    $self->{'store'}->_dbh()->do( "INSERT INTO dlf_test ( time_start ) VALUES ( ? )", {}, $mar11_2004 + 86400 );
    $self->assert_num_equals( 2, $stream->nrecords() );
    $stream->clean( $mar11_2004 + 86400);
    $self->assert_num_equals( 1, $stream->nrecords() );
    $stream->clean();
    $self->assert_num_equals( 0, $stream->nrecords() );
}

1;
