package tests::ConfigSpecParserTest;

use strict;

use base qw/ Test::Unit::TestCase /;

use Lire::Config::SpecParser;
use Lire::Config::TypeSpec;

use File::Temp qw/mktemp tempdir/;
use Carp;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->{'specdir'} = tempdir( "config-spec_XXXXXX", 'CLEANUP' => 1 );
    $self->{'expected_spec'} = $self->build_expected_spec;
    $self->_make_config_spec;
    $self;
}

sub build_expected_spec {
    my $self = $_[0];

    $self->{'bool_desc'} = "<para>This is an experimental option</para>";
    my $cfg = new Lire::Config::ConfigSpec();
    my $bool = new Lire::Config::BooleanSpec( 'name' => "bool_param" );
    $bool->summary( "This is a boolean parameter" );
    $bool->description( $self->{'bool_desc'} );
    $cfg->add( $bool );
    $cfg->add( new Lire::Config::IntegerSpec( 'name' => "int_param" ) );
    $cfg->add( new Lire::Config::DlfConverterSpec( 'name' => "dlfconverter_param" ) );
    $cfg->add( new Lire::Config::CommandSpec( 'name' => "command_param" ) );
    $cfg->add( new Lire::Config::DirectorySpec( 'name' => "dir_param" ) );
    $cfg->add( new Lire::Config::FileSpec( 'name' => "file_param" ) );
    $cfg->add( new Lire::Config::ExecutableSpec( 'name' => "executable_param" ) );

    my $select = new Lire::Config::SelectSpec( 'name' => "select_param" );
    $cfg->add( $select );
    $select->add( new Lire::Config::OptionSpec( 'name' => "option_1" ) );
    $select->add( new Lire::Config::OptionSpec( 'name' => "option_2" ) );

    my $list = new Lire::Config::ListSpec( 'name' => "list_param" );
    $cfg->add( $list );
    $list->add( new Lire::Config::StringSpec( 'name' => "string_param" ));

    return $cfg;
}

sub _make_config_spec {
    my $self = $_[0];

    $self->{'spec_file'} = "$self->{'specdir'}/spec.xml";
    my $fh;
    open $fh, "> $self->{'spec_file'}"
      or croak "open of $self->{'spec_file'} failed: $!";
    print $fh <<SPEC;
<lrcsml:config-spec xmlns:lrcsml="http://www.logreport.org/LRCSML/">
 <lrcsml:boolean name="bool_param">
  <lrcsml:summary>This is a boolean parameter</lrcsml:summary>
  <lrcsml:description>$self->{'bool_desc'}</lrcsml:description>
 </lrcsml:boolean>
 <lrcsml:integer name="int_param">
 </lrcsml:integer>
 <lrcsml:dlf-converter name="dlfconverter_param"/>
 <lrcsml:command name="command_param"/>
 <lrcsml:directory name="dir_param"/>
 <lrcsml:file name="file_param"/>
 <lrcsml:executable name="executable_param"/>
 <lrcsml:select name="select_param">
  <lrcsml:option name="option_1"/>
  <lrcsml:option name="option_2"/>
 </lrcsml:select>
 <lrcsml:list name="list_param">
  <lrcsml:string name="string_param"/>
 </lrcsml:list>
</lrcsml:config-spec>
SPEC
    close $fh;
}

sub set_up {
    my $self = $_[0];
}

sub tear_down {
    my $self = $_[0];

}

sub test_new {
    my $self = $_[0];

    my $parser = new Lire::Config::SpecParser;
    $self->assert_not_null( $parser, "new() returned undef" );
    $self->assert( $parser->isa( "Lire::Config::SpecParser" ),
                                 "new() returned wrong type" );
}

sub test_merge_specifications_dir {
    my $self = $_[0];

    my $parser = new Lire::Config::SpecParser;
    $parser->merge_specifications_dir( $self->{'specdir'} );

    $self->assert_not_null( $parser->configspec,
                            "configspec() returned undef" );

    $self->assert_deep_equals( $self->{'expected_spec'},
                               $parser->configspec,
                             );
}

sub test_merge_specification {
    my $self = $_[0];

    my $parser = new Lire::Config::SpecParser;
    $parser->merge_specification( "$self->{'spec_file'}" );

    $self->assert_not_null( $parser->configspec,
                            "configspec() returned undef" );
    $self->assert_deep_equals( $self->{'expected_spec'},
                               $parser->configspec,
                             );
}

1;
