package tests::ConfigParserTest;

use strict;

use base qw/ Lire::Test::TestCase /;

use Lire::Config::Value;
use Lire::Config::Parser;
use Lire::Config::SpecParser;
use Lire::Config::ConfigSpec;
use Lire::Config::RecordSpec;
use Lire::Config::PluginSpec;
use Lire::Config::Dictionary;
use Lire::Config::Plugin;
use File::Temp qw/mktemp tempdir/;
use Carp;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->{'tmpdir'} = tempdir( "config_XXXXXX", 'CLEANUP' => 1 );
    $self->_make_config_spec;
    $self->_make_config;
    $self->{'config_spec'}     = $self->_load_spec;
    $self->{'expected_config'} = $self->_build_expected_config;

    my $spec = $self->{'other_spec'} = new Lire::Config::ConfigSpec();
    my $plugin = new Lire::Config::PluginSpec( 'name' => 'plugin' );
    $spec->add( $plugin );
    $plugin->add( new Lire::Config::OptionSpec( 'name' => 'option_1' ) );
    $spec->add( new Lire::Config::RecordSpec( 'name' => 'option_1_properties') );

    return $self;
}

sub _make_config_spec {
    my $self = $_[0];

    $self->{'spec_file'} = "$self->{'tmpdir'}/spec.xml";
    my $fh;
    open $fh, "> $self->{'spec_file'}"
      or croak "open of $self->{'spec_file'} failed: $!";
    print $fh <<SPEC;
<config-spec xmlns="http://www.logreport.org/LRCSML/">
 <boolean name="bool_param"/>
 <integer name="int_param"/>
 <directory name="dir_param"/>
 <executable name="executable_param"/>
 <file name="file_param"/>
 <select name="select_param">
  <option name="option_1"/>
  <option name="option_2"/>
 </select>
 <list name="list_param">
  <string name="string_param"/>
 </list>
</config-spec>
SPEC
    close $fh;
}

sub _make_config {
    my $self = $_[0];

    my $fh;
    $self->{'config_file'} = "$self->{'tmpdir'}/config.xml";
    open $fh, "> $self->{'config_file'}"
      or croak "open of $self->{'config_file'} failed: $!\n";
    print $fh <<CONFIG;
<config xmlns="http://www.logreport.org/LRCML/">
 <global>
  <param name="bool_param">yes</param>
  <param name="int_param">10</param>
  <param name="file_param">$self->{'config_file'}</param>
  <param name="dir_param">$self->{'tmpdir'}</param>
  <param name="executable_param">/bin/sh</param>

  <param name="list_param">
   <param name="string_param">string1</param>
   <param name="string_param">string2 &lt; a long string with an entity</param>
  </param>

  <param name="select_param">option_2</param>

 </global>
 <job name="testjob">
  <param name="bool_param">yes</param>
  <param name="int_param">10</param>

  <param name="list_param">
   <param name="string_param">string2</param>
  </param>

  <param name="select_param">option_1</param>
 </job>
</config>
CONFIG
    close $fh;
}


sub _load_spec {
    my $self = $_[0];
    my $spec_parser = new Lire::Config::SpecParser();
    $spec_parser->merge_specification( $self->{'spec_file'} );
    return $spec_parser->configspec;
}

sub _build_expected_config {
    my $self = $_[0];

    my $spec = $self->{'config_spec'};
    my $cfg = new Lire::Config::ConfigFile( 'spec'     => $self->{'config_spec'},
                                            'filename' => $self->{'config_file'},
                                          );
    my $global = $self->{'config_spec'}->instance;
    $cfg->global( $global );

    $global->set( $spec->get( "bool_param" )->instance( 'value' => "yes" ));
    $global->set( $spec->get( "int_param" )->instance( 'value' => 10 ));
    $global->set( $spec->get( "dir_param" )->instance( 'value' => $self->{'tmpdir'} ));
    $global->set( $spec->get( "file_param" )->instance( 'value' => $self->{'config_file'} ));
    $global->set( $spec->get( "executable_param" )->instance( 'value' => "/bin/sh" ));
    $global->set( $spec->get( "select_param" )->instance( 'value' => "option_2" ));
    my $list_spec = $spec->get( 'list_param' );
    my $list = $list_spec->instance;
    $global->set( $list );
    $list->append( $list_spec->get( "string_param" )->instance( 'value' => "string1" ));
    $list->append( $list_spec->get( "string_param" )->instance( 'value' => "string2 < a long string with an entity" ));

    return $cfg;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->{'cfg'}{'_lr_config_spec'} = $self->{'other_spec'};

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    return;
}

sub test_new {
    my $self = $_[0];

    my $parser = new Lire::Config::Parser( 'spec' => $self->{'config_spec'} );
    $self->assert_not_null( $parser, "new() returned undef" );
    $self->assert( $parser->isa( "Lire::Config::Parser" ),
                                 "new() returned wrong type" );
}

sub test_load_config_file {
    my $self = $_[0];

    my $parser = new Lire::Config::Parser( 'spec' => $self->{'config_spec'} );
    my $conf = $parser->load_config_file( "$self->{'config_file'}" );
    $self->assert_not_null( $conf,
                            "load_config_file() returned undef" );

    $self->assert_deep_equals( $self->{'expected_config'}, $conf );
}

sub test_param_start_plugin {
    my $self = $_[0];

    my $parser = new Lire::Config::Parser( 'spec' => $self->{'other_spec'} );
    push @{$parser->{'confstack'}}, $self->{'other_spec'}->instance();
    push @{$parser->{'specstack'}}, $self->{'other_spec'};

    $parser->param_start( undef, 'param', 'name' => 'plugin',
                          'value' => 'option_1' );
    $self->assert_str_equals( $self->{'other_spec'}->get( 'option_1_properties' ),
                              $parser->{'specstack'}[0] );
    $self->assert_deep_equals( $self->{'other_spec'}->get( 'plugin' )->instance( 'value' => 'option_1' ),
                               $parser->{'confstack'}[0] );
}
1;
