package Lire::UI::StoreWindow;

use strict;

use base qw/ Curses::UI::Window /;

use Carp;
use Curses::UI::Common;
use Locale::TextDomain 'lire';
use Time::Local;

use Lire::Utils qw/ check_param check_object_param max /;
use Lire::UI::Utils qw/ text_for_width button_box_width /;

use POSIX qw/strftime/;

use vars qw/ @CARP_NOT %_list_coordinates %_list_labels /;

@CARP_NOT = qw/ Curses::UI::Container /;

%_list_labels = ( 'import_jobs' => { '-text' => 'Import jobs',
                                     '-x' => 15, '-y' => 3 },
                  'report_jobs' => { '-text' => 'Report jobs',
                                     '-x' => 54, '-y' => 3 }, );

%_list_coordinates = ( 'import_jobs' => { '-x' => 3, '-y' => 4,
                                          '-height' => 9, '-width' => 35 },
                       'report_jobs' => { '-x' => 42, '-y' => 4,
                                          '-height' => 9, '-width' => 35 } );

sub new {
    my $class = shift;
    my %userargs = @_;
    keys_to_lowercase( \%userargs );

    check_object_param( $userargs{'store'}, 'store',
                        'Lire::DlfStore' );

    my $self = $class->Curses::UI::Window::new( %userargs );

    my $store_path = text_for_width( $self->{'store'}->path(),
                                     $self->canvaswidth() - 14 );
    $self->add( 'store_label', 'Curses::UI::Label',
                '-text' => "Store: $store_path",
                '-x' => 3, '-y' => 1 );

    foreach my $list ( qw/ import_jobs report_jobs / ) {
        $self->add( "${list}_label", 'Label',
                    %{$_list_labels{$list}} );
        $self->add( $list, 'Lire::UI::Widget',
                    'value' => $self->{'store'}->config()->get( $list ),
                    %{$_list_coordinates{$list}} );
    }

    $self->_add_stream_widgets();

    return $self;
}

sub _add_stream_widgets {
    my $self = $_[0];

    my @streams = $self->{'store'}->dlf_streams();
    my $values;
    my $selected;
    my $focusable;

    if ( @streams ) {
        $values = \@streams;
        $selected = 0;
        $focusable = 1;
    } else {
        $values = [ __( '-- no stream --' ) ];
        $selected = undef;
        $focusable = 0;
    }

    $self->add( 'streams_label', 'Label',
                '-text' => __( 'DLF Streams' ),
                '-x' => 4, '-y' => 14 );

    $self->add( 'streams', 'Listbox',
                '-onchange' => \&_change_cb,
                '-onselchange' => \&_sel_change_cb,
                '-values' => $values,
                '-selected' => $selected,
                '-ypos' => $selected,
                '-focusable' => $focusable,
                '-border' => 1,
                '-x' => 3, '-y' => 15,
                '-width' => 18, '-height' => 7 );

    my $start_label = $self->add( 'stream_start_label', 'Label',
                                  '-text' => __( 'Starts:' ),
                                  '-x' => 23, '-y' => 16 );
    my $end_label = $self->add( 'stream_end_label', 'Label',
                                '-text' => __( 'Ends:' ),
                                '-x' => 23, '-y' => 17 );
    my $nrecords_label = $self->add( 'nrecords_label', 'Label',
                                     '-text' => __( 'Records:' ),
                                     '-x' => 23, '-y' => 18 );

    my $x = max( $start_label->width(), $end_label->width(),
                 $nrecords_label->width() ) + 24;
    $self->add( 'stream_start_value', 'Label',
                '-x' => $x, '-y' => 16 );
    $self->add( 'stream_end_value', 'Label',
                '-x' => $x, '-y' => 17 );
    $self->add( 'nrecords_value', 'Label',
                '-x' => $x, '-y' => 18 );

    $self->add( 'stream_clean_btn', 'Buttonbox',
                '-buttons' => [ { '-label' => __( '[Clean]' ),
                                  '-onpress' => sub { 
                                      $self->_clean_cb_helper()
                                  } } ],
                '-x' => 23 , '-y' => 20 );

    $self->_update_labels();

    return;
}

sub _clean_ok_cb_helper {
    my ( $self, $time_field ) = @_;

    my $dialog = $self->getobj( 'stream_clean' );
    my $streams = $self->getobj( 'streams' );
    my $stream_name = $streams->get();
    my $stream = $self->{'store'}->open_dlf_stream( $stream_name, 'w' );
    my $time = _date_to_epoch( $time_field->text() );
    $stream->clean( $time );
    $stream->close();
    $self->_update_labels();

    return;
}

sub _clean_cb_helper {
    my $self = $_[0];

    my $streams = $self->getobj( 'streams' );
    return unless $streams->{'-focusable'};

    my $title = __x( "Cleaning stream '{stream}'",
                     'stream' => $streams->get() );
    my $dialog = $self->add( 'stream_clean', 'Window',
                             '-title' => $title, '-ipad' => 1,
                             '-x' => 7, '-y' => 8,
                             '-height' => 9, '-width' => 60,
                             '-border' => 1 );
    $dialog->add( 'question', 'Label',
                  '-text' => __( 'Enter the cleaning date in the form "YYYY-MM-DD":' ),
                  '-x' => 0, '-y' => 0 );
    my $time_field = $dialog->add( 'time_field', 'TextEntry',
                                   '-regexp' => '/^[0-9-]*$/',
                                   '-sbborder' => 1, '-x' => 20,
                                   '-y' => 2, '-width' => 14 );

    my $buttons = [ { '-label' => __( '< Cancel >' ),
                      '-onpress' => sub {
                          $dialog->loose_focus()
                      } },
                    { '-label' => __( '< OK >' ),
                      '-onpress' => sub {
                          if ( _test_date ( $time_field->text() ) ) {
                              $self->_clean_ok_cb_helper( $time_field );
                              $dialog->loose_focus();
                          }
                      } } ];
    my $bb_width = button_box_width( $buttons );
    $dialog->add( 'buttons', 'Buttonbox',
                  '-buttons' => $buttons,
                  '-x' => -20, '-y' => 4,
                  '-height' => 1, '-width' => $bb_width );

    $dialog->modalfocus();
    $self->delete( 'stream_clean' );
    $self->draw( 1 );

    return;
}

sub _update_labels {
    my $self = $_[0];

    my $streams = $self->getobj( 'streams' );
    my $stream_start_value = $self->getobj( 'stream_start_value' );
    my $stream_end_value = $self->getobj( 'stream_end_value' );
    my $nrecords_value = $self->getobj( 'nrecords_value' );
    my $stream_name = $streams->get();

    my $start_value;
    my $end_value;
    my $nrecords;
    if ( defined $stream_name ) {
        my $stream = $self->{'store'}->open_dlf_stream( $stream_name );
        $start_value = _epoch_to_date( $stream->start_time() );
        $end_value = _epoch_to_date( $stream->end_time() );
        $nrecords = $stream->nrecords();
        $stream->close();
    } else {
        $start_value = '';
        $end_value = '';
        $nrecords = '';
    }

    $stream_start_value->text( $start_value );
    $stream_end_value->text( $end_value );
    $nrecords_value->text( $nrecords );

    return;
}

sub store {
    return $_[0]->{'store'};
}

# callbacks
sub _sel_change_cb {
    my $list_widget = $_[0];

    $list_widget->{'-selected'} = $list_widget->{'-ypos'};
    $list_widget->run_event( '-onchange' );

    return;
}

sub _change_cb {
    my $list_widget = $_[0];

    croak "'-selected' can never be undefined"
      unless defined $list_widget->{'-selected'};

    my $self = $list_widget->parent();
    $self->_update_labels();

    return;
}

# helper functions
sub _test_date {
    my $text = $_[0];

    return 0
      unless ( $text =~ m/^(\d{2}|\d{4})-(\d{2})-(\d{2})$/ );
    my $month = $2;
    my $day = $3;
    return 0
      if ( $month < 1 || $month > 12 );
    return 0
      if ( $day < 1 || $day > 31 );

    return 1;
}

sub _date_to_epoch {
    my $date = $_[0];

    check_param( $date, 'date', \&_test_date,
                 "wrong format for 'date' (yyyy-mm-dd)" );

    $date =~ m/^(\d{2}|\d{4})-(\d{2})-(\d{2})$/;
    my $year = $1;
    my $month = $2 - 1;
    my $day = $3;

    return timelocal( 0, 0, 0, $day, $month, $year );
}

sub _epoch_to_date {
    my $epoch = $_[0];

    return __( '-- no date --' )
      unless $epoch;
    check_param( $epoch, 'epoch', qr/^\d+$/,
                 "'epoch' should be a positive integer" );

    return strftime( '%Y-%m-%d %H:%S', localtime $epoch );
}

1;
