package Lire::ReportSpec;

use strict;

use base qw/ Lire::XMLSpecContainer /;

use Carp;

use Lire::Config;
use Lire::DataTypes qw/ :special /;
use Lire::XMLSpecContainer;
use Lire::Report::Subreport;
use Lire::Report::TableInfo;
use Lire::Utils qw/ check_object_param /;

=pod

=head1 NAME

Lire::ReportSpec - API to the XML report specifications.

=head1 SYNOPSIS

    use Lire::ReportSpec;

=head1 DESCRIPTION

FIXME: Write some docs!

=cut


########################################################################
#                        Lire::XMLSpecContainer METHODS
########################################################################

sub root_element {
    my ( $self ) = @_;

    return "report-spec";
}

sub root_xml_attrs {
    my ( $self ) = @_;
    if (defined $self->{'charttype'}) {
        return qq{ charttype="$self->{'charttype'}"}
    } else {
        return "";
    }
}

sub file_from_id {
    my ( $self, $super, $id ) = @_;

    my $file;
    my @report_path = @{ Lire::Config->get( 'lr_reports_path' ) };
    foreach my $dir ( @report_path ) {
        next unless -d $dir;
        if ( -e $dir . "/$super/$id.xml" ) {
            $file = $dir . "/$super/$id.xml";
            last;
        }
    }
    croak "can't find XML report specification for $id in ",
      join (":", @report_path ), "\n"
      unless defined $file;

    return $file;
}

sub print_children {
    my ( $self, $fh, $indent ) = @_;

    my $pfx = ' ' x $indent;
    if ( defined $self->{'filter_spec'} ) {
        print $fh "$pfx<lire:filter-spec>\n";
        $self->{'filter_spec'}->print( $fh, $indent + 1);
        print $fh "\n$pfx</lire:filter-spec>\n\n";
    }

    print $fh "$pfx<lire:report-calc-spec>\n";
    $self->{'calc_spec'}->print( $fh, $indent + 1 );
    print $fh "$pfx</lire:report-calc-spec>\n";
}

=pod

=head1 OBJECT METHODS

=cut

sub charttype {
    my ( $self, $chart ) = @_;

    if ( @_ == 2 ) {
        if ( $chart) {
            croak "invalid chart type : $chart"
              unless ( check_chart( $chart ));

            $self->{'charttype'} = $chart;
        } else {
            delete $self->{'charttype'};
        }
    }

    return $self->{'charttype'};
}

sub filter_spec {
    my ( $self, $filter_spec ) = @_;

    if ( @_ == 2 ) {
        if ( defined $filter_spec ) {
            croak "filter specification must be of type Lire::FilterExpr"
              unless UNIVERSAL::isa( $filter_spec, "Lire::FilterExpr" );
            $self->{'filter_spec'} = $filter_spec;
        } else {
            delete $self->{'filter_spec'};
        }
    }

    return $self->{'filter_spec'};
}

sub calc_spec {
    my ( $self, $calc_spec ) = @_;

    if ( @_ == 2 ) {
        croak "calc specification must be of type Lire::Aggregator"
          unless UNIVERSAL::isa( $calc_spec, "Lire::Aggregator" );
        $self->{'calc_spec'} = $calc_spec;
    }

    return $self->{'calc_spec'};
}

=pod

=head2 mark_missing( $reason )

Mark that the report that should be generated by this report
specification should be marked as missing. The $reason parameter gives
the reason why the subreport will be missing.

=cut

sub mark_missing {
    my ( $self, $reason ) = @_;

    $self->{'missing'} = $reason;

    return;
}

=pod

=head2 is_missing()

Returns true if the subreport generated by this report specification
should be marked missing.

=cut

sub is_missing {
    return defined $_[0]{'missing'};
}

=pod

=head2 set_store( $store )

Sets the DLF store upon which this ReportSpec will be generated when
create_subreport() will be called.

=cut

sub set_store {
    my ( $self, $store ) = @_;

    check_object_param( $store, 'store', 'Lire::DlfStore' );

    croak "store doesn't contain a '", $self->schema()->id(), "' DLF stream "
      unless $store->has_dlf_stream(  $self->schema()->id() );

    $self->calc_spec()->_set_store( $store );

    return;
}

=pod

=head2 create_subreport( $report )

Creates a Lire::Report::Subreport object based on this report
specification. This will be a missing subreport if the mark_missing()
method was called. The $report parameter should be of type
Lire::Report and is needed by the Lire::Report::Subreport constructor.

Called by Lire::ReportSection::create_report_section().

=cut

sub create_subreport {
    my ( $self, $report ) = @_;

    check_object_param( $report, 'report', 'Lire::Report' );

    if ( $self->{'missing'} ) {
        return new_missing Lire::Report::Subreport( $report, $self->id,
                                                    $self->{'missing'}) ;
    }

    my $subreport = new Lire::Report::Subreport( $report, $self->id() );
    $subreport->title( $self->expanded_display_title() );
    $subreport->charttype( $self->charttype() );
    $subreport->description( $self->expanded_display_description() );

    $subreport->table_info( $self->create_table_info() );

    $self->calc_spec->create_entries( $subreport );
    $subreport->finalize();

    return $subreport;
}

=pod

=head2 create_table_info()

Returns a Lire::Report::TableInfo object that describes the table
generated by subreport of this type.

=cut

sub create_table_info {
    my ( $self ) = @_;

    my $info = new Lire::Report::TableInfo();

    $self->calc_spec->create_group_info( $info );
    $info->compute_group_layout();

    return $info;
}

# keep perl happy
1;

__END__

=pod

=head1 VERSION

$Id: ReportSpec.pm,v 1.46 2004/03/26 00:27:34 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001,2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
