package Lire::ReportGenerator;

use strict;

use Carp;
use POSIX qw/ strftime /;

use Lire::ReportConfig;
use Lire::Logger qw/lr_info lr_notice lr_warn/;
use Lire::DlfSchema;
use Lire::AsciiDlf::DlfInfo;
use Lire::Utils qw/check_param check_object_param/;

=pod

=head1 NAME

Lire::ReportGenerator - Generate a Report from a DLF store and a ReportConfig

=head1 SYNOPSIS

    use Lire::ReportGenerator;

    my $gen = new Lire::ReportGenerator( $$report_cfg, $dlf_store );
    my $report = $gen->generate_report();

=head1 DESCRIPTION

This method object will generate a report from a report configuration
file and a DLF store.

=cut

sub new {
    my ( $class, $report_cfg, $store ) = @_;

    check_object_param( $report_cfg, 'report_cfg', 'Lire::ReportConfig' );
    check_object_param( $store, 'store', 'Lire::DlfStore' );

    return bless { 'store'        => $store,
                   'report_cfg'   => $report_cfg,
                   'report_spec_order' => [],
                   'report_spec_by_key' => {},
                   'start_time' => undef,
                   'end_time' => undef,
                 }, $class;
}

sub load_dlf_stats {
    my $self = $_[0];

    my $dlf_stream =
      $self->{'store'}->open_dlf_stream( $self->{'report_cfg'}->superservice(), "r" );

    $self->{'start_time'} = $dlf_stream->start_time();
    $self->{'end_time'}   = $dlf_stream->end_time();
    $self->{'nrecords'}   = $dlf_stream->nrecords();

    $dlf_stream->close;

    return;
}

sub generate_report {
    my $self = $_[0];

    $self->load_report_cfg();
    $self->load_dlf_stats();
    $self->print_dlf_stats();
    $self->create_other_dlf_streams();

    foreach my $r ( @{$self->{'report_spec_order'}} ) {
        my $report_spec = $self->{'report_spec_by_key'}{$r};
        next unless $report_spec;

        unless ( $self->{'store'}->has_dlf_stream( $report_spec->schema()->id()))
        {
	    lr_notice( "report '$r' will be skipped because of unavailable input");
            $self->cancel_one_report( $r, "unavailable input" );
            next;
        }
        eval {
            $report_spec->set_store( $self->{'store'} );
        };
        if ( $@ ) {
            lr_warn( $@ );
            $self->cancel_one_report( $r, 'set_store() failed: $@' );
        }
    }

    my $report = eval {
        lr_info( "creating lire XML report..." );
        $self->{'report_cfg'}->create_report( $self->{'start_time'},
                                              $self->{'end_time'} );
    };
    lr_warn( $@ ) if $@;
    $self->{'store'}->close();

    return $report;
}

sub load_report_cfg {
    my $self = $_[0];

    # Merge the sections' filter specifications
    $self->{'report_cfg'}->merge_filters();

    foreach my $section ( $self->{'report_cfg'}->sections() ) {
        my @reports = $section->reports();
        foreach my $r ( @reports ) {
            push @{$self->{'report_spec_order'}}, $r->key();
            $self->{'report_spec_by_key'}{ $r->key() } = $r;
        }
    }

    return;
}

sub print_dlf_stats {
    my $self = $_[0];

    if ( ! defined $self->{'start_time'} ) {
        lr_info( "DLF contains ", $self->{'nrecords'},
                 " records; start and end time are unavailable" );
    } else {
        lr_info( "DLF contains ", $self->{'nrecords'},
                 " records; starts on ",
                 strftime( "%Y-%m-%d %H:%M:%S",
                           localtime $self->{'start_time'}),
                 "; ends on ",
                 strftime( "%Y-%m-%d %H:%M:%S",
                           localtime $self->{'end_time'} ),
               );
    }
}

sub create_other_dlf_streams {
    my $self = $_[0];

    foreach my $report ( @{$self->{'report_spec_order'}} ) {
        my $report_spec = $self->{'report_spec_by_key'}{$report};

        my $schema_id  = $report_spec->schema()->id();
        unless ($self->{'store'}->has_dlf_stream( $schema_id ) ) {
            $self->create_dlf_stream( $schema_id );
        }
    }
}

sub create_dlf_stream {
    my ( $self, $schema_id ) = @_;

    my $schema = Lire::DlfSchema::load_schema( $schema_id );
    if ( $schema->can( 'base' ) ) {
        unless ($self->{'store'}->has_dlf_stream( $schema->base()->id() ) ) {
            $self->create_dlf_stream( $schema->base()->id() );
        }
    }
    eval {
        if ( $schema->isa( "Lire::ExtendedSchema" ) ) {
            $self->create_extended_dlf_stream( $schema );
        } elsif ( $schema->isa( "Lire::DerivedSchema" ) ) {
            $self->create_derived_dlf_stream( $schema );
        }
    };
    lr_warn( $@ )
      if $@;
}
sub cancel_one_report {
    my ( $self, $report_key, $reason ) = @_;

    return unless $self->{'report_spec_by_key'}{$report_key};

    $self->{'report_spec_by_key'}{$report_key}->mark_missing( $reason );
    delete $self->{'report_spec_by_key'}{$report_key};
}

sub create_writer_cb {
    my ( $self, $schema, $out_stream ) = @_;

    my @fields = map { $_->name } $schema->fields;
    return sub {
        my $i = 0;
        my %dlf = map { $fields[$i++] => 
                          defined $_ && $_ ne 'LIRE_NOTAVAIL' ? $_ : undef }
          @{$_[0]};
        $out_stream->write_dlf( \%dlf );
    };
}

sub create_derived_dlf_stream {
    my ( $self, $schema ) = @_;
    lr_info( "creating derived DLF stream '", $schema->id(), "' ",
             "from '", $schema->base->id(), "'" );

    my $dlf_stream =
      $self->{'store'}->open_dlf_stream( $schema->base->id(), "r" );
    my $out_stream =
      $self->{'store'}->open_dlf_stream( $schema->id(), "w" );
    $out_stream->clean();

    my $writer_cb = $self->create_writer_cb( $schema, $out_stream );
    my $module = $schema->module();
    $module->init_computation( new Lire::AsciiDlf::DlfInfo( $dlf_stream ),
                               $writer_cb );

    my $dlf;
    my $count = 0;
    my $rtotal = $dlf_stream->nrecords();
    my $required_fields_checker = 
      $self->create_required_fields_checker_func( $schema );
    while ( $dlf = $dlf_stream->read_dlf_aref ) {
        next unless $required_fields_checker->( $dlf );
        $module->dlf_record( $dlf, $writer_cb );
        $count++;

        lr_info( sprintf( "%.2f%%", $count*100 / $rtotal ),
                 " of DLF source '", $schema->id, "' completed" )
          unless $count % 10_000;
    }
    $module->end_computation( $writer_cb );
    lr_info( $out_stream->nrecords,
             " records in '", $schema->id, "' derived schema" );
    $out_stream->close;
    $dlf_stream->close;
}

sub create_extended_dlf_stream {
    my ( $self, $schema ) = @_;

    lr_info( "creating extended DLF source '", $schema->id ,"' ",
                 "from '", $schema->base->id, "'" );

    my $dlf_stream =
      $self->{'store'}->open_dlf_stream( $schema->base->id, "r" );
    my $out_stream =
      $self->{'store'}->open_dlf_stream( $schema->id, "w" );
    $out_stream->clean();

    my $module = $schema->module;
    $module->init_computation( new Lire::AsciiDlf::DlfInfo( $dlf_stream ) );

    my $dlf;
    my $count = 0;
    my $rtotal  = $dlf_stream->nrecords;
    my $required_fields_checker = 
      $self->create_required_fields_checker_func( $schema );
    my @fields = map { $_->name } $schema->fields;
    while ($dlf = $dlf_stream->read_dlf_aref ) {
        next unless $required_fields_checker->( $dlf );
        my $fields = $module->create_extended_fields( $dlf );

        my $i = 0;
        my %r = map { $fields[$i++] => 
                        defined $_ && $_ ne 'LIRE_NOTAVAIL' ? $_ : undef }
          @$dlf, @$fields;
        $out_stream->write_dlf( \%r );
        $count++;
        lr_info( sprintf( "%.2f%%", $count*100 / $rtotal ),
                 " of DLF source '", $schema->id, "' completed" )
          unless $count % 10_000;
    }
    $module->end_computation;
    lr_info( $out_stream->nrecords, " records in '", $schema->id, 
             "' extended schema" );

    $out_stream->close;
    $dlf_stream->close;
}

sub create_required_fields_checker_func {
    my ( $self , $schema ) = @_;

    my @fields_pos = ();
    foreach my $field ( @{$schema->required_fields} ) {
        push @fields_pos, $schema->base->field( $field )->pos;
    }
    return sub {
        foreach my $f ( @fields_pos ) {
            return 0 unless defined $_[0][$f];
        }
        return 1;
    }
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::ReportConfig(3pm), Lire::DlfStore(3pm), lr_dlf2xml(1)

=head1 VERSION

$Id: ReportGenerator.pm,v 1.22 2004/03/28 19:43:39 flacoste Exp $

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2003, 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut

