package Lire::Report::Note;

use strict;

use Carp;

use Lire::Utils qw/xml_encode/;

=pod

=head1 NAME

Lire::Report::Note - Interface to the note element.

=head1 SYNOPSIS

    print $image->content, "\n";

=head1 DESCRIPTION

This class offers an API to the note elements of a Lire report.

=head1 CONSTRUCTOR

=head2 new( [$xref_id] )

Creates a new Lire::Report::Note object. The $xref_id parameter may be
set to the ID of an element which is linked to this note.

=cut

sub new {
    my $proto = shift;
    my $class = ref $proto || $proto;

    my ( $xref_id ) = @_;

    my $self = bless {}, $class;
    $self->xref( $xref_id )
      if $xref_id;

    $self
}

=pod

=head1 OBJECT METHODS

=head2 xref([$xref_id])

Returns the ID of the element linked to this note, if it has one. If
the $xref_id parameter is set, the note's xref attribute will be set
to this new value..

=cut

sub xref {
    $_[0]{'xref'} = $_[1] if defined $_[1];
    $_[0]{'xref'};
}

=pod

=head2 content([$new_content])

Returns this note's content. The content is encoded in DocBook XML.

If the $new_content parameter is set, this method will set this note's
content to this new value.

=cut

sub content {
    $_[0]{'content'} = $_[1] if defined $_[1];

    $_[0]{'content'};
}

sub write_report {
    my ( $self, $fh, $indent ) = @_;

    $fh ||= *STDOUT;
    my $pfx = ' ' x $indent;

    print $fh $pfx, "<lire:note";
    print $fh qq{ xref="$self->{'xref'}"}
      if $self->{'xref'};
    print $fh ">\n", $self->content, $pfx, "</lire:note>\n";;
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::ReportParser::ReportBuilder(3pm) Lire::Report(3pm)
Lire::Report::Subreport(3pm) Lire::Report::Entry(3pm)
Lire::Report::Group(3pm)

=head1 VERSION

$Id: Note.pm,v 1.5 2004/03/26 00:27:33 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
