package Lire::Report;

use strict;

use Carp;
use POSIX qw/ strftime /;

use Lire::DataTypes qw/ :basic :special /;
use Lire::I18N qw/ set_fh_encoding /;
use Lire::Utils qw/ xml_encode check_param check_object_param /;

=pod

=head1 NAME

Lire::Report - Interface to a Lire report.

=head1 SYNOPSIS

    use Lire::ReportParser::ReportBuilder;
    my $parser = new Lire::ReportParser::ReportBuilder;
    my $report = $parser->parse( "report.xml" );

    print $report->superservice, " report\n";
    print "Generated on ", scalar localtime $report->date(), "\n";
    print "Timespan starts on ", scalar localtime $report->timespan_start(), "\n";
    print "Timespan ends on ", scalar localtime $report->timespan_start(), "\n";
    print "Hostname: ", $report->hostname(), "\n";
    foreach my $s ( $report->sections() ) {
        print "Section: '", $s->title(), "' has ", scalar $s->subreports(),
            " subreports in it\n";
    }

=head1 DESCRIPTION

This module offers an API to the report generated by Lire. The
Lire::ReportParser::ReportBuilder(3pm) can be used to create a
Lire::Report object from an XML file that complies with the Lire Report
Markup Language DTD.

=head1 CONSTRUCTOR

A Lire::Report object can be built from an XML report file, from the
new() method or from a Lire::ReportConfig object.

=head2 new( $superservice, $timespan_start, $timespan_end )

Creates a new Lire::Report. The $superservice, $timespan_start and
$timespan_end parameters initialize the attributes of the same name.

=cut

sub new {
    my ( $class, $superservice, $start, $end ) = @_;

    check_param( $superservice, 'superservice', \&check_superservice,
                 "invalid 'superservice' parameter" );
    check_param( $start, 'timespan_start', \&check_number,
                 "'timespan_start' parameter should be an integer" )
      if ( $start );
    check_param( $end, 'timespan_end', \&check_number,
                 "'timespan_end' parameter should be an integer" )
      if ( $end );

    croak "'timespan_end' should be greater than 'timespan_start': $end < $start"
      if ( $start && $start > $end );

    return bless { 'superservice' => $superservice,
                   'version'      => "2.0",
                   'generator'    => __PACKAGE__ . "(3pm)",
                   'date'         => time,
                   'timespan_start' => $start ? $start : "",
                   'timespan_end' => $end ? $end : "",
                   'sections'     => [],
                   'notes'        => [],
                 }, $class;
}

=pod

=head1 OBJECT METHODS

=head2 version( )

Returns the version number of the report format which was used in the
external representation of this report. This is alway 2.0.

=cut

sub version {
    return $_[0]{'version'};
}

=pod

=head2 generator([$generator)

Returns the generator string that will be outputted in comments when
the write_report() method is used. One can change that value by
passing a new value through the $generator parameter.

=cut

sub generator {
    $_[0]{'generator'} = $_[1] if defined $_[1];
    return $_[0]{'generator'};
}

=pod

=head2 superservice( )

Returns the superservice of this report.

=cut

sub superservice {
    return $_[0]{'superservice'};
}

=pod

=head2 hostname([$hostname])

Returns the hostname of this report.

If the $hostname is parameter is set, the report's hostname will be
set to this new value.

=cut

sub hostname {
    $_[0]{'hostname'} = $_[1] if @_ == 2;
    return $_[0]{'hostname'};
}

=pod

=head2 date( )

Returns the date in seconds since epoch on which this report was
generated.

=cut

sub date {
    return $_[0]{'date'};
}

=pod

=head2 timespan_start( )

Returns the start of the timespan covered by this report in seconds
since epoch.

=cut

sub timespan_start {
    return $_[0]{'timespan_start'};
}

=pod

=head2 timespan_end( )

Returns the end of the timespan covered by this report in seconds
since epoch.

=cut

sub timespan_end {
    return $_[0]{'timespan_end'};
}

=pod

=head2 title( [$title] )

Returns the report's title, if it has one.

If the $title is parameter is set, the report's title will be set to
this new value.

=cut

sub title {
    $_[0]{'title'} = $_[1] if @_ == 2;
    return $_[0]{'title'};
}

=pod

=pod

=head2 description( [$new_desc] )

Returns the report description. This description is encoded in DocBook
XML.

If the $description parameter is set, this method will set the
description to this new value. If the $description parameter is undef,
that description will be removed.

=cut

sub description {
    my ( $self, $new_desc ) = @_;

    # Mutator
    $self->{'description'} = $new_desc 
      if @_ == 2;

    return $self->{'description'};
}

=pod

=head2 notes( )

Returns the top-level report's notes as an array. This will be an
array of Lire::Reprot::Note objects.

=cut

sub notes {
    return @{$_[0]{'notes'}};
}

=pod

=head2 add_note( $note )

Add a top-level note. The $note parameter must be of the
Lire::Report::Note type.

=cut

sub add_note {
    my ( $self, $note ) = @_;

    check_object_param( $note, 'note', 'Lire::Report::Note' );

    push @{$self->{'notes'}}, $note;

    return;
}

=pod

=head2 sections( )

Returns the report's sections as an array. This will be an array of
Lire::Report::Section objects.

=cut

sub sections {
    return @{$_[0]{'sections'}};
}

=pod

=head2 add_section( $section )

Adds a section to this report. The $section parameter should be a
Lire::Report::Section object.

=cut

sub add_section {
    my ( $self, $section ) = @_;

    check_object_param( $section, 'section', 'Lire::Report::Section' );

    push @{$self->{'sections'}}, $section;

    return;
}

=pod

=head2 subreports_by_type( $type )

Returns all the subreports of type $type. The subreport's type
is the id of the report specification used to compute the reports.
This will search for subreports with this type in all the report's
sections. The subreports are returned as an array of
Lire::Report::Subreport objects.

=cut

sub subreports_by_type {
    my ( $self, $type ) = @_;

    my @subreports = ();
    foreach my $s ( $self->sections() ) {
        push @subreports, $s->subreports_by_type( $type );
    }

    return @subreports;
}


=pod

=head2 write_report( [FH] );

Write the report in XML format on the FH filehandle or STDOUT if
omitted.

=cut

sub write_report {
    my ( $self, $fh ) = @_;

    $fh ||= \*STDOUT;

    set_fh_encoding( $fh, 'UTF-8' );

    my $time  = strftime '%Y-%m-%d %H:%M:%S %Z', localtime $self->date();
    my $period_string = "";
    if ( $self->timespan_start() ) {
        my $stime = strftime( '%Y-%m-%d %H:%M:%S %Z',
                              localtime $self->timespan_start() );
        my $etime = strftime( '%Y-%m-%d %H:%M:%S %Z', 
                              localtime $self->timespan_end() );
        $period_string = "$stime - $etime";
    } else {
        $period_string = "Unknown Period";
    }

    print $fh <<EOF;
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE lire:report PUBLIC
  "-//LogReport.ORG//DTD Lire Report Markup Language V$self->{'version'}//EN"
  "http://www.logreport.org/LRML/$self->{'version'}/lrml.dtd">
<lire:report xmlns:lire="http://www.logreport.org/LRML/" version="$self->{'version'}" superservice="$self->{'superservice'}">
 <!-- generated by $self->{'generator'} -->
EOF
    print $fh " <lire:title>", xml_encode( $self->title() ), "</lire:title>\n"
      if defined $self->title();

    print $fh <<EOF;
 <lire:date time="$self->{'date'}">$time</lire:date>
 <lire:timespan start="$self->{'timespan_start'}" end="$self->{'timespan_end'}">$period_string</lire:timespan>
EOF

    print $fh " <lire:hostname>", xml_encode( $self->hostname() ), "</lire:hostname>\n"
      if defined $self->hostname();

    print $fh " <lire:description>\n", $self->description(),
      " </lire:description>\n\n"
        if ( $self->description() );

    if ( $self->notes() ) {
        print $fh " <lire:notes>\n";
        foreach my $n ( $self->notes() ) {
            $n->write_report( $fh, 1 );
        }
        print $fh " </lire:notes>\n\n"
    }

    foreach my $s ( $self->sections() ) {
        $s->write_report( $fh, 1 );
    }
    print $fh "</lire:report>\n";

    return;
}

=pod

=head2 delete( )

Removes all circular references so that the object can be freed.

=cut

sub delete {
    my ( $self ) = @_;

    foreach my $s ( $self->sections() ) {
        foreach my $r ( $s->subreports() ) {
            $r->delete;
        }
    }

    return;
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::ReportParser::ReportBuilder(3pm) Lire::Report::Section(3pm)
Lire::Report::Subreport(3pm) Lire::Report::Entry(3pm)
Lire::Report::Group(3pm) Lire::Report::Note(3pm)
Lire::ReportParser(3pm)

=head1 VERSION

$Id: Report.pm,v 1.20 2004/03/26 00:27:34 wsourdeau Exp $

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2002, 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
