package Lire::Group;

use strict;

use base qw/ Lire::Aggregator /;

use Carp;

use Lire::DlfQuery;
use Lire::DataTypes qw/ check_int /;

=pod

=head1 NAME

Lire::Group - Base class for implementation of the group aggregator

=head1 SYNOPSIS

    use Lire::Group;

=head1 DESCRIPTION

This module is the base class for implementation of the group
aggregator. This aggregator will split the DLF records in groups having
the same key. The key is made up of one or more several non-numerical
fields. The number of keys to keep in the report can be configured
through the limit attribute. The keys can also be sorted using the
sort attribute.

=head1 CONSTRUCTOR

=head2 new( %params )

Creates a new instance of a group aggregator. In addition to the
normal report operator parameters, the limit attribute can be
initialized at creation time.

=cut

sub new {
    my $proto = shift;
    my $class = ref $proto || $proto;

    my $self = bless { 'fields' => [] }, $class;

    my %args = @_;
    $self->SUPER::init( @_, 'op' => 'group' );

    $self->limit( $args{'limit'} )
      if exists $args{'limit'};

    return $self;
}

=pod

=head1 METHODS

=head2 limit( [$new_limit] )

Returns the number of keys to display in the report. This can be an
integer or the name of one of the report specification parameter (when
it starts by $).

The limit attribute can be changed by passing a $new_limit parameter.

=cut

sub limit {
    my ( $self, $limit ) = @_;

    if ( @_ == 2 ) {
	if ( defined $limit && $limit =~ /^\$/ ) {
	    my $name = substr $limit, 1;
	    croak "$limit isn't a defined parameter"
	      unless $self->report_spec->has_param( $name );
	    my $type = $self->report_spec->param( $name )->type();
	    croak "$limit parameter isn't of type int"
	      unless $type eq "int";
	    $self->{'limit'} = $limit;
	} elsif ( defined $limit ) {
	    croak "limit must be a positive integer"
	      unless check_int( $limit );
	    $self->{'limit'} = $limit;
	} else {
	    delete $self->{'limit'};
	}
    }
    return $self->{'limit'};
}

sub _max_entries {
    my $self = $_[0];

    my $ratio;

    my $limit = $self->report_spec->resolve_param_ref( $self->{'limit'} );
    return 0 unless ( $limit );

    if ( $limit < 10 ) {
	$ratio = 2;
    } elsif ( $limit < 25 ) {
	$ratio = 1.5
    } elsif ( $limit < 50 ) {
	$ratio = 1.25
    } elsif ( $limit < 1000 ) {
	$ratio = 1.1
    } else {
	$ratio = 1.05;
    }

    return int( $limit * $ratio );
}

=pod

=head2 sort_fields( [$new_sort_fields] )

Returns fields and/or operator values which are going to be used to
sort the entries generated by this aggregator. This is an array
reference containing group field names and/or operator's name.

The sorting fields can be changed by passing a new array reference.

=cut

sub sort_fields {
    my ( $self, $fields ) = @_;

    if (@_ == 2 && defined $fields ) {
	croak "$fields isn't an array reference"
	  unless UNIVERSAL::isa( $fields, "ARRAY" );

	foreach my $f ( @$fields ) {
	    croak "$f isn't a defined sort field name"
	      unless $self->is_valid_sort_field( $f );
	}
	if ( @$fields ) {
	    $self->{'sort_fields'} = $fields;
	} else {
	    delete $self->{'sort_fields'};
	}
    } elsif ( @_ == 2 ) {
	delete $self->{'sort_fields'};
    }

    return $self->{'sort_fields'};
}

=pod

=head2 group_fields( [$new_fields] )

Returns as a reference to an array of Lire::GroupField objects the DLF
fields that are used to group the records.

If the $new_fields is set, the DLF fields used to group the records
will be changed. This parameter should be a reference to an array of
Lire::GroupField objects.

=cut

sub group_fields {
    my ( $self, $fields ) = @_;

    if ( @_ == 2 ) {
	croak "$fields isn't an array reference"
	  unless UNIVERSAL::isa( $fields, "ARRAY" );

	croak "group fields array is empty"
	  unless @$fields;

	foreach my $f ( @$fields ) {
	    croak "$f isn't of type Lire::GroupField"
	      unless UNIVERSAL::isa( $f, "Lire::GroupField" );
	}
	$self->{'fields'} = $fields;
    }
    return $self->{'fields'};
}

# Overrides Lire::Aggregator::build_query
sub build_query {
    my ( $self, $query ) = @_;

    $self->SUPER::build_query( $query );

    foreach my $field ( @{$self->{'fields'}} ) {
        $query->add_group_field( $field->name );
    }

    $query->set_sort_spec( join( " ", @{$self->sort_fields()} ) )
      if $self->sort_fields();

    return;
}

# Overrides Lire::Aggregator::set_group_summary
sub set_group_summary {
    my ( $self, $group, $row ) = @_;

    $self->SUPER::set_group_summary( $group, $row );

    $group->show( $self->report_spec->resolve_param_ref( $self->{'limit'} ) )
      if defined $self->{'limit'};

    return;
}

# Implements Lire::Aggregator::create_entry
sub create_entry {
    my ( $self, $group, $row ) = @_;

    my $max_entries = $self->_max_entries();
    return undef if $max_entries && $group->entries() >= $max_entries;

    my @names = ();
    foreach my $field ( @{$self->{'fields'}} ) {
        my $value = $row->{ $field->name() };
        unless ( defined $value ) {
            $group->missing_cases( $group->missing_cases() +
                                   $row->{'_lr_nrecords'} );
            return undef;
        }
        push @names, $row->{ $field->name() };
    }
    if ( @names ) {
        my $entry = $group->create_entry();
        foreach my $n ( @names ) {
            $entry->add_name( $n );
        }
        return $entry;
    }
    # Not reached
}

# ------------------------------------------------------------------------
# Method is_name_defined()
#
# Overide Lire:::Aggregator one to also check among the group fields
sub is_name_defined {
    my ( $self, $name ) = @_;

    # Check in fields
    foreach my $f ( @{$self->{'fields'}} ) {
	return 1 if $f->name eq $name;
    }

    # Chain up
    $self->SUPER::is_name_defined( $name );
}

#------------------------------------------------------------------------
# Method is_valid_sort_field( $name )
#
# Method that checks if the name is valid as a sorting field. It is
# only if the name is the name of one of the operator in this group.
# It can optionnally be prefixed by - to specify descending sort
# order.
sub is_valid_sort_field {
    my ( $self, $name ) = @_;

    # Remove descending control char
    $name =~ s/^-//g;

    $self->is_name_defined( $name );
}

# ------------------------------------------------------------------------
# Method xml_attrs()
#
# Implementation required by Lire::Aggregator
sub xml_attrs {
    my ( $self ) = @_;

    my $attr = "";
    if ( defined $self->{'sort_fields'} &&
	 @{$self->{'sort_fields'}})
    {
	my $fields = join " ", @{$self->{'sort_fields'}};
	$attr .= qq{ sort="$fields"};
    }
    $attr .= qq{ limit="$self->{'limit'}"}
      if exists $self->{'limit'};

    $attr;
}

# ------------------------------------------------------------------------
# Method print_content( $fh, $pfx )
#
# Override to output the field elements.
sub print_content {
    my ( $self, $fh, $pfx ) = @_;

    foreach my $f ( @{$self->{'fields'}} ) {
	$f->print( $fh, $pfx + 1);
    }

    $self->SUPER::print_content( $fh, $pfx );
}

# Implements Lire::ReportOperator::name()
sub name {
    return 'group:' . join( ".", map { $_->name() } @{$_[0]->group_fields()} );
}

# Implements Lire::Aggregator::create_categorical_info
sub create_categorical_info {
    my ( $self, $info ) = @_;

    foreach my $field ( @{$self->group_fields} ) {
	$field->create_categorical_info( $info );
    }
}


1;

__END__

=head1 SEE ALSO

Lire::ReportSpec(3pm), Lire::GroupField(3pm), Lire::ReportOperator(3pm)

=head1 VERSION

$Id: Group.pm,v 1.32 2004/03/26 00:27:34 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001, 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut

