package Lire::Config::SpecParser;

use strict;

use Lire::Config::TypeSpec;
use Lire::Utils qw/ check_param /;

use XML::Parser;
use Carp;

use vars qw($LRCSML_NS %LRCSML_ELEMENTS);

BEGIN {
    $LRCSML_NS = 'http://www.logreport.org/LRCSML/';

    my @elmnts = qw/ config-spec list select option command dlf-converter dlf-schema file directory executable integer boolean string object record plugin summary description /;

    %LRCSML_ELEMENTS = map { $_ => 1 } @elmnts;
}

sub new {
    my $proto = shift;
    my $class = ref($proto) || $proto;

    my $self = bless {
        'configspec' => new Lire::Config::ConfigSpec(),
        'paramstack' => [],
        'collect_cdata' => undef,
    }, $class;

    return $self;
}

sub configspec {
    $_[0]->{'configspec'};
}

sub merge_specifications_dir {
    my ( $self, $dir ) = @_;

    croak "missing 'dir' argument"
      unless defined $dir;

    opendir(DIR, $dir)
      or croak("Can't open directory $dir: $!");
    foreach my $file (readdir(DIR)) {
        next unless $file =~ /\.xml$/;

        $self->merge_specification("$dir/$file");
    }
    close(DIR);
}

sub merge_specification {
    my ( $self, $file ) = @_;

    croak "missing 'file' argument"
      unless defined $file;

    my $parser = new XML::Parser(
        'Handlers' => {
            'Init'  => \&Init,
            'Final' => \&Final,
            'Start' => \&Start,
            'End' => \&End,
            'Char' => \&Char,
        },
        'Namespaces' => 1,
        'NoLWP'    => 1,
    );

    # XML::Parser does not provide a real way to pass our own
    # data along, so we'll make do with a hack. sigh.
    $parser->{'LireConfigSpecParser'} = $self;

    open(SPEC, $file)
      or croak("can't open Lire Config specification file $file: $!");
    eval { $parser->parse(\*SPEC); };

    croak "error while parsing XML Config spec $file: $@"
       if $@;

    close SPEC;
}

sub Char {
    my ($expat, $text) = @_;

    my $self = $expat->{'LireConfigSpecParser'};
    if ( defined $self->{'collect_cdata'}) {
        $self->{'collect_cdata'} .= $text;
    }
}

sub Init {
    my ($self) = @_;
}

sub Final {
    my ($self) = @_;
    return $self->{'configspec'};
}

sub error {
    my ($self, $expat, $msg) = @_;

    # Remove other at line message
    # $msg =~ s/(at.*?line \d+\n*)//gm;

    my $line = $expat->current_line;

    croak("$msg at line $line");
}

sub Start {
    my ($expat, $name) = @_;

    my $self = $expat->{'LireConfigSpecParser'};

    my $ns = $expat->namespace($name);
    $ns ||= ''; # Remove warning
    if($ns eq $LRCSML_NS) {
        # This is one of our elements
        croak($self, "unknown Lire ConfigSpec element: <$name>")
            unless exists $LRCSML_ELEMENTS{$name};

        {
            no strict 'refs';

            my $sub = $name . "_start";
            $sub =~ s/-/_/g;    # Hyphen aren't allowed in perl sub names

            eval { $sub->($self, @_); };

            $self->error($expat, $@) if $@;
        };
     } else {
        my $desc_name = $expat->generate_ns_name( "description",
                                                     $LRCSML_NS );
        if ( $expat->within_element( $desc_name ) ) {
            $self->{'collect_cdata'} .= $expat->original_string;
        } else {
            $self->error( $expat, "unknown element: $name" );
        }
    }
}

sub End {
    my ($expat, $name) = @_;

    my $self = $expat->{'LireConfigSpecParser'};

    my $ns = $expat->namespace($name);
    $ns ||= ''; # Remove warning
    if($ns eq $LRCSML_NS) {
        # This is one of our element
        croak($self, "unknown Lire ConfigSpec element: <$name>")
            unless exists $LRCSML_ELEMENTS{$name};

        {
            no strict 'refs';

            my $sub = $name . "_end";
            $sub =~ s/-/_/g;    # Hyphen aren't allowed in perl sub names

            eval { $sub->($self, @_); };

            $self->error($expat, $@) if $@;
        }
    } else {
        my $desc_name = $expat->generate_ns_name( "description",
                                                $LRCSML_NS );
        if ( $expat->within_element( $desc_name ) ) {
            $self->{'collect_cdata'} .= $expat->original_string;
        } else {
            $self->error( $expat, "unknown element: $name" );
        }
    }
}

sub _peek_param {
    my $self = $_[0];

    return $self->{'paramstack'}[-1];
}

sub scalar_start {
    my ($self, $elem) = @_;

    my $pelem = $self->_peek_param();
    $pelem->add( $elem )
      if defined $pelem;

    push @{$self->{'paramstack'}}, $elem;
}

sub scalar_end {
    pop @{$_[0]->{'paramstack'}};
}

sub config_spec_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( $self->{'configspec'});
}

sub config_spec_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub list_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( new Lire::Config::ListSpec( %attr ) );
}

sub list_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub select_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( new Lire::Config::SelectSpec( %attr ) );
}

sub select_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub option_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( new Lire::Config::OptionSpec( %attr ) );
}

sub option_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub string_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( new Lire::Config::StringSpec( %attr ) );
}

sub string_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub dlf_converter_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( new Lire::Config::DlfConverterSpec( %attr ) );
}

sub dlf_converter_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub dlf_schema_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( new Lire::Config::DlfSchemaSpec( %attr ) );
}

sub dlf_schema_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub command_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( new Lire::Config::CommandSpec( %attr ) );
}

sub command_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub integer_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( new Lire::Config::IntegerSpec( %attr ) );
}

sub integer_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub directory_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( new Lire::Config::DirectorySpec( %attr ) );
}

sub directory_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub file_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( new Lire::Config::FileSpec( %attr ) );
}

sub file_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub executable_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( new Lire::Config::ExecutableSpec( %attr ) );
}

sub executable_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub boolean_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( new Lire::Config::BooleanSpec( %attr ) );
}

sub boolean_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub object_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( new Lire::Config::ObjectSpec( %attr ) );
}

sub object_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub record_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( new Lire::Config::RecordSpec( %attr ) );
}

sub record_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub plugin_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->scalar_start( new Lire::Config::PluginSpec( %attr ) );
}

sub plugin_end {
    my $self = shift;
    $self->scalar_end( @_ );
}

sub summary_start {
    my ( $self, $expat, $name, %attr ) = @_;

    $self->{'collect_cdata'} = "";
}

sub summary_end {
    my ( $self, $expat, $name ) = @_;

    my $spec = $self->_peek_param();
    $spec->summary( $self->{'collect_cdata'} );
    $self->{'collect_cdata'} = undef;
}

sub description_start {
    my ( $self, $expat, $name, %attr ) = @_;

    $self->{'collect_cdata'} = "";
}

sub description_end {
    my ( $self, $expat, $name ) = @_;
    my $spec = $self->_peek_param();
    $spec->description( $self->{'collect_cdata'} );
    $self->{'collect_cdata'} = undef;
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::Config::SpecParser -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: SpecParser.pm,v 1.31 2004/03/24 11:11:34 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Wessel Dankers <wsl@logreport.org>

=cut
