package Lire::Config::Parser;

use strict;

use Carp;
use XML::Parser;

use Lire::Utils qw/ check_object_param /;
use Lire::Config::TypeSpec;
use Lire::Config::Value;

use vars qw/$LRCML_NS %LRCML_ELEMENTS/;

BEGIN {
    $LRCML_NS = 'http://www.logreport.org/LRCML/';

    my @elmnts = qw(config global report template job param);

    %LRCML_ELEMENTS = map { $_ => 1 } @elmnts;
}

sub new {
    my $proto = shift;
    my $class = ref($proto) || $proto;

    my %args = @_;

    check_object_param( $args{'spec'}, 'spec',
                        'Lire::Config::ConfigSpec' );

    my $self = bless {
        'specstack' => [],
        'confstack' => [],
        'spec' => $args{'spec'},
    }, $class;

    return $self;
}

sub load_config_file {
    my ( $self, $file ) = @_;

    my $parser = new XML::Parser(
        'Handlers' => {
            'Init' => \&Init,
            'Final' => \&Final,
            'Start' => \&Start,
            'End' => \&End,
            'Char' => \&Char
        },
        'Namespaces' => 1,
        'NoLWP'   => 1,
    );

    # XML::Parser does not provide a real way to pass our own
    # data along, so we'll make do with a hack. sigh.
    $parser->{'LireConfigParser'} = $self;

    $self->{'conf'} = new Lire::Config::ConfigFile( 'spec' => $self->{'spec'},
                                                    'filename' => $file,
                                                  );

    open( my $fh, $file )
      or croak("can't open Lire Config file $file: $!");

    eval { $parser->parse( $fh ); };

    croak "error while parsing XML Config $file: $@"
      if $@;

    close( $fh );

    return $self->{'conf'};
}

sub Init {
    my $self = $_[0];

    return;
}

sub Final {
    return $_[0]->{'conf'};
}

sub error {
    my ( $self, $expat, $msg) = @_;

    # Remove other at line message
    # $msg =~ s/(at.*?line \d+\n*)//gm;

    my $line = $expat->current_line;

    croak("$msg at line $line");
}

sub Start {
    my ($expat, $name) = @_;

    my $self = $expat->{'LireConfigParser'};

    my $ns = $expat->namespace($name);
    $ns ||= ''; # Remove warning
    if($ns eq $LRCML_NS) {
        # This is one of our elements
        croak($self, "unknown Lire ConfigSpec element: <$name>")
            unless exists $LRCML_ELEMENTS{$name};

        {
            no strict 'refs';

            my $sub = $name . "_start";
            $sub =~ s/-/_/g;    # Hyphen aren't allowed in perl sub names

            eval { $sub->($self, @_); };

            $self->error($expat, $@) if $@;
        };
    }

    return;
}

sub End {
    my ($expat, $name) = @_;

    my $self = $expat->{'LireConfigParser'};

    return unless exists $expat->{'LireConfigParser'}; # wtf?

    my $ns = $expat->namespace($name);
    $ns ||= ''; # Remove warning
    if($ns eq $LRCML_NS) {
        # This is one of our element
        croak("unknown Lire Config element: <$name>")
            unless exists $LRCML_ELEMENTS{$name};

        {
            no strict 'refs';

            my $sub = $name . "_end";
            $sub =~ s/-/_/g;    # Hyphen aren't allowed in perl sub names

            eval { $sub->($self, @_); };

            $self->error($expat, $@) if $@;
        }
    }

    return;
}

sub Char {
    my ($expat, $text) = @_;

    my $self = $expat->{'LireConfigParser'};

    my $cont = $self->{'confstack'}[0];

    if( defined($cont) && $cont->isa( "Lire::Config::Scalar" ) ) {
        local $SIG{'__WARN__'} = sub {};
        my $old_text = $cont->get();
        if ( defined $text ) {
            $cont->set( $old_text . $text )
        } else {
            $cont->set($text)
        }
    }

    return;
}

sub param_start {
    my ( $self, $expat, $param, %attr ) = @_;

    my $name = $attr{'name'};
    my $pspec = $self->{'specstack'}[0];
    my $pconf = $self->{'confstack'}[0];

    croak "root element must be <config>"
        unless defined($pspec) && defined($pconf);

    my $spec = $pspec->get($name);
    my $conf = $spec->instance(%attr);

    if($pconf->isa("Lire::Config::List")) {
        $pconf->append( $conf );
    } elsif( $pconf->isa("Lire::Config::Dictionary") ) {
        $pconf->set($conf);
    } else {
        croak "unknown parameter container type: $pconf";
    }

    if( $conf->isa('Lire::Config::Scalar') ) {
        $conf->set('');
    } elsif ( $conf->isa( 'Lire::Config::Plugin' ) ) {
        $spec = $conf->get_properties_spec();
    }

    unshift @{$self->{'specstack'}}, $spec;
    unshift @{$self->{'confstack'}}, $conf;

    return;
}

sub param_end {
    my $self = $_[0];
    shift @{$self->{'specstack'}};
    shift @{$self->{'confstack'}};

    return;
}

sub config_start {
    my ( $self, $expat, $name, %attr ) = @_;

    croak "<config> must be the root element"
        if @{$self->{'confstack'}};

    $self->{'specstack'} = [$self->{'spec'}];
    $self->{'confstack'} = undef;

    return;
}

sub config_end {
    my $self = $_[0];

    shift @{$self->{'specstack'}};
    shift @{$self->{'confstack'}};

    croak "<config> must be the root element"
        if @{$self->{'confstack'}};

    return;
}

# Obsolete stuff, kept for backward compatibility
# with older config files.
sub template_start {
    shift->obsolete_start( @_ );
    return;
}

sub template_end {
    shift->obsolete_end( @_ );
    return;
}

sub job_start {
    shift->obsolete_start( @_ );
    return;
}

sub job_end {
    shift->obsolete_end( @_ );
    return;
}

sub report_start {
    shift->obsolete_start( @_ );
    return;
}

sub report_end {
    shift->obsolete_end( @_ );
    return;
}

sub obsolete_start {
    my ( $self, $expat, $job, %attr ) = @_;

    my $spec = $self->{'specstack'}[0];
    my $conf = $spec->instance(%attr);

    # This is an orphaned node, it isn't part of
    # the config anymore.

    $self->{'confstack'} = [ $conf ];

    return;
}

sub obsolete_end {
    my $self = $_[0];

    $self->{'confstack'} = undef;

    return;
}

sub global_start {
    my ( $self, $expat, $global, %attr ) = @_;

    my $spec = $self->{'specstack'}[0];
    my $global_conf = $spec->instance(%attr);
    $self->{'conf'}->global( $global_conf );
    $self->{'confstack'} = [ $global_conf ];

    return;
}

sub global_end {
    my $self = $_[0];

    $self->{'confstack'} = undef;

    return;
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::Config::Parser -

=head1 SYNOPSIS

Parses a single Lire configuration file.

=head1 DESCRIPTION

Rather straightforward XML parser that follows the configuration
specification to parse an XML file.

=head1 VERSION

$Id: Parser.pm,v 1.41 2004/03/31 19:02:25 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Wessel Dankers <wsl@logreport.org>

=cut
