package Lire::AsciiDlf::ReportOperator;

use strict;

use base qw/ Exporter /;

use vars qw/ @EXPORT_OK /;

use Carp;

BEGIN {
    @EXPORT_OK = qw/ group_data_value /;
}

=pod

=head1 NAME

Lire::AsciiDlf::ReportOperator - Base interface for AsciiDlf operator implementation

=head1 SYNOPSIS

    # This module is only an interface description

=head1 DESCRIPTION

The Lire::AsciiDlf::ReportOperator module describes the method that
must be implemented by operator that can be used to compute Lire
report from Ascii DFL file.

It also defines the methods needed for the operator to properly
support merging of XML reports.

This interface only defines the method used by all operator that are
nested into an aggregator. The Lire::AsciiDlf::Aggregator modules
defines other methods that will need to be implemented by aggregator
operator.

=head1 REPORT COMPUTATION INTERFACE

=pod

=head2 init_merge( $period_start, $period_end )

This method is called once before merging of XML reports is started.

The $period_start and $period_end parameters gives the time spanned by
all the XML reports.

=cut

sub init_merge {
    croak "Unimplemented init_merge() method in ", ref $_[0], "\n";
}

=pod

=head2 end_merge()

This method is once after all the XML reports were merged.

=cut

sub end_merge {
    croak "Unimplemented end_merge() method in ", ref $_[0], "\n";
}

=pod

=head2 init_group_data()

This method should return a data structure that will be passed the
call to update_group_data() and merge_group_data(). There will be one
such data structure for every group that exists in the aggregator
containing this operator.

There are a few restrictions placed on this data structure in order to
assure that it can be sorted by aggregator that need to sort their
entries. If the data structure is

=over

=item SCALAR

That value can be used for sorting purpose.

=item SCALAR REFERENCE

The scalar value will be used as sorting key.

=item ARRAY REFERENCE

The first item in the array reference will be used as sorting key.

=item HASH REFERENCE

The C<sort_value> items in the hash will be used as sorting key.

=item OBJECT REFERENCE

The value returned by the sort_value() method will be used as sorting
key.

=back

A function group_data_value() can be used to extract the sorting key
value according to these rules. It can be imported:

    use Lire::AsciiDlf::ReportOperator qw/group_data_value/;

    my $v = group_data_value( $data );

=cut

sub init_group_data {
    croak "Unimplemented init_group_data() method in ", ref $_[0], "\n";
}

=pod

=head2 merge_group_data( $value, $data )

This method will be called once for each value generated by this
operator in the XML reports to be merged. 

The $value parameter can be either an hash reference or a
Lire::Report::Group object. It will be the same things that was
generated by this operator.

$data is the group data structure returned by init_group_data() for
the group where this $value should be merged.

=head2 end_group_data( $data )

This method will be called once for every group that was created by
the aggregator containing this operator. The $data parameter is the
data structure that was returned by init_group_data() for the current
group.

One possible use of this method would be, for example, for the avg
operator to compute the actual average since it cannot be done until
all the records are processed.

=cut

sub end_group_data {
    croak "Unimplemented end_group_data() method in", ref $_[0], "\n";
}

=pod

=head2 add_entry_value( $entry, $data )

This method will be called once for each group created in by the
aggregator. In this method, the operator should add the appropriate
name and value for this entry. $entry is the Lire::Report::Entry
object to which the names and/or values should be added. $data is the
data structure returned by init_group_data() corresponding for the
current group.

For example, its in this method that the sum operator will add the
value containing the sum for the current entry.

=cut

sub add_entry_value {
    croak "Unimplemented add_entry_value() method in ", ref $_[0], "\n";
}

=pod

=head1 UTILITY FUNCTION

=head2 group_data_value($data)

Returns the "value" from one data item created by a init_group_data
method.

=cut

sub group_data_value {
    if ( ref $_[0] eq 'SCALAR' ) {
	return ${$_[0]};
    } elsif (ref $_[0] eq 'ARRAY' ) {
	return $_[0][0];
    } elsif (ref $_[0] eq 'HASH' ) {
	return $_[0]{'sort_value'};
    } elsif (ref $_[0]) {
	return $_[0]->sort_value;
    } else {
	return$_[0];
    }
}

# keep perl happy
1;

__END__

=head1 SEE ALSO

Lire::AsciiDlf::Aggregator(3pm), Lire::Ascii::DlfInfo(3pm)

=head1 VERSION

$Id: ReportOperator.pm,v 1.11 2004/03/26 00:27:34 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut


