package Lire::AsciiDlf::Aggregator;

use strict;

# We do not inherit from Lire::Aggregator so as not to confuse the
# overidding mechanism. Otherwise, perl might find methods in
# Lire::Aggregator instead of the specific subclass in the Lire::
# package
use base qw/ Lire::AsciiDlf::ReportOperator /;

use Carp;

use Lire::DataTypes qw/:basic/;
use Lire::Logger qw/lr_notice/;

use constant INITED	  => 1;
use constant FINISHED	  => 2;
use constant MERGE_INITED => 3;

=pod

=head1 NAME

Lire::AsciiDlf::Aggregator - Base AsciiDlf implementation of Aggregator

=head1 SYNOPSIS

    use Lire::AsciiDlf::Aggregator;

    use base qw/Lire::AsciiDlf::Aggregator/;

=head1 DESCRIPTION

This module provides common implementation of all the methods required
by Lire::AsciiDlf::ReportOperator. It defines additional methods
required by Aggregator implementation to be able to compute reports
from Ascii DLF file.

The base Aggregator implementation takes care of computing the summary
information included in the Lire XML reports.

The computation specific to the aggregator should be implemented in
the init_agggregator_data(), update_aggregator_data() and
end_aggregator_data() methods.

This class also takes care of the case when the aggregator is the
top-level aggregator, that is the immediate child of the
report-calc-spec element in the report specification),

=head1 IMPLEMENTATION OF Lire::AsciiDlf::ReportOperator METHODS

=pod

=head1 init_merge( $period_start, $period_end )

The default implementation makes sure that all contained operators are
inited.

Implementation of specific aggregator must chain up to this method, if
they override it.

=cut

sub init_merge {
    my ($self, $period_start, $period_end ) = @_;

    foreach my $op ( @{$self->ops} ) {
	$op->init_merge( $period_start, $period_end );
    }

    $self->{'data'}   = $self->init_group_data;
    $self->{'_state'} = MERGE_INITED;

    return $self;
}

#------------------------------------------------------------------------
# Methode merge_subreport( $subreport )
#
# This method will be called once for every subreport to be merged.
# 
# $subreport is the Lire::Report::Subreport to merged.
#
# Method called by lr_xml_merge(1)
sub merge_subreport {
    my ( $self, $subreport ) = @_;

    croak "init_merge() wasn't called"
      unless $self->{'_state'} == MERGE_INITED;

    $self->merge_group_data( $subreport, $self->{'data'} );

    return $self;
}

=pod

=head2 end_merge()

The default implementation makes sure that all operators gets the
end_report() event.

Subclasses should chain up to this method, if they override it.

=cut

sub end_merge {
    my ( $self ) = @_;

    croak "init_merge() wasn't called"
      unless $self->{'_state'} == MERGE_INITED;

    $self->end_group_data( $self->{'data'} );
    $self->{'_state'} = FINISHED;

    return $self;
}

=pod

=head2 create_entries( $subreport )

This class implements the create_entries() method declard by
Lire::Aggregator so that actual entries get added to the subreport.

Subclasses only need to implement the create_group_entries() method to
add their entry to the Subreport.

=cut

sub create_entries {
    my ( $self, $subreport ) = @_;

    croak "end_report() wasn't called"
      unless $self->{'_state'} == FINISHED;

    $subreport->nrecords( $self->{'data'}[1] );
    $subreport->missing_cases( $self->{'data'}[3] );
    $self->set_summary_values( $subreport, $self->{'data'}[0] );
    $self->create_group_entries( $subreport, $self->{'data'}[2] );
}

=pod

=head2 init_group_data()

The Aggregator implements init_group_data(). It takes care of
computing the summary information. Subclass does the equivalent in
init_aggregator_data().

=cut

sub init_group_data {
    my ( $self ) = @_;

    # Elements of the array
    # 0 = summary data
    # 1 = nrecords
    # 2 = subclass' data
    # 3 = missing-cases
    return [ $self->init_summary_data, 0, $self->init_aggregator_data, 0 ];
}


=pod

=head2 merge_group_data( $value, $data )

The Aggregator implements merge_group_data(). It takes care of merging
the summary information. Subclass does the equivalent in
merge_aggregator_data().

=cut

sub merge_group_data {
    my ( $self, $value, $data ) = @_;

    croak "value should be of type Lire::Report::Group, not $value\n"
      unless UNIVERSAL::isa( $value, "Lire::Report::Group" );

    $data->[1] += $value->nrecords;
    $data->[3] += $value->missing_cases;
    $self->merge_summary_data( $value, $data->[0] );
    $self->merge_aggregator_data( $value, $data->[2] );
}

=pod

=head2 end_group_data($data)

The Aggregator implements end_group_data(). It takes care of
computing the summary information. Subclass does the equivalent in
end_aggregator_data().

=cut

sub end_group_data {
    my ( $self, $data ) = @_;

    $self->end_summary_data( $data->[0] );
    $self->end_aggregator_data( $data->[2] );
}

=pod

=head2 add_entry_value( $entry, $data )

This method will make sure that the entries of nested aggregator are
wrapped up in a Lire::Report::Group element.

There is no reason to override that method since the entries of the
aggregator are added in the create_group_entries() method.

=cut

sub add_entry_value {
    my ( $self, $entry, $data ) = @_;

    my $group = $entry->create_group;
    $group->nrecords( $data->[1] );
    $group->missing_cases( $data->[3] );
    $self->set_summary_values( $group, $data->[0] );
    $self->create_group_entries( $group, $data->[2] );
}

# ------------------------------------------------------------------------
# METHODS FOR SUMMARY STATISTICS COMPUTATION
#
# This class defines several methods that are to be used in subclasses
# to compute the summary statistics that are included in Lire XML
# report. Those summary statistics are only computed by GroupOp
# operator (those that compute an aggregated value like avg, sum,
# etc.). The summary value is computed over all the records seen by the
# aggregator instead of only the grouped records.

#------------------------------------------------------------------------
# Method init_summary_data ()
#
# Returns a summary data structure. 
sub init_summary_data {
    my ( $self, $data ) = @_;

    $data ||= {};

    foreach my $op ( @{$self->ops} ) {
	if ( $op->isa( 'Lire::AsciiDlf::GroupOp' ) ) {
	    $data->{$op->name} = $op->init_group_data;
	} elsif ( $op->isa( 'Lire::AsciiDlf::Aggregator' ) ) {
	    $op->init_summary_data( $data );
	}
    }

    return $data;
}

#------------------------------------------------------------------------
# Method merge_summary_data ( $group, $summary_data )
#
# This method updates the GroupOp's summary data structures.
sub merge_summary_data {
    my ( $self, $group, $data ) = @_;

    foreach my $op ( @{$self->ops} ) {
	if ( $op->isa( 'Lire::AsciiDlf::GroupOp' ) ) {
	    my $value = $group->get_summary_value( $op->name );
	    unless ($value) {
		lr_notice( "missing summary value for ", $op->name,
			   " operator");
		next;
	    }
	    $op->merge_group_data( $value, $data->{$op->name} );
	} elsif ( $op->isa( 'Lire::AsciiDlf::Aggregator' ) ) {
	    $op->merge_summary_data( $group, $data );
	}
    }
}

#------------------------------------------------------------------------
# end_summary_data( $summary_data )
#
# This method calls end_group_data() on all GroupOp's data structures.
sub end_summary_data {
    my ( $self, $data ) = @_;

    foreach my $op ( @{$self->ops} ) {
	if ( $op->isa( 'Lire::AsciiDlf::GroupOp' ) ) {
	    $op->end_group_data( $data->{$op->name} );
	} elsif ( $op->isa( 'Lire::AsciiDlf::Aggregator' ) ) {
	    $op->end_summary_data( $data );
	}
    }
}

#------------------------------------------------------------------------
# set_summary_values( $group, $summary_data )
#
# Make sure all operators sets their summary value
sub set_summary_values {
    my ( $self, $group, $data ) = @_;

    foreach my $op ( @{$self->ops} ) {
	if ( $op->isa( 'Lire::AsciiDlf::GroupOp' ) ) {
            my $v = $op->create_value( $group,
                                       $op->data2dlf( $data->{$op->name()} ) );
            $group->set_summary_value( $op->name(), %$v );
	} elsif ( $op->isa( 'Lire::AsciiDlf::Aggregator' ) ) {
	    $op->set_summary_values( $group, $data );
	}
    }
}

#------------------------------------------------------------------------
# Method get_summary_value_string( $name )
#
# Returns a code string which can be used to access the summary
# data item of operator $name.
sub get_summary_value_string {
    my ( $self, $name ) = @_;

    return "->{'" . $name . "'}";
}


=pod

=head1 METHODS THAT SHOULD BE IMPLEMENTED BY SUBCLASSES

=head2 init_aggregator_data()

This is the equivalent of init_group_data() and is called from
Aggregator's implementation of init_group_data().

=cut

sub init_aggregator_data {
    croak "Unimplemented init_aggregator_data() method in ", ref $_[0], "\n";
}

=pod

=head2 merge_aggregator_data( $value, $data )

This method is the equivalent than merge_group_data() and is called
from Aggregator's implementation of merge_group_data().

=cut

sub merge_aggregator_data {
    croak "Unimplemented merge_aggregator_data() method in ", ref $_[0], "\n";
}

=pod

=head2 end_aggregator_data( $data )

This method is the equivalent of the end_group_data() and is called
from Aggregator's implementation of end_group_data().

=cut

sub end_aggregator_data {
    croak "Unimplemented end_aggregator_data() method in ", ref $_[0], "\n";
}

=head2 create_group_entries( $group, $data )

In this method, the aggregator should add one Lire::Report::Entry
object for every group created by the aggregator.

$group is an instance of Lire::Report::Group to which the entries
should be added. $data is the data structure returned by
init_group_data() for the group in which the aggregator is nested (or
the only structure that was created when the aggregator is the
top-level aggregator).

In this method, the aggregator must make sure to call
add_entry_value() on its contained operators for each entry created.

=cut

sub create_group_entries {
    croak "Unimplemented create_group_entries method in ", ref $_[0], "\n";
}

# keep perl happy
1;

__END__

=head1 SEE ALSO

Lire::AsciiDlf::Aggregator(3pm), Lire::Ascii::DlfInfo(3pm)

=head1 VERSION

$Id: Aggregator.pm,v 1.29 2004/03/26 00:27:34 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001-2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut

