/*
 * dvb_ca.h: generic DVB CA functions
 *
 * Copyright (C) 2004 Andrew de Quincey
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef _DVB_CA_H_
#define _DVB_CA_H_

#include <linux/list.h>

#include "dvbdev.h"

#define DVB_CA_MAX_SLOTS 4

#define DVB_CA_SLOTSTATUS_CAM_PRESENT 1
#define DVB_CA_SLOTSTATUS_CAM_CHANGED 2

#define DVB_CA_FLAG_EN50221 1
#define DVB_CA_FLAG_EN50221_IRQ_CAMCHANGE 2
#define DVB_CA_FLAG_EN50221_IRQ_READ 4

#define DVB_CA_TASK_CAM_CHANGED 1
#define DVB_CA_TASK_CAM_READ 2
#define DVB_CA_TASK_CAM_WRITE 4



/* A connection id to a CAM */
struct dvb_ca_connection {

        u8 connection_id; /* ID of this connection */

        struct dvb_ringbuffer rx_buffer; /* queue of data */
        struct dvb_ringbuffer tx_buffer; /* queue of data */

        int rx_partial_pkt; /* pointer to the position the current RX partial packet starts at, or -1 for none */

        int tx_partial_pkt_size; /* amount of data remaining from the current partial packet, or -1 for none */

        unsigned long last_used; /* last time in jiffies that this connection was used. */

        list_head next; /* next connection in list */
};

/* Information on a CA slot */
struct dvb_ca_slot {

        u8 cam_present; /* is a CAM present in this slot or not? */

        u16 cam_manfid; /* manufacturer ID of the CAM */
        u16 cam_devid; /* device ID of the CAM */

        u32 config_base; /* base register of CAM config */

        u8 config_option; /* value to write into Config Control register */

        u8 irq_supported; /* if 1, the CAM supportes IRQs */

        int link_buf_size; /* size of the buffer to use when talking to the CAM */

        struct semaphore sem; /* semaphore for syncing access to slot structure */

        int usage_counter; /* current usage counter of this slot */

        atomic_t camchange;
        atomic_t write;
        atomic_t read;

        list_head connections; /* list of dvb_ca_connection, one per connection_id */
};


/* Structure describing a CA interface */
struct dvb_ca {

        int index; //???

        dvb_adapter* adapter; //???

        /* Flags describing the interface (DVB_CA_FLAG_*) */
        u32 flags;

        /* number of slots supported by this CA interface */
        unsigned int slot_count;

        /* information on each slot */
        struct dvb_ca_slot slot_info[DVB_CA_MAX_SLOTS];

        /* functions for accessing attribute memory on the CAM */
        int (*read_attribute_mem)(struct dvb_ca* ca, int slot, int address);
        int (*write_attribute_mem)(struct dvb_ca* ca, int slot, int address, u8 value);

        /* functions for accessing the control interface on the CAM */
        int (*read_cam_control)(struct dvb_ca* ca, int slot, u8 address);
        int (*write_cam_control)(struct dvb_ca* ca, int slot, u8 address, u8 value);

        /* Functions for controlling slots */
        int (*slot_reset)(struct dvb_ca* ca, int slot);
        int (*slot_status)(struct dvb_ca* ca, int slot);
        int (*slot_ts_bypass)(struct dvb_ca* ca, int slot, int bypass);

        /* Hardware-specific IOCTL implementation. Set to NULL if you don't care */
        int (*ioctl)(void *, unsigned int cmd, void *parg);


        /* private data, used by caller */
        void* data;

        /* wait queues for read() and write() operations */
        wait_queue_head_t read_queue;
        wait_queue_head_t write_queue;

        /* PID of the monitoring thread */
        pid_t thread_pid;

        /* Semaphore for syncinc access when modifying driver structures */
        struct semaphore thread_sem;

        /* Wait queue used when shutting thread down */
        wait_queue_head_t thread_queue;

        /* Flag indicating when thread should exit */
        int exit;

        /* Flag indicating if the CA device is open */
        int open;
};




/* ******************************************************************************** */
/* Functions for controlling access to slots */

/**
 * Safely increment the usage counter for a CA slot.
 *
 * @param ca CA instance.
 * @param slot Slot concerned.
 *
 * @return 0 on success, <0 on failure.
 * */
extern int dvb_ca_slot_acquire(struct dvb_ca* ca, int slot);


/**
 * Safely decrement the usage counter for a CA slot.
 *
 * @param ca CA instance.
 * @param slot Slot concerned.
 *
 * @return 0 on success, <0 on failure.
 * */
extern int dvb_ca_slot_release(struct dvb_ca* ca, int slot);

/**
 * Acquire a slot exclusively. The slot semaphore will be left locked on successful
 * exit of this function.
 *
 * @param ca CA instance.
 * @param slot Slot concerned.
 *
 * @return 0 on success, <0 on failure.
 * */
extern int dvb_ca_slot_acquire_exclusive(struct dvb_ca* ca, int slot);


/**
 * Release an exclusively owned slot. The slot semaphore will be unlocked by this function.
 *
 * @param ca CA instance.
 * @param slot Slot concerned.
 *
 * @return 0 on success, <0 on failure.
 * */
extern int dvb_ca_slot_release_exclusive(struct dvb_ca* ca, int slot);



/* ******************************************************************************** */
/* Packet buffer functions */


/**
 * Flush buffers associated with a particular CA slot.
 *
 * @param ca CA instance.
 * @param slot Slot whose buffers are to be to flushed.
 *
 * @return 0 on success, <0 on failure.
 */
extern int dvb_ca_buf_flush_buffers(struct dvb_ca* ca, int slot);


/**
 * This function does not talk to a CAM; it is a utility function for grabbing data from the
 * connection buffers in chunks suitable for use by hardware-dependent code.
 *
 * Copy data to be sent to the CAM from the cache into the
 * supplied buffer. This call is for lower layers to get data still to
 * be sent to the CAM. The buffer is filled by the layers which receive data
 * from userspace.
 *
 * @param ca CA instance.
 * @param dest Where to put the data.
 * @param len Size of destination buffer.
 * @param last_fragment. This will be set to 1 if this is the last fragment of a packet,
 *                       or 0 if there is still more data in this packet to be transmitted.
 * @param slot Slot concerned.
 * @param connection_id Connection concerned.
 *
 * @return Number of bytes copied to dest, or <0 on error.
 */
extern int dvb_ca_buf_copy_to_cam(dvb_ca* ca, u8 *dest, int len, u8* last_fragment, u8 slot, u8 connection_id);


/**
 * This function does not talk to a CAM; it is a utility function for caching data retrieved
 * from a CAM by some hardware-dependent method and storing it in the connection buffers.
 *
 * Copy data recieved from the CAM into the cache. This call is for lower layers
 * to cache data received from the CAM. The buffer is emptied by the higher layers
 * which send data to userspace.
 *
 * @param ca CA instance.
 * @param data Buffer containing data.
 * @param len Number of bytes.
 * @param last_fragment 1 if this is the last fragment of a packet, or 0 if not.
 * @param slot Slot the data was received from.
 * @param connection_id Connection id the data was received from.
 *
 * @return 0 on success, or -1 if there was not enough space.
 * If -1 is returned, the packet will be automatically discarded. Any further data written
 * for this packet (last_fragment==0) will also be discarded until the next packet (i.e. the
 * write AFTER the write with last_fragment==1).
 */
extern int dvb_ca_buf_copy_from_cam(dvb_ca* ca, u8 *data, int len, u8 last_fragment, u8 slot, u8 connection_id);


/* ******************************************************************************** */
/* Initialisation/shutdown functions */

/**
 * Initialise a new DVB CA device.
 *
 * @param dvb_adapter DVB adapter to attach the new CA device to.
 * @param ca_dev The dvb_device_t to return.
 * @param ca The dvb_ca instance.
 *
 * @return 0 on success, nonzero on failure
 */
extern int dvb_ca_init(dvb_adapter_t *dvb_adapter, dvb_device_t **ca_dev, struct dvb_ca* ca);

/**
 * Release a DVB CA device.
 *
 * @param ca_dev The dvb_device_t instance for the CA device.
 * @param ca The associated dvb_ca instance.
 */
extern void dvb_ca_release(dvb_device_t *ca_dev, struct dvb_ca* ca);



#endif
