/*b
 * Copyright (C) 2001,2002,2003  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

/*
 * schwab.com
 *
 * line-oriented ASCII, using custom buffering.
 *
 * This is the nicest streamer protocol I've seen so far.  My only
 * complaints are there is no tradetype indicator, and you cannot request
 * the most recent N news articles.  That may simply be a lack of
 * complete understanding on my part, though.
 *
 * 63-82-176-11.cybertrader.com port 1081 ...
 *
 * > 83;CLIENT;{l13nT;192.168.1.106;241677464;SSPro;2.206;;
 * > 82;7;
 * < 78;7;63.82.176.46;9000
 * > 82;3;
 * < 78;1;63.82.178.8;1007
 * > 82;8;0;1;
 * < 78;2;63.82.178.8;2007
 * < 78;8;63.82.176.62;1040;1
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <regex.h>
#include "debug.h"
#include "error.h"
#include "rc.h"
#include "streamer.h"
#include "srpref.h"
#include "linuxtrade.h"
#include "optchain.h"
#include "chart.h"
#include "info.h"
#include "util.h"
#include "news.h"
#include "article.h"
#include "alert.h"
#include "futs.h"
#include "l2sr.h"

#ifndef MIN
    #define MIN(A,B) (((A)<(B)) ? (A) : (B))
#endif
#ifndef MAX
    #define MAX(A,B) (((A)>(B)) ? (A) : (B))
#endif

#define		BUFLEN		65536
#define		SYMCNT		200
#define		SYMBUFLEN	(SYMCNT * (SYMLEN+1) + 1)

#define		NLB	2

typedef struct streamerpriv
{
	LINEBUF	lb[NLB];

	int	l2ok;		// Allowed to get L2 quotes
	time_t	last_keepalive;	// Time of last keepalive message
	time_t	last_flush;	// Time of last writefile flush

	int	last_time;	// Time in last $TIME update message
	time_t	last_utime;	// Unix time when the above happened

	char	*l2sym;

	/*
	 * News
	 */
	int	hotnews;	// Display hot news items
	int	reflags;
		#define RE_INC	1
		#define	RE_EXC	2
	regex_t	include_re;
	regex_t	exclude_re;

	int	headseqnum;	// Sequence number for news requests
	int	headpagenum;	// Page number for news requests
	int	numhead;	// Number of headlines requested
	char	headcmd[80];	// Buffer for headline command

	int	artseqnum;	// Sequence number for news requests

	/*
	 * Auxilliary server: Top 10 list, market movers
	 */
	int			haveaux;
	struct sockaddr_in      auxaddr;
	TOP10			top10[10][10];

	/*
	 * Chart data
	 */
	struct tm	*charttm;
	int		chartdays;
	char		chartsym[SYMLEN+1];
} STREAMERDATA;

static char	FutsES[16];
static char	FutsNQ[16];
static char	FutsDJ[16];

typedef struct
{
	char	*canon, *sym;
} SYMMAP;

static SYMMAP SymMap[] =
{
	{	"$DJI",		"$DJI",		},
	{	"$DJT",		"$DJT",		},
	{	"$DJU",		"$DJU",		},

	{	"$NYA",		"$NYA",		},
	{	"$TRIN",	"$TRIN",	},
	{	"$TICK",	"$TICK",	},

	{	"$COMP",	"$COMPX",	},
	{	"$NDX",		"$NDX",		},
	{	"$SPX",		"$SPX",		},
	{	"$OEX",		"$OEX",		},
	{	"$MID",		"$MID",		},
	{	"$SML",		"$SML",		},
	{	"$RLX",		"$RLX",		},

	{	"$XAL",		"$XAL",		},
	{	"$BTK",		"$BTK",		},
	{	"$XBD",		"$XBD",		},
	{	"$XAX",		"$XAX",		},
	{	"$XCI",		"$XCI",		},
	{	"$IIX",		"$IIX",		},
	{	"$NWX",		"$NWX",		},
	{	"$XOI",		"$XOI",		},
	{	"$DRG",		"$DRG",		},
	{	"$XTC",		"$XTC",		},

	{	"$GSO",		"$GSO",		},
	{	"$HWI",		"$HWI",		},
	{	"$RUI",		"$RUI",		},
	{	"$RUT",		"$RUT",		},
	{	"$RUA",		"$RUA",		},
	{	"$SOX",		"$SOX",		},
	{	"$OSX",		"$OSX",		},

	{	"$BKX",		"$BKX",		},
	{	"$GOX",		"$GOX",		},
	{	"$XAU",		"$XAU",		},
	{	"$YLS",		"$YLS",		},
	{	"$DDX",		"$DDX",		},
	{	"$DOT",		"$DOT",		},

	{	"$RXP",		"$RXP",		},
	{	"$RXH",		"$RXH",		},
	{	"$XNG",		"$XNG",		},
	{	"$FPP",		"$FPP",		},
	{	"$DJR",		"$DJR",		},
	{	"$UTY",		"$UTY",		},

	{	"$TYX",		"$TYX",		},
	{	"$TRINQ",	"$TRIQE",	},
	{	"$TICKQ",	"$TICQE",	},
	{	"$TRINA",	"$TRIA",	},
	{	"$TICKA",	"$TICA",	},
	{	"$VIX",		"$VIX",		},
	{	"$VXN",		"$VXN",		},

	{	"$NQ",		FutsNQ,		},
	{	"$ES",		FutsES,		},
	{	"$DJ",		FutsDJ,		},
	{	NULL,		NULL		}
};

static void streamer_send_headlines(STREAMER sr, char *sym, int numlines);
static void streamer_send_article(STREAMER sr, char *artkey);

//
// Convert canonical index names to/from streamer index names
//
static void
streamer_canon2sym(char *out, char *in)
{
	char	*ip, *op;
	char	*p;
	int	len;
	SYMMAP	*map;

	ip = in;
	op = out;
	for (;;)
	{
		p = strchr(ip, '|');
		if (!p) p = strchr(ip, ',');
		if (!p) p = strchr(ip, ' ');
		if (!p) p = strchr(ip, 0);

		len = p - ip;
		if (ip[len-1] != '.')
		{
			memcpy(op, ip, len);
			op[len] = 0;
		}
		else
		{
			// Long option symbol
			sprintf(op, ".%.*s", len-1, ip);
		}

		for (map = SymMap; map->canon; ++map)
			if (strcmp(op, map->canon) == 0)
			{
				strcpy(op, map->sym);
				break;
			}

		if (*p == 0)
			break;

		ip += len + 1;
		op = strchr(op, 0);
		*op++ = *p;
		*op = 0;
	}
}

static void
streamer_sym2canon(char *out, char *in)
{
	SYMMAP	*map;

	for (map = SymMap; map->sym; ++map)
		if (strcmp(in, map->sym) == 0)
		{
			strcpy(out, map->canon);
			return;
		}

	if (in != out)
		strcpy(out, in);
}

static int
streamer_select(STREAMER sr,
		int numfd, fd_set *readfds, fd_set *writefds,
		fd_set *exceptfds, struct timeval *timeout)
{
	int	i;

	for (i = 0; i < NLB; ++i)
	{
		if (!readfds)
			continue;
		if (sr->fd[i] < 0)
			continue;
		if (FD_ISSET(sr->fd[i], readfds) && sr->priv->lb[i].bufp)
		{
			FD_ZERO(readfds);
			FD_SET(sr->fd[i], readfds);
			if (writefds)
				FD_ZERO(writefds);
			if (exceptfds)
				FD_ZERO(exceptfds);
			return 1;
		}
	}

	if (0)
		fprintf(stderr, "do select\n");
	return select(numfd, readfds, writefds, exceptfds, timeout);
}

static void
streamer_close(STREAMER sr)
{
	int	i;

	for (i = 0; i < NLB; ++i)
		linebuf_close(&sr->priv->lb[i]);

	for (i = 0; i < sr->nfd; ++i)
	{
		close(sr->fd[i]);
		sr->fd[i] = -1;
	}
}

static void
streamer_record(STREAMER sr, FILE *fp)
{
	sr->writefile = fp;
}

static void
streamer_init(STREAMER sr)
{
	int	i;

	sr->refresh = 60;
	sr->id[0] = 0;

	sr->nfd = 2;
	for (i = 0; i < sr->nfd; ++i)
		sr->fd[i] = -1;
	for (i = 0; i < NLB; ++i)
		linebuf_init(&sr->priv->lb[i]);
	if (NLB > 1)
		linebuf_tag(&sr->priv->lb[1], "2< ");

	time(&sr->priv->last_keepalive);
	time(&sr->priv->last_flush);
	sr->priv->last_time = 0;
	sr->priv->last_utime = 0;
	sr->priv->headseqnum = 100;
	sr->priv->artseqnum = 100;

	sr->priv->haveaux = FALSE;

	if (sr->priv->charttm)
	    free(sr->priv->charttm);
	sr->priv->charttm = NULL;
	sr->priv->chartdays = 0;
	sr->priv->chartsym[0] = 0;
}

int
open_aux(STREAMER sr)
{
	int	rc;

	if (!sr->priv->haveaux)
		return (0);

	sr->fd[1] = socket(AF_INET, SOCK_STREAM, 0);
	if (sr->fd[1] < 0)
		return -2;

	rc = connect_timeout(sr->fd[1], (SA *) &sr->priv->auxaddr,
					sizeof(sr->priv->auxaddr), 15);
	if (rc < 0)
	{
		close(sr->fd[1]);
		sr->fd[1] = -1;
		return -3;
	}

	linebuf_open(&sr->priv->lb[1], sr->fd[1], 8192);

	streamer_printf2(sr->fd[1], "8 1||2.2|\n");

	sprintf(sr->id, "schwab.com+aux");

	return (0);
}

/*
 * Open connection to the streamer
 */
static int
streamer_open(STREAMER sr, RCFILE *rcp, FILE *readfile)
{
	struct hostent		*hep;
	struct sockaddr_in	sockaddr;
	int			rc;
	char			buf[512];
	int			len;
	int			conn_tries;
	int			num_tries;
	int			sleep_time;
	time_t			now;
	struct tm		tm;
	char			*username = get_rc_value(rcp, "username");
	char			*password = get_rc_value(rcp, "password");
	char			*hostname = get_rc_value(rcp, "hostname");
	int			port = atoi(get_rc_value(rcp, "port"));
	char			*software = get_rc_value(rcp, "software");
	int			useL2 = atoi(get_rc_value(rcp, "useL2"));
	int			usenews = atoi(get_rc_value(rcp, "usenews"));
	int			hotnews = atoi(get_rc_value(rcp, "hotnews"));
	char			*include = get_rc_value(rcp, "hotinclude");
	char			*exclude = get_rc_value(rcp, "hotexclude");
	char			*auxname = get_rc_value(rcp, "auxname");
	int			auxport = atoi(get_rc_value(rcp, "auxport"));
	regex_t			re;
	char			*sp, *dp;

	/*
	 * I wish I could do a stricter job of authorization here,
	 * but its not my problem since no method is provided.
	 */
	for (sp = username, dp = buf; *sp; ++sp)
		if (*sp >= '0' && *sp <= '9')
			*dp++ = *sp;
	*dp = 0;
	if (strlen(buf) != 8 && strlen(buf) != 9)
		return SR_AUTH;
	if (strcmp(password, "XXXXXX") == 0 || strlen(password) < 4)
		return SR_AUTH;

	streamer_init(sr);
	++sr->cnt_opens;
	++sr->cnt_realopens;
	time(&sr->time_open);
	time(&sr->time_realopen);

	sr->usenews = usenews;
	sr->priv->hotnews = hotnews;

	sr->priv->reflags = 0;
	if (include[0])
	{
		rc = regcomp(&re, include, REG_EXTENDED|REG_NOSUB);
		if (rc == 0)
		{
			debug(1, "IncludeNews: <%s>\n", include);
			sr->priv->include_re = re;
			sr->priv->reflags |= RE_INC;
		}
	}

	if (exclude[0])
	{
		rc = regcomp(&re, exclude, REG_EXTENDED|REG_NOSUB);
		if (rc == 0)
		{
			debug(1, "ExcludeNews: <%s>\n", exclude);
			sr->priv->exclude_re = re;
			sr->priv->reflags |= RE_EXC;
		}
	}

	if (readfile)
	{
		debug(1, "Reading from a file...\n");
		sr->fd[0] = fileno(readfile);
		sr->readfile = readfile;
		linebuf_open(&sr->priv->lb[0], sr->fd[0], BUFLEN);
		return sr->fd[0];
	}

	sr->readfile = NULL;

	hep = mygethostbyname(hostname);
	if (!hep)
		return (-1);

	memcpy(&sockaddr.sin_addr, hep->h_addr, hep->h_length);
	sockaddr.sin_family = AF_INET;
	sockaddr.sin_port = htons(port);

	/*
	 * The schwab streamer goes down for about 1 minute around
	 * 4:30 AM Eastern.  Adjust connection timeouts to account
	 * for this.
	 */
	time(&now);
	putenv("US/Eastern");
	tm = *localtime(&now);
	set_timezone();
	if (tm.tm_hour >= 4 && tm.tm_hour <= 6)
	{
	    sleep_time = 10;
	    num_tries = 30;
	}
	else
	{
	    sleep_time = 1;
	    num_tries = 5;
	}

	conn_tries = 0;
reconnect:
	if (++conn_tries >= num_tries)
		return -4;

	debug(5, "Open socket...\n");

	sr->fd[0] = socket(AF_INET, SOCK_STREAM, 0);
	if (sr->fd[0] < 0)
		return -2;

	debug(5, "Socket fd=%d...\n", sr->fd[0]);
	if (Debug >= 5)
	{
	    timestamp(stderr);
	    debug(0, "Connect to '%s', try %d...\n", hostname, conn_tries);
	}

	rc = connect_timeout(sr->fd[0], (SA *) &sockaddr, sizeof(sockaddr), 15);
	if (rc < 0)
	{
		if (Debug >= 5)
			syserror(0, "Couldn't connect\n");
		sleep(sleep_time);
		close(sr->fd[0]);
		goto reconnect;
	}

	linebuf_open(&sr->priv->lb[0], sr->fd[0], BUFLEN);

	debug(5, "Initialize...\n");

	// General hello message format...
	// Y|???|username|software|version|hostip
	if (software[0] != 'V' && useL2)
	{
		// No sense in doing the work to get the real IP address,
		// since any sane person is behind a firewall anyway.
		char *ipaddr = "192.168.1.105";

		// Street Smart Pro users...
		// Hmm, don't send the registration symbol, and we get futs
		if (0)
			streamer_printf(sr->fd[0],
				"Y|2|%s|StreetSmart Pro%c|2.109|%s\n",
				username, 0xae, ipaddr);
		else
			streamer_printf(sr->fd[0],
				"Y|2|%s|StreetSmart Pro|2.109|%s\n",
				username, ipaddr);
		sr->priv->l2ok = TRUE;
		futscode(FutsNQ, "/NQ", FUTS_SCHWAB, NULL);
		futscode(FutsES, "/ES", FUTS_SCHWAB, NULL);
		// not yet available..
		futscode(FutsDJ, "/DJ", FUTS_SCHWAB, NULL);
	}
	else
	{
		// Velocity users...
		streamer_printf(sr->fd[0], "Y|2.0|BOGUS_TOKEN|Velocity\n");
		sr->priv->l2ok = FALSE;
		strcpy(FutsNQ, "$NQ");
		strcpy(FutsES, "$ES");
		// not yet available..
		strcpy(FutsDJ, "$DJ");
	}

	// General get quote format...
	// 4|nsyms|sym|sym|...
	// General add streamer format...
	// 2|nsyms|sym|sym|...
	len = streamer_printf(sr->fd[0],
			"4|1|$TIME\n"
			"2|1|$TIME\n");

	if (hotnews)
		len = streamer_printf(sr->fd[0], "R|1|!NEWSHOT|\n");

	sprintf(sr->id, "schwab.com");

	/*
	 * Now get address and open auxilliary server
	 */
	hep = mygethostbyname(auxname);
	if (!hep)
		return (0);

	memcpy(&sr->priv->auxaddr.sin_addr, hep->h_addr, hep->h_length);
	sr->priv->auxaddr.sin_family = AF_INET;
	sr->priv->auxaddr.sin_port = htons(auxport);
	sr->priv->haveaux = TRUE;

	open_aux(sr);

	return 0;
}

static void
streamer_send_quickquote(STREAMER sr, char *sym)
{
	char	sbuf[32];

	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_canon2sym(sbuf, sym);

	streamer_printf(sr->fd[0], "4|1|%s\n", sbuf);
}

static void
streamer_send_livequote(STREAMER sr, char *sym)
{
	streamer_send_quickquote(sr, sym);
}

static void
streamer_send_symbols(STREAMER sr, char *symbols, int add)
{
	char	sbuf[SYMBUFLEN];
	char	*s, *p;

	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_canon2sym(sbuf, symbols);

	for (p = s = sbuf; *p; s = p+1)
	{
		int	len;
		p = strchr(s, '|');
		if (!p) p = strchr(s, ',');
		if (!p) p = strchr(s, ' ');
		if (!p) p = strchr(s, 0);
		len = p - s;
		if (add)
			streamer_printf(sr->fd[0],
				"3|1|%.*s\n"
				"4|1|%.*s\n"
				"2|1|%.*s\n"
				"R|1|%.*s\n",
				len, s, len, s, len, s, len, s);
		else
			streamer_printf(sr->fd[0], "3|1|%s\n", s);
	}
}

static void
streamer_send_symbols_end(STREAMER sr, int add, int all)
{
}

static void
streamer_send_top10(STREAMER sr, char market, int type, int add)
{
	if (sr->fd[1] < 0 || sr->readfile)
		return;

	//
	// > 8 1|2.2|			initial ID
	// > 9 0|			clear decks?
	// > 9 ?|			send NAS %
	// > 8 Q||2.2|			???
	//
	if (add)
	{
		streamer_printf2(sr->fd[1], "9 0|\n");
		switch (type)
		{
		case TOP_NETGAIN:
		case TOP_NETLOSS:
			if (market == 'Q')
				streamer_printf2(sr->fd[1], "9 7|\n");
			else
				streamer_printf2(sr->fd[1], "9 1g|\n");
			break;
		case TOP_PCTGAIN:
		case TOP_PCTLOSS:
			if (market == 'Q')
				streamer_printf2(sr->fd[1], "9 ?|\n");
			else
				streamer_printf2(sr->fd[1], "9 3g|\n");
			break;
		}
	}
}

static void
streamer_send_movers(STREAMER sr, int on)
{
	static int	lo_near_16ths = 2;	// in .0625's
	static int	hi_near_16ths = 2;	// in .0625's

	// >10 ?o|01|038|2|0|0|		???
	//
	// >10 ??|??|???|?|lower|upper
	if (on)
	{
		streamer_printf2(sr->fd[1], "8 Q||2.2|\n");
		streamer_printf2(sr->fd[1],
			"10 ?o|01|038|2|%d|%d|\n",
				lo_near_16ths, hi_near_16ths);
	}
}

static inline int
le32(unsigned char *p)
{
    return p[0] + (p[1]<<8) + (p[2]<<16) + (p[3]<<24);
}
static inline int
le16(unsigned char *p)
{
    return p[0] + (p[1]<<8);
}

static double
dec6(unsigned char *p)
{
    int val = le32(p);
    switch (p[5])
    {
    case 0: return val;
    case 1: return val/10.0;
    case 2: return val/100.0;
    case 3: return val/1000.0;
    case 4: return val/10000.0;
    case 5: return val/100000.0;
    case 6: return val/1000000.0;
    case 7: return val/10000000.0;
    case 8: return val/100000000.0;
    default:	return 987.65;
    }
}

typedef struct
{
    char	sym[8+1];
    char	unk1;		// Command? 0x31 for chart?
    char	unk2;		// Always 0x02?
    char	type;		// 5=intraday, 11=daily
    int		len;		// total length, e.g. sizeof(CHARTREQ)
    int		seqnum;		// User defined sequence number of request
    short	days;		// Number of days to retrieve
    char	interval;	// e.g. 1/3/5 mins
    char	unk3;		// Sometimes 0, sometimes 0x40
} CHARTREQ;
typedef struct
{
    char	sym[8+1];
    char	unk1;
    char	unk2;
    char	type;	// 5=intraday, 11=daily
    int		len;	// total length of response
    int		seqnum;
    short	days;
    char	interval;	// e.g. 1/3/5 mins
    char	unk3;
} RSP;

static unsigned int	ChartSeqNum = 0;

static int
chart_response(int fd, unsigned char **bufp)
{
	*bufp = NULL;

	for (;;)
	{
	    int			rc;
	    int		  	len, datalen;
	    unsigned char	resphdr[16];
	    unsigned char 	*databuf;
	    unsigned char 	*datap;

	    rc = read_timeout(fd, resphdr, 16, 10);
	    if (rc <= 0)
		return -1;
	    if (Debug >= 5)
		hexdump(stderr, "<C ", "   ", resphdr, rc);

	    len = le32(resphdr+12);
	    len -= 16;
	    datalen = len;
	    datap = databuf = malloc(len);
	    if (!databuf)
		error(1, "Could not allocate %d bytes\n", len);
	    while (len)
	    {
		rc = read_timeout(fd, datap, len, 10);
		if (rc <= 0)
		{
		    free(databuf);
		    return -1;
		}
		if (Debug >= 5)
		    hexdump(stderr, "<C ", "   ", datap, rc);
		len -= rc;
		datap += rc;
	    }

	    if (resphdr[10] != 'V')
	    {
		*bufp = databuf;
		return datalen;
	    }
	    else
		free(databuf);
	}
}

/*
 * Return the mm/dd/yy of the most recent available chart data
 *
 * We need this because the intraday data does not tell us which
 * day it is for.  Why is it that every streamer has awful warts
 * like this?
 */
static int
chart_most_recent_days(int fd, char *sym, int days, struct tm *tmp)
{
	CHARTREQ	req;
	int		rc;
	int	  	datalen;
	unsigned char	*databuf;
	unsigned char	*datap;
	unsigned char	*datae;
	int		n;

	strncpy(req.sym, sym, 8);
	req.sym[8] = 0;
	req.unk1 = 0x31;
	req.unk2 = 0x02;
	req.seqnum = ++ChartSeqNum;
	req.days = days;
	req.unk3 = 0;
	req.type = 11;
	req.interval = 0;
	if (Debug >= 5)
	    hexdump(stderr, "C> ", "   ", &req, sizeof(req));
	rc = write(fd, &req, sizeof(req));
	if (rc < 0)
		return -1;

	rc = chart_response(fd, &databuf);
	if (rc < (24+0))
	{
	    if (databuf)
		free(databuf);
	    return -1;
	}

	datalen = le32(databuf+16) - 6;
	datap = databuf+16+6;
	datae = datap + datalen;
	n = 0;
	while (datap < datae)
	{
	    sscanf(datap, "%d/%d/%d",
		    &tmp->tm_mon, &tmp->tm_mday, &tmp->tm_year);
	    tmp->tm_mon -= 1;
	    tmp->tm_year += 2000 - 1900;
	    tmp->tm_hour = 0;
	    tmp->tm_min = 0;
	    tmp->tm_sec = 0;
	    debug(5, "most recent %d: %02d/%02d/%02d\n", n,
		    tmp->tm_mon+1, tmp->tm_mday, tmp->tm_year);
	    datap += 37;
	    ++tmp;
	    ++n;
	    if (n == days)
		break;
	}
	free(databuf);
	return n;
}

static int
within_period(int lasthhmm, int hhmm, int period)
{
    lasthhmm /= period;
    lasthhmm *= period;
    return (hhmm >= lasthhmm && hhmm < (lasthhmm+period));
}

static int
streamer_send_chart(STREAMER sr, char *sym, int freq, int periods, int days)
{
	RCFILE			*rcp = SrCur->rcfile;
	char			*chartname = get_rc_value(rcp, "chartname");
	int			chartport = atoi(get_rc_value(rcp,"chartport"));
	struct hostent		*hep;
	struct sockaddr_in	sockaddr;
	int			fd;
	int			rc;
	char			symbuf[32];
	int			mergecnt = 0;
	CHARTREQ		req;
	static int		seqnum = 0;
	int		  	datalen;
	unsigned char 		*databuf;
	unsigned char 		*datap;
	unsigned char 		*datae;

	int			mi;
	OHLCV			mergedata;
	int			mergehhmm = 0;
	time_t			now;
	struct tm		tm;
	int			curhhmm, lasthhmm;
	int			dayindex = 0;

	streamer_canon2sym(symbuf, sym);

	hep = mygethostbyname(chartname);
	if (!hep)
		return SR_ERR;

	memcpy(&sockaddr.sin_addr, hep->h_addr, hep->h_length);
	sockaddr.sin_family = AF_INET;
	sockaddr.sin_port = htons(chartport);

	fd = socket(AF_INET, SOCK_STREAM, 0);
	rc = connect_timeout(fd, (SA *) &sockaddr, sizeof(sockaddr), 10);
	if (rc < 0)
		return SR_ERR;

	if (1)
	{
	    static unsigned char firstreq[] =
	    {
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x09,0x00,0x18,0x00,0x00,0x00,
		0x2e,0x00,0x0b,0x00,0x0a,0x04,0x00,0x00
	    };
	    if (Debug >= 5)
		hexdump(stderr, "C> ", "   ", &firstreq, sizeof(firstreq));
	    rc = write(fd, firstreq, sizeof(firstreq));
	    if (rc < 0)
		    return SR_ERR;
	}

	if (freq == FREQ_HOUR &&
		(!sr->priv->charttm || strcmp(sr->priv->chartsym, symbuf)
		 || sr->priv->chartdays < days) )
	{
	    if (sr->priv->charttm)
		free(sr->priv->charttm);
	    sr->priv->charttm = (struct tm *) malloc(sizeof(struct tm) * days);
	    if (!sr->priv->charttm)
		error(1, "No space for %d days of chart data\n", days);
	    rc = chart_most_recent_days(fd, symbuf, days, sr->priv->charttm);
	    if (rc <= 0)
		return SR_ERR;
	    debug(1, "%d days requested, %d chart days retrieved\n", days, rc);
	    sr->priv->chartdays = rc;
	    strncpy(sr->priv->chartsym, symbuf, SYMLEN);
	    sr->priv->chartsym[SYMLEN] = 0;
	}

	strncpy(req.sym, symbuf, 8);
	req.sym[8] = 0;
	req.unk1 = 0x31;
	req.unk2 = 0x02;
	req.seqnum = ++seqnum;
	req.days = days;
	req.unk3 = 0;
	req.type = 0;
	req.interval = 0;

	switch(freq)
	{
	case FREQ_1MIN:	req.type = 5; req.interval = 1; mergecnt = 1; break;
	case FREQ_5MIN:	req.type = 5; req.interval = 5; mergecnt = 1; break;
	case FREQ_10MIN:req.type = 5; req.interval = 5; mergecnt = 2; break;
	case FREQ_HOUR: req.type = 5; req.interval = 5; mergecnt = 12; break;
	case FREQ_DAY:  req.type = 11; req.interval = 0; break;
	case FREQ_WEEK:  req.type = 11; req.interval = 0; break;
	default: return SR_ERR;
	}

	if (Debug >= 5)
	    hexdump(stderr, "C> ", "   ", &req, sizeof(req));
	rc = write(fd, &req, sizeof(req));
	if (rc < 0)
		return SR_ERR;

	rc = chart_response(fd, &databuf);
	if (rc < (24+0))
	{
	    if (databuf)
		free(databuf);
	    goto out;
	}

	datalen = le32(databuf+16) - 6;
	datap = databuf+16+6;
	datae = datap + datalen;
	lasthhmm = 0;
	time(&now);
	now -= days * 24 * 60 * 60;
	if (1)
	    now += 24*60*60;
	tm = *localtime(&now);
	mi = 0;
	if (freq == FREQ_HOUR)
	{
	    int	lastdtime = 99999999;
	    int	gotdays;

	    dayindex = sr->priv->chartdays - 1;
	    gotdays = 0;
	    for (datap = databuf+16+6; datap < datae; datap += 37)
	    {
		int dtime = atoi(datap);
		if (dtime < lastdtime)
		    ++gotdays;
		lastdtime = dtime;
	    }
	    debug(1, "got %d days of intraday data\n", gotdays);
	    dayindex = gotdays - 1;
	    tm = sr->priv->charttm[dayindex--];
	}

	datap = databuf+16+6;
	while (datap < datae)
	{
	    OHLCV		cdata;
	    struct tm	lasttm;

	    cdata.open = dec6(datap+9+0);
	    cdata.high = dec6(datap+9+6);
	    cdata.low = dec6(datap+9+12);
	    cdata.close = dec6(datap+9+18);
	    cdata.volume = le32(datap+9+24);

	    if (freq == FREQ_DAY)
	    {
		sscanf(datap, "%d/%d/%d",
			&tm.tm_mon, &tm.tm_mday, &tm.tm_year);
		tm.tm_mon -= 1;
		tm.tm_year += 2000 - 1900;
		chart_data(symbuf,
				tm.tm_mon+1, tm.tm_mday, tm.tm_year+1900,
				0, &cdata);
	    }
	    else if (freq == FREQ_WEEK)
	    {
		time_t	this;

		// Get the date in the current days ohlcv record.
		sscanf(datap, "%d/%d/%d",
			&tm.tm_mon, &tm.tm_mday, &tm.tm_year);
		tm.tm_mon -= 1;
		tm.tm_year += 2000 - 1900;
		tm.tm_hour = 0;
		tm.tm_min = 0;
		tm.tm_sec = 0;
		this = mktime(&tm);
		tm = *localtime(&this);

		// If a new week has started, output the last week
		// Note that we get the day ohlcv records in reverse
		// date order.
		if (mi && tm.tm_wday > lasttm.tm_wday)
		{
		    debug(1, "%d/%d/%d: Start new week\n",
			    tm.tm_mon+1, tm.tm_mday, tm.tm_year);
		    lasttm = *tm2monday(&lasttm);
		    chart_data(symbuf, lasttm.tm_mon+1, lasttm.tm_mday,
				lasttm.tm_year+1900, 0, &mergedata);
		    mi = 0;
		}

		// Begin or update the weeks data
		if (mi++ == 0)
			mergedata = cdata;
		else
		{
		    mergedata.high = MAX(mergedata.high, cdata.high);
		    mergedata.low = MIN(mergedata.low, cdata.low);
		    mergedata.open = cdata.open;
		}
		lasttm = tm;
	    }
	    else
	    {
		int dtime = atoi(datap);
		int hr = dtime / 10000;
		int min = (dtime / 100) % 100;

		// See if we are on a new days worth of data
		curhhmm = hr*60 + min;
		if (curhhmm < lasthhmm)
		{
		    // Output last sample
		    if (mergecnt && mi)
			chart_data(symbuf,
				tm.tm_mon+1, tm.tm_mday, tm.tm_year+1900,
				mergehhmm, &mergedata);
		    mi = 0;

		    // Advance to next weekday
		    for (;;)
		    {
			now += 24*60*60;
			tm = *localtime(&now);
			if (tm.tm_wday >=1 && tm.tm_wday <= 5)
			    break;
		    }
		    if (freq == FREQ_HOUR && dayindex >= 0)
		    {
			tm = sr->priv->charttm[dayindex--];
			now = mktime(&tm);
		    }
		    debug(5, "New day: %02d/%02d/%02d, dayindex=%d\n",
			    tm.tm_mon+1, tm.tm_mday, tm.tm_year,
			    dayindex);
		}
		lasthhmm = curhhmm;

		if (mergecnt > 1)
		{
		    if (mi && !within_period(mergehhmm, curhhmm,
				mergecnt*req.interval))
		    {
			chart_data(symbuf,
				tm.tm_mon+1, tm.tm_mday, tm.tm_year+1900,
				mergehhmm, &mergedata);
			mi = 0;
		    }

		    if (mi == 0)
		    {
			mi = min % (mergecnt*req.interval);
			mi /= req.interval;
			mergedata = cdata;
			mergehhmm = curhhmm;
			debug(5, "	smerge %02d/%02d %02d:%02d mi=%d\n",
				tm.tm_mon+1, tm.tm_mday,
				mergehhmm/60, mergehhmm%60, mi);
			mergehhmm /= mergecnt*req.interval;
			mergehhmm *= mergecnt*req.interval;
		    }
		    else
		    {
			if (cdata.high > mergedata.high)
			    mergedata.high = cdata.high;
			if (cdata.low < mergedata.low)
			    mergedata.low = cdata.low;
			mergedata.close = cdata.close;
			debug(5, "	merge %02d/%02d %02d:%02d\n",
				tm.tm_mon+1, tm.tm_mday,
				curhhmm/60, curhhmm%60);
		    }
		    ++mi;
		}
		else
		    chart_data(symbuf,
				tm.tm_mon+1, tm.tm_mday, tm.tm_year+1900,
				curhhmm, &cdata);
	    }

	    datap += 37;
	}
	free(databuf);

	if (freq <= FREQ_HOUR && mi)
	{
	    chart_data(symbuf, tm.tm_mon+1, tm.tm_mday, tm.tm_year+1900,
		    mergehhmm, &mergedata);
	}
out:
	chart_data_complete(TRUE);
	close(fd);
	return 0;
}

static void
streamer_send_keepalive(STREAMER sr)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_printf(sr->fd[0],
			"V\n"
			"4|1|$TIME\n"
			"2|1|$TIME\n");
}

static void
streamer_send_headlines(STREAMER sr, char *sym, int numlines)
{
	time_t	then;
	struct tm *tmp;

	if (sr->fd[0] < 0 || sr->readfile)
		return;

	time(&then);
	then -= 4 * 24 * 60 * 60;	// Subtract four days
	tmp = localtime(&then);

	sprintf(sr->priv->headcmd,
			"Q|%%d|%s|%02d%02d%02d|%02d%02d|||%d|%%d\n",
			sym,
			tmp->tm_year % 100, tmp->tm_mon+1, tmp->tm_mday,
			tmp->tm_hour, tmp->tm_min,
			numlines);

	sr->priv->headpagenum = 1;
	streamer_printf(sr->fd[0], sr->priv->headcmd,
			++sr->priv->headseqnum, sr->priv->headpagenum++);

	sr->priv->numhead = numlines;
}

static void
streamer_send_article(STREAMER sr, char *artkey)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_printf(sr->fd[0], "P|%d|%s\n", ++sr->priv->artseqnum, artkey);
}

static int
streamer_send_l2(STREAMER sr, char *sym, int add)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return 0;
	if (!sr->priv->l2ok)
		return SR_AUTH;

	if (add)
	{
		sr->priv->l2sym = sym;
		streamer_printf(sr->fd[0], "7|1|%s|-1|\n5|1|%s|\n", sym, sym);
	}
	else
	{
		sr->priv->l2sym = NULL;
		streamer_printf(sr->fd[0], "6|1|%s|\n", sym);
	}
	return 1;
}

static int
decode64(unsigned char *s)
{
	int	val = 0;

	if (!s || *s == 0)
		return 0;

	while (*s >= '0' && *s < ('0' + 64))
	{
		val *= 64;
		val += *s++ - '0';
	}
	return val;
}

static double
decode(char *s)
{
	char	code;
	double	val;

	if (!s || *s == 0)
		return 0.0;

	code = *s++;
	switch (code)
	{
	case ',': val = 0.0; break;	// what does ",N" mean?
	case '1': val = decode64(s); val /= 10.0; break;
	case '2': val = decode64(s); val /= 100.0; break;
	case '3': val = decode64(s); val /= 1000.0; break;
	case '4': val = decode64(s); val /= 10000.0; break;
	case '5': val = decode64(s); val /= 100000.0; break;
	case '6': val = decode64(s); val /= 1000000.0; break;
	case '7': val = decode64(s); val /= 10000000.0; break;
	case '8': val = decode64(s); val /= 100000000.0; break;
	case '$': val = atoi(s); break;
	default: val = decode64(s); break;
	}

	//fprintf(stderr, "decode '%s' = %f\n", s, val);
	return val;
}

static int
tradetime(STREAMER sr)
{
	int	delta;
	time_t	now;

	if (sr->priv->last_utime)
	{
		time(&now);
		delta = now - sr->priv->last_utime;
		if (delta >= 60) delta = 59;
	}
	else
		delta = 0;

	return (sr->priv->last_time + delta);
}

static void
do_time(STREAMER sr, char *buf)
{
	// K|$TIME|$113200|0|0||||||A\|$101400|$60100|0|$60100|$60000
	// I|$TIME|$113300||||||||A]
	// I|$TIME||||||||||$101900

	char		*fld[32];
	int		rc;
	int		cybtime;

	rc = strsplit(fld, asizeof(fld), buf, '|');
	if (rc < 11)
	{
		if (Debug)
			error(0, "Bogus $TIME record (%d fields) '%s'\n",
					rc, buf);
		return;
	}

	if (fld[2] && fld[2][0] == '$')
	{
	    // cybtime = decode64(fld[10]);
	    // sr->priv->last_time = (cybtime/100)*3600 + (cybtime%100)*60 + 0;
	    cybtime = atoi(fld[2] + 1);
	    sr->priv->last_time = (cybtime/10000)*3600
				    + ((cybtime/100)%100)*60 + cybtime%60;
	    if (buf[0] == 'I')
		    time(&sr->priv->last_utime);
	}
}

static void
do_fullquote(STREAMER sr, char *buf)
{
	QUICKQUOTE	qq;
	LIVEQUOTE	lq;
	QUOTE		q;
	//STOCK		*sp;
	char		*fld[32];
	int		rc;
	int		time;

	memset(fld, 0, sizeof(fld));
	rc = strsplit(fld, asizeof(fld), buf, '|');
	if (rc < 16)
	{
		if (Debug)
			error(0, "Bogus 4 record (%d fields) '%s'\n", rc, buf);
		return;
	}

	memset(&q, 0, sizeof(q));
	memset(&qq, 0, sizeof(qq));
	memset(&lq, 0, sizeof(lq));

	streamer_sym2canon(q.sym, fld[1]);
	strcpy(qq.sym, q.sym);
	strcpy(lq.sym, q.sym);

	lq.last = qq.last = q.last = decode(fld[2]);
	if (q.last == 0)
		lq.last = qq.last = q.last = decode(fld[15]);
	q.last_size = decode64(fld[3]);
	qq.volume = q.volume = decode64(fld[4]) * 100;
	qq.ask_size = q.ask_size = decode64(fld[5]);
	qq.bid_size = q.bid_size = decode64(fld[6]);
	qq.ask = q.ask = decode(fld[7]);
	qq.bid = q.bid = decode(fld[8]);
	q.bid_tick = fld[9][0];

	time = decode64(fld[10]);
	qq.timetrade = q.time = (time/100)*3600 + (time%100)*60 + 0;

	qq.high = q.high = decode(fld[11]);
	if (fld[12] && fld[12][0] && fld[12][0] != ',')
	{
		qq.low = q.low = decode(fld[12]);
		debug(5, "LOW is %.2f\n", q.low);
	}

	// FIXME: If no low value is present, then figure out one

	// unk1 = fld[13];
	// open = fld[14];
	lq.close = qq.prev_close = q.close = decode(fld[15]);
	// unk2 = fld[16];

	display_quote(&q, 0);
	optchain_quote(&q);
	info_quickquote(&qq);
	display_livequote(&lq);
}

static void
do_quoteupdate(STREAMER sr, char *buf)
{
	QUICKQUOTE	qq;
	LIVEQUOTE	lq;
	QUOTE		q;
	STOCK		*sp;
	char		*fld[32];
	int		rc;
	int		tradetype = 0;
	double		lasthigh, lastlow;

	memset(fld, 0, sizeof(fld));
	rc = strsplit(fld, asizeof(fld), buf, '|');
	if (rc < 2)
	{
		if (Debug)
			error(0, "Bogus 2 record (%d fields) '%s'\n", rc, buf);
		return;
	}

	if (rc == 2)
	{
		// Just "2|SYMBOL" means a trade occurred with the exact
		// same values as the previous one.
		tradetype = 1;
	}

	memset(&q, 0, sizeof(q));
	memset(&qq, 0, sizeof(qq));
	memset(&lq, 0, sizeof(lq));

	streamer_sym2canon(q.sym, fld[1]);
	strcpy(qq.sym, q.sym);
	strcpy(lq.sym, q.sym);

	sp = find_stock(q.sym);
	if (!sp)
		return;

	copy_quote(&sp->cur, &q, &qq, &lq);

	lasthigh = q.high;
	lastlow = q.low;

	if (fld[2] && fld[2][0] && fld[2][0] != ',')
	{
		lq.last = qq.last = q.last = decode(fld[2]);
		qq.timetrade = q.time = tradetime(sr);
		tradetype = 1;
	}
	if (fld[3] && fld[3][0])
	{
		q.last_size = decode64(fld[3]);
		qq.timetrade = q.time = tradetime(sr);
		tradetype = 1;
	}
	if (fld[4] && fld[4][0])
	{
		qq.volume = q.volume = decode64(fld[4]) * 100;
		qq.timetrade = q.time = tradetime(sr);
		tradetype = 1;
	}
	if (fld[5] && fld[5][0])
		qq.ask_size = q.ask_size = decode64(fld[5]);
	if (fld[6] && fld[6][0])
		qq.bid_size = q.bid_size = decode64(fld[6]);
	if (fld[7] && fld[7][0] && fld[7][0] != ',')
		qq.ask = q.ask = decode(fld[7]);
	if (fld[8] && fld[8][0] && fld[8][0] != ',')
		qq.bid = q.bid = decode(fld[8]);
	if (fld[9] && fld[9][0])
		q.bid_tick = fld[9][0];
	if (fld[10] && fld[10][0])
	{
		int	time = decode64(fld[10]);
		qq.timetrade = q.time = (time/100)*3600 + (time%100)*60 + 0;
	}
	if (fld[11] && fld[11][0] && fld[11][0] != ',')
	{
		qq.high = q.high = decode(fld[11]);
		debug(1, "HI %s, rc=%d, 11=%s, hi=%.2f\n",
				q.sym, rc, fld[11], q.high);
	}
	if (fld[12] && fld[12][0] && fld[12][0] != ',')
	{
		qq.low = q.low = decode(fld[12]);
		debug(5, "update LOW is %.2f\n", q.low);
	}
	if (fld[15] && fld[15][0] && fld[15][0] != ',')
		lq.close = qq.prev_close = q.close = decode(fld[15]);

	display_quote(&q, tradetype);
	optchain_quote(&q);
	info_quickquote(&qq);
	display_livequote(&lq);

	if (tradetype && sr->priv->l2sym && strcmp(sr->priv->l2sym, q.sym) == 0)
	{
	    int	tstype = TS_SPREAD;

	    if (q.last > lasthigh) tstype = TS_ATHIGH;
	    else if (q.last > q.ask) tstype = TS_ABOVE;
	    else if (q.last >= q.ask) tstype = TS_ATASK;
	    else if (q.last < lastlow) tstype = TS_ATLOW;
	    else if (q.last < q.bid) tstype = TS_BELOW;
	    else if (q.last <= q.bid) tstype = TS_ATBID;
	    l2sr_trade(sr, q.last, q.last_size, q.time, -1, tstype);
	}
}

static void
do_optionquote(STREAMER sr, char *buf)
{
	QUICKQUOTE	qq;
	QUOTE		q;
	//STOCK		*sp;
	char		*fld[32];
	int		rc;
	int		time;

	rc = strsplit(fld, asizeof(fld), buf, '|');
	if (rc < 22)
	{
		if (Debug)
			error(0, "Bogus N record (%d fields) '%s'\n", rc, buf);
		return;
	}

	memset(&q, 0, sizeof(q));
	memset(&qq, 0, sizeof(qq));

	if (fld[1][0] != '.')
	{
		if (Debug)
			error(0, "Bogus N record (not an option sym) '%s'\n",
					rc, buf);
		return;
	}
	streamer_sym2canon(q.sym, fld[1]);
	memmove(q.sym, q.sym+1, SYMLEN);
	strcat(q.sym, ".");
	strcpy(qq.sym, q.sym);

	qq.last = q.last = decode(fld[2]);
	if (q.last == 0)
		qq.last = q.last = decode(fld[15]);
	q.last_size = decode64(fld[3]);
	// unk4 = fld[4];
	qq.volume = q.volume = decode64(fld[5]) * 100;
	qq.bid = q.bid = decode(fld[6]);
	qq.bid_size = q.bid_size = decode64(fld[7]);
	qq.ask = q.ask = decode(fld[8]);
	qq.ask_size = q.ask_size = decode64(fld[9]);
	// unk10 = fld[10];
	time = decode64(fld[11]);
	qq.timetrade = q.time = (time/100)*3600 + (time%100)*60 + 0;

	qq.high = q.high = decode(fld[12]);
	qq.low = q.low = decode(fld[13]);

	// unk14 = fld[14];
	qq.prev_close = q.close = decode(fld[15]);
	// unk16 = fld[16];
	// unk17 = fld[17];
	// unk18 = fld[18];
	// strike = fld[19];
	// date = fld[20];
	// root = fld[21];

	display_quote(&q, 0);
	optchain_quote(&q);
}

static void
do_headlines(STREAMER sr, char *buf)
{
	char		*fld[4 + 25*8 + 99];
	int		i;
	int		rc;
	int		numhead;
	int		numgot;
	char		**fldp;

	// Headline resp (all on 1 line):
	// 	Q|R|315|1|02091305c7PB  |
	// 	BARRON'S: 13D Filings: Investors Report To The SEC|
	// 	020913|2255|020913|2255|021013|H|
	// 	type R key nlines ...
	// 	artcode headline date time date time date H ...
	rc = strsplit(fld, asizeof(fld), buf, '|');
	if (rc > 1 && strcmp(fld[1], "E") == 0)
	{
		// Error, most likely no such symbol
		return;
	}
	numhead = (rc - 4) / 8;
	if (numhead == 0)
	{
		if (Debug)
			error(0, "Bogus Q record (rc = %d) '%s'\n",
					rc, buf);
		return;
	}
	if (atoi(fld[2]) != sr->priv->headseqnum)
	{
		if (Debug)
			error(0, "Bogus Q (wanted seq %d, got <%s>) '%s'\n",
					sr->priv->headseqnum, fld[2], buf);
		return;
	}

	numgot = atoi(fld[3]);
	if (numhead < numgot)
		numgot = numhead;

	if (numgot == sr->priv->numhead && sr->priv->headpagenum < 10)
	{
		// Possibly more headlines available, issue next page command
		streamer_printf(sr->fd[0], sr->priv->headcmd,
			++sr->priv->headseqnum, sr->priv->headpagenum++);
	}

	fldp = fld + 4;
	for (i = 0; i < numgot; ++i, fldp += 8)
	{
		add_headline("", fldp[1], fldp[0],
				datetime2unix(fldp[2], fldp[3]),
				"", sr, FALSE);
	}
}

static void
do_article(STREAMER sr, char *buf)
{
	char		*fld[32];
	int		rc;

	// Article resp:
	// 	P|316|02091305c7PB  |00|^]  13Ds are filed with
	rc = strsplit(fld, asizeof(fld), buf, '|');
	if (rc < 5)
	{
		if (Debug)
			error(0, "Bogus P record (%d fields) '%s'\n", rc, buf);
		return;
	}

	article_display_schwab(fld[4]);
}

static void
do_l2full(STREAMER sr, char *buf)
{
	char		*fld[3 + 300*7];
	int		i;
	int		rc;
	int		numgot;
	int		numwant;
	char		**fldp;

	//      7|21|DGII|BTRD||0||0|AQh|O|PIPR|4550|1|45AP|1|AQh|K|
        //      type nquot sym    mmid bid sz ask sz time code   ...
	rc = strsplit(fld, asizeof(fld), buf, '|');
	numgot = (rc - 3) / 7;
	if (numgot == 0)
	{
		if (Debug)
			error(0, "Bogus 7 record (rc = %d)\n", rc);
		return;
	}

	numwant = atoi(fld[1]);
	if (numwant > numgot)
	{
		if (Debug)
			error(0, "Bogus 7 record want=%d, got=%d\n",
					numwant, numgot);
		return;
	}

	fldp = fld + 3;
	for (i = 0; i < numgot; ++i, fldp += 7)
	{
		L2BA	ba;
		int	tim;

		strncpy(ba.mmid, fldp[0], MMIDLEN);
		ba.mmid[MMIDLEN] = 0;
		ba.bid = decode(fldp[1]);
		ba.bidsz = decode64(fldp[2]);
		ba.ask = decode(fldp[3]);
		ba.asksz = decode64(fldp[4]);
		tim = decode64(fldp[5]);
		ba.code = fldp[6][0];

		ba.time = (tim/10000) * 3600
			    + ((tim/100) % 100) * 60
			    + tim % 60;

		debug(3, "%s\t%.2f/%d\t%.2f/%d %02d:%02d:%02d %c\n",
				ba.mmid, ba.bid, ba.bidsz, ba.ask, ba.asksz,
				ba.time/3600, (ba.time/60)%60, ba.time%60,
				ba.code);

		l2sr_ba(sr, fld[2], &ba, 1);
	}
}

static void
do_l2update(STREAMER sr, char *buf)
{
	char	*fld[3 + 100*7];
	int	rc;
	int	numgot;
	char	**fldp;
	L2BA	ba;
	int	tim;

	//      5|INTC|MRLN||I||D|A8h
        //      type sym    mmid bid sz ask sz time code
	rc = strsplit(fld, asizeof(fld), buf, '|');
	if (rc < 3)
	{
		if (Debug)
			error(0, "Bogus 5 record (rc = %d)\n", rc);
		return;
	}

	fldp = fld + 2; numgot = rc - 2;

	//fprintf(stderr, "getlast '%s' '%s'\n", fld[1], fld[2]);

	rc = l2sr_getlast(sr, fld[1], fld[2], &ba);
	if (rc < 0)
		return;

	strncpy(ba.mmid, fldp[0], MMIDLEN);
	ba.mmid[MMIDLEN] = 0;
	if (numgot > 1 && fldp[1][0])
		ba.bid = decode(fldp[1]);
	if (numgot > 2 && fldp[2][0])
		ba.bidsz = decode64(fldp[2]);
	if (numgot > 3 && fldp[3][0])
		ba.ask = decode(fldp[3]);
	if (numgot > 4 && fldp[4][0])
		ba.asksz = decode64(fldp[4]);
	if (numgot > 5 && fldp[5][0])
	{
		tim = decode64(fldp[5]);
		ba.time = (tim/10000) * 3600
			    + ((tim/100) % 100) * 60
			    + tim % 60;
	}
	if (numgot > 6 && fldp[6][0])
		ba.code = fldp[6][0];

	//fprintf(stderr, "bid %.2f ask %.2f\n", ba.bid, ba.ask);

	l2sr_ba(sr, fld[1], &ba, 1);
}

static int
filter_headline(STREAMER sr, char *hline)
{
	int	rc;
	int	myrc = 0;

	switch (sr->priv->reflags)
	{
	default:
		return 0;
	case RE_INC:
		rc = regexec(&sr->priv->include_re, hline, 0, NULL, 0);
		return rc;
	case RE_EXC:
		return !regexec(&sr->priv->exclude_re, hline, 0, NULL, 0);
	case RE_INC|RE_EXC:
		rc = regexec(&sr->priv->exclude_re, hline, 0, NULL, 0);
		if (rc == 0)
			myrc = 1;
		rc = regexec(&sr->priv->include_re, hline, 0, NULL, 0);
		if (rc == 0)
			myrc = 0;
		return myrc;
	}
}

static int
duplicate_headline(char	*hline)
{
	static char	oldhline[256];

	// Simple attempt to get rid of some, but not all, duplicates
	if (strcmp(hline, oldhline) == 0)
		return TRUE;

	strncpy(oldhline, hline, sizeof(oldhline));
	oldhline[sizeof(oldhline)-1] = 0;
	return FALSE;
}

static void
do_newnews(STREAMER sr, char *buf)
{
	char		*fld[32];
	int		rc;
	time_t		tim;

	// 	artcode headline date time date time date H ...
	// 	R|INTC|020917x427Pmd |IBM Blade Server Alliance
	// 	|020917|0933|020917|0648|020917|N|
	rc = strsplit(fld, asizeof(fld), buf, '|');
	if (rc < 10)
	{
		if (Debug)
			error(0, "Bogus R record (%d fields) '%s'\n", rc, buf);
		return;
	}

	// get rid of some duplicates
	if (duplicate_headline(fld[3]))
		return;

	tim = datetime2unix(fld[4], fld[5]);

	if (strcmp(fld[1], "!NEWSHOT") == 0)
	{
		if (filter_headline(sr, fld[3]))
			return;
		if (!ToolMode)
		    display_scrollnews("hot", fld[2], fld[3], tim, "", 0);
		save_headline("hot", fld[2], fld[3], tim, "", sr, 0);
	}
	else
	{
		if (!ToolMode)
		    display_scrollnews(fld[1], fld[2], fld[3], tim, "", 0);
		save_headline(fld[1], fld[2], fld[3], tim, "", sr, 0);
		if (!ToolMode)
		    alert_news_symbol(fld[1]);
	}
}

static int
streamer_process_main(STREAMER sr)
{
	int	len;
	char	buf[16384];

	len = linebuf_gets(&sr->priv->lb[0], buf, sizeof(buf));
	if (len <= 0)
	{
		if (sr->readfile)
		{
			if (sr->fd[0] >= 0)
				fclose(sr->readfile);
			sr->fd[0] = -1;
			return (0);
		}
		return (-1);
	}

	sr->cnt_rx += len + 1;

	if (sr->writefile)
	{
		fputs(buf, sr->writefile);
		putc('\n', sr->writefile);
	}

	switch (buf[0])
	{
	case 's':
		// Sync message?
		// s|*|VVVVVVVVVVVUVVVVVVVVV|TUAUBUXUIUCUNURUUU
		if (0)
		{
			// Request new sync line
			streamer_printf(sr->fd[0], "s|\n");
		}
		break;
	case '2':
		// Quote update
		// 2|INTC|,N||||||||QO
		do_quoteupdate(sr, buf);
		break;
	case '4':
		// Stocks
		// 4|QLGC|41BD`|0|0|0|0|41BIL|41BFD|D|QS|||27||41BD`
		do_fullquote(sr, buf);
		break;
	case '5':
		do_l2update(sr, buf);
		break;
	case '7':
		do_l2full(sr, buf);
		break;
	case 'I':
		// K|$TIME|$113200|0|0||||||A\|$101400|$60100|0|$60100|$60000
		// I|$TIME|$113300||||||||A]
		// I|/NQZG2||||0|0|2FFN|2FEa
		if (strncmp(buf+2, "$TIME|", 6) == 0)
			do_time(sr, buf);
		else
			do_quoteupdate(sr, buf);
		break;
	case 'G':
		// Long name Request:
		//	G|1|LTRX
		// Long name Response:
		// 	G|LTRX|LANTRONIX INC|0|0|0|0|0|
		break;
	case 'N':
		// Options
		do_optionquote(sr, buf);
		break;
	case 'K':
		// Fullquote for Indexes, Futures, $TIME
		if (strncmp(buf+2, "$TIME|", 6) == 0)
			do_time(sr, buf);
		else
			do_fullquote(sr, buf);
		break;
	case 'Q':
		// Request news mesg:
		// 	Q|315|LTRX|020913|1956|||25|1
		// 	  key sym  sdate  stime edate etime nlines artoffset
		//	key seems to be aribtrary
		//	Most recent headline: Q|195|MSFT|||000000|0000|1|1
		//	artoffset == pagenumber?
		// Symbol error resp:
		// 	Q|E|315|0
		// Headline resp (all on 1 line):
		// 	Q|R|315|1|02091305c7PB  |
		// 	BARRON'S: 13D Filings: Investors Report To The SEC|
		// 	020913|2255|020913|2255|021013|H|
		// 	type R key nlines ...
		// 	artcode headline date time date time date H/N
		// Article request mesg:
		// 	P|316|02091305c7PB
		// Article resp:
		// 	P|316|02091305c7PB  |00|^]  13Ds are filed with
		// 	January 13 and 19. ^H ^]  Source:
		do_headlines(sr, buf);
		break;
	case 'R':
		// Streaming news add...
		// 	R|1|INTC
		// 	R|1|!NEWSHOT|
		// Streaming news response...
		// 	R|INTC|020917x427Pmd |IBM Blade Server Alliance
		// 	|020917|0933|020917|0648|020917|N|
		do_newnews(sr, buf);
		break;
	case 'P':
		do_article(sr, buf);
		break;
	case 'V':
		// keepalive
		if (sr->priv->haveaux && sr->fd[1] < 0)
		{
			// Aux server died unexpectedly, try to reopen streamer
			return (-1);
		}
		break;
	case 'Y':
		// Y|2|9	(have seen 0, 9, 10, 24, 50, ...)
		break;
	}

	return (0);
}

static void
do_top10(STREAMER sr, char *buf)
{
	char	*fld[10*4 + 6];
	int	nfld;
	int	n10;
	int	i;
	int	type;
	int	t10type, t10mkt;

	// 	00SUNW|24X|1mU]|24h|1INTC|2Ji|dWb|2J;|2CSCO|2E0|]NT|33@a| ...
	type = buf[0];
	if (type < '0' || type > '9')
	{
		debug(1, "Unknown top10 type '%c'\n", type);
		return;
	}
	
	// Convert to decimal
	type -= '0';

	switch (type)
	{
	case 0:	t10type = TOP_VOL;     t10mkt = 'Q'; break;
	case 1:	t10type = TOP_NETGAIN; t10mkt = 'Q'; break;
	case 2:	t10type = TOP_NETLOSS; t10mkt = 'Q'; break;
	case 3:	t10type = TOP_PCTGAIN; t10mkt = 'Q'; break;
	case 4:	t10type = TOP_PCTLOSS; t10mkt = 'Q'; break;
	case 5:	t10type = TOP_VOL;     t10mkt = 'N'; break;
	case 6:	t10type = TOP_NETGAIN; t10mkt = 'N'; break;
	case 7:	t10type = TOP_NETLOSS; t10mkt = 'N'; break;
	case 8:	t10type = TOP_PCTGAIN; t10mkt = 'N'; break;
	case 9:	t10type = TOP_PCTLOSS; t10mkt = 'N'; break;
	default:
		break;
	}

	nfld = strsplit(fld, asizeof(fld), buf+1, '|');
	nfld &= ~3;
	if (nfld == 0)
	{
		// Clear list
		return;
	}

	if (nfld < 4)
	{
		if (Debug)
			error(0, "Bogus top10 record (%d fields) '%s'\n",
					nfld, buf);
		return;
	}

	n10 = nfld / 4;
	for (i = 0; i < nfld; i += 4)
	{
		int	rank;
		TOP10	*t;
		
		rank = fld[i+0][0] - '0';
		if (rank < 0 || rank > 9)
			return;

		t = &sr->priv->top10[type][rank];

		t->rank = rank + 1;
		t->type = t10type;
		t->market = t10mkt;

		if (fld[i+0][1])
		{
			strncpy(t->sym, fld[i+0] + 1, SYMLEN);
			t->sym[SYMLEN] = 0;
		}
		if (fld[i+1][0])
			t->prev = decode(fld[i+1]);;
		if (fld[i+2][0])
			t->volume = 100 * decode64(fld[i+2]);;
		if (fld[i+3][0])
			t->curr = decode(fld[i+3]);;
		t->change = t->curr - t->prev;
		display_top10(t);
	}
}

static void
do_mover(STREAMER sr, char *buf)
{
	char		*fld[10*4 + 6];
	int		nfld;
	MKTMOVER	mm;
	char		type;
	time_t		now;
	struct tm	*tmp;

	type = buf[0];
	fld[2] = "";
	nfld = strsplit(fld, asizeof(fld), buf+1, '|');
	if (type >= 'I' && type <= 'P' && nfld < 2)
	{
		debug(1, "Bogus mover record '%c' (%d fields)\n", type, nfld);
		return;
	}
	else if (nfld < 3)
	{
		debug(1, "Bogus mover record '%c' (%d fields)\n", type, nfld);
		return;
	}

	strncpy(mm.sym, fld[0], SYMLEN);
	mm.sym[SYMLEN] = 0;

	mm.last = decode(fld[1]);
	mm.change = 0;
	mm.count = decode64(fld[2]);

	time(&now); tmp = localtime(&now);
	mm.time = tmp->tm_hour*3600 + tmp->tm_min*60 + tmp->tm_sec;

	switch (type)
	{
	case 'A': mm.hilo = LO_DAILY; break;// NAS daily low (red on black)
	case 'B': mm.hilo = HI_DAILY; break;// NAS daily high (green on black)
	case 'C': mm.hilo = LO_52; break;   // NAS 52 low (black on red)
	case 'D': mm.hilo = HI_52; break;   // NAS 52 high (black on green)
	case 'E': mm.hilo = LO_DAILY; break;// NY  daily low (red on black)
	case 'F': mm.hilo = HI_DAILY; break;// NY  daily high (green on black)
	case 'G': mm.hilo = LO_52; break;   // NY  52 low (black on red)
	case 'H': mm.hilo = HI_52; break;   // NY  52 high (black on green)
	case 'I': mm.hilo = NEAR_LO; break;     // NAS (purple on black)
	case 'J': mm.hilo = NEAR_HI; break;     // NAS (yellow on black)
	case 'K': mm.hilo = NEAR_LO_52; break; // NAS (black on purple)
	case 'L': mm.hilo = NEAR_HI_52; break; // NAS (black on yellow)
	case 'M': mm.hilo = NEAR_LO; break;    // NY  (purple on black)
	case 'N': mm.hilo = NEAR_HI; break;    // NY  (yellow on black)
	case 'O': mm.hilo = NEAR_LO_52; break; // NY  (black on purple)
	case 'P': mm.hilo = NEAR_HI_52; break; // NY  (black on yellow)
	default:
		  debug(1, "Unknown mover code '%c' <%s> <%s> <%s>\n",
				  type, fld[0], fld[1], fld[2]);
		  return;
	}

	display_mktmover(&mm);
}

static int
streamer_process_aux(STREAMER sr)
{
	int	len;
	char	buf[16384];

	len = linebuf_gets(&sr->priv->lb[1], buf, sizeof(buf));
	if (len <= 0)
	{
		sprintf(sr->id, "schwab,com");
		debug(1, "Aux streamer closed: len=%d\n", len);
		linebuf_close(&sr->priv->lb[1]);
		close(sr->fd[1]);
		sr->fd[1] = -1;
		return -1;
	}

	switch (buf[0])
	{
	case '$':
		// Keep alive
		streamer_printf2(sr->fd[1], "$\n");
		if (0)
		{
			// Test aux server restart
			sprintf(sr->id, "schwab,com");
			debug(1, "Aux streamer closed: len=%d\n", len);
			linebuf_close(&sr->priv->lb[1]);
			close(sr->fd[1]);
			sr->fd[1] = -1;
		}
		break;
	case '0':
		// Top 10 stocks
		do_top10(sr, buf+2);
		break;
	case '1':
		// New high's and low's
		do_mover(sr, buf+2);
		break;
	}

	return 0;
}

static int
streamer_process(STREAMER sr, int fdindex)
{
	if (fdindex == 0)
		return streamer_process_main(sr);
	else if (fdindex == 1)
		return streamer_process_aux(sr);
	else
		error(1, "Process on unknown fdindex %d.\n", fdindex);

	return 0;
}

static void
streamer_timetick(STREAMER sr, time_t now)
{
	if (now > sr->priv->last_keepalive + 300)
	{
		streamer_send_keepalive(sr);
		sr->priv->last_keepalive = now;
	}

	if (now > sr->priv->last_flush + 5)
	{
		sr->priv->last_flush = now;
		if (sr->writefile)
			fflush(sr->writefile);
	}
}

static void
nullsr(STREAMER sr)
{
}

STREAMER
schwab_new(void)
{
	STREAMER	sr;

	sr = (STREAMER) malloc(sizeof(*sr));
	if (!sr)
		return NULL;
	memset(sr, 0, sizeof(*sr));

	sr->open = streamer_open;
	sr->select = streamer_select;
	sr->close = streamer_close;
	sr->record = streamer_record;
	sr->timetick = streamer_timetick;

	sr->send_quickquote = streamer_send_quickquote;
	sr->send_livequote = streamer_send_livequote;
	sr->send_livequote_end = nullsr;
	sr->send_symbols = streamer_send_symbols;
	sr->send_symbols_end = streamer_send_symbols_end;

	// sr->send_disconnect = streamer_send_disconnect;
	sr->send_top10 = streamer_send_top10;
	sr->send_movers = streamer_send_movers;
	// sr->send_info = streamer_send_info;
	// sr->send_optchain = streamer_send_optchain;
	sr->send_chart = streamer_send_chart;

	sr->send_headlines = streamer_send_headlines;
	sr->send_article = streamer_send_article;

	sr->send_l2 = streamer_send_l2;

	sr->process = streamer_process;

	sr->priv = (STREAMERDATA *) malloc(sizeof(*sr->priv));
	if (!sr->priv)
	{
		free(sr);
		return NULL;
	}
	memset(sr->priv, 0, sizeof(*sr->priv));

	time(&sr->time_start);
	sr->priv->charttm = NULL;

	streamer_init(sr);

	return (sr);
}
