/*
  The osipua library is a library based on oSIP that implements CallLeg and User Agent
  level.
  Copyright (C) 2001  Simon MORLAT simon.morlat@free.fr
  											Aymeric MOIZARD jack@atosc.org
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include "osipua.h"
#include "udp.h"
#include "osipmanager.h"
#include "resolver.h"

#ifdef INET6  
#include <sys/types.h>  
#include <sys/socket.h>  
#include <netdb.h>  
#endif

static void set_all_callbacks (osip_t * cf);
static int osip_global_initialized=0;
#define unblock_udp_thread(manager) write(manager->udp_control_fd,(void*)manager,1)


void kill_all_transaction(list_t *transactions)
{
   transaction_t *transaction;
   while (!list_eol(transactions, 0))
     {
       transaction = list_get(transactions,0);
       printf("closing transaction : %x", transaction->your_instance);
 
       /* loop on all SIP events not already consumed by the transaction */
       /* TODO */
 
       transaction_free(transaction);
       sfree(transaction);
     }
}

OsipManager *
osip_manager_new ()
{
	OsipManager *manager;
	int control_fds[2];
	int err;

	manager = smalloc (sizeof (OsipManager));
	memset (manager, 0, sizeof (OsipManager));
	if (!osip_global_initialized)
	{
		if (-1 == osip_global_init ())
		{
			osip_trace (OSIP_ERROR, ("error: Failed to init mutex.\n"));
			return NULL;	/* mutex is not initialised properly */
		}
		osip_global_initialized=1;
	}
	osip_init (&manager->config);

#if defined(THREAD_PTH)
	pth_init ();
#endif
	set_all_callbacks (manager->config);
	/* create the control file descriptor (used to unblock the udp daemon sometimes */
	err = pipe (control_fds);
	if (err != 0)
	{
		perror ("Error creating pipe");
		exit (1);
	}
	FD_SET (control_fds[0], &manager->udpfdset);
	manager->udp_unblock_fd = control_fds[0];
	manager->udp_control_fd = control_fds[1];	/* the file descriptor where to write something to control the daemon */
	manager->max_udpfd = control_fds[0];
	manager->recv_tout.tv_sec = 0;
	manager->recv_tout.tv_usec = 500000;	/*500 ms lets good precision for retransmission that should occur every 500 ms */
	/* allocate the memory to receive packets */
	manager->udp_buf =
		(char *) smalloc (SIP_MESSAGE_MAX_LENGTH * sizeof (char) + 1);
	manager->resolv_fifo = smalloc (sizeof (fifo_t));
	fifo_init (manager->resolv_fifo);
	manager->mutex=smutex_init();
	fifo_init(&manager->garbage_trn);
	return manager;
}

void
osip_manager_destroy (OsipManager * manager)
{
	int i;

	osip_manager_stop_udp_daemon (manager);
	sfree (manager->udp_buf);
	/* close all file descripors */
	for (i = 0; i < OSIP_MAX_UDP_PORTS; i++)
	{
		if (manager->udpfds[i] != 0)
			close (manager->udpfds[i]);
	}
	kill_all_transaction(manager->config->ict_transactions);
	kill_all_transaction(manager->config->nict_transactions);
	kill_all_transaction(manager->config->ist_transactions);
	kill_all_transaction(manager->config->nist_transactions);
	/* free the osip_t structure */
	sfree (manager->config);
	fifo_free (manager->resolv_fifo);
	sfree (manager->resolv_fifo);
	smutex_destroy(manager->mutex);
	fifo_free(&manager->garbage_trn);
	sfree (manager);
}


int
osip_manager_add_udpport (OsipManager * manager, int port)
{
	int i, newfd, err;
#ifdef INET6  
	struct addrinfo hints, *lres0, *lres;  
	char num[8];  
#else
	struct sockaddr_in laddr;
#endif
	int option = 1;

	for (i = 0; i < OSIP_MAX_UDP_PORTS; i++)
	{
		if (port == manager->udpports[i])
		{
			osip_trace (OSIP_INFO1,
				    ("port already listened\n"));
			return -EALREADY;
		}
	}
	if (port != manager->send_port)
	{			/* create a new file descriptor */
#ifdef INET6
                /* XXX I need to rewrite!! */  
                memset(&hints, 0, sizeof(hints));  
                hints.ai_family = PF_UNSPEC;  
                hints.ai_socktype = SOCK_DGRAM;  
                hints.ai_protocol = IPPROTO_UDP;  
                hints.ai_flags = AI_PASSIVE;  
  
                snprintf(num, sizeof(num), "%d", port);  
                err = getaddrinfo(NULL, num, &hints, &lres0);  
                if (err) {  
                  osip_trace (OSIP_ERROR,("error: %s\n",gai_strerror(err)));  
                  return -1;  
                }  
  
                newfd = -1;  
                for (lres = lres0; lres; lres = lres->ai_next) {  
                        newfd = socket (lres->ai_family, lres->ai_socktype, lres->ai_protocol);  
                        if (newfd < 0)  
                                continue;  
  
                        if (bind(newfd, lres->ai_addr, lres->ai_addrlen) < 0) {  
                                osip_trace (OSIP_ERROR,  
                                    ("Failed to bind socket !\n"));  
                                close (newfd);  
                                newfd = -1;  
                                continue;  
                        } else {  
                                err = setsockopt (newfd, SOL_SOCKET, SO_REUSEADDR,  
                                                  (void *) &option, sizeof (option));  
                                if (err != 0)  
                                  osip_trace (OSIP_ERROR,  
                                              ("warning: port cannot be reused.\n"));  
                                break;  
                        }  
                }  
                freeaddrinfo(lres0);  
                if (newfd < 0) {  
                          osip_trace (OSIP_ERROR,  
                                      ("warning: bad socket.\n"));  
                          return -errno;  
                }  
#else
		newfd = socket (PF_INET, SOCK_DGRAM, IPPROTO_UDP);

		laddr.sin_addr.s_addr = htons (INADDR_ANY);
		laddr.sin_port = htons ((short) port);
		laddr.sin_family = AF_INET;
		if (bind (newfd, (struct sockaddr *) &laddr, sizeof (laddr)) <
		    0)
		{
			osip_trace (OSIP_ERROR,
				    ("Failed to bind socket !\n"));
			close (newfd);
			return -errno;
		}
		err = setsockopt (newfd, SOL_SOCKET, SO_REUSEADDR,
				  (void *) &option, sizeof (option));
		if (err != 0)
			osip_trace (OSIP_ERROR,
				    ("warning: port cannot be reused.\n"));
#endif
	}
	else
	{
		/* if the socket used to send data has the same port as the new port to scan, then use it */
		newfd = manager->send_sock;
	}
	for (i = 0; i < OSIP_MAX_UDP_PORTS; i++)
	{
		if (manager->udpports[i] == 0)
		{
			/* this is a free place to store the port number and file descriptor */
			manager->udpports[i] = port;
			manager->udpfds[i] = newfd;
			FD_SET (newfd, &manager->udpfdset);
			if (newfd > manager->max_udpfd)
				manager->max_udpfd = newfd;
			/* unblock the udp daemon thread */
			unblock_udp_thread (manager);
			return 0;
		}
	}
	/* no more port availlable for listening */
	close (newfd);
	osip_trace (OSIP_ERROR,
		    ("info: The table of scanned file descriptor is full.\n"));
	return -1;
}

int
osip_manager_remove_udpport (OsipManager * manager, int port)
{
	int i;

	for (i = 0; i < OSIP_MAX_UDP_PORTS; i++)
	{
		if (port == manager->udpports[i])
		{
			manager->udpports[i] = 0;
			FD_CLR (manager->udpfds[i], &manager->udpfdset);
			/* unblock the udp dameon */
			unblock_udp_thread (manager);
			if (port != manager->send_port)
				close (manager->udpfds[i]);
			manager->udpfds[i] = 0;
			return 0;
		}
	}
	osip_trace (OSIP_ERROR, ("error: Could not close Udp Port.\n"));
	return -ENOENT;
}


void
osip_manager_stop_udp_daemon (OsipManager * manager)
{
	if (manager->udp_run_cond)
	{
		manager->udp_run_cond = 0;
		unblock_udp_thread (manager);
		sthread_join (&manager->udp_thread);
	}
}

void
osip_manager_start_udp_daemon (OsipManager * manager)
{
	/* start a daemon */
	if (manager->udp_run_cond == 1)
		return;
	manager->udp_run_cond = 1;
	/* create daemon thread */
	sthread_create (0, &manager->udp_thread, sipd_thread,
			(void *) manager);
	return;			/* ok */
}

void
osip_manager_start_resolver (OsipManager * manager)
{
	/* start the resolver thread */
	if (manager->resolv_run_cond == 1)
		return;
	manager->resolv_run_cond = 1;
	/* create daemon thread */
	sthread_create (0, &manager->resolv_thread, resolver_thread,
			(void *) manager);
	return;			/* ok */
}

void
osip_manager_stop_resolver (OsipManager * manager)
{
	if (manager->resolv_run_cond)
	{
		manager->resolv_run_cond = 0;
		sthread_join (&manager->resolv_thread);
	}
}

static void undefined_callback(transaction_t *trn, sip_t *sipmsg)
{
	
}

static void undefined_callback2(transaction_t *trn, int error)
{
	
}

static void
set_all_callbacks (osip_t * cf)
{
	
	/* callbacks for ict */
	osip_setcb_ict_kill_transaction(cf,ict_kill_transaction);
	osip_setcb_ict_invite_sent(cf,undefined_callback);
	osip_setcb_ict_invite_sent2(cf,undefined_callback);
	osip_setcb_ict_ack_sent(cf,undefined_callback);
	osip_setcb_ict_ack_sent2(cf,undefined_callback);
	osip_setcb_ict_1xx_received(cf,ict_1xx_received);
	osip_setcb_ict_2xx_received(cf,ict_2xx_received);
	osip_setcb_ict_2xx_received2(cf,undefined_callback);
	osip_setcb_ict_3xx_received(cf,ict_3xx_received);
	osip_setcb_ict_4xx_received(cf,ict_4xx_received);
	osip_setcb_ict_5xx_received(cf,ict_5xx_received);
	osip_setcb_ict_6xx_received(cf,ict_6xx_received);
	osip_setcb_ict_3456xx_received2(cf,undefined_callback);
	osip_setcb_ict_transport_error(cf,undefined_callback2);

	/* callbacks for nict */
	osip_setcb_nict_kill_transaction(cf,nict_kill_transaction);
	osip_setcb_nict_register_sent(cf,undefined_callback);
	osip_setcb_nict_bye_sent(cf,undefined_callback);
	osip_setcb_nict_options_sent(cf,undefined_callback);
	osip_setcb_nict_info_sent(cf,undefined_callback);
	osip_setcb_nict_cancel_sent(cf,undefined_callback);
	osip_setcb_nict_notify_sent(cf,undefined_callback);
	osip_setcb_nict_subscribe_sent(cf,undefined_callback);
	osip_setcb_nict_unknown_sent(cf,undefined_callback);
	osip_setcb_nict_request_sent2(cf,undefined_callback);
	osip_setcb_nict_1xx_received(cf,nict_1xx_received);
	osip_setcb_nict_2xx_received(cf,nict_2xx_received);
	osip_setcb_nict_2xx_received2(cf,undefined_callback);
	osip_setcb_nict_3xx_received(cf,nict_3xx_received);
	osip_setcb_nict_4xx_received(cf,nict_4xx_received);
	osip_setcb_nict_5xx_received(cf,nict_5xx_received);
	osip_setcb_nict_6xx_received(cf,nict_6xx_received);
	osip_setcb_nict_3456xx_received2(cf,undefined_callback);
	osip_setcb_nict_transport_error(cf,undefined_callback2);

	/* callbacks for ist */
	osip_setcb_ist_kill_transaction(cf,ist_kill_transaction);
	osip_setcb_ist_invite_received(cf,ist_invite_received);
	osip_setcb_ist_invite_received2(cf,undefined_callback);
	osip_setcb_ist_ack_received(cf,ist_ack_received);
	osip_setcb_ist_ack_received2(cf,undefined_callback);
	osip_setcb_ist_1xx_sent(cf,undefined_callback);
	//osip_setcb_ist_1xx_sent2(cf,undefined_callback);
	osip_setcb_ist_2xx_sent(cf,undefined_callback);
	osip_setcb_ist_2xx_sent2(cf,undefined_callback);
	osip_setcb_ist_3xx_sent(cf,undefined_callback);
	osip_setcb_ist_4xx_sent(cf,undefined_callback);
	osip_setcb_ist_5xx_sent(cf,undefined_callback);
	osip_setcb_ist_6xx_sent(cf,undefined_callback);
	osip_setcb_ist_3456xx_sent2(cf,undefined_callback);
	osip_setcb_ist_transport_error(cf,undefined_callback2);

	/* callbacks for nist */
	osip_setcb_nist_kill_transaction(cf,nist_kill_transaction);
	osip_setcb_nist_register_received(cf,nist_register_received);
	osip_setcb_nist_bye_received(cf,nist_bye_received);
	osip_setcb_nist_options_received(cf,nist_options_received);
	osip_setcb_nist_info_received(cf,undefined_callback);
	osip_setcb_nist_cancel_received(cf,nist_cancel_received);
	osip_setcb_nist_notify_received(cf,undefined_callback);
	osip_setcb_nist_subscribe_received(cf,undefined_callback);
	osip_setcb_nist_unknown_received(cf,undefined_callback);
	/* ... TO BE ADDED: All known and used method extansions */
	osip_setcb_nist_request_received2  (cf,undefined_callback);
	osip_setcb_nist_1xx_sent(cf,undefined_callback);
	osip_setcb_nist_2xx_sent(cf,undefined_callback);
	osip_setcb_nist_2xx_sent2(cf,undefined_callback);
	osip_setcb_nist_3xx_sent(cf,undefined_callback);
	osip_setcb_nist_4xx_sent(cf,undefined_callback);
	osip_setcb_nist_5xx_sent(cf,undefined_callback);
	osip_setcb_nist_6xx_sent(cf,undefined_callback);
	osip_setcb_nist_3456xx_sent2(cf,undefined_callback);
	osip_setcb_nist_transport_error(cf,undefined_callback2);

	osip_setcb_send_message (cf, udp_send);
}

int
osip_manager_set_send_port (OsipManager * manager, int port)
{
	int newfd, err;
#ifdef INET6  
        struct addrinfo hints, *lres0, *lres;  
        char num[8];  
        int success = -1;  
#else
	struct sockaddr_in laddr;
#endif
	int option = 1;

	if (manager->send_port == port)
		return 0;
	else if (manager->send_sock != 0)
		close (manager->send_sock);

	/* create a new file descriptor */
#ifdef INET6
        memset(&hints, 0, sizeof(hints));  
        hints.ai_family = PF_UNSPEC;  
        hints.ai_socktype = SOCK_DGRAM;  
        hints.ai_protocol = IPPROTO_UDP;  
	hints.ai_flags = AI_PASSIVE;  
   
        snprintf(num, sizeof(num), "%d", port);  
        err = getaddrinfo(NULL, num, &hints, &lres0);  
        if (err)  
          osip_trace (OSIP_ERROR,("error: %s\n",gai_strerror(err)));  
          
        newfd = -1;  
        for (lres = lres0; lres; lres = lres->ai_next) {  
                newfd = socket (lres->ai_family, lres->ai_socktype, lres->ai_protocol);  
                if (newfd < 0)  
                  continue;  
                err = setsockopt (newfd, SOL_SOCKET, SO_REUSEADDR,  
                                 (void *) &option, sizeof (option));  
                if (err != 0)  
                  osip_trace (OSIP_ERROR,  
                              ("warning: port cannot be reused.\n"));  
                if (bind(newfd, lres->ai_addr, lres->ai_addrlen) < 0) {  
                  /* XXX bind fail message */  
                  osip_trace (OSIP_ERROR,  
                              ("Could not to bind socket for sending messages: %s\n",strerror(errno)));  
                  close (newfd);  
                  newfd = -1;  
                 continue;  
                }  
                
                manager->send_sock = newfd;  
                manager->send_port = port;  
                freeaddrinfo(lres0);  
                return 0;  
        }  
        freeaddrinfo(lres0);  
        osip_trace (OSIP_ERROR,  
                    ("warning: bad socket.\n"));  
        return -errno;
#else
	newfd = socket (PF_INET, SOCK_DGRAM, IPPROTO_UDP);

	laddr.sin_addr.s_addr = htons (INADDR_ANY);
	laddr.sin_port = htons ((short) port);
	laddr.sin_family = AF_INET;
	if (bind (newfd, (struct sockaddr *) &laddr, sizeof (laddr)) < 0)
	{
		osip_trace (OSIP_WARNING,
			    ("Could not to bind socket for sending messages.\n"));
		close (newfd);
		return -errno;
	}
	err = setsockopt (newfd, SOL_SOCKET, SO_REUSEADDR, (void *) &option,
			  sizeof (option));
	if (err != 0)
		osip_trace (OSIP_ERROR,
			    ("port cannot be reused.\n"));
	manager->send_sock = newfd;
	manager->send_port = port;
	return 0;
#endif
}
