///	\file	dirreader.cpp
///	\brief	dirreader.cpp
#include "dirreader.h"

using namespace std;
using namespace MLS;

///	\brief	생성자
DirReader::DirReader()
{
	m_handle = NULL;
	m_lpath = new char[FILENAME_MAX];
	UpdateConfig();
}

void DirReader::UpdateConfig()
{
	date_type = config.GetValue("date_type");	
}

void DirReader::SaveConfig()
{
	
}

///	\brief	현재 directory가 '/'폴더인가를 조사
///	\return	true	:	'/' directory임\n
///			false	:	'/' directory가 아님.
bool DirReader::isRoot()
{
	return (m_path == "/");		
}

///	\brief	초기화 함수
///	\param	path	읽어들일 path
///	\return	초기화 성공 여부
bool DirReader::Init(const string &path)
{
	// access 에 R_OK 도 들어가야 제대로 Mls 가 작동 된다.
	if (access(path.c_str(), R_OK | X_OK) == -1)
	{
		return false;
	}

	if (chdir(path.c_str())==-1)
	{
		return false;
	}
	if ((m_handle = opendir("."))==NULL) return false;
	
	m_path = path;
			
	return true;
}

///	\brief	다음 파일이나 directory로 이동한다.
///	\return	true	:	성공적으로 이동했음.\n
///			false	:	이동하는것을 실패했음.
bool DirReader::Next()
{
	if (!m_handle) return false;
	m_pEntry = readdir(m_handle);	
	
	if (!m_pEntry) return false;
	
	return true;
}

///	\brief	현재 파일정보를 얻음
///	\param	file	파일정보 반환값
///	\return	현재 파일정보얻기 성공여부
bool DirReader::GetInfo(File &file)
{
	bool bLink;		// link 파일여부
	int  n;
	
	// . 파일정보를 얻는다. 
	if (lstat(m_pEntry->d_name, &m_statbuf)==-1) return false;
	
	// . 파일이 link 파일이면
	if (S_ISLNK(m_statbuf.st_mode))
	{	
		if ((n = readlink(m_pEntry->d_name, m_lpath, FILENAME_MAX)) == -1) return false;
		m_lpath[n] = '\0';
		
		// .. 링크 대상인 진짜 파일 정보를 얻는다.
		if (lstat(m_lpath, &m_statbuf) == -1) return false;

		bLink = true;
	}
	else bLink = false;

	
	// . 파일 이름 알아냄
	file.name = m_pEntry->d_name;
	
	// . directory 여부를 알아냄
	if (S_ISDIR(m_statbuf.st_mode)) file.bDir = true;
	else file.bDir = false;
	
	// . file attribute(drwxrwxrwx)를 설정함	===========
	// file S_ISUID, S_ISGID, S_ISVTX 추가
	// zip과 맞추기 위해서 '.' 에서 '-' 로 바꿈. 
	if (bLink)
		file.attr[0]= 'l';
	else 
		file.attr[0]= file.bDir ? 'd' : '-';
		
	file.attr[1]= S_IRUSR & m_statbuf.st_mode ? 'r' : '-';
	file.attr[2]= S_IWUSR & m_statbuf.st_mode ? 'w' : '-';
	file.attr[3]= S_IXUSR & m_statbuf.st_mode ? 'x' : '-';
	if (S_ISUID & m_statbuf.st_mode) file.attr[3] = 'S';
	if (S_ISUID & m_statbuf.st_mode && 
		S_IXUSR & m_statbuf.st_mode) file.attr[3] = 's';
	file.attr[4]= S_IRGRP & m_statbuf.st_mode ? 'r' : '-';
	file.attr[5]= S_IWGRP & m_statbuf.st_mode ? 'w' : '-';
	file.attr[6]= S_IXGRP & m_statbuf.st_mode ? 'x' : '-';
	if (S_ISGID & m_statbuf.st_mode) file.attr[6] = 'S';
	if (S_ISGID & m_statbuf.st_mode &&
		S_IXGRP & m_statbuf.st_mode) file.attr[6] = 's';
	file.attr[7]= S_IROTH & m_statbuf.st_mode ? 'r' : '-';
	file.attr[8]= S_IWOTH & m_statbuf.st_mode ? 'w' : '-';
	file.attr[9]= S_IXOTH & m_statbuf.st_mode ? 'x' : '-';
	if (S_ISVTX & m_statbuf.st_mode) file.attr[9] = 'S';
	if (S_ISVTX & m_statbuf.st_mode &&
		S_IXUSR & m_statbuf.st_mode) file.attr[9] = 's';
	file.attr[10]='\0';
	// ====================================================
	
	// . 파일 크기를 알아냄
	file.size = m_statbuf.st_size;

	// . 파일의 시간(?)을 알아냄
	file.mtime = m_statbuf.st_mtime;		
	
	// . 파일 생성 날짜를 알아냄
	strftime(file.date, 9, date_type.c_str(), localtime(&m_statbuf.st_mtime));

	// . 파일 생성 시간을 알아냄
	strftime(file.time, 6, "%H:%M", localtime(&m_statbuf.st_mtime));
	
	
	return true;
}

///	\brief	제거 함수
void DirReader::Destroy()
{
	// . directory 닫기
	closedir(m_handle);
	m_handle = NULL;

	// . 메모리 해제
	delete []m_lpath;
}
