///	@file	mlslocale.cpp
///	@brief	mlslocale.cpp

#include <cctype>

#include <ncurses.h>
#include "mlslocale.h"
#include "strutil.h"

chtype MLS::HLINE=ACS_HLINE;		///< 가로 그래픽 문자 '-'
chtype MLS::VLINE=ACS_VLINE;		///< 세로 그래픽 문자 '|'
chtype MLS::ULCORNER=ACS_ULCORNER;	///< 왼쪽 위 사각 귀뚱이 그래픽 문자
chtype MLS::LLCORNER=ACS_LLCORNER;	///< 왼쪽 아래 사각 귀뚱이 그래픽 문자
chtype MLS::URCORNER=ACS_URCORNER;	///< 오른쪽 위 사각 귀뚱이 그래픽 문자
chtype MLS::LRCORNER=ACS_LRCORNER;	///< 오른쪽 아래 사각 귀뚱이 그래픽 문자
chtype MLS::LTEE=ACS_LTEE;			///< ?????
chtype MLS::RTEE=ACS_RTEE;			///< ?????
chtype MLS::BTEE=ACS_BTEE;			///< 'ㅗ'그래픽 문자
chtype MLS::TTEE=ACS_TTEE;			///< 'ㅜ'그래픽 문자

MLS::ENCODING	MLS::e_nCurLang     = AUTO;		///< langeuage seting
MLS::LINECODE	MLS::e_nBoxLineCode = ACSLINE;	///< BOX LINE MODE

/// @brief	Locale을 설정하는 함수
///
/// locale에 따라 그래픽 문자들을 초기화 한다.
///
/// @param nEncode	LOCALE
void MLS::Set_Locale(
	MLS::ENCODING nEncode			///< Locale
	)
{
	if (nEncode == AUTO)
	{
		string sLang;

		if ( getenv("LANG") ) sLang = getenv("LANG");

		// Convert to lower case.
		sLang = strutil::tolower(sLang);
		
		if  (	sLang == "ko_kr.euckr" || sLang == "ko_kr" )
		{
			e_nCurLang = KO_EUCKR;
			setlocale(LC_ALL, getenv("LANG"));
			setlocale(LC_CTYPE, getenv("LANG"));
			textdomain("mls_euckr");
		}
		else if (	sLang == "ko_kr.utf-8" || sLang == "ko_kr.utf8" )
		{
			e_nCurLang = KO_UTF8;
			setlocale(LC_ALL, getenv("LANG"));
			setlocale(LC_CTYPE, getenv("LANG"));
			textdomain("mls_utf8");
		}
		else
		{
			e_nCurLang = US;
			setlocale(LC_MESSAGES, "C");
			setlocale(LC_CTYPE, "ko_KR.UTF-8");
			bind_textdomain_codeset("mls_utf8", "C");
		}
	}
	else
	{
		e_nCurLang = nEncode;

		if (nEncode == KO_EUCKR)
		{
			LOG("KO_EUCKR !!!");
			setlocale(LC_ALL, "ko_KR.eucKR");
			setlocale(LC_CTYPE, "ko_KR.eucKR");
			textdomain("mls_euckr");
		}
		else if (nEncode == KO_UTF8)
		{
			setlocale(LC_ALL, "ko_KR.utf-8");
			setlocale(LC_CTYPE, "ko_KR.utf-8");
			textdomain("mls_utf8");
		}
		else
		{
			setlocale(LC_MESSAGES, "C");
			setlocale(LC_CTYPE, "ko_KR.UTF-8");
			bind_textdomain_codeset("mls_utf8", "C");
		}
	}
}

/// @brief	그래픽모드 라인이나 ASCII모드 라인을 설정한다.
void MLS::Set_Line(
	MLS::LINECODE nCode			///< 그래픽모드, ASCII모드
	)
{
	switch(nCode)
	{
		case CHARLINE:
			HLINE='-';
			VLINE='|';
			ULCORNER='+';
			LLCORNER='+';
			URCORNER='+';
			LRCORNER='+';
			LTEE='+';
			RTEE='+';
			BTEE='+';
			TTEE='+';
			e_nBoxLineCode = CHARLINE;
			break;

		case ACSLINE:
			HLINE=ACS_HLINE;
			VLINE=ACS_VLINE;
			ULCORNER=ACS_ULCORNER;
			LLCORNER=ACS_LLCORNER;
			URCORNER=ACS_URCORNER;
			LRCORNER=ACS_LRCORNER;
			LTEE=ACS_LTEE;
			RTEE=ACS_RTEE;
			BTEE=ACS_BTEE;
			TTEE=ACS_TTEE;
			e_nBoxLineCode = ACSLINE;
	}
}

/// @brief	CodeConvert 함수 : utf-8 -> euckr로 바꿀때 유용 반대 가능 c함수
///
/// code 형식을 바꿀 경우 사용한다. 이것을 끝낼때는 리턴된 char*를 free 해야 한다.
///
/// @param str			source
/// @param to_codeset	원시 codeset
/// @param from_codeset	바꿀 coddset
/// @return	바뀐 char*
#ifdef _ICONV_USE
char* MLS::code_convert(const char *str,
						const char *to_codeset,
						const char *from_codeset)
{
	iconv_t cd;
	char *dest;
	char *outp;
	char *p, *startp;
	size_t inbytes_remaining;
	size_t outbytes_remaining;
	size_t err;
	size_t outbuf_size;
	int have_error = 0;
	int len;

	if (strcmp (to_codeset, from_codeset) == 0)
	{
		char *p;
		p = (char*)malloc (sizeof(char) * (strlen (str) + 1));
		if (!p)
			return NULL;
			strcpy (p, str);
		return p;
	}

	cd = iconv_open (to_codeset, from_codeset);

	if (cd == (iconv_t) - 1)
		return NULL;

	p = (char*)malloc (sizeof(char) * (strlen (str) + 1));
	strcpy (p, str);

	if (p == NULL) return NULL;

	len = strlen (p);
	startp = p;
	inbytes_remaining = len;
	outbuf_size = len + 1;        /* + 1 for nul in case len == 1 */

	outbytes_remaining = outbuf_size - 1; /* -1 for nul */
	outp = dest = (char*)malloc (outbuf_size);

	again:

	err = iconv (cd, (ICONV_CONST char **) &p, &inbytes_remaining,	&outp, &outbytes_remaining);

	if (err == (size_t) - 1)
	{
		switch (errno)
		{
			case EINVAL:
				/* Incomplete text, do not report an error */
				LOG("code_convert error EINVAL\n");
				break;

				case E2BIG:
				{
					size_t used = outp - dest;
					outbuf_size *= 2;
					dest = (char*)realloc (dest, outbuf_size);

					outp = dest + used;
					outbytes_remaining = outbuf_size - used - 1;        /* -1 for nul */

					goto again;
				}
				break;

			case EILSEQ:
				have_error = 1;
				LOG("code_convert error EILSEQ\n");
				break;

			default:
				LOG("code_convert error default\n");
				have_error = 1;
				break;
		}
	}

	*outp = '\0';

	if ((p - startp) != len)
		have_error = 1;

	free (startp);

	iconv_close (cd);

	if (have_error)
	{
		free (dest);
		dest = NULL;
	}

	return dest;
}
#endif

/// @brief	CodeConvert 함수 : 예를 들어 utf-8 -> euckr로 바꿀때 유용 C++ 함수
///
/// code 형식을 바꿀 경우 사용한다.
/// 다른 이유가 없으면 이것을 이용해서 사용한다.
///
/// @param sSource		source
/// @param to_codeset	원시 codeset
/// @param from_codeset	바꿀 coddset
/// @return	바뀐 string 데이터
string MLS::CodeConvert(const string& sSource, const char *to_codeset, const char *from_codeset)
{
	char*	cStr = NULL;
	string	sCodeConvertStr = "";

	#ifdef _ICONV_USE
	cStr = code_convert((char*)sSource.c_str(), to_codeset, from_codeset);
	#endif

	if (cStr != NULL) 
	{
		sCodeConvertStr = cStr;
		free(cStr);
	}
	return sCodeConvertStr;
}

/// @brief	영문과 한글을 자동으로 보여주기 위한 함수.
/// @param	sEnglish	영문
/// @param	sKorean		영문에 대응한 한글 내용
/// @return	언어 설정에 따라 영어 아니면 한글 메시지를 반환한다.
string MLS::ChgEngKor(const string& sEnglish, const string& sKorean)
{
	#ifndef _ICONV_USE
		return sEnglish;
	#endif

	if (e_nCurLang == KO_EUCKR)
	{
		return CodeConvert(sKorean, "EUC-KR", "UTF-8");
	}
	else if (e_nCurLang == KO_UTF8)
	{
		return sKorean;
	}
	else
	{
		return sEnglish;
	}
}

/// @brief	현재 사용하는 코드로 바꿔준다. US 면 utf-8로 바꾼다.
/// @param	sSrc	문자열
/// @param	eEncode	리턴 현재 인코드
/// @param	nSize	한글 길이 사이즈 (NULL이면 반환않함)
/// @return	 바꾼 문자열
string	MLS::isKorCode(const string& sSrc, MLS::ENCODING* eEncode)
{
	string	sCodeConvertStr = "";
	if (eEncode != NULL) *eEncode = US;
	if (sSrc == "") return "";

	bool bUS = true;
	wstring	wstr = strutil::strtowstr(sSrc);
	if (wstr.size() != 0)
	{
		for (int n = 0; n < wstr.size(); n++)
		{
			if (wstr[n] > 0x80) {
				bUS = false;
				break;
			}
		}
		if (bUS == true) {
			*eEncode = US;
			return sSrc;
		}
	}
	if (e_nCurLang == KO_EUCKR)
	{
		#ifndef _ICONV_USE
			sCodeConvertStr = sSrc;
			*eEncode = KO_EUCKR;
			return sCodeConvertStr;
		#endif
		// euckr로 바꿈.
		sCodeConvertStr = CodeConvert(sSrc, "EUC-KR", "UTF-8");
		if (sCodeConvertStr == "")
		{
			// utf8로 바꿔봄
			sCodeConvertStr = CodeConvert(sSrc, "UTF-8", "EUC-KR");
			if (sCodeConvertStr == "") return sSrc;
			if (eEncode != NULL) *eEncode = KO_EUCKR;
			return sSrc;
		}
		if (eEncode != NULL) *eEncode = KO_UTF8;
		return sCodeConvertStr;
	}
	else // 나머지는 utf-8로 바꿈.
	{
		#ifndef _ICONV_USE
			sCodeConvertStr = sSrc;
			*eEncode = KO_UTF8;
			return sCodeConvertStr;
		#endif
		// utf8 로 바꿈.
		sCodeConvertStr = CodeConvert(sSrc, "UTF-8", "EUC-KR");
		if (sCodeConvertStr == "")
		{
			// euckr로 바꿔봄
			sCodeConvertStr = CodeConvert(sSrc, "EUC-KR", "UTF-8");
			if (sCodeConvertStr == "") return sSrc;
			if (eEncode != NULL) *eEncode = KO_UTF8;
			return sSrc;
		}
		if (eEncode != NULL) *eEncode = KO_EUCKR;
		return sCodeConvertStr;	
	}
	return sSrc;
}
