///	@file	menu.cpp
///	@brief	menu.cpp
#include <ncurses.h>

#include <algorithm>
#include <functional>

#include "mainframe.h"
#include "menu.h"
#include "drawutil.h"
#include "mlslocale.h"
#include "strutil.h"

#include "panel.h"

using namespace strutil;
using namespace std;

namespace MLS {

///	@brief	생성자
Menu::Menu()
{
	_num = 0;
	_win = NULL;
}

/// @brief	소멸자
Menu::~Menu()
{
	// . 윈도우 삭제
	if (_win) delwin(_win);
}

///	@brief	메뉴를 화면에 그린다.
void Menu::Draw()
{
	if (!_win) return;

	// 주 메뉴를 그린다.
	// 예 "Mls  Run  File  Directory  Archive  Option" 메뉴
	setcol(g_Color.Func, _win);
	wmove(_win, 0, 0);
	whline(_win,' ', g_nCOLS);
	for (int t=0; t<_item.size(); t++)
		mvwprintw(_win, 0, _item[t].GetLocation() , "%-11s", _item[t].GetName().c_str());

	// 주 메뉴에서 첫글자를 노란글자로 쓴다.(구분)
	setcol(g_Color.FuncA, _win);
	for (int t=0; t<_item.size(); t++)
	{
		mvwprintw(_win, 0, _item[t].GetLocation(),
				(char*)krstrncpy(_item[t].GetName(), 0, 1).c_str());
	}

	// 선택 된 메뉴에 검은색바탕에 글자를 씀
	setcol(7, 0, _win);
	mvwprintw(_win, 0, _item[_num].GetLocation()-1 , " %s ", _item[_num].GetName().c_str());

	// 선택 된 메뉴에 첫번째 글자를 노란색으로 쓴다.
	setcol(11, 0, _win);
	mvwprintw(_win, 0, _item[_num].GetLocation(),
			(char*)krstrncpy(_item[_num].GetName(), 0, 1).c_str());

	redrawwin(_win);
	wrefresh(_win);
}

///	@brief	메뉴에서 행해지는 행동들을 처리한다.
///	@param	rp		MLS Panel
string Menu::Process(KeyBind*	pKeyBind)
{
	int key;
	bool bRef=true;
	string	sRtCmd = "";

	//touchline(stdscr, 0, g_nLINES);

	while(1)
	{
		// 필요할 때만 화면 원상복귀
		if (bRef)
		{
			MenuRefresh();
			bRef=false;
		}

		// .. Menu 화면을 그린다.
		Draw();
		// 메뉴 뭉치를 그린다.
		_item[_num].Draw(pKeyBind);

		// .. 키입력을 받는다.
		key = getch();

		// .. '`'입력을 받으면 메뉴를 없애고 메뉴를 없앤다.
		if ('`' == key) break;

		// ESC 누르면 메뉴를 닫는다.
		if (key == 27) break;

		// .. 엔터를 입력하면 해당 메뉴의 명령을 실행한다.
		if (13 == key || 10 == key)
		{
			MenuRefresh();
			sRtCmd = _item[_num].GetCmd();
			break;
		}

		// .. 왼쪽 화살표를 누르면 왼쪽에 있는 메뉴로 이동한다.
		if (KEY_LEFT == key)
		{
			_num = (_num+ _item.size() -1) % _item.size();
			bRef = true;
		}

		// .. 오른쪽 화살표를 누르면 오른쪽에 있는 메뉴로 이동한다.
		if (KEY_RIGHT == key)
		{
			_num = (_num + 1) % _item.size();
			bRef = true;
		}

		// .. 위쪽 화살표를 누르면 메뉴_item위치 위로 변경
		if (KEY_UP == key)	_item[_num].CursorUp();
		// .. 아래쪽 화살표를 누르면 메뉴_item위치 아래로 변경
		if (KEY_DOWN == key) _item[_num].CursorDown();

		// 마우스
		if (key == KEY_MOUSE)
		{
			if (MouseProc()) bRef = true;
		}

		if (key == KEY_RESIZE)
		{
			g_nLINES = LINES;
			g_nCOLS = COLS;
			MenuRefresh();
		}
	}
	return sRtCmd;
}

///	@brief	메뉴를 생성한다.
void Menu::Create()
{
	_item.clear();

	MenuAddCatecory();

	// Disable 된 데이터를 바꾼다.
	for(int c=0; c<_vCmdDisable.size(); c++)
		for(int b=0; b<_item.size(); b++)
			_item[b].SetDisable(_vCmdDisable[c]);

	if (_win) delwin(_win);
	_win = newwin(1, g_nCOLS, 0, 0);
	
	CreateWindow();
}

///	@brief	메뉴윈도우를 만들다.
void Menu::CreateWindow()
{
	for_each(_item.begin(), _item.end(), mem_fun_ref(&MenuCategory::Create));
}

/// @brief	마우스 이벤트 처리 함수
bool Menu::MouseProc()
{
	MEVENT event;
	getmouse(&event);

	if (event.bstate & BUTTON1_CLICKED)
	{// 메뉴 윗부분 클릭 검출
		if (event.y == 0)
		for (int t=0; t<_item.size(); t++)
		{
			int width = krstrlen(_item[t].GetName());

			if (_item[t].GetLocation() <= event.x &&
					event.x <= _item[t].GetLocation() + width)
			{
				_num = t;
				return true;
			}
		}
		else if( 2 <= event.y && event.y <= 2 + _item[_num].GetItemSize())
		{
			if ( _item[_num].GetLocation() <= event.x &&
					event.x <= _item[_num].GetLocation() + 24)
			{
				_item[_num].SetCursor(event.y - 2);
				ungetch(13);
			}
		}

	}
	// 아랫부분 클릭 검출
	return false;
}

///	@brief	메뉴 뭉치를 생성한다.
void MenuCategory::Create()
{
	if (_win)
	{
		delwin(_win);
	}
	_win = newwin(_item.size() + 2, 28, 1, _location-3);// (wy, wx, y, x)
}

///	@brief	메뉴 뭉치를 화면에 그린다.
void MenuCategory::Draw(KeyBind* pKeyBind)
{
	int 	nFuncKey;
	string	sCtlKey;
	if (!_win) return;

	// ----- 긋기
	setcol(g_Color.Func, _win);
	wmove(_win, 0, 0);
	waddch(_win, ' ');
	waddch(_win, ULCORNER);
	whline(_win, HLINE, 24);
	mvwaddch(_win, 0, 26, URCORNER);
	waddch(_win, ' ');

	for(int c=0; c<_item.size(); c++)
	{
		wmove(_win, c+1, 0);
		waddch(_win, ' ');
		waddch(_win, VLINE);
		if (_item[c].GetName().empty())
		{// 빈 문자열은 줄긋기

			setcol(0, g_Color.Func.back, _win);
			whline(_win, HLINE, 24);
			wmove(_win, c+1, 26);

			setcol(g_Color.Func, _win);
		}
		else
		{
			whline(_win, ' ', 24);
			if (_item[c].GetEnable() == false) setcol(8, g_Color.Func.back, _win);
			mvwprintw(_win, c+1, 2, " %s ", _item[c].GetName().c_str());
			
			// Ctrl, Alt 정보를 넣는다.
			if (_item[c].GetEnable() == false)
				setcol(8, g_Color.Func.back, _win);
			else
				setcol(g_Color.FuncA, _win);
			if (pKeyBind)
				sCtlKey = pKeyBind->GetKeyStr(_item[c].GetCmd());
			if (sCtlKey != "")
			{
				int nPos = 25 - krstrlen(sCtlKey);
				mvwprintw(_win, c+1, nPos, "%s", sCtlKey.c_str());
			}
			
			setcol(g_Color.Func, _win);
		}

		mvwaddch(_win, c+1, 26, VLINE);
		mvwaddch(_win, c+1, 27, ' ');
	}

	// ----- 긋기
	wmove(_win, _item.size()+1, 0);
	waddch(_win, ' ');
	waddch(_win, LLCORNER);
	whline(_win, HLINE, 24);
	mvwaddch(_win, _item.size()+1, 26, LRCORNER);
	waddch(_win, ' ');

	// . 선택 된 item를 반전 시켜 보여줌
	setcol(7, 0, _win);
	wmove(_win, _index+1, 2);
	whline(_win, ' ', 24);
	mvwprintw(_win, _index+1, 2, " %s ", _item[_index].GetName().c_str());
	if (pKeyBind)
		sCtlKey = pKeyBind->GetKeyStr(_item[_index].GetCmd());
	if (sCtlKey != "")
	{
		int nPos = 25 - krstrlen(sCtlKey);
		mvwprintw(_win, _index+1, nPos, "%s", sCtlKey.c_str());
	}

	redrawwin(_win);
	wrefresh(_win);
}

void MenuCategory::SetCursor(int idx)
{
	if (idx < 0 || _item.size() <= idx) return;
	if (_item[idx].GetName().empty()) return;
	_index = idx;
}

///	@brief	현재 선택 되어 있는 메뉴 항목위치을 위로 하나 이동시킴
void MenuCategory::CursorUp( )
{
	if (_item.empty()) return;

	_index = (_index +_item.size() - 1) % _item.size();

	// 빈 스트링은 건너뛴다
	// 메뉴전체가 빈 스트링일 경우 무한 루프 빠질위험 있음
	if (_item[_index].GetName().empty()) CursorUp();
	if (_item[_index].GetEnable() == false) CursorUp();
}

///	@brief	현재 선택 되어 있는 메뉴 항목위치을 아래로 하나 이동시킴
void MenuCategory::CursorDown( )
{
	if (_item.empty()) return;

	_index = (_index + 1) % _item.size();

	// 빈 스트링은 건너뛴다
	// 메뉴전체가 빈 스트링일 경우 무한 루프 빠질위험 있음
	if (_item[_index].GetName().empty()) CursorDown();
	if (_item[_index].GetEnable() == false) CursorDown();
}

} // namespace
