/* *************************************************************************
 *   Copyright (C) 2004 by Byoungyoung, La                                 *
 *   la9527@yahoo.co.kr                                                    *
 *                                                                         *
 ************************************************************************* */

///	\file	TargzUtil.cpp
///	\brief	TargzUtil.cpp

#include "TargzUtil.h"

using namespace MLS;

///	\brief	생성자
///	\param	sDir_Filename	Full file name(directory 포함)
///	\param	isZip			zip 파일 여부
TargzUtil::TargzUtil(const string& sDir_Filename, bool isZip)
: m_TarCommand(DetectTarCommand())
{
	string::size_type tSize;

	tSize = sDir_Filename.find("/");
	
	if (tSize != string::npos)
	{
		// . direcotory name 설정
		m_Dir = sDir_Filename.substr(0, sDir_Filename.find_last_of("/"));
	
		// . file name  설정
		m_Filename = sDir_Filename.substr(	sDir_Filename.find_last_of("/"),
						sDir_Filename.length() - sDir_Filename.find_last_of("/"));
	}
	else
	{
		m_Dir = "./";
		m_Filename = sDir_Filename;
	}

	m_sCwd = m_Dir;
					
	// . 변수 초기화
	m_isZip = isZip;
	
	m_GzipDir = "";
	m_Bzip2Dir = "";
	m_TarDir = "";
	
	
	m_tFileList.clear();
	m_ErrorInfo.clear();
	
	GetFileInfo();
}


///	\brief	생성자
///	\param	sDir		file direictory name
///	\param	sFilename	file name(directory 미포함)
///	\param	isZip		zip 파일 여부
TargzUtil::TargzUtil(const string& sDir, const string& sFilename, bool isZip)
: m_TarCommand(DetectTarCommand())
{
	m_Dir = sDir;
	m_sCwd = m_Dir;
	m_Filename = sFilename;
	m_isZip = isZip;	
	
	m_GzipDir = "";
	m_Bzip2Dir = "";
	m_TarDir = "";
	
	m_tFileList.clear();
	m_ErrorInfo.clear();
}

///	\brief	소멸자
TargzUtil::~TargzUtil()
{
	File*	pFile = NULL;
	uint	nCount;
	
	for (nCount = 0; nCount < (uint)m_tFileList.size(); nCount++)
	{
		pFile = m_tFileList[nCount];
		delete pFile;
	}
}

///	\brief	파일이 있는지 확인
///	\param	sFilename	filename
///	\return	파일이 있는지 여부
bool
TargzUtil::IsFileCreate(const string& sFilename)
{
	string sDir = m_Dir + sFilename;
	LOG("CHECK FILE [%s]", sDir.c_str());

	FILE*	pf = fopen(sDir.c_str(), "rb");
	if (!pf) return true;
	fclose(pf);
	return false;
}

///	\brief	 확장자를 제외한 기본파일 이름 반환 (Test.tar.gz -> Test.tar)
///	\return	변환된 파일 이름
string
TargzUtil::GetTarFileName(void)
{
	string sFilename = "";
	string::size_type tSize;
	
	tSize = m_Filename.rfind(".tar");
	if (tSize != string::npos)
	{
		sFilename = m_Filename.substr(0, tSize+4);
		return sFilename;
	}
	
	tSize = m_Filename.rfind(".tgz");
	if (tSize != string::npos) 
	{
		sFilename = m_Filename;
		sFilename = m_Filename.substr(0, tSize);
		sFilename = sFilename+".tar";
		return sFilename;
	}
		
	tSize = m_Filename.rfind(".tgz");
	if (tSize != string::npos) 
	{
		sFilename = m_Filename;
		sFilename = m_Filename.substr(0, tSize);
		sFilename = sFilename+".tar";
		return sFilename;
	}
	
	LOG("FileName Convert :: %s", sFilename.c_str());
	return m_Filename;
}

///	\brief	확장자 체크 
///	\param	sExt	체크할 파일 확장자
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int
TargzUtil::FileExtCheck(const char* sExt)
{
	string::size_type tStringSize = m_Filename.rfind(sExt);
	
	if (tStringSize == string::npos) 
		return ERROR;
	return SUCCESS;
}

///	\brief	압축 파일 구분.(m_isFileInfo 에 저장)
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int	
TargzUtil::GetFileInfo(void)
{
	if (GetTarFileName() == "") 
	{
		LOG("GetTarFileName NULL");
		return ERROR;
	}
	
	if (FileExtCheck(".tar.gz") == SUCCESS 	|| FileExtCheck(".tgz") == SUCCESS 
	 || FileExtCheck(".tar.z")  == SUCCESS  || FileExtCheck(".tar.Z") == SUCCESS) 
	{
		LOG("m_isFileInfo :: TAR_GZ");
		m_isFileInfo = TAR_GZ;
	}
	else if (FileExtCheck(".tar.bz2") == SUCCESS || FileExtCheck(".tbz") == SUCCESS
		|| FileExtCheck(".tar.bz") == SUCCESS)
	{
		LOG("m_isFileInfo :: TAR_BZ");
		m_isFileInfo = TAR_BZ;
	}
	else if (FileExtCheck(".tar") == SUCCESS) 
	{
		LOG("m_isFileInfo :: TAR");
		m_isFileInfo = TAR;
	}
	else if (FileExtCheck(".gz") == SUCCESS)
	{
		m_isFileInfo = GZ;
	}
	else if (FileExtCheck(".bz2") == SUCCESS || FileExtCheck(".bz") == SUCCESS)
	{
		m_isFileInfo = BZ;
	}
	else if (FileExtCheck(".zip") == SUCCESS) 
	{
		LOG("m_isFileInfo :: ZIP");
		m_isFileInfo = ZIP;
	}
	else if (FileExtCheck(".rpm") == SUCCESS)
	{
		LOG("m_isFileInfo :: .rpm");
		m_isFileInfo = RPM;
	}
	else if (FileExtCheck(".deb") == SUCCESS)
	{
		LOG("m_isFileInfo :: .deb");
		m_isFileInfo = DEB;
	}
	else if (FileExtCheck(".alz") == SUCCESS)
	{
		LOG("m_isFileInfo :: .alz");
		m_isFileInfo = ALZ;
	}
	else 
	{
		LOG("m_isFileInfo :: ERROR");
		return ERROR;
	}
	return SUCCESS;
}

///	\brief	Command 명령 실행. 
///	\param	sCommand	실행할 명령어
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int
TargzUtil::CommandExecute(const string& sCommand)
{
	char	cLine[1024];
	FILE*	pfFile = NULL;
	m_ErrorInfo.clear();

	LOG("Command Execute [%s]", sCommand.c_str());
	
	string sCmd = sCommand;
	sCmd.append(" 2> /dev/null");
	pfFile = popen(sCmd.c_str(), "r");
	
	if (pfFile) {
		rewind(pfFile);
		// 줄단위로 데이터 읽음.
		while (fgets(cLine, sizeof(cLine), pfFile))
		{
			LOG("Command Error :: [%s]", cLine);
			// 에러가 나면 m_ErrorInfo로 저장.
			m_ErrorInfo.push_back(cLine);
		}
		pclose(pfFile);
	}
	else
	{
		throw Exception(gettext("File open error !!!"));
		return ERROR;
	}
	
	if (m_ErrorInfo.size() != 0) return ERROR;	
	return SUCCESS;
}

///	\brief	파일 전체 경로에서 디렉토리와 파일이름을 빼낸다.
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int
TargzUtil::Fullname_To_Filename(string* sFullFile, string* sFilename, bool* bDir)
{
	
	string				sTmp;
	string::size_type	tStringSize;
	string::size_type	tStringSize_tmp_1;	
	
	tStringSize_tmp_1 = 0;
	sTmp = "";
	
	if (*sFullFile == "") return ERROR;

	for(;;)
	{
		tStringSize_tmp_1 = sFullFile->find("/./", tStringSize_tmp_1);
		if (tStringSize_tmp_1 != string::npos)
		{
			sFullFile->erase(tStringSize_tmp_1, 2);
		}
		else 
			break;
	}

	// 앞에 '.'로 시작되는 파일이면 '.'을 떼어낸다.
	if (sFullFile->substr(0, 1) == ".") sFullFile->erase(0, 1);
	
	// 앞에 '/'로 시작되는 파일이면 '/'를 떼어낸다.
	if (sFullFile->size() != 1)
		if (sFullFile->substr(0, 1) == "/") sFullFile->erase(0, 1);
	
	// 마지막에 '/' 가 있는지 찾는다. (디렉토리인지 확인)
	tStringSize = sFullFile->find("/", sFullFile->length() - 1);
	
	// Fullname에서 name 구분.
	if (tStringSize != string::npos)
	{
		sTmp = sFullFile->substr(0, sFullFile->length() - 1);
		
		tStringSize_tmp_1 = sTmp.rfind("/");
					
		if (tStringSize_tmp_1 == string::npos)
		{
			*sFilename = *sFullFile;
		}
		else
		{
			*sFilename = sFullFile->substr(	tStringSize_tmp_1 + 1, sFullFile->length() - tStringSize_tmp_1);
		}
		*bDir = true;
	}
	else
	{
		sTmp = *sFullFile;
		tStringSize_tmp_1 = sTmp.rfind("/");
		*sFilename = sFullFile->substr(	tStringSize_tmp_1 + 1, sFullFile->length() - tStringSize_tmp_1);
		
		*bDir = false;
	}
	return SUCCESS;
}

///	\brief	압축 확장자 별로 파일리스트를 읽는다.
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int
TargzUtil::LineFormatRead(vector<string>&	vLineToken, File* pFileInfo, int nFormat)
{
	string	sFullFilename = "";
	string	sFilename = "";
	string	sDate = "";
	bool	bDir = false;
	
	switch(nFormat)
	{
		case	TAR_GZ:
		case	TAR:
		case	TAR_BZ:
			// It is not same date and time format every OS.
			// So if you get information from text output, user maybe
			// get incorrect screen.
			//
			// Consider process tar header. It is not real solution.
			// cf) http://www.gnu.org/software/tar/manual/html_node/tar_134.html
			
			// 파일 정보 채움.
			#ifdef __FreeBSD__
			if (vLineToken.size() >= 7)
			{ 
				snprintf(pFileInfo->date, sizeof(pFileInfo->date), "%4s%4s", vLineToken[3].c_str(), vLineToken[4].c_str()); 
				strncpy(pFileInfo->time, vLineToken[5].c_str(), 5);
				for (int n = 0; n < vLineToken.size()-7; n++)
					if (n == 0)
						sFullFilename = vLineToken[7];
					else
						sFullFilename = sFullFilename + " " + vLineToken[7+n];
			#else
			if (vLineToken.size() >= 6)
			{ 
				strncpy(pFileInfo->date, (vLineToken[3].c_str())+2, 8);
				strncpy(pFileInfo->time, vLineToken[4].c_str(), 5);
				for (int n = 0; n < vLineToken.size()-5; n++)
					if (n == 0)
						sFullFilename = vLineToken[5];
					else
						sFullFilename = sFullFilename + " " + vLineToken[5+n];
			#endif
				strncpy(pFileInfo->attr, vLineToken[0].c_str(), 10);
				pFileInfo->size = atol(vLineToken[2].c_str());

				if (!strncmp(pFileInfo->attr, "d", 1))
					pFileInfo->bDir = true;
				else
					pFileInfo->bDir = false;

				pFileInfo->TarFullName = sFullFilename;
				//LOG("TarFullName [%s]", (pFileInfo->TarFullName).c_str());

				// '/./' 제거
				Fullname_To_Filename(&sFullFilename, &sFilename, &bDir);
				
				//pFileInfo->bDir = bDir;
				if (pFileInfo->bDir == true)
					if (sFilename.substr(sFilename.length()-1, 1) == "/")
						sFilename.erase(sFilename.length()-1, 1);
				
				if (pFileInfo->bDir == true)
					if (sFullFilename.substr(sFullFilename.length()-1, 1) != "/")
						sFullFilename = sFullFilename + "/";
					
				pFileInfo->FullName = sFullFilename;
				pFileInfo->name = sFilename;

				// Link setting
				if (!strncmp(pFileInfo->attr, "l", 1))
				{
					#ifdef __FreeBSD__
					if (vLineToken.size() == 10)
						pFileInfo->name = sFilename + " -> " + vLineToken[9];
					#else
					if (vLineToken.size() == 8)
						pFileInfo->name = sFilename + " -> " + vLineToken[7];
					#endif
				}
			}
			else
			{
				return ERROR;
			}
			break;
		
		case RPM:
			if (vLineToken.size() >= 9)
			{
				// 파일 정보 채움.
				strncpy(pFileInfo->attr, vLineToken[0].c_str(), 10);
				pFileInfo->size = atol(vLineToken[4].c_str());
				
				sDate = vLineToken[5] + " " + vLineToken[6];
				memset(pFileInfo->date, ' ', 8);
    			strncpy(pFileInfo->date, sDate.c_str(), sDate.size());
				strncpy(pFileInfo->time, vLineToken[7].c_str(), 5);
				
				if (!strncmp(pFileInfo->attr, "d", 1))
					pFileInfo->bDir = true;
				else
					pFileInfo->bDir = false;

				for (int n = 0; n < vLineToken.size()-8; n++)
					if (n == 0)
						sFullFilename = vLineToken[8];
					else
						sFullFilename = sFullFilename + " " + vLineToken[8+n];
					
				pFileInfo->TarFullName = sFullFilename;
				
				// '/./' 제거
				Fullname_To_Filename(&sFullFilename, &sFilename, &bDir);
				
				//pFileInfo->bDir = bDir;
				if (pFileInfo->bDir == true)
					if (sFilename.substr(sFilename.length()-1, 1) == "/")
						sFilename.erase(sFilename.length()-1, 1);
				
				// 디렉토리 중에 끝에 '/' 가 없으면 만들어준다.
				if (pFileInfo->bDir == true)
					if (sFullFilename.substr(sFullFilename.length()-1, 1) != "/")
						sFullFilename = sFullFilename + "/";
				pFileInfo->FullName = sFullFilename;
				pFileInfo->name = sFilename;

				// Link setting
				if (!strncmp(pFileInfo->attr, "l", 1))
					if (vLineToken.size() == 11)
						pFileInfo->name = sFilename + " -> " + vLineToken[10];
			}
			else
			{
				return ERROR;
			}
			break;

		case DEB:
			if (vLineToken.size() >= 6)
			{
				// lrwxrwxrwx root/root         0 2005-03-03 20:00:59 ./usr/lib/libXft.so.2 -> libXft.so.2.1.1
				// 파일 정보 채움.
				strncpy(pFileInfo->attr, vLineToken[0].c_str(), 10);
				pFileInfo->size = atol(vLineToken[2].c_str());
				strncpy(pFileInfo->date, (vLineToken[3].c_str())+2, 8);
				strncpy(pFileInfo->time, vLineToken[4].c_str(), 5);

				if (!strncmp(pFileInfo->attr, "d", 1))
					pFileInfo->bDir = true;
				else
					pFileInfo->bDir = false;

				for (int n = 0; n < vLineToken.size()-5; n++)
					if (n == 0)
						sFullFilename = vLineToken[5+n];
					else
						sFullFilename = sFullFilename + " " + vLineToken[5+n];
					
				pFileInfo->TarFullName = sFullFilename;

				// '/./' 제거
				Fullname_To_Filename(&sFullFilename, &sFilename, &bDir);

				// Root 디토리가 있으면 지운다.
				if (sFilename == "/") return ERROR;
				
				//pFileInfo->bDir = bDir;
				if (pFileInfo->bDir == true)
					if (sFilename.substr(sFilename.length()-1, 1) == "/")
						sFilename.erase(sFilename.length()-1, 1);

				pFileInfo->name = sFilename;
				
				// 디렉토리 중에 끝에 '/' 가 없으면 만들어준다.
				if (pFileInfo->bDir == true)
					if (sFullFilename.substr(sFullFilename.length()-1, 1) != "/")
						sFullFilename = sFullFilename + "/";
				pFileInfo->FullName = sFullFilename;
				pFileInfo->name = sFilename;

				// Link Setting	
				if (!strncmp(pFileInfo->attr, "l", 1))
					if (vLineToken.size() == 8)
						pFileInfo->name = sFilename + " -> " + vLineToken[7];
			}
			else
			{
				return ERROR;
			}
			break;
			
		case ZIP:
			if (vLineToken.size() >= 4)
			{
				// 파일 정보 채움.
				pFileInfo->size = atol(vLineToken[0].c_str());
				strncpy(pFileInfo->date, (vLineToken[1].c_str()), 8);
				strncpy(pFileInfo->time, vLineToken[2].c_str(), 5);
				
				if (vLineToken[1] == "Date" && vLineToken[2] == "Time") return ERROR;
				if (vLineToken[1] == "----" && vLineToken[2] == "----") return ERROR;
				
				// '/./' 제거
				for (int n = 0; n < vLineToken.size()-3; n++)
					if (n == 0)
						sFullFilename = vLineToken[3];
					else
						sFullFilename = sFullFilename + " " + vLineToken[3+n];
					
				Fullname_To_Filename(&sFullFilename, &sFilename, &bDir);
				
				pFileInfo->bDir = bDir;
				if (pFileInfo->bDir == true)
					if (sFilename.substr(sFilename.length()-1, 1) == "/")
						sFilename.erase(sFilename.length()-1, 1);
				pFileInfo->name = sFilename;
				pFileInfo->FullName = sFullFilename;
				pFileInfo->TarFullName = sFullFilename;
				
				// zip은 내용을 알수 없음. 나중에 수정할수 있음 수정.
				if (pFileInfo->bDir == true)
					strncpy(pFileInfo->attr, "drwxr-xr-x", 10);
				else
					strncpy(pFileInfo->attr, "-rw-r--r--", 10);
			}
			else
			{
				return ERROR;
			}
			break;
		case ALZ:
			if (vLineToken.size() >= 6)
			{
				if (vLineToken[0].substr(0, 9) == "Copyright") return ERROR;
				
				// 파일 정보 채움.
				if (vLineToken[0].size() > 8)
				{
					vLineToken[0] = strutil::Replace(vLineToken[0], ":", "-");
					strncpy(pFileInfo->date, vLineToken[0].substr(2, 8).c_str(), 8);
				}
				else
				{
					strncpy(pFileInfo->date, vLineToken[0].c_str(), 8);
				}
				strncpy(pFileInfo->time, vLineToken[1].c_str(), 5);
				
				if (strutil::tolower(vLineToken[0]) == "date" &&
					strutil::tolower(vLineToken[1]) == "time") return ERROR;
				if (vLineToken[0].substr(0, 4) == "----" &&
					vLineToken[1].substr(0, 4) == "----") return ERROR;

				string sFileSize = vLineToken[3];
				for (int n = 0; n < sFileSize.size(); n++)
					if (isdigit((int)sFileSize[n]) == 0) return ERROR;
				
				pFileInfo->size = atol(sFileSize.c_str());
				
				for (int n = 0; n < vLineToken.size()-5; n++)
					if (n == 0)
						sFullFilename = vLineToken[5];
					else
						sFullFilename = sFullFilename + " " + vLineToken[5+n];

				pFileInfo->FullName = sFullFilename;
				pFileInfo->TarFullName = sFullFilename;

				// \ -> /
				sFullFilename = strutil::Replace(sFullFilename, "\\", "/");
				Fullname_To_Filename(&sFullFilename, &sFilename, &bDir);

				pFileInfo->bDir = bDir;
				if (pFileInfo->bDir == true)
					if (sFilename.substr(sFilename.length()-1, 1) == "/")
						sFilename.erase(sFilename.length()-1, 1);
				pFileInfo->name = sFilename;
				pFileInfo->FullName = vLineToken[5];
				pFileInfo->TarFullName = vLineToken[5];

				// AD_H 형태로 나오는데 디렉토리만 우선 구성.
				if (pFileInfo->bDir == true)
					strncpy(pFileInfo->attr, "drwxr-xr-x", 10);
				else
					strncpy(pFileInfo->attr, "-rw-r--r--", 10);
			}
			else
				return ERROR;
			break;
	}
	
	return SUCCESS;
}

///	\brief	압축 화일의 정보 읽음.
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int
TargzUtil::
FileListRead(void)
{
	FILE*    pfFile = NULL;
	string::size_type tStringSize;
	string::size_type tStringSize_tmp_1;
	string::size_type tStringSize_tmp_2;
	string	sCommand;
	string 	sLine;
	string	sToken;
 	string	sTmp;
	int		nCount = 0;
	
	vector<string>	tTmp;
	m_ErrorInfo.clear();
	
	File*	pFileInfo = NULL;
	
	char	cLine[1024];
	memset(&cLine, 0, sizeof(cLine));
	
	if (m_Filename.length() == 0) {
		throw Exception("Error :: filename is NULL");
		return ERROR;
	}
	
	SetCounter(20);	
	
	// 파일 확장자 체크
	if (GetFileInfo() == ERROR) 
	{
		//throw Exception("Error :: Filename Info ERROR");
		LOG("file GetFileInfo ERROR :: %s", sCommand.c_str());
		return ERROR;
	}

	if (m_isFileInfo == GZ || m_isFileInfo == BZ)
	{
		File*	pFile;
		pFile = new File;
		strncpy(pFile->attr, "dr--r--r--", 10);
		pFile->size = 0;
		pFile->bDir = true;
		strncpy(pFile->date, "--------", 8);
		strncpy(pFile->time, "--:--", 5);
		pFile->name = "..";
		pFile->FullName = "/";
		m_tFileList.push_back(pFile);
		
		pFile = new File;
		strncpy(pFile->attr, "-rw-r--r--", 10);
		pFile->size = 0;
		pFile->bDir = false;
		strncpy(pFile->date, "--------", 8);
		strncpy(pFile->time, "--:--", 5);
		pFile->name = m_Filename.substr(0, m_Filename.rfind("."));
		pFile->TarFullName = pFile->FullName = m_Filename.substr(0, m_Filename.rfind("."));
		m_tFileList.push_back(pFile);
		LOG("Check !!! - [%s] [%s] [%d]", pFile->name.c_str(), pFile->FullName.c_str(), m_tFileList.size());
		return SUCCESS;
	}

	// . 압축 파일에 해당하는 명령어를 알아낸다.
	switch(m_isFileInfo)
	{
		case	TAR_GZ:
			sCommand = m_TarDir + m_TarCommand + " tvfz " + m_Dir + strutil::addslash(m_Filename);
			break;
		case	TAR_BZ:
			sCommand = m_TarDir + m_TarCommand + " tvfj " + m_Dir + strutil::addslash(m_Filename);
			break;
		case	TAR:
			sCommand = m_TarDir + m_TarCommand + " tvf " + m_Dir + strutil::addslash(m_Filename);
			break;
		case	ZIP:
			sCommand = m_TarDir + "unzip -l " + m_Dir + strutil::addslash(m_Filename);
			break;
		case	RPM:
			sCommand = m_TarDir + "rpm2cpio " + m_Dir + strutil::addslash(m_Filename) + " | cpio -tv ";
			break;
		case	DEB:
			sCommand = m_TarDir + "dpkg-deb -c " + m_Dir + strutil::addslash(m_Filename);
			break;
		case	ALZ:
			sCommand = "export LANG=c; " + m_TarDir + "unalz -l " + m_Dir + strutil::addslash(m_Filename);
			break;
		default:
			return ERROR;
	}
	
	//cout << "Command :: " << sCommand << endl;
	LOG("COMMAND Data :: [%s]", sCommand.c_str());
	sCommand = sCommand + " 2> /dev/null";
	
	m_tFileList.clear();
	// . 실행
	pfFile = popen(sCommand.c_str(), "r");

	if (pfFile) 
	{
		rewind(pfFile);
		// -rw-r--r-- root/root       730 2004-01-12 23:15:53 project/smstest/smstest.kdevelop.pcs
		//
		// In FreeBSD bsdtar (c locale)
		//
		// -rw-rw-r--  0 500    500      3827 Nov 27 15:13 mls-0.4.5/README
		//
		// In FreeBSD gnu tar (c locale)
		//
		// -rw-rw-r-- 500/500        3827 Nov 27 15:13 2004 mls-0.4.5/README
		//
		SetCounter(50);
		// 줄단위로 데이터 읽음.
		while (fgets(cLine, sizeof(cLine), pfFile))
		{
			sLine = cLine;
			if (sLine == "") continue;
			// 줄바꿈(\n) 은 지운다.
			sLine.erase(sLine.length() - 1, 1);
			if (sLine == "") continue;
			
			LOG("COMMAND Data :: [%s]", sLine.c_str());
			
			tTmp.clear();
			sToken = "";

			if (sLine.substr(0, 1) != " ")
			{
				tStringSize = sLine.find(" ");
				if (tStringSize == string::npos) 
				{
					tTmp.push_back(sToken);
					break;
    			}
				
				sToken = sLine.substr(0, tStringSize);
			}
			else
			{
				tStringSize = sLine.find_first_not_of(" ");
				if (tStringSize == string::npos)
				{
					tTmp.push_back(sToken);
					break;
				}
				
				tStringSize_tmp_1 = sLine.find(" ", tStringSize);

				if (tStringSize_tmp_1 == string::npos)
				{
					tTmp.push_back(sToken);
					break;
				}
				
				sToken = sLine.substr(tStringSize, tStringSize_tmp_1 - tStringSize);
			}

			// 데이터 하나씩 넣음.
			tTmp.push_back(sToken);
			
			// 공백단위로 끈는다. 
			while(1)
			{
				tStringSize_tmp_1 = sLine.find(" ", tStringSize);
				tStringSize_tmp_2 = sLine.find(" ", tStringSize_tmp_1+1);
				
				if (tStringSize_tmp_2 == string::npos)
				{
					sToken = sLine.substr(tStringSize_tmp_1);
					sToken.erase(0, 1);
					tTmp.push_back(sToken);
					break;
				}
				else
				{
					sToken = sLine.substr(tStringSize_tmp_1, tStringSize_tmp_2 - tStringSize_tmp_1);
					if (sToken != " ") 
					{
						sToken.erase(0, 1);
						tTmp.push_back(sToken);
					}
				}
				
				tStringSize = tStringSize_tmp_1 + 1;
			}
			
			sToken = "";
			
			pFileInfo = new File;
			
			if (LineFormatRead(tTmp, pFileInfo, m_isFileInfo) == SUCCESS)
			{
				m_tFileList.push_back(pFileInfo);
			}
			else
			{
				LOG("COMMAND ERROR :: %s", sLine.c_str());
				m_ErrorInfo.push_back(sLine);
				delete pFileInfo;
			}
		}
		
		pclose(pfFile);
		SetCounter(80);
		
		LOG("COMMAND END :: datasize :: %d", m_tFileList.size());
		
		if (m_tFileList.size() == 0)
		{
			throw Exception(gettext(	"'%s' File reading failure !!!"),
							m_Filename.c_str());
			return ERROR;
		}
		
		// 에러정보가 있으면
		if (m_ErrorInfo.size() != 0)
		{
			LOG("m_ErrorInfo :: %s", m_ErrorInfo[0].c_str());
		}
	}
	else
	{
		throw Exception("%s - %s", gettext("File reading failure !!!"), strerror(errno));
		return ERROR;
	}
    SetCounter(90);
	if (FileListCheck_DirInsert() == ERROR) return ERROR;
	return SUCCESS;
}

///	\brief	압축 화일에서 없는 디렉토리를 찾아 넣어준다.
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int
TargzUtil::FileListCheck_DirInsert(void)
{
	File*				pFile  = NULL;
	File*				pFile_2  = NULL;
	int					nCount = 0, nCount_2 = 0;
	
	string				sFullName;
	string				sTmpFullName, sTmpFilename;
	bool 				bTmpDir;
	
	vector<string>		vDir;
	vector<string>		vNullDir;
	
	string::size_type	tSizeType;
	string::size_type	tStringSize_tmp_1;
	string::size_type	tStringSize_tmp_2;

	// 디렉토리를 우선 찾아서 vDir 넣는다.
	for (nCount = 0; nCount < (uint)m_tFileList.size(); nCount++)
	{
		pFile = m_tFileList[nCount];
		
		if (pFile->bDir == true)
		{
			vDir.push_back(pFile->FullName);
		}
	}
	
	// 파일리스트 중 디렉토리가 없는 것을 찾아 디렉토리를 만들어준다.
	for (nCount = 0; nCount < (uint)m_tFileList.size(); nCount++)
	{
		pFile = m_tFileList[nCount];
		
		// 디렉토리는 검사하지 않는다.
		if (pFile->bDir == true) continue;
		
		sFullName = pFile->FullName;
		
		tSizeType = sFullName.rfind("/");
		
		// 디렉토리가 없는 것은 검사하지 않음.
		if (tSizeType == string::npos) continue;
		
		do
		{
			sFullName = sFullName.substr(0, tSizeType+1);
			
			if (vDir.size() != 0)
			{
				for (nCount_2 = 0; nCount_2 < (uint)vDir.size(); nCount_2++)
				{
					if (vDir[nCount_2] == sFullName) break;
					
					if (nCount_2 == vDir.size() - 1)
					{
						vNullDir.push_back(sFullName);
						vDir.push_back(sFullName);
						break;
					}
				}
			}
			else
			{
				vNullDir.push_back(sFullName);
				vDir.push_back(sFullName);
			}
			
			tSizeType = sFullName.rfind("/", tSizeType-1);
			if (tSizeType == 0) break;
		}
		while(tSizeType != string::npos);
	}
	
	for (nCount_2 = 0; nCount_2 < (uint)vNullDir.size(); nCount_2++)
	{
		pFile = new File;
		strncpy(pFile->attr, "dr--r--r--", 10);
		pFile->size = 0;
		pFile->bDir = true;
		strncpy(pFile->date, "--------", 8);
		strncpy(pFile->time, "--:--", 5);
		sTmpFullName = vNullDir[nCount_2];
		Fullname_To_Filename(&sTmpFullName, &sTmpFilename, &bTmpDir);
		pFile->name = sTmpFilename;
		pFile->FullName = sTmpFullName;
		m_tFileList.push_back(pFile);
		//LOG("NULL Dir [%s] [%s] [%s]", vNullDir[nCount_2].c_str(), sTmpFullName.c_str(), sTmpFilename.c_str());
	}
	
	pFile = new File;
	strncpy(pFile->attr, "dr--r--r--", 10);
	pFile->size = 0;
	pFile->bDir = true;
	strncpy(pFile->date, "--------", 8);
	strncpy(pFile->time, "--:--", 5);
	pFile->name = "..";
	pFile->FullName = "/";
	m_tFileList.push_back(pFile);
	return SUCCESS;
}

///	\brief	디렉토리 별로 파일 정보를 읽어 옴, Return Data :: pRe_Dir
///	\param	s_Dir	directory name
///	\param	pRe_Dir	파일정보 반환값
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int
TargzUtil::GetDir_Files(string& s_Dir, vector<MLS::File*>* pRe_Dir)
{
	//LOG(" GetDir_Files :: %s", s_Dir.c_str());
	File*	pFile = NULL;
	uint	nCount = 0;
	string::size_type	tSizeType;
	string::size_type	tSizeType_tmp_1;
	string::size_type	tSizeType_tmp_2;
	
	vector<MLS::File*>	tDir_Files;
	
	tDir_Files.clear();
	
	if (s_Dir.length() == 0)
	{
		for (nCount = 0; nCount < (uint)m_tFileList.size(); nCount++)
		{
			pFile = NULL;
			pFile = m_tFileList[nCount];
			
			tSizeType = pFile->FullName.find("/");
			
			if (tSizeType == string::npos)
			{
				tDir_Files.push_back(pFile);
				continue;
			}
			
			tSizeType_tmp_1 = pFile->FullName.find("/", tSizeType + s_Dir.length() + 1);
			
			if (tSizeType_tmp_1 == string::npos)
			{
				if (pFile->FullName.substr(pFile->FullName.length() - 1) == "/")
				{
					tDir_Files.push_back(pFile);
				}
			}
		}
	}
	else
	{
		for (nCount = 0; nCount < (uint)m_tFileList.size(); nCount++)
		{
			pFile = m_tFileList[nCount];
			tSizeType = pFile->FullName.find(s_Dir);
			if (tSizeType == string::npos) 	continue;
			
            tSizeType_tmp_1 = pFile->FullName.find("/", s_Dir.length() + tSizeType);
			
			// 파일 넣기
			if (tSizeType_tmp_1 == string::npos)
			{
				tDir_Files.push_back(pFile);
				continue;
			}
			
			// 디렉토리 넣기
			if (tSizeType_tmp_1 == pFile->FullName.length() - 1)
			{
				tDir_Files.push_back(pFile);
			}
		}
	}
	
	if (tDir_Files.size() == 0) 
	{
		LOG("tDir_files size :: 0 :: ERROR :: %d", m_tFileList.size());
		return ERROR;
	}
	
	pRe_Dir->clear();
	*pRe_Dir = tDir_Files;
	return SUCCESS;
}

///	\brief	디렉토리 별로 파일 정보를 읽어 옴, Return Data :: pRe_Dir
///	\param	s_Dir	directory name
///	\param	pRe_Dir	파일정보 반환값
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int
TargzUtil::GetDir_AllFiles(string& s_Dir, vector<MLS::File*>* pRe_Dir)
{
	
	File*	pFile = NULL;
	uint	nCount;
	string::size_type	tSizeType;
	pRe_Dir->clear();
	
	for (uint nCount = 0; nCount < (uint)m_tFileList.size(); nCount++)
	{
		pFile = m_tFileList[nCount];
		tSizeType = pFile->FullName.find(s_Dir);
		if (tSizeType == string::npos) 	continue;
		
		pRe_Dir->push_back(pFile);
	}	
	return SUCCESS;
}

///	\brief	전체 정보 읽어옴.
///	\param	rRe_File	반환할 파일정보
///	\return	SUCCESS	:	성공
int 	
TargzUtil::GetAll_Files(vector<MLS::File*>* rRe_File)
{
	*rRe_File = m_tFileList;
	return SUCCESS;
}

///	\brief	압축 풀었을때의 전체 파일 사이즈를 알아낸다.
///	\return	압축 풀었을때의 전체 파일 사이즈
uint	TargzUtil::GetAllDataSize(void)
{
	uint	uTotal = 0;
	
	File*	pFile = NULL;
	uint	nCount;
	string::size_type	tSizeType;
	
	for (uint nCount = 0; nCount < (uint)m_tFileList.size(); nCount++)
	{
		pFile = m_tFileList[nCount];
		uTotal = uTotal + pFile->size;
	}
	return uTotal;
}

///	\brief	 화일하나에 대해 Append나 delete를 한다.
///
///	 Append : true, Delete : false
///
///	\param	sFullName	full file name
///	\param	nAppendDel	TAR_DELETE 	:	파일 지움\n
///						TAR_APPEND	:	파일 추가
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int		
TargzUtil::FileControl(const string& sFullName, int nAppendDel)
{
	string	sCommand = "";	
	
	string	sTarFilename = GetTarFileName();

	LOG("TARFileNAME [%s]", sTarFilename.c_str());

	if (m_isFileInfo == ZIP)
	{
		if (IsFileCreate(m_Filename) == true)
		{
			if (nAppendDel == TAR_DELETE)
			{
				throw Exception("ERROR : '"+ m_Filename +"' file not found... :: ZIP_DELETE");
				return ERROR;
			}
		} else {
			if (nAppendDel == TAR_DELETE)
			{
				sCommand = "cd "+m_sCwd+"; zip -q -Fd " + m_Dir + strutil::addslash(m_Filename) + " " + sFullName;
				if (CommandExecute(sCommand) == ERROR) return ERROR;
				return SUCCESS;
			}
		}

		sCommand = "cd "+m_sCwd+"; zip -q -Fr " + m_Dir + strutil::addslash(m_Filename) + " " + sFullName;
		if (CommandExecute(sCommand) == ERROR) return ERROR;
		return SUCCESS;
	}
	
	// 우선 파일이 있는지 확인. 없으면 만듦. :: 있으면 true
	if (IsFileCreate(sTarFilename) == true)
	{
		if (nAppendDel == TAR_DELETE)
		{
			throw Exception("ERROR : '"+ m_Filename +"' file not found... :: TAR_DELETE");
			return ERROR;
		}

		sCommand = "cd "+m_sCwd+"; " + m_TarCommand + " cf " + m_Dir + sTarFilename + " " + sFullName;
		if (CommandExecute(sCommand) == ERROR) return ERROR;
	}
	else
	{
		switch(nAppendDel)
		{
			case TAR_APPEND:
				sCommand = "cd "+m_sCwd+"; " + m_TarCommand + " rf " + m_Dir + sTarFilename +
							" " + sFullName;
				if (CommandExecute(sCommand) == ERROR) return ERROR;
				break;
				
			case TAR_DELETE:
				sCommand = "cd "+m_sCwd+"; " + m_TarCommand + " --delete --file " +
							m_Dir + sTarFilename + " " + sFullName;
				LOG("DELETE :: %s", sCommand.c_str());
				if (CommandExecute(sCommand) == ERROR) return ERROR;
				break;
		}
	}
	return SUCCESS;
}

///	\brief	파일을 압축한다.
///
/// File들은 name으로만 판별.
///
///	\param	t_FileList	압축할 파일 리스트 
///	\param	nAppendDel :: TAR_APPEND, TAR_DELETE 가 있음. 
///	\param	nAppendDel	TAR_DELETE 	:	파일 지움\n
///						TAR_APPEND	:	파일 추가
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int
TargzUtil::Compress(vector<MLS::File*>& t_FileList, int nAppendDel)
{
	string	sCommand = "";
	SetCounter(20);

	if (GetFileInfo() == ERROR) return ERROR;
	
	// 우선 파일이 있는지 확인. 있으면 압축을 푼다.
	if (IsFileCreate(strutil::addslash(m_Filename)) == false)
	{
		// tar.gz 파일은 압축을 풀고 Append나 Delete를 해야 한다. 
		switch(m_isFileInfo)
		{
			case	TAR_GZ:
				sCommand = "cd "+m_sCwd+"; " + "gunzip " + m_Dir + strutil::addslash(m_Filename);
				if (CommandExecute(sCommand) == ERROR) return ERROR;
				break;
			case 	TAR_BZ:
				sCommand = "cd "+m_sCwd+"; " + "bunzip2 " + m_Dir + strutil::addslash(m_Filename);
				if (CommandExecute(sCommand) == ERROR) return ERROR;
				break;
		}
	}
	SetCounter(50);	
	string::size_type	tSizeType;
	File*	pFile;
		
	for (uint nCount = 0; nCount < (uint)t_FileList.size(); nCount++)
	{
		pFile = t_FileList[nCount];
		if (FileControl(pFile->FullName, nAppendDel) == ERROR) return ERROR;
	}	
	SetCounter(70);	
	switch(m_isFileInfo)
	{
		case	TAR_GZ:
			sCommand = "cd "+m_sCwd+"; " + "gzip " + m_Dir + GetTarFileName();
			if (CommandExecute(sCommand) == ERROR) return ERROR;
			break;
		case 	TAR_BZ:
			sCommand = "cd "+m_sCwd+"; " + "bzip2 " + m_Dir + GetTarFileName();
			if (CommandExecute(sCommand) == ERROR) return ERROR;
			break;
	}
	SetCounter(90);	
	return SUCCESS;
}

///	\brief	File이 아닌 파일이름을 string으로
///	\param	t_FileList	압축할 파일 리스트 
///	\param	nAppendDel	TAR_DELETE 	:	파일 지움\n
///						TAR_APPEND	:	파일 추가
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int
TargzUtil::Compress(vector<string>& t_FileList, int nAppendDel)
{
	string	sCommand = "";
	SetCounter(20);
	
	if (GetFileInfo() == ERROR) return ERROR;
	
	// 우선 파일이 있는지 확인. 있으면 압축을 푼다.
	if (IsFileCreate(strutil::addslash(m_Filename)) == false)
	{
		// tar.gz 파일은 압축을 풀고 Append나 Delete를 해야 한다. 
		switch(m_isFileInfo)
		{
			case	TAR_GZ:
				sCommand = "cd "+m_sCwd+"; " + "gunzip " + m_Dir + strutil::addslash(m_Filename);
				if (CommandExecute(sCommand) == ERROR) return ERROR;
				break;
			case 	TAR_BZ:
				sCommand = "cd "+m_sCwd+"; " + "bunzip2 " + m_Dir + strutil::addslash(m_Filename);
				if (CommandExecute(sCommand) == ERROR) return ERROR;
				break;
		}
	}

	LOG("UNZIP PASS !!!");
		
	SetCounter(50);
	string::size_type	tSizeType;
	string	sFullFileName;
	SetCounter(70);	
	for (uint nCount = 0; nCount < (uint)t_FileList.size(); nCount++)
	{
		sFullFileName = t_FileList[nCount];
		if (FileControl(sFullFileName, nAppendDel) == ERROR) return ERROR;
	}

	LOG("FileControl PASS !!!");
	
	switch(m_isFileInfo)
	{
		case	TAR_GZ:
			sCommand = "cd "+m_sCwd+"; " + "gzip " + m_Dir + GetTarFileName();
			if (CommandExecute(sCommand) == ERROR) return ERROR;
			break;
		case 	TAR_BZ:
			sCommand = "cd "+m_sCwd+"; " + "bzip2 " + m_Dir + GetTarFileName();
			if (CommandExecute(sCommand) == ERROR) return ERROR;
			break;
	}
	SetCounter(90);	
	return SUCCESS;
}

  
///	\brief	파일 하나를 압축한다.
///	\param	pFile		압축할 파일명
///	\param	nAppendDel	TAR_DELETE 	:	파일 지움\n
///						TAR_APPEND	:	파일 추가
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int
TargzUtil::Compress(MLS::File* pFile, int nAppendDel)
{
	string	sCommand = "";
	
	SetCounter(20);
	
	if (GetFileInfo() == ERROR) return ERROR;

	string sInTarget = "cd " + m_sCwd + "; ";
	
	// 우선 파일이 있는지 확인. 있으면 압축을 푼다.
	if (IsFileCreate(strutil::addslash(m_Filename)) == false)
	{
		// tar.gz 파일은 압축을 풀고 Append나 Delete를 해야 한다.
		switch(m_isFileInfo)
		{
			case	TAR_GZ:
				sCommand = sInTarget + "gunzip " + m_Dir + strutil::addslash(m_Filename);
				if (CommandExecute(sCommand) == ERROR) return ERROR;
				break;
			case 	TAR_BZ:
				sCommand = sInTarget + "bunzip2 " + m_Dir + strutil::addslash(m_Filename);
				if (CommandExecute(sCommand) == ERROR) return ERROR;
				break;
		}
	}
	SetCounter(50);
	if (FileControl(pFile->FullName, nAppendDel) == ERROR) return ERROR;
	SetCounter(70);
	
	switch(m_isFileInfo)
	{
		case	TAR_GZ:
			sCommand = sInTarget + "gzip " + m_Dir + GetTarFileName();
			if (CommandExecute(sCommand) == ERROR) return ERROR;
			break;
		case 	TAR_BZ:
			sCommand = sInTarget + "bzip2 " + m_Dir + GetTarFileName();
			if (CommandExecute(sCommand) == ERROR) return ERROR;
			break;
	}
	SetCounter(90);
	return SUCCESS;
}

///	\brief	directory를 압축한다.
///	\param	s_Dir	압축할 디렉토리
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int
TargzUtil::Compress(const string& s_Dir)
{
	string	sCommand = "";

	if (access(s_Dir.c_str(), R_OK) == -1)
    {
        throw Exception(strerror(errno));
		return ERROR;
    }

	string sInTarget = "cd "+m_sCwd+"; ";
	
	SetCounter(20);
	
	if (GetFileInfo() == ERROR) return ERROR;
	
	// 우선 파일이 있는지 확인. 있으면 압축을 푼다.
	if (IsFileCreate(strutil::addslash(m_Filename)) == false)
	{
		// tar.gz 파일은 압축을 풀고 Append나 Delete를 해야 한다. 
		switch(m_isFileInfo)
		{
			case	TAR_GZ:
				sCommand = sInTarget + "gunzip -c " + m_Dir + strutil::addslash(m_Filename) + " > " + strutil::addslash(m_Filename);
				if (CommandExecute(sCommand) == ERROR) return ERROR;
				break;
			case 	TAR_BZ:
				sCommand = sInTarget + "bunzip2 " + m_Dir + strutil::addslash(m_Filename);
				if (CommandExecute(sCommand) == ERROR) return ERROR;
				break;
		}
		
		sCommand = sInTarget + m_TarCommand + " rf " + m_Dir + GetTarFileName() + " " + s_Dir; // Append Directory
	}
	else
	{
		sCommand = sInTarget + m_TarCommand + " cf " + m_Dir + GetTarFileName() + " " + s_Dir; // create
	}
	
	SetCounter(50);
	if (CommandExecute(sCommand) == ERROR) return ERROR;
	
	SetCounter(70);
	
	switch(m_isFileInfo)
	{
		case	TAR_GZ:
			sCommand = sInTarget + "gzip " + m_Dir + GetTarFileName();
			if (CommandExecute(sCommand) == ERROR) return ERROR;
			break;
		case 	TAR_BZ:
			sCommand = sInTarget + "bzip2 " + m_Dir + GetTarFileName();
			if (CommandExecute(sCommand) == ERROR) return ERROR;
			break;
	}
	
	SetCounter(90);
	return SUCCESS;
}

///	\brief	압축을 푼다.
///	\param	s_Dir	압축풀 디렉토리
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int
TargzUtil::Uncompress(const string& s_Dir)
{
	string	sCommand = "";
	string	sZip = "";
	
	if (access(s_Dir.c_str(), W_OK) == -1)
    {
        throw Exception(strerror(errno));
		return ERROR;
    }

	SetCounter(20);

	if (GetFileInfo() == ERROR) return ERROR;

	// 우선 파일이 있는지 확인. 있으면 압축을 푼다.
	if (IsFileCreate(strutil::addslash(m_Filename)) == false)
	{
		// tar.gz 파일은 압축을 풀고 Append나 Delete를 해야 한다. 
		SetCounter(50);

		string sInTarget = "cd "+s_Dir+"; ";
		
		switch(m_isFileInfo)
		{
			case BZ:
				sCommand = sInTarget + "bunzip2 " + m_Dir + strutil::addslash(m_Filename);
				break;
			case GZ:
				sCommand = sInTarget + "gunzip " + m_Dir + strutil::addslash(m_Filename);
				break;
			case TAR:
				sCommand = sInTarget + m_TarCommand + " xf " + m_Dir + strutil::addslash(m_Filename)
							+ " -C " + s_Dir; // extract Directory
				break;
			case TAR_BZ:
				sCommand = sInTarget + m_TarCommand + " xfj " + m_Dir + strutil::addslash(m_Filename)
							+ " -C " + s_Dir; // extract Directory
				 break;
			case TAR_GZ:
				sCommand = sInTarget + m_TarCommand + " xfz " + m_Dir + strutil::addslash(m_Filename)
							+ " -C " + s_Dir; // extract Directory
				break;
			case ZIP:
				sCommand = sInTarget + "unzip -F -o -qq" + sZip + " " + m_Dir + strutil::addslash(m_Filename)
							+ " -d " + s_Dir; // extract Directory
				break;
			case RPM:
				sCommand = sInTarget + "rpm2cpio " + m_Dir + strutil::addslash(m_Filename) +
							" | cpio -iumd --quiet ";  // extract file
				break;
			case DEB:
					//dpkg-deb --fsys-tarfile $archive | tar xOf - $filename ./$filename
				sCommand = sInTarget + "dpkg-deb --fsys-tarfile " + m_Dir + strutil::addslash(m_Filename) +
								" | tar xf - -C " + s_Dir;  // extract file
				break;
			case ALZ:
				sCommand = sInTarget + "unalz " + m_Dir + strutil::addslash(m_Filename); // extract file
		}

		SetCounter(70);
		
		LOG("Command :: [%s]", sCommand.c_str());
		
		if (CommandExecute(sCommand) == ERROR) return ERROR;
	}
	else
	{
		throw Exception(gettext("Error :: '") + m_Dir + m_Filename + gettext("' File not found"));
		return ERROR;
	}
	
	SetCounter(90);
	return SUCCESS;
}

///	\brief	해당되는 파일의 압축을 푼다.
///	\param	t_FileList	압축풀 파일 리스트 
///	\param	sTargetDir	압풀을 폴 장소(Directory)
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int
TargzUtil::Uncompress(vector<MLS::File*>& t_FileList, const string& sTargetDir)
{
	string	sCommand = "";
	string	sZip = "";

	if (access(sTargetDir.c_str(), W_OK) == -1)
    {
        throw Exception(strerror(errno));
		return ERROR;
    }

	SetCounter(20);
	
	if (GetFileInfo() == ERROR) return ERROR;
	
	LOG("Uncompress [%s]", sTargetDir.c_str());
	
	// 우선 파일이 있는지 확인. 있으면 압축을 푼다.
	if (IsFileCreate(strutil::addslash(m_Filename)) == false)
	{
		LOG("IsFileCreate :: false");

		// tar.gz 파일은 압축을 풀고 Append나 Delete를 해야 한다.
		SetCounter(50);
		
		string::size_type	tSizeType;
		File*	pFile;

		string sInTarget = "cd "+sTargetDir+"; ";
		
		for (uint nCount = 0; nCount < (uint)t_FileList.size(); nCount++)
		{
			pFile = t_FileList[nCount];
			
			switch(m_isFileInfo)
			{
				case BZ:
					sCommand = sInTarget + "bunzip2 " + m_Dir + strutil::addslash(m_Filename);
					break;
				case GZ:
					sCommand = sInTarget + "gunzip -c " + m_Dir + strutil::addslash(m_Filename) +
							" > " + pFile->TarFullName + " 2> /dev/null";
					break;
				case TAR:
					sCommand = 	sInTarget +  m_TarCommand + " xf " + m_Dir + strutil::addslash(m_Filename) +
							" -C " + sTargetDir + " " + pFile->TarFullName; // extract file
					break;
				case TAR_BZ:
					sCommand = 	sInTarget +  m_TarCommand + " xfj " + m_Dir + strutil::addslash(m_Filename) +
							" -C " + sTargetDir + " " + pFile->TarFullName; // extract file
					break;
				case TAR_GZ:
					sCommand = 	sInTarget +  m_TarCommand + " xfz " + m_Dir + strutil::addslash(m_Filename) +
							" -C " + sTargetDir + " " + pFile->TarFullName; // extract file
					break;
				case ZIP:
					sCommand = 	sInTarget +  "unzip -F -o -qq " + m_Dir + strutil::addslash(m_Filename) +
							" " + strutil::addslash(pFile->TarFullName) + " -d " + sTargetDir;  // extract file
					break;
				case RPM:
					sCommand =  sInTarget +  "rpm2cpio " + m_Dir + strutil::addslash(m_Filename) +
							" | cpio -iumd --quiet " + strutil::addslash(pFile->TarFullName);  // extract file
					break;
				case DEB:
					//dpkg-deb --fsys-tarfile $archive | tar xOf - $filename ./$filename
					sCommand =  sInTarget +  "dpkg-deb --fsys-tarfile " + m_Dir +
							strutil::addslash(m_Filename) + " | tar xf - -C " + sTargetDir +
							" " + strutil::addslash(pFile->TarFullName);  // extract file
					break;
				case ALZ:
					sCommand = 	sInTarget +  "unalz " + m_Dir + strutil::addslash(m_Filename) +
								" \"" + strutil::addslash(pFile->TarFullName) + "\"";  // extract file
			}

			LOG("Command :: [%d]", m_isFileInfo);
			LOG("Command :: [%s]", sCommand.c_str());
						
			// 에러무시
			
			if (CommandExecute(sCommand) == ERROR) continue;
		}
		
		SetCounter(70);
	}
	else
	{
		throw Exception(gettext("Error :: '") + m_Dir + m_Filename + gettext("' File not found"));
		return ERROR;
	}
	SetCounter(90);
	return SUCCESS;
}

///	\brief	압축을 푼다.
///	\param	t_FileList	File list (String 방식)
///	\param	sTargetDir	압풀을 폴 장소(Directory)
///	\return	SUCCESS	:	성공\n
///			ERROR	:	실패
int
TargzUtil::Uncompress(vector<string>& t_FileList, const string& sTargetDir)
{
	string	sCommand = "";
	string	sZip = "";
	
	if (access(sTargetDir.c_str(), W_OK) == -1)
    {
        throw Exception(strerror(errno));
		return ERROR;
    }
	
	SetCounter(20);

	LOG("Uncompress 1 [%s]", sTargetDir.c_str());

	if (GetFileInfo() == ERROR) return ERROR;

	LOG("Uncompress 2 Dir [%s] Filename [%s]", m_Dir.c_str(), m_Filename.c_str());
	
	// 우선 파일이 있는지 확인. 있으면 압축을 푼다.
	if (IsFileCreate(strutil::addslash(m_Filename)) == false)
	{
		// tar.gz 파일은 압축을 풀고 Append나 Delete를 해야 한다. 
		SetCounter(50);
		
		string::size_type	tSizeType;
		string	sFullFileName;

		string sInTarget = "cd "+sTargetDir+"; ";
		
		for (uint nCount = 0; nCount < (uint)t_FileList.size(); nCount++)
		{
			sFullFileName = t_FileList[nCount];

			LOG("Uncompress 2 [%s]", sTargetDir.c_str());

			switch(m_isFileInfo)
			{
				case BZ:
					sCommand = sInTarget + "bunzip2 " + m_Dir + strutil::addslash(m_Filename);
					break;
				case GZ:
					sCommand = sInTarget + "gunzip -c " + m_Dir + strutil::addslash(m_Filename) +
								" > " + sFullFileName + " 2> /dev/null";
					break;
				case TAR:
					sCommand = sInTarget + m_TarCommand + " xfj " + m_Dir + strutil::addslash(m_Filename) +
								" -C " + sTargetDir + " " + sFullFileName; // extract file
					break;
				case TAR_BZ:
					sCommand = sInTarget + m_TarCommand + " xfj " + m_Dir + strutil::addslash(m_Filename) +
								" -C " + sTargetDir + " " + sFullFileName; // extract file
					break;
				case TAR_GZ:
					sCommand = sInTarget + m_TarCommand + " xfz " + m_Dir + strutil::addslash(m_Filename) +
								" -C " + sTargetDir + " " + sFullFileName; // extract file
					break;
				case ZIP:
					sCommand = sInTarget + "unzip -F -o -qq " + m_Dir + strutil::addslash(m_Filename) +
								" " + sFullFileName + " -d " + sTargetDir;  // extract file
					break;
				case RPM:
					sCommand = sInTarget + "rpm2cpio " + m_Dir + strutil::addslash(m_Filename) +
								" | cpio -iumd --quiet " + sFullFileName;  // extract file
					break;
				case DEB:
					//dpkg-deb --fsys-tarfile $archive | tar xOf - $filename ./$filename
					sCommand = sInTarget + "dpkg-deb --fsys-tarfile " + m_Dir + strutil::addslash(m_Filename) +
								" | tar xf - -C " + sTargetDir + " " + sFullFileName;  // extract file
					break;
				case ALZ:
					sCommand = sInTarget + "unalz " + m_Dir + strutil::addslash(m_Filename) +
								" \"" + strutil::addslash(sFullFileName) + "\"";  // extract file
					break;
					
				default:
					LOG("ERROR FILEINFO");
					break;
			}
			
			LOG("COMMAND :: [%s]", sCommand.c_str());
						
			// 에러무시
			if (CommandExecute(sCommand) == ERROR) continue;
		}
		
		SetCounter(70);
	}
	else
	{
		LOG("COMMAND :: %s", sCommand.c_str());
		throw Exception(gettext("Error :: '") + m_Dir + m_Filename + gettext("' File not found"));
		return ERROR;
	}
	
	SetCounter(90);
	return SUCCESS;
}

string
TargzUtil::DetectTarCommand() const
{
	// In FreeBSD, there is bsd tar(bsdtar) and gnu tar(gtar).
	// Default is bsdtar.
	// Need consider m_TarDir ?

	#ifndef TARNAME
		FILE* const tarVersionCommand = popen("tar --version", "r");
		if(tarVersionCommand != NULL) {
	
			char tarVersion[7];
			memset(tarVersion, 0, 7);
	
			rewind(tarVersionCommand);
			if(fgets(tarVersion, sizeof(tarVersion), tarVersionCommand) != NULL) {
	
				if(strncmp(tarVersion, "bsdtar", sizeof(tarVersion)) == 0) {
					pclose(tarVersionCommand);
					// Need check existance of gtar ?
					return "gtar";
				}
	
			}
			pclose(tarVersionCommand);
	
			return "tar";
	
		} else
			return "tar";
	#else
		return TARNAME;
	#endif
}

