/***************************************************************************
 *   Copyright (C) 2005 by Byoungyoung, La                                 *
 *   la9527@yahoo.co.kr                                                    *
 ***************************************************************************/

#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/time.h>
#include <stdlib.h>

#include "strutil.h"
#include "colorset.h"
#include "editor.h"
#include "editorcmd.h"
#include "strlinetoken.h"

using namespace std;
using namespace strutil;

namespace MLS
{
Editor*	g_pEditor;

Editor::Editor(const string& sTitle, int height, int width)
	: _title(sTitle), _height(height), _width(width)
{
	_editorwin = NULL;
	_nTabSize = 8;
	_sFile = "";
	_bLineNumView = false;
	_bInsert = true;
	_EditMode = EDIT;
	_nCurCulumn_Max = 0;
	_bReadOnly = false;
	_bDosMode = false;
	_bIndentMode = true;
	_bKeyCfgLoad = false;

	int len = krstrlen(_title);
	if (_width < len) _width = len;
	g_pEditorCmd = new EditorCmd;
}

Editor::~Editor()
{
	if(g_pEditorCmd) delete g_pEditorCmd; g_pEditorCmd = NULL;	
}

void Editor::Init()
{
	g_pEditorCmd->Init(g_pEditor);
}

bool Editor::KeyLoad(const string& sKeyFile)
{
	if (_tEditKeyBind.Load(sKeyFile) == false)
	{
		_bKeyCfgLoad = false;
		return false;
	}

	_bKeyCfgLoad = true;
	return true;
}

void Editor::SetEditor(int nTabSize, bool bBackup, bool bLineNumView)
{
	_nTabSize = nTabSize;
	_bBackup = bBackup;
	_bLineNumView = bLineNumView;
}

void Editor::New(const string& sFile)
{
	_sFile = sFile;
	_vText.clear();
	_eEncode = e_nCurLang;
	_nFirstLine = 0;
	_nCurLine = 0;
	_nCurCulumn = 0;
	_EditMode = EDIT;
	_nCurCulumn_Max = 0;
	_bInsert = true;
	_nConvInfo = 0;

	// find 초기화
	_sFindStr = L"";
	_nFindPosX = 0;
	_nFindPosY = 0;
	
	for (int n = 0; n<_vDoInfo.size();n++)
	{
		DoInfo*	pDoInfo = _vDoInfo[n];
		delete pDoInfo;
	}
	_vDoInfo.clear();
	_vText.push_back(L" ");
}

bool Editor::Load(const string& sFile, bool bReadOnly)
{
	ifstream in(sFile.c_str());
	if (!in) return false;
	if (!in.good()) return false;

	string	line, var, val;
	string	sConvert, sConvert2, sTab;
	wstring	wConv;
	int		nSize = 0;
	
	sTab.append(1, ' ');
	sTab.append(_nTabSize-1, '\t');

	int 	nUS = 0, nKO_EUCKR = 0, nKO_UTF8 = 0;

	// 새로운 파일 여는 것과 같다.
	New(sFile);
	ENCODING	eEncode;

	_bReadOnly = bReadOnly;
	
	do
	{
		getline(in, line);
		
		if (line == "")
		{
			if (in.eof()) break;
			continue;
		}

		eEncode = AUTO;
		sConvert = isKorCode(line, &eEncode);
		if (eEncode == US) nUS++;
		else if (eEncode == KO_EUCKR) nKO_EUCKR++;
		else if (eEncode == KO_UTF8) nKO_UTF8++;
	}
	while(!in.eof());
	in.close();

	LOG("US [%d] EUCKR [%d] UTF8 [%d]", nUS, nKO_EUCKR, nKO_UTF8);
	if (nKO_EUCKR > nKO_UTF8)
		_eEncode = KO_EUCKR;
	else if (nKO_EUCKR < nKO_UTF8)
		_eEncode = KO_UTF8;
	else
		_eEncode = US;
	
	_vText.clear();

	ifstream in2(sFile.c_str());
	if (!in2) return false;
	if (!in2.good()) return false;
	line = "";
	bool bNotSave = false;
	bool bDosMode = false;
	do
	{
		getline(in2, line);
		if (line == "") {
			if (in2.eof()) break;
			_vText.push_back(L"");
			continue;
		}
		
		if (e_nCurLang == KO_UTF8) {
			if (_eEncode == KO_EUCKR) {
				sConvert = CodeConvert(line, "UTF-8", "EUC-KR"); // utf-8 로 바꿈.
				if (sConvert == "")	sConvert = line;
			}
			else sConvert = line;
		}
		else if (e_nCurLang == KO_EUCKR) {
			if (_eEncode == KO_UTF8) {
				sConvert = CodeConvert(line, "EUC-KR", "UTF-8"); // euc-kr 로 바꿈.
				if (sConvert == "")	sConvert = line;
			}
			else sConvert = line;
		}
		else 
		{
			// US 로 되는데, 이때는 utf-8로 바꾸는 것과 동일하게 한다. (CTYPE : en_US.utf-8)
			if (_eEncode == KO_EUCKR) {
				sConvert = CodeConvert(line, "UTF-8", "EUC-KR"); // utf-8 로 바꿈.
				if (sConvert == "")	sConvert = line;
			}
			else sConvert = line;
		}

		sConvert = Replace(sConvert, "\t", sTab.c_str());
		sConvert2 = Replace(sConvert, "\r", "");
		if (sConvert2.size() != sConvert.size()) bDosMode = true;
		wConv = strtowstr(sConvert2);
		if (sConvert2.size() != 0 && wConv.size() == 0) bNotSave = true;
		_vText.push_back(wConv);
	}
	while(!in2.eof());
	in2.close();

	if (_vText.size() == 0) _vText.push_back(L"");

	if (bDosMode == true) _bDosMode = true;

	if (bNotSave == true)
	{
		MsgBox(gettext("Error"), gettext("This document locale error. current file read only."));
		_bReadOnly = true;
	}
	return true;
}

bool	Editor::Save(const string& sFile, ENCODING	Encode, bool bDosMode, bool bBackup)
{
	string	sTmpFile;
	_sFile = sFile;
	sTmpFile = "." + sFile + ".tmp";
	ofstream out(sTmpFile.c_str());
	if (!out)
	{
		MsgBox(gettext("Error"), strerror(errno));
		return false;
	}

	string			sSrc, sConvert;
	ENCODING		eEncode;
	
	if (Encode == AUTO)
		eEncode = _eEncode;
	else
		eEncode = Encode;

	string	sTab;
	sTab.append(1, ' ');
	sTab.append(_nTabSize-1, '\t');
		
	for (uint n = 0; n < _vText.size(); n++)
	{
		sSrc = wstrtostr(_vText[n]);
		sSrc = Replace(sSrc, sTab.c_str(), "\t");
		if (bDosMode)	sSrc = sSrc + "\r";

		if (sSrc.size() == 0 && _vText[n].size() > 0)
		{
			LOG("sSrc [%s]", sSrc.c_str());
			return false;
		}
		
		if (eEncode == KO_EUCKR)
		{
			sConvert = "";
			if (e_nCurLang == KO_UTF8) // euckr로 바꿈.
				sConvert = CodeConvert(sSrc, "EUC-KR", "UTF-8");
			if (e_nCurLang == KO_EUCKR)
				sConvert = sSrc;
			if (e_nCurLang == US)
				sConvert = CodeConvert(sSrc, "EUC-KR", "UTF-8");
			if (sConvert == "") out << sSrc << endl;
			else				out << sConvert << endl;
		}
		else if (eEncode == KO_UTF8)
		{
			sConvert = "";
			if (e_nCurLang == KO_EUCKR) // utf-8 로 바꿈.
				sConvert = CodeConvert(sSrc, "UTF-8", "EUC-KR");
			if (e_nCurLang == KO_UTF8)
				sConvert = sSrc;
			if (e_nCurLang == US)
				sConvert = sSrc;
			if (sConvert == "") out << sSrc << endl;
			else				out << sConvert << endl;
		}
		else
		{
			out << sSrc << endl;
		}
	}
	out.close();

	if (bDosMode)	_bDosMode = true;
	else			_bDosMode = false;

	struct stat	t_stat;
	if (stat(sFile.c_str(), &t_stat) == -1)
	{
		t_stat.st_mode = 0644;
	}

	if (bBackup == true)
	{
		string sBackFile = "." + sFile + ".bak";
		if (rename(sFile.c_str(), sBackFile.c_str()) == -1 )
		{
			if (errno != 2) // 파일이 존재하지 않을 경우 그냥 넘긴다.(새로운 파일일 경우)
			{
				MsgBox(COLOR_RED, gettext("Error"), "%s (%d)", strerror(errno), errno);
				return false;
			}
		}
	}
	
	if (rename(sTmpFile.c_str(), sFile.c_str()) == -1)
	{
		MsgBox(COLOR_RED, gettext("Error"), "%s (%d)", strerror(errno), errno);
		return false;
	}
	
	if (chmod(sFile.c_str(), t_stat.st_mode) == -1)
	{
		MsgBox(COLOR_RED, gettext("Error"), "%s (%d)", strerror(errno), errno);
	}
	_eEncode = eEncode;
	return true;
}

void	Editor::DrawEtc()
{
	wbkgd(_editorwin, COLOR(g_Color.MEdit.font, g_Color.MEdit.back));

	string	str;

	setcol(g_Color.Func, _editorwin);
	wmove(_editorwin, 0,0);
	whline(_editorwin, ' ', _width);
	// Funtion panel draw
	for (int i=0; i<8; i++)
	{
		str = _tEditKeyBind.GetHelpCmd(_tEditKeyBind.GetFuncKey(i+2));
		int pos = GetColumnIndex(_width, 8, i);
		if (str != "")
			mvwprintw(_editorwin, 0, 2+pos, "%s", gettext(str.c_str()));
	}

	setcol(g_Color.FuncA, _editorwin);
	mvwprintw (_editorwin, 0, 0, "F");
	for (int i=0; i<8; i++)
	{
		int pos = GetColumnIndex(_width, 8, i);
		mvwprintw(_editorwin, 0, 1+pos, "%d", i+2);
	}

	setcol(COLOR_BLACK, g_Color.Func.back, _editorwin);
	for (int i=0; i<8;i++)
	{
		int pos = GetColumnIndex(_width, 8, i);
		mvwaddch(_editorwin, 0, pos, VLINE);
	}

	// title 출력
	setcol(g_Color.Stat, _editorwin);
	wmove(_editorwin, 1, 0);
	whline(_editorwin, ' ', _width);
	mvwprintw(_editorwin, 1, (_width - krstrlen(_title))/2 , "%s", _title.c_str());
}

void Editor::ScreenMemSave(int nLine, int nCulumn)
{
	wstring 		sViewWStr, sLineWStr;
	StrLineToken	tStrLineToken;

	if (_nCurLine > 0)
	{
		if (_nCurLine >= _vText.size()) _nCurLine = _vText.size()-1;
		if (_nCurLine <= _nFirstLine) _nFirstLine = _nFirstLine-1;
		if (_nFirstLine <= 0) _nFirstLine = 0;
		if (_nLastLine-_nFirstLine >= 10 && _nLastLine-_nCurLine <= 0)
		{
			if (_vViewString.size() >= nLine)
				if (_nFirstLine <= _vText.size())
					_nFirstLine = _nFirstLine+1;

			if (_vText.size() <= _nLine-5)
				_nFirstLine = 0;
		}
	}

	LOG("_FirstLine [%d] [%d]", _nCurLine, _nFirstLine);

	for(;;)
	{
		int nViewLine = _nFirstLine;
		
		if (nViewLine < 0) return;
		
		_vViewString.clear();
	
		LineInfo	tLineInfo;
		bool 		bNext = false;
	
		for (int t=0; t< nLine; t++)
		{
			if (!tStrLineToken.NextChk())
			{
				if (nViewLine >= _vText.size()) break;
				sLineWStr = _vText[nViewLine];
				tStrLineToken.SetWString(sLineWStr, nCulumn);
				nViewLine++;
			}

			if (tStrLineToken.LineSize()-1 == tStrLineToken.GetCurNum())
				bNext = false;
			else
				bNext = true;
			
			sViewWStr = tStrLineToken.Get();
				
			tLineInfo.nViewLine = t;
			tLineInfo.nTextLine = nViewLine-1;
			tLineInfo.sWString = sViewWStr;
			tLineInfo.bNext = bNext;
			tLineInfo.nNextLineNum = tStrLineToken.GetCurNum();
			_vViewString.push_back(tLineInfo);
			tStrLineToken.Next();
		}
		_nLastLine = nViewLine-1;
		
		if (_vViewString.size() > nLine-3)
		{
			if (_nLastLine == _nCurLine && tLineInfo.bNext == true)
			{
				_nFirstLine++;
				continue;
			}
			if (_nLastLine < _nCurLine)
			{		
				_nFirstLine++;
				continue;
			}
		}
		break;
	}	
	return;
}

void Editor::SelectionDraw(const wstring& sViewWStr, int nY, int nN, int x1, int x2)
{
	string	sViewStr, sViewStr2, sViewStr3;
	wstring sWStr1, sWStr2, sWStr3;

	if (sViewWStr.size() == 0) return;

	if (x1 >= sViewWStr.size()) x1 = sViewWStr.size();
	if (x2 >= sViewWStr.size()) x2 = sViewWStr.size();
	
	sWStr1 = sViewWStr.substr(0, x1);
	sWStr2 = sViewWStr.substr(x1, x2-x1);
	sWStr3 = sViewWStr.substr(x2, sViewWStr.size());

	if (sWStr1.size() != 0)
	{
		sViewStr = wstrtostr(sWStr1);
		sViewStr = Replace(sViewStr, "\t", " "); // Tab -> Space
	
		setcol(g_Color.MEdit, _editorwin);
		mvwprintw(_editorwin, nY, nN, "%s", sViewStr.c_str());
	}
	if (sWStr2.size() != 0)
	{
		sViewStr2 = wstrtostr(sWStr2);
		sViewStr2 = Replace(sViewStr2, "\t", " "); // Tab -> Space
	
		setrcol(g_Color.MEdit, _editorwin);// 반전
		mvwprintw(_editorwin, nY, nN+wstrlen(sWStr1), "%s", sViewStr2.c_str());
	}
	if (sWStr3.size() != 0)
	{
		sViewStr3 = wstrtostr(sWStr3);
		sViewStr3 = Replace(sViewStr3, "\t", " "); // Tab -> Space
	
		setcol(g_Color.MEdit, _editorwin);
		mvwprintw(_editorwin, nY, nN+wstrlen(sWStr1)+wstrlen(sWStr2), "%s", sViewStr3.c_str());
	}
}

void Editor::LineDraw(	const wstring&	sViewWStr,
						int nY, 
						int nN, 
						int nTextNum,
						int nNum)
{
	setcol(g_Color.MEdit, _editorwin);
	wmove(_editorwin, nY, nN);
	whline(_editorwin, ' ', _width);
	
	if (_bLineNumView == true)
	{
		char 	sFormat[20];
		sprintf(sFormat, "%%%dd", _nLineWidth);
		setcol(g_Color.MEditInfo, _editorwin);
		mvwprintw(_editorwin, nY, nN+1, sFormat, nTextNum+1);
		setcol(g_Color.MEdit, _editorwin);
		mvwhline(_editorwin, nY, nN+_nLineWidth+1, VLINE, 1);
		nN = 3+_nLineWidth+nN;
	}

	if (_EditMode == EDIT)
	{
		string sViewStr = wstrtostr(sViewWStr);
		string sViewStr2 = Replace(sViewStr, "\t", " "); // Tab -> Space
		setcol(g_Color.MEdit, _editorwin);
		mvwprintw(_editorwin, nY, nN, "%s", sViewStr2.c_str());
	}
	else if (_EditMode == SELECT || _EditMode == SHIFT_SELECT)
	{
		EditSelect	tEditSelect = _EditSelect;
		SelectSort(&tEditSelect);

		int x1 = tEditSelect.x1;
		int x2 = tEditSelect.x2;

		wstring 		sLineWStr = _vText[nTextNum];
		wstring			sWStr;
		StrLineToken	tStrLineToken;
		tStrLineToken.SetWString(sLineWStr, _nCulumn);
		
		int				nCurSize = 0;
		for (int n = 0; n<tStrLineToken.LineSize(); n++)
		{
			sWStr = tStrLineToken.GetLineStr(n);
			if (n == nNum) break;
			nCurSize=nCurSize+sWStr.size();
		}

		int	nCurEndSize = nCurSize+sViewWStr.size();
		
		if (tEditSelect.x1 > nCurSize)	x1 = tEditSelect.x1 - nCurSize;
		if (tEditSelect.x2 > nCurSize)	x2 = tEditSelect.x2 - nCurSize;
		
		if (tEditSelect.y1 == nTextNum && tEditSelect.y2 == nTextNum)
		{
			if (tEditSelect.x1 >= nCurSize && tEditSelect.x2 <= nCurEndSize)
			{
				SelectionDraw(sViewWStr, nY, nN, x1, x2);
			}
			else if (tEditSelect.x1 >= nCurSize && tEditSelect.x2 > nCurEndSize)
			{
				SelectionDraw(sViewWStr, nY, nN, x1, sViewWStr.size());
			}
			else if (tEditSelect.x1 < nCurSize && tEditSelect.x2 > nCurSize)
			{
				SelectionDraw(sViewWStr, nY, nN, 0, x2);
			}
			else if (tEditSelect.x1 < nCurSize && tEditSelect.x2 > nCurEndSize)
			{
				string sViewStr = wstrtostr(sViewWStr);
				string sViewStr2 = Replace(sViewStr, "\t", " "); // Tab -> Space
				setrcol(g_Color.MEdit, _editorwin); // 반전
				mvwprintw(_editorwin, nY, nN, "%s", sViewStr2.c_str());
			}
			else
			{
				string sViewStr = wstrtostr(sViewWStr);
				string sViewStr2 = Replace(sViewStr, "\t", " "); // Tab -> Space
				setcol(g_Color.MEdit, _editorwin);
				mvwprintw(_editorwin, nY, nN, "%s", sViewStr2.c_str());
			}
		}
		else if (tEditSelect.y1 == nTextNum)
		{
			if (tEditSelect.x1 >= nCurSize)
			{
				SelectionDraw(sViewWStr, nY, nN, x1, sViewWStr.size());
				
				string sViewStr = wstrtostr(sViewWStr);
				string sViewStr2 = Replace(sViewStr, "\t", " "); // Tab -> Space
				wmove(_editorwin, nY, nN+krstrlen(sViewStr2));
				whline(_editorwin, ' ', _width-nN-krstrlen(sViewStr2));
			}
			else if (tEditSelect.x1 < nCurSize)
			{
				string sViewStr = wstrtostr(sViewWStr);
				string sViewStr2 = Replace(sViewStr, "\t", " "); // Tab -> Space
				setrcol(g_Color.MEdit, _editorwin); // 반전
				mvwprintw(_editorwin, nY, nN, "%s", sViewStr2.c_str());
			}
			else
			{
				string sViewStr = wstrtostr(sViewWStr);
				string sViewStr2 = Replace(sViewStr, "\t", " "); // Tab -> Space
				
				mvwprintw(_editorwin, nY, nN, "%s", sViewStr2.c_str());
			}
		}
		else if (tEditSelect.y2 == nTextNum)
		{
			if (tEditSelect.x2 > nCurSize)
			{
				SelectionDraw(sViewWStr, nY, nN, 0, x2);
			}
			else if (tEditSelect.x2 > nCurEndSize)
			{
				string sViewStr = wstrtostr(sViewWStr);
				string sViewStr2 = Replace(sViewStr, "\t", " "); // Tab -> Space
				setrcol(g_Color.MEdit, _editorwin); // 반전
				mvwprintw(_editorwin, nY, nN, "%s", sViewStr2.c_str());
			}
			else
			{
				string sViewStr = wstrtostr(sViewWStr);
				string sViewStr2 = Replace(sViewStr, "\t", " "); // Tab -> Space
				setcol(g_Color.MEdit, _editorwin);
				mvwprintw(_editorwin, nY, nN, "%s", sViewStr2.c_str());
			}
		}
		else if (tEditSelect.y1 < nTextNum && tEditSelect.y2 > nTextNum)
		{
			string sViewStr = wstrtostr(sViewWStr);
			string sViewStr2 = Replace(sViewStr, "\t", " "); // Tab -> Space
			setrcol(g_Color.MEdit, _editorwin); // 반전
			wmove(_editorwin, nY, nN);
			whline(_editorwin, ' ', _width);
			mvwprintw(_editorwin, nY, nN, "%s", sViewStr2.c_str());
		}
		else
		{
			string sViewStr = wstrtostr(sViewWStr);
			string sViewStr2 = Replace(sViewStr, "\t", " "); // Tab -> Space
			setcol(g_Color.MEdit, _editorwin);
			mvwprintw(_editorwin, nY, nN, "%s", sViewStr2.c_str());
		}
	}
}

void Editor::CurserDraw(int nY, int nN)
{
	LineInfo			tLineInfo;
	vector<LineInfo>	vLineInfo;

	for (uint n = 0; n < _vViewString.size(); n++)
	{
		tLineInfo = _vViewString[n];
		if (tLineInfo.nTextLine == _nCurLine) 
			vLineInfo.push_back(tLineInfo);
	}

	int		cur = _nCurCulumn;
	int		length = 0;
	int 	line1 = 0;
	
	for (uint n = 0; n < vLineInfo.size(); n++)
	{
		tLineInfo = vLineInfo[n];
		length = length + tLineInfo.sWString.size();
		if (length >= _nCurCulumn)
		{
			line1 = tLineInfo.nViewLine;
			break;
		}
		cur = cur - tLineInfo.sWString.size();
	}
	cur = wstrlen((tLineInfo.sWString.substr(0, cur)));
	wmove(_editorwin, nY+line1, nN+cur);
	LOG("CURSOR :: [%d] [%d] [%d] [%d]", line1, cur, _nCurCulumn, vLineInfo.size());
}

void Editor::Draw(bool bReDraw)
{
	if (bReDraw)	DrawEtc();

	uint 		nViewLine = 0;
	wstring 	sLineStr, sViewStr;
	string		sSpaceTab;
	string		sInfo1, sInfo2, sInfo3;
	char		sLineFormat[50];

	int nViewWidth, nKrStrSize;

	LineInfo	tLineInfo;
	
	for (int t=0; t< _nLine; t++)
	{
		if (_vViewString.size() > t)
		{
			tLineInfo = _vViewString[t];
			LineDraw(tLineInfo.sWString, 2+t, 0, tLineInfo.nTextLine, tLineInfo.nNextLineNum);
		}
		else
		{
			setcol(g_Color.MEdit, _editorwin);
			wmove(_editorwin, 2+t, 0);
			whline(_editorwin, ' ', _nCulumn);

			if (_bLineNumView == true)
			{
				setcol(COLOR_BLACK, g_Color.MEdit.back, _editorwin);
				mvwprintw(_editorwin, 2+t, 3+_nLineWidth, "~");
			}
			else
			{
				setcol(COLOR_BLACK, g_Color.MEdit.back, _editorwin);
				mvwprintw(_editorwin, 2+t, 0, "~");
			}
		}
		nViewLine++;
	}

	setcol(g_Color.MEdit, _editorwin);
	wmove(_editorwin, _height-1, 0);
	whline(_editorwin, ' ', _width);

	if (_bInsert)
		sInfo1 = gettext("[Ins]");
	else
		sInfo1 = gettext("[Ovr]");
		
	if (_EditMode == SELECT)
		sInfo1 = gettext("[Select]");

	if (_bDosMode)	sInfo2 = "[DOS]";

	if (_eEncode == KO_EUCKR)		sInfo3 = "[EUC-KR]";
	else if (_eEncode == KO_UTF8)	sInfo3 = "[UTF-8]";
	else							sInfo3 = "[US]";

	setcol(g_Color.MEditInfoA, _editorwin);
	mvwprintw(_editorwin, _height-1, _width-krstrlen(sInfo1), (char*)sInfo1.c_str());
	if (_bDosMode)
	{
		setcol(g_Color.MEditInfoA, _editorwin);
		mvwprintw(_editorwin, _height-1, _width-krstrlen(sInfo2)-krstrlen(sInfo1), (char*)sInfo2.c_str());
	}
	setcol(g_Color.MEditInfo, _editorwin);
	mvwprintw(_editorwin, _height-1, _width-krstrlen(sInfo3)-krstrlen(sInfo2)-krstrlen(sInfo1), (char*)sInfo3.c_str());

	setcol(g_Color.MEditInfo, _editorwin);
	snprintf(sLineFormat, sizeof(sLineFormat), "Line: %d Col: %d (%d%%) ",
				_nCurLine+1, _nCurCulumn, (int)(((float)(_nCurLine+1)/(float)_vText.size())*100));
	mvwprintw(_editorwin, _height-1,
			  _width-krstrlen(sLineFormat)-krstrlen(sInfo3)-krstrlen(sInfo2)-krstrlen(sInfo1),
			  sLineFormat);
	
	if (_bReadOnly == true)
	{
		setcol(g_Color.MEditInfoA, _editorwin);
		mvwprintw(_editorwin, _height-1, 10, gettext("[Read Only]"));
	}
	else
	{
		if (_nConvInfo != _vDoInfo.size())
		{
			setcol(g_Color.MEditInfoA, _editorwin);
			mvwprintw(_editorwin, _height-1, 10, gettext("[Change]"));
		}
	}
	
	//  커서 위치를 옮긴다.
	if (_bLineNumView == true)
		CurserDraw(2, 3+_nLineWidth);
	else
		CurserDraw(2, 0);

	redrawwin(_editorwin);
	wrefresh(_editorwin);
}

inline long DiffTime(timeval* tBefore, timeval* tAfter)
{
	char	cAftTime[10];
	char	cBefTime[10];
	
	string	sAftTime, sBefTime;
	long lDiffTime = (long)(tAfter->tv_sec - tBefore->tv_sec);

	LOG("DiffTime long [%ld] [%ld] [%ld]", lDiffTime, tAfter->tv_sec, tBefore->tv_sec);
	snprintf(cAftTime, sizeof(cAftTime), "%-6ld", tAfter->tv_usec);
	snprintf(cBefTime, sizeof(cBefTime), "%-6ld", tBefore->tv_usec);
	
	sAftTime = cAftTime;	sBefTime = cBefTime;

	if (sAftTime.size() - 6 > 0) sAftTime.append(sAftTime.size()-6, '0');
	if (sBefTime.size() - 6 > 0) sBefTime.append(sBefTime.size()-6, '0');

	lDiffTime = lDiffTime * 1000000;
	lDiffTime = (lDiffTime + atol(sAftTime.c_str())) - atol(sBefTime.c_str());
	return lDiffTime;
}

int Editor::Do()
{
	if (_editorwin == NULL)
	{
		_editorwin = newwin(_height, _width, ((g_nLINES-_height)/2), (g_nCOLS-_width)/2);
	}

	_shutdown = false;

	Refresh();

	timeval		bfTime, afTime;
	
	gettimeofday(&afTime, 0); time(&afTime.tv_sec);
	gettimeofday(&bfTime, 0); time(&bfTime.tv_sec);
	
	while(!_shutdown)
	{// 출력
		_nLine = _height - 3;

		_nLineWidth = itoa(_vText.size()).size();
		if (_bLineNumView == true)
		{
			_nCulumn = _width - (_nLineWidth+5);
		}
		else
			_nCulumn = _width - 2;

		//LOG("Check :: [%d] [%d] [%d]", _nLine, _nCulumn, _nLineWidth);
		ScreenMemSave(_nLine, _nCulumn);
		
		Draw();
		// 커서를 보이게 하고 마우스를 없앤다. 붙여 넣기 쉽게
		mousemask(0, NULL);
		if (_EditMode == EDIT) curs_set(1); // 에디트 모드일때만 커서를 켠다.
		
		int nKey=getch();
		LOG("KEY INPUT [%d]", nKey);

		curs_set(0); // 커서를 보이게 한다.
		mousemask( 	BUTTON1_CLICKED	| BUTTON2_CLICKED | BUTTON3_CLICKED |
				BUTTON_SHIFT | BUTTON_CTRL, NULL);
		
		if (nKey == ERR)
		{
			LOG("Editor Key Err !!!");
			continue;
		}

		if (nKey == KEY_RESIZE)
		{
			g_nLINES = LINES;	g_nCOLS = COLS;
			_height = LINES; 	_width = COLS;
			Refresh();
			continue;
		}

		if (nKey == KEY_MOUSE)
		{
			Key_Mouse();
			continue;
		}

		if (KEY_F(0) <= nKey && nKey <= KEY_F(12))
		{
			string command = _tEditKeyBind.GetFuncKey(nKey-KEY_F(0));
			LOG("Editor Func Key [%d] [%s]", nKey-KEY_F(0), command.c_str());
			g_pEditorCmd->Execute(command);
			continue;
		}
		
		// Execute
		if (nKey != 27)
		{
			string command = _tEditKeyBind.GetCommand(nKey);
			
			if (g_pEditorCmd->Execute(command) == ERROR)
			{
				if (_bReadOnly == true) continue;

				string sKey;
				sKey.append(1, (char)nKey);
				
				// ascii
				if (nKey < 0x80)
				{
					// Konsole 버그 제거 (글자 마다 입력되는 버그 제거)
					if (nKey != 8)
						InputData(sKey);
					else
						Key_BS();
				}
				else
				{
					nodelay(stdscr, TRUE);
					int nKey2 = getch();
					int nKey3 = getch();
					nodelay(stdscr, FALSE);

					// utf8 -> 3char, etc -> 2char 
					if (nKey2 != ERR) sKey.append(1, (char)nKey2);
					if (nKey3 != ERR) sKey.append(1, (char)nKey3);
					InputData(sKey);
					LOG("KR Input :: [%d][%d][%d] [%s]", nKey, nKey2, nKey3, sKey.c_str());
				}
				continue;
			}
			LOG("Key [%d]", nKey);
		}
		else
		{
			// Alt + 문자는 이렇게 받는다.
			nodelay(stdscr, TRUE);
			int nKey2 = getch();
			int nKey3 = getch();
			int nKey4 = getch();
			int nKey5 = getch();
			nodelay(stdscr, FALSE);
			LOG("Input :: [%d][%d][%d][%d][%d]", nKey, nKey2, nKey3, nKey4, nKey5);

			if (nKey2 != ERR)
			{
				if (nKey3 == ERR)
					g_pEditorCmd->Execute(_tEditKeyBind.GetMetaCommand(nKey2));
				else
				{
					if (nKey2 == 79 && nKey3 == 50)	{
						// 쉬프트 처리 --> 각 콘솔마다 다름.. Gnome Console 기준
						switch(nKey4)
						{
							case 68: g_pEditorCmd->Execute("<shift_left>"); break;
							case 67: g_pEditorCmd->Execute("<shift_right>"); break;
							case 65: g_pEditorCmd->Execute("<shift_up>"); break;
							case 66: g_pEditorCmd->Execute("<shift_down>"); break;
						}	
					}
					// 리모트 접속 End, Home 키
					// 27, 79, 72 (linux) ::  27, 91, 72 (vt100)
					if (nKey2 == 91 || nKey2 == 79)
					{
						if (nKey3 == 72)	g_pEditorCmd->Execute("<home>");
						if (nKey3 == 70)	g_pEditorCmd->Execute("<end>");
						// Konsole. End Home 키
						if (nKey3 == 49 && nKey4 == 126) g_pEditorCmd->Execute("<home>");
						if (nKey3 == 52 && nKey4 == 126) g_pEditorCmd->Execute("<end>");
					}
					if (nKey3 == 91 && nKey4 >= 'A' && nKey4 <= 'E')  // Konsole 에서 F1~F5 키 (konsole)
					{
						string command = _tEditKeyBind.GetFuncKey(nKey4-'A'+1);
						g_pEditorCmd->Execute(command);
					}
					if (nKey3 == 49 && nKey4 >= 49 && nKey4 <= 52) // Konsole 에서 F1~F5 키 (xterm xfree 3.x)
					{
						string command = _tEditKeyBind.GetFuncKey(nKey4-49+1);
						g_pEditorCmd->Execute(command);
					}
					if (nKey2 == 79 && nKey3 >= 80 && nKey4 <= 83) // 일부 Remote 콘솔 F1~F4
					{
						string command = _tEditKeyBind.GetFuncKey(nKey3-80+1);
						g_pEditorCmd->Execute(command);
					}
				}
			}
			else
			{
				g_pEditorCmd->Execute(_tEditKeyBind.GetCommand(27));
			}
		}
	}

	for (int n = 0; n<_vDoInfo.size();n++)
	{
		DoInfo*	pDoInfo = _vDoInfo[n];
		delete pDoInfo;
	}
	_vDoInfo.clear();
	_bReadOnly = false;
	_bDosMode = false;
	if (_editorwin) delwin(_editorwin); _editorwin = NULL;
	return 0;
}

void	Editor::Refresh()
{
	g_nLINES = LINES; g_nCOLS = COLS;
	_height = LINES; _width = COLS;
	wresize(_editorwin, g_nLINES, g_nCOLS);
	mvwin(_editorwin, 0, 0);
	wclear(_editorwin);
	Draw();
	redrawwin(_editorwin);
	wrefresh(_editorwin);
}

void	Editor::InputData(const string& sKrStr)
{
	if (_bReadOnly == true) return;

	wstring sLine, sLine2, sChar;
	
	int				nKrStrSize;

	if (_EditMode != EDIT)
	{
		Selected_Del();
		_EditMode = EDIT;
	}
	
	if (_vText.size() > _nCurLine)
	{
		sLine = _vText[_nCurLine];
		// Undo를 위한 데이터 저장
		_vDoInfo.push_back(new DoInfo(_nCurLine, _nCurCulumn, sLine));
		
		sChar = strtowstr(sKrStr);

		if (_bInsert)
			sLine.insert(_nCurCulumn, sChar);
		else
			sLine.replace(_nCurCulumn, sChar.size(), sChar);

		_vText[_nCurLine] = sLine;
		_nCurCulumn = _nCurCulumn + sChar.size();
	}
	_nCurCulumn_Max = _nCurCulumn;
}

void	Editor::Key_Del()
{
	if (_bReadOnly == true) return;

	wstring sLine, sLine2;
	
	string			sTab;

	if (_EditMode != EDIT)
	{
		Selected_Del();
		_EditMode = EDIT;
		return;
	}

	sLine = _vText[_nCurLine];
	int nStrSize = sLine.size();

	if (_nCurCulumn < nStrSize)
	{
		// Undo를 위한 데이터 저장
		_vDoInfo.push_back(new DoInfo(_nCurLine, _nCurCulumn, sLine));

		sLine2 = sLine.substr(_nCurCulumn, _nTabSize);

		sTab.append(1, ' ');
		sTab.append(_nTabSize-1, '\t');

		if (wstrtostr(sLine2) == sTab)
		{
			sLine2 = sLine.substr(0, _nCurCulumn);
			sLine2 = sLine2 + sLine.substr(_nCurCulumn+_nTabSize, nStrSize-(_nCurCulumn+_nTabSize));
		}
		else
		{
			sLine2 = sLine.substr(0, _nCurCulumn);
			sLine2 = sLine2 + sLine.substr(_nCurCulumn+1, nStrSize-(_nCurCulumn+1));
		}

		_vText[_nCurLine] = sLine2;
	}
	else
	{
		if (_vText.size() > _nCurLine+1)
		{
			sLine2 = _vText[_nCurLine+1];
			// Undo를 위한 데이터 저장
			_vDoInfo.push_back(new DoInfo(_nCurLine, _nCurCulumn, sLine, sLine2));

			sLine2 = sLine + sLine2;
			_vText[_nCurLine] = sLine2;
			_vText.erase(_vText.begin()+_nCurLine+1);
		}
	}
	_nCurCulumn_Max = _nCurCulumn;
}

void	Editor::Key_BS()
{
	if (_bReadOnly == true) return;

	wstring sLine, sLine2;
	int				nStrSize;

	if (_EditMode != EDIT)
	{
		Selected_Del();
		_EditMode = EDIT;
		_nCurLine = _EditSelect.y1;
		_nCurCulumn = _EditSelect.x1;
		_nCurCulumn_Max = _nCurCulumn;
		return;
	}

	if (_vText.size() > _nCurLine)
	{
		sLine = _vText[_nCurLine];
		
		if (_nCurCulumn == 0 && _vText.size() > 0 && _nCurLine > 0)
		{
			sLine2 = _vText[_nCurLine-1];
			// Undo를 위한 데이터 저장
			_vDoInfo.push_back(new DoInfo(_nCurLine-1, _nCurCulumn, sLine2, sLine));
			
			nStrSize = sLine2.size();
			sLine2 = sLine2 + sLine;
			
			_vText[_nCurLine-1] = sLine2;
			_vText.erase(_vText.begin() + _nCurLine);
			
			Key_Up();
			_nCurCulumn = nStrSize;
		}
		else
		{
			nStrSize = _vText[_nCurLine].size();
			if (_nCurCulumn <= nStrSize)
			{
				// Undo를 위한 데이터 저장
				_vDoInfo.push_back(new DoInfo(_nCurLine, _nCurCulumn, sLine));

				wstring sTabCheck;
				string	sTab;
				if (_nCurCulumn >= _nTabSize)
				{
					sTabCheck = _vText[_nCurLine].substr(_nCurCulumn-_nTabSize, _nTabSize);
				}

				sTab.append(1, ' ');
				sTab.append(_nTabSize-1, '\t');
				if (wstrtostr(sTabCheck) == sTab)
				{
					sLine2 = _vText[_nCurLine].substr(0, _nCurCulumn-_nTabSize);
					sLine2 = sLine2 + _vText[_nCurLine].substr(_nCurCulumn, nStrSize-_nCurCulumn);
					_nCurCulumn = _nCurCulumn - (_nTabSize-1);
				}
				else
				{
					sLine2 = _vText[_nCurLine].substr(0, _nCurCulumn-1);
					sLine2 = sLine2 + _vText[_nCurLine].substr(_nCurCulumn, nStrSize-_nCurCulumn);
				}
			}
	
			_vText[_nCurLine] = sLine2;
			Key_Left();
		}
	}
	_nCurCulumn_Max = _nCurCulumn;
}

void	Editor::Key_Enter()
{
	if (_bReadOnly == true) return;

	wstring sLine, sLine2, sLine3;
	
	int				nStrlen;

	if (_EditMode != EDIT)
	{
		Selected_Del();
		_EditMode = EDIT;
	}

	string	p = wstrtostr(_vText[_nCurLine]);
	string	p1;
	int		nNew;
	int 	nOld = _nCurCulumn;

	if (_bIndentMode)
	{
		for (int n = 0; n < p.size(); n++)
		{
			if (p[n] != ' ' && p[n] != '\t') {
				p1 = p.substr(0, n);
				break;
			}
		}
	}

	if (_vText.size() > _nCurLine)
	{
		sLine = _vText[_nCurLine];
		// Undo를 위한 데이터 저장
		_vDoInfo.push_back(new DoInfo(_nCurLine, _nCurCulumn, 2, sLine));

		nStrlen = sLine.size();
		

		sLine2 = sLine2 + sLine.substr(0, _nCurCulumn);
		sLine3 = strtowstr(p1) + sLine3 + sLine.substr(_nCurCulumn, nStrlen-_nCurCulumn);

		_vText[_nCurLine] = sLine2;
		_vText.insert(_vText.begin()+_nCurLine+1, sLine3);
	}
	else
	{
		// Undo를 위한 데이터 저장
		_vDoInfo.push_back(new DoInfo(_nCurLine, _nCurCulumn, 2, sLine));

		_vText.push_back(strtowstr(p1));
		ScreenMemSave(_nLine, _nCulumn);
	}
	_nCurCulumn = p1.size();
	_nCurCulumn_Max = _nCurCulumn;
	Key_Down();
}


void	Editor::Key_Tab()
{
	if (_bReadOnly == true) return;

	string	sTab;
	sTab.append(1, ' ');
	sTab.append(_nTabSize-1, '\t');

	if (_EditMode != EDIT)
	{
		//Selected_Del();
		//_EditMode = EDIT;
		wstring			WStr;
		vector<wstring>	vSave;

		SelectSort(&_EditSelect);

		for (int y = _EditSelect.y1; y <= _EditSelect.y2; y++)
		{
			WStr = _vText[y];
			vSave.push_back(WStr);
		}
		
		// Undo를 위한 데이터 저장
		_vDoInfo.push_back(new DoInfo(_EditSelect.y1, 0, vSave, -1));

		for (int y = _EditSelect.y1; y <= _EditSelect.y2; y++)
		{
			WStr = _vText[y];
			WStr = strtowstr(sTab) + WStr;
			_vText[y] = WStr;
		}
		ScreenMemSave(_nLine, _nCulumn);
		return;
	}
	
	for (int n = 0; n<sTab.size();n++)
	{
		InputData(sTab.substr(n, 1));
		ScreenMemSave(_nLine, _nCulumn);
	}
}

void	Editor::Key_Untab()
{
	if (_bReadOnly == true) return;
	
	if (_EditMode != EDIT)
	{
		wstring			WStr;
		vector<wstring>	vSave;

		SelectSort(&_EditSelect);

		for (int y = _EditSelect.y1; y <= _EditSelect.y2; y++)
		{
			WStr = _vText[y];
			vSave.push_back(WStr);
		}
		
		// Undo를 위한 데이터 저장
		_vDoInfo.push_back(new DoInfo(_EditSelect.y1, 0, vSave, -1));

		string	sTab;
		sTab.append(1, ' ');
		sTab.append(_nTabSize-1, '\t');

		for (int y = _EditSelect.y1; y <= _EditSelect.y2; y++)
		{
			WStr = _vText[y];
			if (WStr.substr(0, strtowstr(sTab).size()) == strtowstr(sTab))
			{
				_vText[y] = WStr.substr(strtowstr(sTab).size());
			}
		}
		ScreenMemSave(_nLine, _nCulumn);
	}
}

void	Editor::GotoLine()
{
	string	sNumber;
	int		nLine;
	bool	bContinue = false;

	if (InputBox(gettext("go to line number."), sNumber) == -1) return;

	for (int n=0; n< sNumber.size(); n++)
	{
		if (isdigit((char)sNumber[n]) == 0)
		{
			MsgBox(1, gettext("Error"), gettext("not digit [%s]"), sNumber.c_str());
			return;
		}
	}

	if (sNumber.size() == 0) return;
	nLine = atoi(sNumber.c_str());

	if (nLine < _vText.size())
	{
		_nCurLine = nLine-1;
		if (_nCurLine <= 0) _nCurLine = 0;
		_nFirstLine = _nCurLine - 10;
		if (_nFirstLine <= 0) _nFirstLine = 0;
	}
	else
	{
		_nCurLine = _vText.size()-1;
		_nFirstLine = _nCurLine - 10;
	}
	_EditMode = EDIT;
}

void	Editor::GotoFirst()
{
	_nCurLine = 0;
	_nFirstLine = 0;
	_EditMode = EDIT;
}

void	Editor::GotoEnd()
{
	_nCurLine = _vText.size()-1;
	_nFirstLine = _nCurLine - 10;
	_EditMode = EDIT;
}

void	Editor::About()
{
	int width = 40;
	int height = 13;

	width +=4;

	WINDOW *win = newwin(height, width, (g_nLINES-height)/2, (g_nCOLS-width)/2);
	//wclear(win);
	wbkgd(win, COLOR(COLOR_WHITE, 3));

	wattron(win ,A_BOLD);
	wborder(win, VLINE, VLINE, HLINE, HLINE, ULCORNER, URCORNER, LLCORNER, LRCORNER);
	wattroff(win, A_BOLD);

	// title 출력
	wattron(win, COLOR(COLOR_BLACK, COLOR_WHITE));
	wmove(win, 1, 1);
	whline(win, ' ', width-2);
	string sTitle = gettext("About Mls Editor");
	mvwprintw(win, 1, (40 - krstrlen(sTitle))/2, (char*)sTitle.c_str());
	wattroff(win, A_BOLD);

	// msg
	wattron (win, COLOR(COLOR_WHITE, 3));
	wattron (win, A_BOLD);
	mvwprintw(win, 3, 2, gettext("Mls Editor %s - Simple File Editor"), "0.1");
	mvwprintw(win, 5, 2, gettext("Author : "));
	mvwprintw(win, 6, 2, gettext("  Byoungyoung, La (la9527@yahoo.co.kr)"));
	mvwprintw(win, 7, 2, gettext("  IOKLO  (leesjung@nownuri.net)"));
	mvwprintw(win, 8, 2, gettext("  fehead (infiniterun@gmail.com)"));
	mvwprintw(win, 9, 2, gettext("  Eunseo, Choi (eunseo.choi@gmail.com)"));
	mvwprintw(win, 11, 2, "(c) 2004 The Mls Developers");
	wattroff(win, A_BOLD);
	wrefresh(win);
	getch();
	delwin(win);
}

};
