/*
 * Line6 Linux USB driver - 0.7.4
 *
 * Copyright (C) 2004-2008 Markus Grabner (grabner@icg.tugraz.at)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation, version 2.
 *
 */

#include "driver.h"

#include <sound/core.h>
#include <sound/control.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>

#include "audio.h"
#include "capture.h"
#include "playback.h"
#include "pod.h"


/* trigger callback */
int snd_pod_trigger(struct snd_pcm_substream *substream, int cmd)
{
	struct snd_pod_pcm *chip = snd_pcm_substream_chip(substream);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 22)
	struct list_head *pos;
#endif
	struct snd_pcm_substream *s;
	int err;
	unsigned long flags;

	spin_lock_irqsave(&chip->lock_trigger, flags);
	clear_bit(BIT_PREPARED, &chip->flags);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 22)
	snd_pcm_group_for_each(pos, substream) {
		s = snd_pcm_group_substream_entry(pos);
#else
	snd_pcm_group_for_each_entry(s, substream) {
#endif
		switch(s->stream) {
		case SNDRV_PCM_STREAM_PLAYBACK:
			err = snd_pod_playback_trigger(s, cmd);

			if(err < 0) {
				spin_unlock_irqrestore(&chip->lock_trigger, flags);
				return err;
			}

			break;

		case SNDRV_PCM_STREAM_CAPTURE:
			err = snd_pod_capture_trigger(s, cmd);

			if(err < 0) {
				spin_unlock_irqrestore(&chip->lock_trigger, flags);
				return err;
			}

			break;

		default:
			dev_err(s2m(substream), "Unknown stream direction %d\n", s->stream);
		}
	}

	spin_unlock_irqrestore(&chip->lock_trigger, flags);
	return 0;
}

/* control info callback */
static int snd_pod_control_info(struct snd_kcontrol *kcontrol, struct snd_ctl_elem_info *uinfo) {
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 2;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 256;
	return 0;
}

/* control get callback */
static int snd_pod_control_get(struct snd_kcontrol *kcontrol, struct snd_ctl_elem_value *ucontrol) {
	int i;
	struct snd_pod_pcm *chip = snd_kcontrol_chip(kcontrol);

	for(i = 2; i--;)
		ucontrol->value.integer.value[i] = chip->volume[i];

	return 0;
}

/* control put callback */
static int snd_pod_control_put(struct snd_kcontrol *kcontrol, struct snd_ctl_elem_value *ucontrol) {
	int i, changed = 0;
	struct snd_pod_pcm *chip = snd_kcontrol_chip(kcontrol);

	for(i = 2; i--;)
		if(chip->volume[i] != ucontrol->value.integer.value[i]) {
			chip->volume[i] = ucontrol->value.integer.value[i];
			changed = 1;
		}

	return changed;
}

/* control definition */
static struct snd_kcontrol_new pod_control = {
	.iface = SNDRV_CTL_ELEM_IFACE_MIXER,
	.name = "PCM Playback Volume",
	.index = 0,
	.access = SNDRV_CTL_ELEM_ACCESS_READWRITE,
	.info = snd_pod_control_info,
	.get = snd_pod_control_get,
	.put = snd_pod_control_put
};

/*
	Cleanup the PCM device.
*/
static void pod_cleanup_pcm(struct snd_pcm *pcm)
{
	int i;
	struct snd_pod_pcm *chip = snd_pcm_chip(pcm);

	for(i = POD_ISO_BUFFERS; i--;) {
		if(chip->urb_audio_out[i]) {
			usb_kill_urb(chip->urb_audio_out[i]);
			usb_free_urb(chip->urb_audio_out[i]);
		}
		if(chip->urb_audio_in[i]) {
			usb_kill_urb(chip->urb_audio_in[i]);
			usb_free_urb(chip->urb_audio_in[i]);
		}
	}
}

/* create a PCM device */
static int snd_pod_new_pcm(struct snd_pod_pcm *chip)
{
	struct snd_pcm *pcm;
	int err;

	if((err = snd_pcm_new(chip->pod->line6.card, (char *)chip->pod->line6.properties->name, 0, 1, 1, &pcm)) < 0) 
		return err;

	pcm->private_data = chip;
	pcm->private_free = pod_cleanup_pcm;
	chip->pcm = pcm;
	strcpy(pcm->name, chip->pod->line6.properties->name);

	/* set operators */
	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_PLAYBACK, &snd_pod_playback_ops);
	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_CAPTURE, &snd_pod_capture_ops);

	/* pre-allocation of buffers */
	snd_pcm_lib_preallocate_pages_for_all(pcm,
																				SNDRV_DMA_TYPE_CONTINUOUS,
																				snd_dma_continuous_data(GFP_KERNEL),
																				64 * 1024, 128 * 1024);

	return 0;
}

/* PCM device destructor */
static int snd_pod_pcm_free(struct snd_device *device)
{
	return 0;
}

/*
	Create and register the PCM device and mixer entries.
	Create URBs for playback and capture.
*/
int pod_init_pcm(struct usb_line6_pod *pod)
{
	static struct snd_device_ops pcm_ops = {
		.dev_free = snd_pod_pcm_free,
	};

	int err;
	int ep_read = 0, ep_write = 0;
	struct snd_pod_pcm *chip;

	if(!(pod->line6.properties->capabilities & LINE6_BIT_PCM))
		return 0;  /* skip PCM initialization and report success */

	/* initialize PCM subsystem based on product id: */
	switch(pod->line6.product) {
	case LINE6_DEVID_BASSPODXT:
  case LINE6_DEVID_BASSPODXTLIVE:
  case LINE6_DEVID_BASSPODXTPRO:
  case LINE6_DEVID_PODXT:
  case LINE6_DEVID_PODXTLIVE:
  case LINE6_DEVID_PODXTPRO:
		ep_read  = 0x82;
		ep_write = 0x01;
		break;

  case LINE6_DEVID_PODX3:
  case LINE6_DEVID_PODX3LIVE:
		ep_read  = 0x86;
		ep_write = 0x02;
		break;

  case LINE6_DEVID_POCKETPOD:
		ep_read  = 0x82;
		ep_write = 0x02;
		break;

	default:
		MISSING_CASE;
	}

	chip = kmalloc(sizeof(struct snd_pod_pcm), GFP_KERNEL);

	if(chip == NULL)
		return -ENOMEM;

	memset(chip, 0, sizeof(struct snd_pod_pcm));
	chip->volume[0] = chip->volume[1] = 128;
	chip->pod = pod;
	chip->ep_audio_read = ep_read;
	chip->ep_audio_write = ep_write;
	chip->max_packet_size = usb_maxpacket(pod->line6.usbdev, usb_rcvintpipe(pod->line6.usbdev, ep_read), 0);
	pod->chip = chip;

	/* PCM device: */
	if((err = snd_device_new(pod->line6.card, SNDRV_DEV_PCM, pod, &pcm_ops)) < 0)
		return err;

	snd_card_set_dev(pod->line6.card, pod->line6.ifcdev);

	if((err = snd_pod_new_pcm(chip)) < 0)
		return err;

	spin_lock_init(&chip->lock_audio_out);
	spin_lock_init(&chip->lock_audio_in);
	spin_lock_init(&chip->lock_trigger);

	if((err = create_audio_out_urbs(chip)) < 0)
		return err;

	if((err = create_audio_in_urbs(chip)) < 0)
		return err;

	/* mixer: */
	if((err = snd_ctl_add(pod->line6.card, snd_ctl_new1(&pod_control, chip))) < 0)
		return err;

	return 0;
}

/* prepare pcm callback */
int snd_pod_prepare(struct snd_pcm_substream *substream)
{
	struct snd_pod_pcm *chip = snd_pcm_substream_chip(substream);

	if(!test_and_set_bit(BIT_PREPARED, &chip->flags)) {
		unlink_wait_clear_audio_out_urbs(chip);
		chip->pos_out = 0;
		chip->pos_out_done = 0;
	
		unlink_wait_clear_audio_in_urbs(chip);
		chip->bytes_out = 0;
		chip->pos_in_done = 0;
		chip->bytes_in = 0;
	}

	return 0;
}
