/***********************************************************************************

    Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include "lifeograph.hpp"
#include "app_window.hpp"
#include "widget_entrylist.hpp"
#include "view_entry.hpp"


using namespace LIFEO;


// WIDGET ENTRY LIST ===========================================================
WidgetEntryList::WidgetEntryList( BaseObjectType* cobject,
                                  const Glib::RefPtr<Gtk::Builder>& )
:   Gtk::TreeView( cobject )
{
    // CELL RENDERERS
    Gtk::CellRendererPixbuf* cellr_pb_icon = Gtk::manage( new Gtk::CellRendererPixbuf );
    Gtk::CellRendererText* cellr_text = Gtk::manage( new Gtk::CellRendererText );
    Gtk::TreeViewColumn* column = Gtk::manage( new Gtk::TreeViewColumn );

    Gtk::CellRendererPixbuf* cellr_pb_icon2 = Gtk::manage( new Gtk::CellRendererPixbuf );

    cellr_text->property_ellipsize() = Pango::ELLIPSIZE_END;
    if( Lifeograph::settings.small_lists )
        cellr_text->property_scale() = .90;
    cellr_text->set_fixed_height_from_font( 1 );

    column->pack_start( *cellr_pb_icon, false );
    column->pack_start( *cellr_text );
    column->pack_start( *cellr_pb_icon2, false );

    column->add_attribute( cellr_pb_icon->property_pixbuf(), ListData::colrec->icon );
    column->add_attribute( cellr_text->property_markup(), ListData::colrec->info );
    column->add_attribute( cellr_pb_icon2->property_pixbuf(), ListData::colrec->icon2 );

    column->set_cell_data_func( *cellr_pb_icon2,
                                sigc::mem_fun( this, &WidgetEntryList::cell_data_func_icon ) );

    // TODO we failed to accomplish fixed height mode with the second icon renderer
    //column->set_sizing( Gtk::TREE_VIEW_COLUMN_FIXED );

    m_treestore = Gtk::TreeStore::create( *ListData::colrec );

    if( Lifeograph::s_color_insensitive.empty() )
    {
        Lifeograph::s_color_insensitive = convert_gdkcolor_to_html(
                get_style_context()->get_color( Gtk::STATE_FLAG_INSENSITIVE ) );
    }

    append_column( *column );
}

void
WidgetEntryList::cell_data_func_icon( Gtk::CellRenderer* cell, const Gtk::TreeModel::iterator& iter )
{
    DiaryElement* elem( ( * iter )[ ListData::colrec->ptr ] );
    if( elem != NULL )
    {
        bool favored( elem->get_status() & ES::FAVORED );
        cell->set_property( "visible", favored );
        if( favored )
            cell->set_property( "pixbuf", Lifeograph::icons->entry_favorite_16 );
    }
}

WidgetEntryList::~WidgetEntryList()
{
}

void
WidgetEntryList::clear()
{
    m_treestore->clear();
    Lifeograph::s_elem_dragged = NULL;
}

void
WidgetEntryList::present_element( const DiaryElement* element )
{
    Gtk::TreePath path( element->m_list_data->treepath );

    expand_to_path( path );
    if( ! get_selection()->is_selected( path ) )
        get_selection()->select( path );

    Gtk::TreePath path_b, path_e;
    get_visible_range( path_b, path_e );

    if( treepath_is_less( path, path_b ) || treepath_is_more( path, path_e ) )
    {
        PRINT_DEBUG( "path is not visible" );
        scroll_to_row( path, 0.0 );   // does nothing if already visible
    }
}

void
WidgetEntryList::expand_element( const DiaryElement* element ) // this function does not do much
{
    expand_to_path( element->m_list_data->treepath );
}

void
WidgetEntryList::go_up( bool flag_entry_operation )
{
    DiaryElement* elem( AppWindow::p->panel_main->get_cur_elem() );

    if( elem == NULL )
        return;
    else
    if( elem->get_type() < DiaryElement::ET_DIARY ) // not list element
        return;

    Gtk::TreePath path( elem->m_list_data->treepath );

    do
    {
        if( ! path.prev() )
        {
            if( path.size() > 1 )
                path.up();
            else    // diary
            if( make_path_deeper_last( path ) )
                make_path_deeper_last( path ); // go still deeper if possible:
        }
        else
            make_path_deeper_last( path );

        elem = ( * m_treestore->get_iter( path ) )[ ListData::colrec->ptr ];
    }
    // repeat until an entry is found if that is what we are looking for
    // BEWARE: do not set this flag when there is no entry in the list
    while( flag_entry_operation && elem->get_type() != DiaryElement::ET_ENTRY );

    PRINT_DEBUG( "previous path: " + path.to_string() );

    Gtk::TreeRow row( * m_treestore->get_iter( path ) );
    DiaryElement* element( row[ ListData::colrec->ptr ] );
    element->show();
}

void
WidgetEntryList::go_down( bool flag_entry_operation )
{
    DiaryElement* elem( AppWindow::p->panel_main->get_cur_elem() );

    if( elem == NULL )
        return;
    else
    if( elem->get_type() < DiaryElement::ET_DIARY ) // not list element
        return;

    Gtk::TreePath path( elem->m_list_data->treepath );

    do
    {
        if( ! m_treestore->get_iter( path )->children().empty() )
            path.down();
        else
        if( ! move_path_next( path ) )
        {
            while( path.size() > 1 )
            {
                path.up();
                if( move_path_next( path ) )
                    break;
            }
        }

        elem = ( * m_treestore->get_iter( path ) )[ ListData::colrec->ptr ];
    }
    // repeat until an entry is found if that is what we are looking for
    // BEWARE: do not set this flag when there is no entry in the list
    while( flag_entry_operation && elem->get_type() != DiaryElement::ET_ENTRY );

    PRINT_DEBUG( "next path: " + path.to_string() );

    Gtk::TreeRow row( * m_treestore->get_iter( path ) );
    DiaryElement *listitem = row[ ListData::colrec->ptr ];
    listitem->show();
}

void
WidgetEntryList::handle_sorting_criteria_changed( SortingCriteria sc )
{
    if( sc == Diary::d->get_sorting_criteria() )
        return;

    Diary::d->set_sorting_criteria( sc );

    switch( sc )
    {
        default: // future-proofing
        case SC_DATE:
            m_treestore->set_default_sort_func(
                    sigc::mem_fun( this, &WidgetEntryList::sort_by_date ) );
            break;
        case SC_SIZE:
            m_treestore->set_default_sort_func(
                    sigc::mem_fun( this, &WidgetEntryList::sort_by_size ) );
            break;
        case SC_CHANGE:
            m_treestore->set_default_sort_func(
                    sigc::mem_fun( this, &WidgetEntryList::sort_by_change ) );
            break;
    }

    AppWindow::p->panel_diary->update_entry_list();
}

void
WidgetEntryList::reset_sorting_criteria()
{
    switch( Diary::d->get_sorting_criteria() )
    {
        default:
        case SC_DATE:
            m_treestore->set_default_sort_func(
                    sigc::mem_fun( this, &WidgetEntryList::sort_by_date ) );
            break;
        case SC_SIZE:
            m_treestore->set_default_sort_func(
                    sigc::mem_fun( this, &WidgetEntryList::sort_by_size ) );
            break;
        case SC_CHANGE:
            m_treestore->set_default_sort_func(
                    sigc::mem_fun( this, &WidgetEntryList::sort_by_change ) );
            break;
    }
}

// not used now but may be used in the future
Gtk::TreePath
WidgetEntryList::find_element( const DiaryElement* element ) const
{
    Gtk::TreeIter iter_diary( m_treestore->children().begin() );
    if( element == ( *iter_diary )[ ListData::colrec->ptr ] )
        return m_treestore->get_path( iter_diary );

    for( Gtk::TreeIter iter = ( *iter_diary )->children().begin();
         iter != ( *iter_diary )->children().end();
         ++iter )
    {
        const DiaryElement *list_elem( ( *iter )[ ListData::colrec->ptr ] );
        if( element->get_id() == list_elem->get_id() )
        {
            return m_treestore->get_path( iter );
        }
        else
        {
            if( element->get_type() == DiaryElement::ET_ENTRY &&
                list_elem->get_type() != DiaryElement::ET_ENTRY )
                if( element->get_date() > list_elem->get_date() )
                    for( Gtk::TreeIter itr2 = ( *iter )->children().begin();
                         itr2 != ( *iter )->children().end();
                         ++itr2 )
                    {
                        if( element == ( *itr2 )[ ListData::colrec->ptr ] )
                            return m_treestore->get_path( itr2 );
                    }
        }
    }

    return m_treestore->get_path( m_treestore->children().begin() );
}

bool
WidgetEntryList::on_button_press_event( GdkEventButton* event )
{
    Lifeograph::s_flag_dragging = false;

    Gtk::TreePath path;
    if( get_path_at_pos( event->x, event->y, path ) )
    {
        Gtk::TreeRow row = * m_treestore->get_iter( path );
        Lifeograph::s_elem_dragged = row[ ListData::colrec->ptr ];
    }
    else
        Lifeograph::s_elem_dragged = NULL;

/*  if( event->button != 3 )
        return Gtk::TreeView::on_button_press_event( event );

    Glib::RefPtr< Gtk::TreeSelection > selection = get_selection();

    if( selection->count_selected_rows() <= 0 )
        return Gtk::TreeView::on_button_press_event( event );

    // treeview issues selection changed signal after button press signal.
    // this causes context menus to be linked to the previously selected item
    // when a row is selected with right click.
    // to get around this we use a somewhat dirty hack and select the row
    // from within the button press event handler
    // p.s.: i dunno but there might be a simpler way of doing this
    Gtk::TreeModel::Path path;
    Gtk::TreeViewColumn  *column;
    int i, j;
    if( ! get_path_at_pos( (int) event->x, (int) event->y, path, column, i, j ) )
        return Gtk::TreeView::on_button_press_event( event );

    set_cursor( path );
    // end of dirty hack

    Gtk::TreeRow row = *( selection->get_selected() );
    DiaryElement *item = row[ ListData::colrec->ptr ];

    if( ! ( item->get_type() == DiaryElement::IT_ENTRY ||
            item->get_type() == DiaryElement::IT_TAG ) )
        return Gtk::TreeView::on_button_press_event( event );

    if( m_menu )
    {
        delete m_menu;
        m_menu = NULL;
    }

    m_menu = new Gtk::Menu;
    Gtk::Image *imageDismiss = Gtk::manage(
            new Gtk::Image( Gtk::Stock::DELETE, Gtk::ICON_SIZE_MENU ) );

    if( item->get_type() == DiaryElement::IT_ENTRY )
    {
        Entry* entry = dynamic_cast< Entry* >( item );


        m_menu->items().push_back(
            Gtk::Menu_Helpers::ImageMenuElem(
                entry->is_favored() ?
                        _( "Remove from _Favorites" ) : _( "Add to _Favorites" ),
                *image_favored,
                sigc::mem_fun( * entry, &Entry::toggle_favoredness ) ) );
    }
    m_menu->accelerate( *this );
    m_menu->popup( event->button, event->time );
*/
    return Gtk::TreeView::on_button_press_event( event );
}

bool
WidgetEntryList::on_button_release_event( GdkEventButton* event )
{
    if( ! Lifeograph::s_flag_dragging /*&& event->button != 1*/ )
    {
        if( Lifeograph::s_elem_dragged )
            Lifeograph::s_elem_dragged->show();
        else
            Diary::d->show();   // if nothing is selected show the diary view
    }

    return Gtk::TreeView::on_button_release_event( event );
}

bool
WidgetEntryList::on_key_press_event( GdkEventKey* event )
{
    if( event->state == 0 )
    {
        switch( event->keyval )
        {
            case GDK_KEY_Home:
                Diary::d->show();
                return true;
            case GDK_KEY_End:
                return true;
            case GDK_KEY_Up:
                go_up( false );
                return true;
            case GDK_KEY_Down:
                go_down( false );
                return true;
        }
    }
    return Gtk::TreeView::on_key_press_event( event );
}

void
WidgetEntryList::on_drag_begin( const Glib::RefPtr< Gdk::DragContext >& )
{
    Lifeograph::s_flag_dragging = ! Diary::d->is_read_only();
}

bool
WidgetEntryList::on_drag_motion( const Glib::RefPtr< Gdk::DragContext >& context,
                                 int x, int y, guint time )
{
    if( ! Lifeograph::s_flag_dragging )
        return false;

    if( ! Lifeograph::s_elem_dragged )
        return false;

    bool retval( false );
    Gtk::TreePath path;
    DiaryElement* element( NULL );

    // AUTOSCROLL
    Gdk::Rectangle rect;
    get_visible_rect( rect );
    int new_y = rect.get_y();
    if( y < ( rect.get_height() * 0.15 ) )
    {
        if( new_y > 0 )
        {
            new_y -= 5;
            if( new_y < 0 )
                new_y = 0;
            scroll_to_point( -1, new_y );
        }
    }
    else
    if( y > ( rect.get_height() * 0.85 ) )
    {
        if( get_vadjustment()->get_value() < get_vadjustment()->get_upper() )
        {
            new_y += 5;
            scroll_to_point( -1, new_y );
        }
    }

    // EVALUATE THE HOVERED ELEMENT
    if( get_path_at_pos( x, y, path ) )
    {
        Gtk::TreeRow row = * m_treestore->get_iter( path );
        element = row[ ListData::colrec->ptr ];
        if( element != NULL && element != Lifeograph::s_elem_dragged )
        {
            switch( Lifeograph::s_elem_dragged->get_type() )
            {
                case DiaryElement::ET_ENTRY:
                    if( element->get_type() != DiaryElement::ET_DIARY )
                        retval = true;
                    break;
                case DiaryElement::ET_CHAPTER:
                    // chapters can only be dropped onto temporal entries
                    if( element->get_type() == DiaryElement::ET_ENTRY &&
                        element->get_date().is_ordinal() == false &&
                        element->get_date().get_pure() !=
                                Lifeograph::s_elem_dragged->get_date().get_pure() )
                        retval = true;
                    break;
                case DiaryElement::ET_TOPIC:
                    // topics can only be dropped onto other topics
                    if( element->get_type() == DiaryElement::ET_TOPIC )
                        retval = true;
                    break;
                case DiaryElement::ET_TODO_GRP:
                    // to-do groups can only be dropped onto other to-do groups
                    if( element->get_type() == DiaryElement::ET_TODO_GRP )
                        retval = true;
                    break;
                case DiaryElement::ET_TAG:
                case DiaryElement::ET_UNTAGGED:
                    if( element->get_type() == DiaryElement::ET_ENTRY )
                        retval = true;
                    break;
                default: // the rest is not accepted
                    break;
            }
        }
    }

    if( retval )
    {
        m_elem_drop_target = element;
        context->drag_status( Gdk::ACTION_MOVE, time );
        get_selection()->select( path );
    }
    else
        get_selection()->unselect_all();

    //return Gtk::TreeView::on_drag_motion( context, x, y, time );
    return retval;
}

bool
WidgetEntryList::on_drag_drop( const Glib::RefPtr< Gdk::DragContext >& context,
                               int x, int y, guint time )
{
    bool retval( false );

    if( Lifeograph::s_flag_dragging )
    {
        if( Lifeograph::s_elem_dragged != NULL && m_elem_drop_target != NULL )
        {
            switch( Lifeograph::s_elem_dragged->get_type() )
            {
                case DiaryElement::ET_ENTRY:
                {
                    Entry* entry( dynamic_cast< Entry* >( Lifeograph::s_elem_dragged ) );
                    switch(m_elem_drop_target->get_type() )
                    {
                        case DiaryElement::ET_CHAPTER:
                        case DiaryElement::ET_TOPIC:
                        case DiaryElement::ET_TODO_GRP:
                        {
                            Chapter *chapter( dynamic_cast< Chapter* >( m_elem_drop_target ) );
                            Diary::d->set_entry_date( entry, chapter->get_free_order() );
                            AppWindow::p->panel_main->refresh_title();
                            AppWindow::p->panel_diary->update_entry_list();
                            retval = true;
                            break;
                        }
                        case DiaryElement::ET_ENTRY:
                            Diary::d->set_entry_date( entry, m_elem_drop_target->get_date() );
                            AppWindow::p->panel_main->refresh_title();
                            AppWindow::p->panel_diary->update_entry_list();
                            retval = true;
                            break;
                        default:
                            break;
                    }
                    break;
                }
                case DiaryElement::ET_CHAPTER:
                {
                    Chapter* chapter( dynamic_cast< Chapter* >( Lifeograph::s_elem_dragged ) );
                    Diary::d->get_current_chapter_ctg()->set_chapter_date(
                            chapter, m_elem_drop_target->get_date().m_date );
                    AppWindow::p->panel_main->refresh_title();
                    AppWindow::p->panel_diary->update_entry_list();
                    retval = true;
                    break;
                }
                case DiaryElement::ET_TOPIC:
                {
                    Chapter* chapter( dynamic_cast< Chapter* >( Lifeograph::s_elem_dragged ) );
                    Diary::d->set_topic_order( chapter, m_elem_drop_target->get_date().m_date );
                    AppWindow::p->panel_main->refresh_title();
                    AppWindow::p->panel_diary->update_entry_list();
                    retval = true;
                    break;
                }
                case DiaryElement::ET_TAG:
                    if( m_elem_drop_target->get_type() == DiaryElement::ET_ENTRY )
                    {
                        Tag *tag( dynamic_cast< Tag* >( Lifeograph::s_elem_dragged ) );
                        Entry* entry( dynamic_cast< Entry* >( m_elem_drop_target ) );
                        if( entry->add_tag( tag ) ) // if does not already have the tag
                        {
                            if( AppWindow::p->panel_main->is_cur_elem( m_elem_drop_target ) )
                            {
                                AppWindow::p->m_entry_view->update_tag_widget();
                                AppWindow::p->m_entry_view->update_theme();
                            }

                            retval = true;
                        }
                    }
                    break;
                case DiaryElement::ET_UNTAGGED:
                    if( m_elem_drop_target->get_type() == DiaryElement::ET_ENTRY )
                    {
                        Entry* entry( dynamic_cast< Entry* >( m_elem_drop_target ) );
                        if( entry->clear_tags() )
                        {
                            if( AppWindow::p->panel_main->is_cur_elem( m_elem_drop_target ) )
                            {
                                AppWindow::p->m_entry_view->update_tag_widget();
                                AppWindow::p->m_entry_view->update_theme();
                            }

                            retval = true;
                        }
                    }
                    break;
                default:
                    break;
            }
        }
    }

    context->drag_finish( retval, false, time );
    return retval;
}

int
WidgetEntryList::sort_by_date( const Gtk::TreeModel::iterator& itr1,
                               const Gtk::TreeModel::iterator& itr2 )
{
    // SORT BY DATE (ONLY DESCENDINGLY FOR NOW)
    DiaryElement* item1 = ( *itr1 )[ ListData::colrec->ptr ];
    DiaryElement* item2 = ( *itr2 )[ ListData::colrec->ptr ];
    if( !( item1 && item2 ) )
        return 0;
    else
    if( item1->get_type() == DiaryElement::ET_DIARY )
    return -1;

    int direction( ( item1->get_date().is_ordinal() && item2->get_date().is_ordinal() ) ? -1 : 1 );

    if( item1->get_date().get_sort() > item2->get_date().get_sort() )
        return( -1 * direction );
    else
    if( item1->get_date().get_sort() < item2->get_date().get_sort() )
        return( 1 * direction );
    else
        return 0;
}

int
WidgetEntryList::sort_by_size( const Gtk::TreeModel::iterator& itr1,
                               const Gtk::TreeModel::iterator& itr2 )
{
    DiaryElement* item1 = ( *itr1 )[ ListData::colrec->ptr ];
    DiaryElement* item2 = ( *itr2 )[ ListData::colrec->ptr ];
    if( !( item1 && item2 ) )
        return 0;

    // group ordinal entries together:
    if( item1->get_date().is_ordinal() != item2->get_date().is_ordinal() )
        return( item1->get_date().is_ordinal() ? -1 : 1 );

    if( item1->get_size() > item2->get_size() )
        return -1;
    else
    if( item1->get_size() < item2->get_size() )
        return 1;
    else
        return 0;
}

int
WidgetEntryList::sort_by_change( const Gtk::TreeModel::iterator& itr1,
                                 const Gtk::TreeModel::iterator& itr2 )
{
    DiaryElement* item1 = ( *itr1 )[ ListData::colrec->ptr ];
    DiaryElement* item2 = ( *itr2 )[ ListData::colrec->ptr ];
    if( !( item1 && item2 ) )
        return 0;

    time_t date1( item1->get_type() == DiaryElement::ET_ENTRY ?
            dynamic_cast< Entry* >( item1 )->get_date_changed() : 0 );
    time_t date2( item2->get_type() == DiaryElement::ET_ENTRY ?
            dynamic_cast< Entry* >( item2 )->get_date_changed() : 0 );

    if( date1 > date2 )
        return -1;
    else
    if( date1 < date2 )
        return 1;
    else
        return 0;
}

inline bool
WidgetEntryList::move_path_next( Gtk::TreePath& path )
{
    if( unsigned( path.back() ) < ( m_treestore->
                get_iter( path )->parent()->children().size() - 1 ) )
    {
        path.next();
        return true;
    }
    else
        return false;
}

inline bool
WidgetEntryList::make_path_deeper_last( Gtk::TreePath& path )
{
    if( ! m_treestore->get_iter( path )->children().empty() )
    {
        path.push_back( m_treestore->get_iter( path )->children().size() - 1 );
        return true;
    }
    return false;
}
