/***********************************************************************************

	Copyright (C) 2010 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include "settings.hpp"

#include <fstream>
#include <glib/gstdio.h>	// g_mkdir()


using namespace LIFEO;


Settings::Settings( void )
:	autologout( true ), idletime( IDLETIME_DEFAULT ), show_formatting_toolbar( true ),
	width( WIDTH_DEFAULT ), height( HEIGHT_DEFAULT ),
	position_x( POSITION_NOTSET ), position_y( POSITION_NOTSET ),
	position_pane( PANEPOS_DEFAULT ), position_pane_tags( PANEPOS_TAGS_DEFAULT )
{
}

bool
Settings::read( void )
{
	std::string path = Glib::build_filename(	Glib::get_user_config_dir(),
												"lifeograph/lifeo.conf" );
	std::ifstream file( path.c_str() );

	if( ! file.is_open() )
	{
		print_info( "No configuration file, maiden voyage!" );
		return false;
	}
	
	reset();

	std::string line;

	while( getline( file, line ) )
	{
		switch( line[ 0 ] )
		{
			case 'R':
				if( recentfiles.size() < LIFEO::MAX_RECENT_FILE_COUNT )
					recentfiles.push_back( line.substr( 2 ) );
				break;

			// TODO: position & size settings may go to the diary
			// TODO: check if these are within screen boundaries
			case 'H':
				height = convert_string( line.substr( 2 ) );
				break;
			case 'W':
				width = convert_string( line.substr( 2 ) );
				break;
			case 'X':
				position_x = convert_string( line.substr( 2 ) );
				break;
			case 'Y':
				position_y = convert_string( line.substr( 2 ) );
				break;
			case 'P':	// uppercase
				position_pane = convert_string( line.substr( 2 ) );
				break;
			case 'p':	// lowercase
				position_pane_tags =  convert_string( line.substr( 2 ) );
				// zero width is not accepted by Gtk::Paned
				if( position_pane_tags == 0 )
				    position_pane_tags = 1;
				break;
			case 'L':
				autologout = ( line[ 2 ] == 'y' || line[ 2 ] == 'Y' );
				break;
			case 'I':
				idletime = convert_string( line.substr( 2 ) );
				if( idletime < IDLETIME_MIN || idletime > IDLETIME_MAX )
					idletime = IDLETIME_DEFAULT;
				break;
			case 'F':
				show_formatting_toolbar = ( line[ 2 ] == 'y' || line[ 2 ] == 'Y' );
				break;

			case 0:		// empty line
			case '#':	// comment
				break;
			default:
				print_info( "unrecognized line:\n" + line );
				break;
		}
	}

	file.close();
	return true;
}

bool
Settings::write( void )
{
	std::string path = Glib::build_filename( Glib::get_user_config_dir(), "lifeograph" );
	if( ! Glib::file_test( path, Glib::FILE_TEST_IS_DIR ) )
		g_mkdir( path.c_str(), 0700 );
	Glib::build_filename( path, "lifeo.conf" );
	std::ofstream file(	Glib::build_filename( path, "lifeo.conf" ).c_str(),
						std::ios::out | std::ios::trunc );

	file << "W " << width;
	file << "\nH " << height;
	file << "\nX " << position_x;
	file << "\nY " << position_y;
	file << "\nP " << position_pane;
	file << "\np " << position_pane_tags;

	// RECENT FILES
	for( std::list< std::string >::iterator iter = recentfiles.begin();
		 iter != recentfiles.end(); ++iter )
		file << "\nR " << *iter;

	file << "\nL " << ( autologout ? 'y' : 'n' );
	file << "\nI " << idletime;
	file << "\nF " << ( show_formatting_toolbar ? 'y' : 'n' );

	file.close();
	return true;
}

void
Settings::reset( void )
{
	recentfiles.clear();
}

