/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <cstdlib>

#include "lifeograph.hpp"
#include "dialog_preferences.hpp"


using namespace LIFEO;


// CONSTRUCTOR
Lifeograph::Lifeograph( int argc, char *argv[] )
:	m_about_dialog( NULL ), m_menu_link( NULL ), m_vbox_login( NULL ), m_hpaned_main( NULL ),
	m_theme_view( NULL ),
	m_programpath( argv[ 0 ] ), m_loginstatus( LOGGED_OUT ),
	m_flag_open_directly( false ),
	m_flag_force_welcome( false )
{
	Cipher::init();

	UndoManager::m = new UndoManager;	// init()??
	// FIXME:
	Diary::d = new Diary;
	ListData::colrec = new ListData::Colrec;

	// INTERFACE INITIALIZATION
	TextviewDiary::m_buffer = new TextbufferDiary; //EntryView::init();
	//m_diary_view->init();

	// CONFIGURATION
	m_flag_firsttimeuser = ! m_settings.read();
	set_default_size( m_settings.width, m_settings.height );
	if( m_settings.position_x != POSITION_NOTSET && m_settings.position_y != POSITION_NOTSET )
		move( m_settings.position_x, m_settings.position_y );

	// COMMANDLINE OPTIONS
	for( int i = 1; i < argc; i++ )
	{
		if( ! strcmp( argv[ i ], "--open" ) || ! strcmp( argv[ i ], "-o" ) )
		{
			if( i < argc )
			{
				if( Glib::file_test( argv[ ++i ], Glib::FILE_TEST_EXISTS ) )
					if( ! Glib::file_test( argv[ i ], Glib::FILE_TEST_IS_DIR ) )
					{
						// TODO: detect duplicates?
						m_settings.recentfiles.push_front( argv[ i ] );
						m_flag_open_directly = true;
					}
			}
			else
				Error( "No path provided" );
		}
		else
		if( ! strcmp( argv[ i ], "--force-welcome" ) )
		{
			m_flag_force_welcome = true;
		}
		else
		if( ! strcmp( argv[ i ], "--ignore-locks" ) )
		{
			Diary::d->m_flag_ignore_locks = true;
		}
		else
		if( Glib::file_test( argv[ i ], Glib::FILE_TEST_EXISTS ) )
		{
			if( ! Glib::file_test( argv[ i ], Glib::FILE_TEST_IS_DIR ) )
			{
				m_settings.recentfiles.push_front( argv[ i ] );
				m_flag_open_directly = true;
			}
		}
	}

	// GTKBUILDER
	load_gui( UIDIR "/lifeograph.ui" );

	// PROGRAM-WIDE VARIABLE INITIALIZATIONS
	Gtk::AboutDialog::set_url_hook( &open_url );
	Gtk::AboutDialog::set_email_hook( &mail_to );

    Gtk::Main::signal_run().connect( sigc::mem_fun( this, &Lifeograph::on_run ) );
}

Lifeograph::~Lifeograph( void )
{
	remove();
	delete UndoManager::m;
	if( m_vbox_login ) delete m_vbox_login;
	if( Diary::d ) delete Diary::d;
}

void
Lifeograph::on_run( void )
{
    // purpose of this function is calling draw functions after Gtk::Main::run()
    if( m_flag_firsttimeuser || m_flag_force_welcome )
        draw_welcomescreen();
    else
        draw_loginscreen();
}

bool
Lifeograph::on_event( GdkEvent* )
{
    if( ! m_settings.autologout || m_loginstatus != LOGGED_IN || ! Diary::d->is_encrypted() )
		return false;
	if( m_secondsremaining < LOGOUT_COUNTDOWN )
		update_title();
	// restart
	m_secondsremaining = LOGOUT_COUNTDOWN;
	m_connection_timeout.disconnect();
	m_connection_timeout = Glib::signal_timeout().connect_seconds(
			sigc::mem_fun( *this, &Lifeograph::handle_idle ),
			m_settings.idletime - LOGOUT_COUNTDOWN );

	return false;
}

bool
Lifeograph::on_delete_event( GdkEventAny* )
{
	// BEWARE: handle_button_quit_clicked() calls this function

	PRINT_DEBUG( "on_delete_event()" );

	if( m_loginstatus == LOGGED_IN )
		if( ! finish_editing() )
			return true;

	// SAVE SETTINGS
	get_size( m_settings.width, m_settings.height );
	get_position( m_settings.position_x, m_settings.position_y );
	m_settings.write();

	return false;
}

bool
Lifeograph::handle_idle()
{
    if( ! m_settings.autologout )
        return false;

	if( m_secondsremaining > 0 )
	{
		set_title( Glib::ustring::compose(
				_( "<<<%1 SECONDS TO LOG OUT>>>" ), m_secondsremaining ) );
		m_connection_timeout = Glib::signal_timeout().connect_seconds(
				sigc::mem_fun( *this, &Lifeograph::handle_idle ), 1 );
		m_secondsremaining--;
		return false;
	}

	m_loginstatus = LOGGED_TIME_OUT;
	logout( true );
	return false;
}

void
Lifeograph::update_login_file( void )
{
	bool flag_open_ready( true );
	bool flag_encrypted( false );

	// FIXME: this is not so elegant here:
    if( m_filebuttonrecent->get_recentlist()->size() <= 0 )
	{
        m_label_login_info->set_label( _( "Welcome new user!" ) );
        flag_open_ready = false;
	}
    else
    if( ! Diary::d->set_path( m_filebuttonrecent->get_filename() ) )
    {
        m_label_login_info->set_text( _( STRING::DIARY_LOCKED ) );
        flag_open_ready = false;
    }

    // TODO: FileButtonRecent should handle this
//    if( Glib::file_test( path, Glib::FILE_TEST_IS_DIR ) )
//        return;

    if( flag_open_ready )
    {
        flag_open_ready = false;
        switch( Diary::d->read_header() )
        {
            case SUCCESS:
                flag_encrypted = Diary::d->is_encrypted();
                if( flag_encrypted )
                {
                    if( m_loginstatus == LOGGED_TIME_OUT )
                        m_label_login_info->set_text( _( STRING::ENTER_PASSWORD_TIMEOUT ) );
                    else
                        m_label_login_info->set_text( _( STRING::ENTER_PASSWORD ) );
                    flag_open_ready = ( m_entry_password->get_text_length() > 0 );
                }
                else
                {
                    m_label_login_info->set_text( _( STRING::DB_IS_NOT_ENCRYPTED ) );
                    flag_open_ready = true;
                }
                break;
            case INCOMPATIBLE_FILE:
                m_label_login_info->set_text( _( STRING::INCOMPATIBLE_DB ) );
                break;
            case CORRUPT_FILE:
                m_label_login_info->set_text( _( STRING::CORRUPT_DB ) );
                break;
            default:
                m_label_login_info->set_text( _( STRING::FAILED_TO_OPEN_DB ) );
                break;
        }
    }

    if( flag_open_ready && ( ! flag_encrypted ) && m_flag_open_directly )
    {
        handle_button_opendb_clicked();
        return;
    }

    m_button_open_diary->set_sensitive( flag_open_ready );
    m_entry_password->set_visible( flag_encrypted );

    // FOCUS ADJUSTMENT
    if( flag_encrypted )
        m_entry_password->grab_focus();
    else
        m_button_open_diary->grab_focus();
}

void
Lifeograph::handle_button_quit_clicked( void )
{
	if( ! on_delete_event( NULL ) )
		Gtk::Main::quit();
}

void
Lifeograph::handle_button_about_clicked( void )
{
	if( m_about_dialog == NULL )
	{
		Lifeobase::builder->get_widget( "aboutdialog", m_about_dialog );
		m_about_dialog->set_name( PROGRAM_NAME );
		m_about_dialog->set_version( PROGRAM_VERSION_STRING );
		m_about_dialog->set_transient_for( *this );
	}

	m_about_dialog->run();
	m_about_dialog->hide();
}

void
Lifeograph::handle_filebutton_db_selectionchanged( void )
{
	if( m_internaloperation ) return;

	update_login_file();
}

void
Lifeograph::handle_entry_password_changed( void )
{
    if( Diary::d->is_path_set() )
        m_button_open_diary->set_sensitive( m_entry_password->get_text_length() > 0 );
}

void
Lifeograph::initialize_new_diary( void )
{
	DialogSaveDiary *dialogsavediary = new DialogSaveDiary;
	dialogsavediary->set_current_folder( Glib::get_home_dir() );

	if( dialogsavediary->run() == Gtk::RESPONSE_ACCEPT )
	{
		Diary::d->init_new( dialogsavediary->get_filename() );
		draw_editscreen();
	}

	delete dialogsavediary;
}

bool
Lifeograph::write_backup( void )
{
	sync_entry();

	std::string path_backup( Diary::d->m_path + LOCK_SUFFIX );
	Result result( Diary::d->m_passphrase.empty() ?
			Diary::d->write_plain( path_backup ) : Diary::d->write_encrypted( path_backup ) );

	return( result == SUCCESS );
}

void
Lifeograph::handle_button_opendb_clicked( void )
{
	if( Diary::d->is_old() )
	{
		Gtk::MessageDialog messagedialog( *this,
										  "",
										  false,
										  Gtk::MESSAGE_WARNING,
										  Gtk::BUTTONS_CANCEL,
										  true );
		messagedialog.set_message(
			_( "Are You Sure You Want to Upgrade The Diary?" ) );
		messagedialog.set_secondary_text(
			_( "You are about to open an old diary that "
			   "has to be upgraded to the new format. \n\n"
			   "It is STRONGLY recommended to back up any diary "
			   "before upgrading.\n\n"
			   "After you upgrade a diary, you won't be able to open it "
			   "in a previous version of Lifeograph." ) );
		messagedialog.add_button( _( "Upgrade The Diary" ), Gtk::RESPONSE_ACCEPT );

		if( messagedialog.run() !=  Gtk::RESPONSE_ACCEPT )
			return;
	}
	if( Diary::d->is_encrypted() )
	{
		std::string passphrase = m_entry_password->get_text();
		Diary::d->set_passphrase( passphrase );
	}

	switch( Diary::d->read_body() )
	{
		case EMPTY_DATABASE:	// no special treatment for now:
		case SUCCESS:
			draw_editscreen();
			break;
		case WRONG_PASSWORD:
		    m_label_login_info->set_text( _( STRING::WRONG_PASSWORD ) );
			m_entry_password->set_text( "" );
			m_entry_password->set_sensitive( false );
			sleep( 1 );
			m_entry_password->set_sensitive( true );
			m_entry_password->grab_focus();
			break;
		case CORRUPT_FILE:
		    m_label_login_info->set_text( _( STRING::CORRUPT_DB ) );
			Diary::d->clear();	// clear partially read content if any
			break;
		default:
			break;
	}
}

void
Lifeograph::handle_textview_link_activated( const Date &date )
{
	Entryvector *entries = Diary::d->get_entries( date.get_pure() );

	// seek and destroy cyclic links
	for( Entryiterv iter = entries->begin(); iter != entries->end(); ++iter )
	{
		if( ( *iter ) == m_panel_main->get_cur_elem() )
		{
			entries->erase( iter );
			break;
		}
	}

	if( entries->size() > 1 )
	{
		if( m_menu_link )
			m_menu_link->items().clear();
		else
			m_menu_link = new Gtk::Menu;

		for( Entryiterv iter = entries->begin(); iter != entries->end(); ++iter )
		{
			Entry *entry = ( *iter );
			m_menu_link->items().push_back(
					Gtk::Menu_Helpers::MenuElem(
							entry->get_name(),
							sigc::mem_fun( entry, &Entry::show ) ) );
		}

		m_menu_link->accelerate( *this );
		m_menu_link->popup( 0, gtk_get_current_event_time() );
	}
	else
	if( entries->size() == 1 )
		entries->at( 0 )->show();
	else
	{
		Entry *entry = Diary::d->create_entry( date.m_date );
		update_entry_list();
		entry->show();
	}

	delete entries;
}

void
Lifeograph::handle_entry_title_changed( DiaryElement *element )
{
	if( m_internaloperation ) return;

	Gtk::TreeRow row = * m_panel_diary->get_row( element->m_list_data->treepath );
	row[ ListData::colrec->info ] = element->get_list_str();
}

void
Lifeograph::handle_settings_changed( void )
{
	m_entry_view->update_formatting_toolbar();
}

void
Lifeograph::handle_undo( void )
{
	UndoManager::m->undo();
}

void
Lifeograph::handle_redo( void )
{
	UndoManager::m->redo();
}

bool
Lifeograph::finish_editing( bool opt_save )
{
	// SAVING
	m_settings.position_pane = m_hpaned_main->get_position();
	m_settings.position_pane_tags = m_hpaned_entry->get_position();
	sync_entry();
	Diary::d->m_last_elem = get_cur_elem()->get_id();

	if( opt_save )
	{
		// files added to recent list here: for newly created files, files that...
		// ...are saved under different names/dirs (a future save as option):
		m_filebuttonrecent->set_filename( Diary::d->get_path() );
		if( Diary::d->write() != SUCCESS )
		{
			Gtk::MessageDialog messagedialog(	*this,
												"",
												false,
												Gtk::MESSAGE_WARNING,
												Gtk::BUTTONS_OK,
												true );
			messagedialog.set_message( _( STRING::CANNOT_WRITE ) );
			messagedialog.set_secondary_text( _( STRING::CANNOT_WRITE_SUB ) );
			messagedialog.run();

			return false;
		}
	}
	else
	{
		Gtk::MessageDialog messagedialog(	*this,
											"",
											false,
											Gtk::MESSAGE_WARNING,
											Gtk::BUTTONS_CANCEL,
											true );
		messagedialog.set_message(
			_( "Are you sure you want to log out without saving?" ) );
		messagedialog.set_secondary_text( Glib::ustring::compose(
			_( "Your changes will be backed up in %1. "
			   "If you exit normally, your diary is saved automatically." ),
			"<b>" + Diary::d->get_path() + ".~unsaved~</b>" ), true );
		messagedialog.add_button( _( "_Log out without Saving" ), Gtk::RESPONSE_ACCEPT );

		if( messagedialog.run() !=  Gtk::RESPONSE_ACCEPT )
			return false;
		// back up changes
		Diary::d->write( Diary::d->get_path() + ".~unsaved~" );
	}

	// CLEARING
	// TODO: m_loginstatus = LOGGING_OUT_IN_PROGRESS;

	m_signal_logout.emit();	// only for DialogEvent

	m_internaloperation++;
	m_panel_diary->handle_logout();
	m_entry_view->m_textviewdiary->m_buffer->handle_logout();
	m_panel_main->clear_history();

	if( m_loginstatus == LOGGED_IN )
		m_loginstatus = LOGGED_OUT;
	Diary::d->clear();
	m_connection_timeout.disconnect();
	m_connection_backup.disconnect();
	m_internaloperation--;

	return true;
}

void
Lifeograph::logout( bool opt_save )
{
	if( finish_editing( opt_save ) )
		draw_loginscreen();
}

void
Lifeograph::show_today( void )
{
	Entry *entry_today = Diary::d->get_entry_today();
	bool flag_add = false;

	if( entry_today == NULL )	// add new entry if no entry exists on selected date
		flag_add = true;
	else						// or current entry is already at today
	if( entry_today->get_date().get_pure() ==
				m_panel_main->get_cur_elem()->get_date().get_pure() )
		flag_add = true;

	if( flag_add )
	{
		entry_today = Diary::d->add_today();
		update_entry_list();
	}

	entry_today->show();
}

void
Lifeograph::go_back( void )
{
	m_panel_main->go_back();
}

void
Lifeograph::sync_entry( void )
{
	m_entry_view->sync();
}

bool
Lifeograph::confirm_dismiss_element( void )
{
	Gtk::MessageDialog message( *Lifeobase::base, "", false,
							    Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE, true );
	message.set_message( _( "Are you sure, you want to dismiss?" ) );
	message.set_secondary_text( _( "This operation cannot be undone!" ) );
	message.add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
	message.add_button( _( "_DISMISS!" ), Gtk::RESPONSE_ACCEPT );

	if( message.run() != Gtk::RESPONSE_ACCEPT )
		return false;
	else
		return true;
}

void
Lifeograph::remove_element_from_history( DiaryElement *element )
{
	m_panel_main->remove_element_from_history( element );
}

bool
Lifeograph::change_current_element( DiaryElement *element, Gtk::Widget *widget )
{
	if( m_panel_main->get_cur_elem() == element )
		return false;

	m_internaloperation++;

	m_panel_main->show( element, widget );
	m_panel_diary->show( element );

	m_internaloperation--;

	return true;
}

DiaryElement*
Lifeograph::get_cur_elem( void )
{
	return m_panel_main->get_cur_elem();
}

DiaryElement::Type
Lifeograph::get_cur_elem_type( void )
{
	return m_panel_main->get_cur_elem_type();
}

DiaryElement*
Lifeograph::get_dragged_elem( void )
{
	return m_panel_diary->m_treeview_entries->get_dragged_elem();
}

Gtk::TreeRow
Lifeograph::get_element_row( DiaryElement *element )
{
	switch( element->get_type() )
	{
		case DiaryElement::ET_ENTRY:
		case DiaryElement::ET_CHAPTER:
			return( m_panel_diary->get_row( element->m_list_data->treepath ) );

		case DiaryElement::ET_TAG:
			return( m_panel_tag->get_row( element->m_list_data->treepath ) );
		default:
			return( Gtk::TreeRow() );
	}
}

void
Lifeograph::update_entry_list( void )
{
	m_panel_diary->update_entry_list();
}

void
Lifeograph::update_tag_list( void )
{
	m_panel_tag->populate_main();
}

void
Lifeograph::update_tag_categories( void )
{
	m_tag_view->update_combobox_categories();
}

void
Lifeograph::update_button_filter_tag( void )
{
    Lifeobase::m_internaloperation++;
    m_tag_view->m_button_filter->set_active( Diary::d->get_filter_tag() != NULL );
    Lifeobase::m_internaloperation--;
}

void
Lifeograph::update_button_remove_filters( void )
{
    m_panel_diary->m_button_remove_filters->set_visible(
            Diary::d->get_filtering_status() &
            ( Diary::FS_FILTER_DATE | Diary::FS_FILTER_TAG ) );
}

void
Lifeograph::update_tag_icon( const Tag *tag )
{
	m_panel_tag->update_tag_icon( tag );
}

void
Lifeograph::update_theme_list( void )
{
	m_widget_panel_exp->populate_themes();
}

void
Lifeograph::update_toolbar()
{
	bool flag_sensitive =
			m_panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY ?
					m_entrycount > 1 :
					m_panel_diary->m_treeview_entries->m_treestore_entries->children().size() > 1;

	m_action_previous_list->set_sensitive( flag_sensitive );
	m_action_next_list->set_sensitive( flag_sensitive );
}

void
Lifeograph::update_calendar( void )
{
	m_panel_diary->update_calendar();
}

void
Lifeograph::update_title( void )
{
	Glib::ustring title( PROGRAM_NAME );

	if( m_loginstatus == LOGGED_IN )
	{
		title += " - ";
		title += Glib::filename_display_basename( Diary::d->get_path() );
	}

	set_title( title );
}

void
Lifeograph::update_entryliststyle( void )
{
	m_internaloperation++;

	update_entry_list();

	if( m_panel_main->get_cur_elem_type() != DiaryElement::ET_ENTRY )
		Diary::d->show();

	m_panel_diary->m_treeview_entries->columns_autosize();

	m_internaloperation--;
}

void
Lifeograph::focus_filter( void )
{
	m_panel_diary->m_entry_filter->grab_focus();
}

void
Lifeograph::focus_tag( void )
{
	m_entry_view->focus_tag();
}

void
Lifeograph::toggle_comments( void )
{
    m_entry_view->m_button_toggle_comments->set_active(
            ! m_entry_view->m_button_toggle_comments->get_active() );
}

void
Lifeograph::select_date_in_calendar( const Date &date )
{
	m_panel_diary->select_date_in_calendar( date );
}

// MODE DRAWING FUNCTIONS
void
Lifeograph::draw_welcomescreen( void )
{
	Gtk::Alignment	*alignment =
			Gtk::manage( new Gtk::Alignment( 0.5, 0.5, 0.1, 0 ) );
	Gtk::Label		*label_salutation =
			Gtk::manage( new Gtk::Label( _( STRING::SALUTATION ), 0.5, 0.5 ) );
	Gtk::Button		*button_start =
			Gtk::manage( new Gtk::Button( _("_Continue"), true ) );
	Gtk::Image		*icon_forward =
			Gtk::manage( new Gtk::Image( Gtk::Stock::GO_FORWARD, Gtk::ICON_SIZE_MENU ) );
	Gtk::VBox		*vbox =
			Gtk::manage( new Gtk::VBox );

	alignment->set_padding( 120, 120, 150, 150 );
	add( *alignment );

	label_salutation->set_use_markup( true );

	button_start->set_image( *icon_forward );
	button_start->set_image_position( Gtk::POS_RIGHT );

	vbox->set_spacing( 50 );

	vbox->pack_start( *label_salutation );
	vbox->pack_start( *button_start );

	alignment->add( *vbox );

	button_start->signal_clicked().connect(
		sigc::mem_fun( *this, &Lifeograph::draw_loginscreen ) );

	show_all();
}

void
Lifeograph::draw_loginscreen( void )
{
    remove();

    if( m_vbox_login == NULL )
    {
        Gtk::Button     *button_about;
        Gtk::Button     *button_quit;
        Gtk::Image      *image_about;

        try
        {
            Lifeobase::builder->get_widget( "vbox_login", m_vbox_login );
            Menubutton::s_builder_name = "button_login_recent";
            Lifeobase::builder->get_widget_derived( "button_login_recent", m_filebuttonrecent );
            Lifeobase::builder->get_widget_derived( "entry_login_password", m_entry_password );
            Lifeobase::builder->get_widget( "label_login_info", m_label_login_info );
            Lifeobase::builder->get_widget( "image_login_about", image_about );
            Lifeobase::builder->get_widget( "button_login_about", button_about );
            Lifeobase::builder->get_widget( "button_login_open", m_button_open_diary );
            Lifeobase::builder->get_widget( "button_login_quit", button_quit );
        }
        catch( ... )
        {
            throw LIFEO::Error( "creation of login  view failed" );
        }

        image_about->set( PIXMAPDIR "/lifeograph.png" );
        m_entry_password->set_idle_text( "Password" );
        m_filebuttonrecent->set( &m_settings.recentfiles );

        // SIGNALS
        m_entry_password->signal_changed().connect(
                sigc::mem_fun( *this, &Lifeograph::handle_entry_password_changed ) );
        m_button_open_diary->signal_clicked().connect(
                sigc::mem_fun( *this, &Lifeograph::handle_button_opendb_clicked ) );
        button_quit->signal_clicked().connect(
                sigc::mem_fun( *this, &Lifeograph::handle_button_quit_clicked ) );
        button_about->signal_clicked().connect(
                sigc::mem_fun( *this, &Lifeograph::handle_button_about_clicked ) );

        m_filebuttonrecent->signal_selection_changed().connect(
                sigc::mem_fun( *this, &Lifeograph::handle_filebutton_db_selectionchanged ) );
        m_filebuttonrecent->signal_create_file().connect(
                sigc::mem_fun( *this, &Lifeograph::initialize_new_diary ) );

        // FILL RECENT FILES LIST
        if( ! m_settings.recentfiles.empty() )
            m_filebuttonrecent->update_filenames();
    }

    set_default( *m_button_open_diary );

    add( *m_vbox_login );
    m_vbox_login->show_all();

    // SETTING DB FILE
    update_login_file();
    set_title( PROGRAM_NAME );
}

void
Lifeograph::draw_editscreen( void )
{
	m_entry_password->set_text( "" );	// clear password
	remove();

	if( m_hpaned_main == NULL )
	{
        Gtk::MenuItem               *menuitem_preferences;
        Gtk::MenuItem               *menuitem_about;

		try
		{
			Lifeobase::builder->get_widget( "hpaned_main", m_hpaned_main );
			// TOOLBAR AND ITS ITEMS
			Lifeobase::builder->get_widget( "button_today", m_button_today );
			Lifeobase::builder->get_widget( "button_logout", m_button_logout );

            Lifeobase::builder->get_widget( "menuitem_preferences", menuitem_preferences );
            Lifeobase::builder->get_widget( "menuitem_about", menuitem_about );

            // TIME BASED NAVIGATION PANE
            Lifeobase::builder->get_widget( "hpaned_second", m_hpaned_entry );
            Lifeobase::builder->get_widget( "label_gui_elem", m_label_gui_elem );
            Lifeobase::builder->get_widget( "label_gui_elem_extra", m_label_gui_elem_extra );
            Lifeobase::builder->get_widget( "button_gui_elem", m_button_gui_elem );
            Lifeobase::builder->get_widget( "image_gui_elem", m_image_gui_elem );
            Lifeobase::builder->get_widget( "hbox_gelem_header", m_hbox_gelem_header );

			m_panel_main = new PanelMain;
			m_panel_diary = new PanelDiary;
			m_panel_tag = new TagPanel;
			m_widget_panel_exp = new WidgetPanelExp;
			m_diary_view = new DiaryView;
			m_theme_view = new ThemeView;
			m_tag_view = new TagView;
			m_category_tags_view = new CategoryTagsView;
			m_chapter_view = new ChapterView;
			m_entry_view = new EntryView;

			create_actions();	// must be after panel initializations
		}
		catch( std::exception &ex )
		{
			throw ex;
		}

        // GEOMETRY
        m_hpaned_main->set_position( m_settings.position_pane );
        m_hpaned_entry->set_position( m_settings.position_pane_tags );

		// ACTIONS
		m_action_today->connect_proxy( *m_button_today );
		m_action_previous_browsing->connect_proxy( *m_button_gui_elem );
		m_action_previous_match->connect_proxy( *m_panel_diary->m_button_match_prev );
		m_action_next_match->connect_proxy( *m_panel_diary->m_button_match_next );
		// becoming proxy widgets makes them visible:
		m_panel_diary->m_button_match_prev->set_visible( false );
		m_panel_diary->m_button_match_next->set_visible( false );

		m_button_logout->add_accelerator(
				"clicked",
				get_accel_group(),
				GDK_Escape,
				Gdk::CONTROL_MASK,
				Gtk::ACCEL_VISIBLE );

		// SIGNALS
		m_action_today->signal_activate().connect(
				sigc::mem_fun( *this, &Lifeograph::show_today ) );

		menuitem_about->signal_activate().connect(
				sigc::mem_fun( *this, &Lifeograph::handle_button_about_clicked ) );
		menuitem_preferences->signal_activate().connect(
				sigc::ptr_fun( &DialogPreferences::create ) );

		m_button_logout->signal_clicked().connect(
				sigc::bind( sigc::mem_fun( *this, &Lifeograph::logout ), true ) );

		LinkEntry::signal_activated().connect(
				sigc::mem_fun( *this, &Lifeograph::handle_textview_link_activated ) );
	}
	else
	{
		m_internaloperation++;
		m_panel_diary->m_entry_filter->set_text( "" );
		m_panel_diary->m_entry_replace->set_text( "" );
		m_internaloperation--;
	}

	m_connection_backup = Glib::signal_timeout().connect_seconds(
			sigc::mem_fun( this, &Lifeograph::write_backup ),
			BACKUP_INTERVAL );

	add( *m_hpaned_main );

	// LOGIN
	m_internaloperation++;

	m_panel_diary->handle_login();
	m_panel_tag->populate_main();
	m_panel_diary->m_button_remove_filters->set_visible( false );
	m_widget_panel_exp->populate_themes();
	m_tag_view->update_combobox_categories();
	m_diary_view->handle_login();
	m_entry_view->handle_login();

	m_internaloperation--;
	m_loginstatus = LOGGED_IN;
	update_title();
}

void
Lifeograph::create_actions( void )
{
	create_action(	m_action_undo,
					"Undo",
					Gtk::Stock::UNDO,
					_( "Undo" ),
					"",		// this tooltip will be dynamic
					Gtk::AccelKey( GDK_z, Gdk::CONTROL_MASK ),
					sigc::mem_fun( *this, &Lifeograph::handle_undo ) );

	create_action(	m_action_redo,
					"Redo",
					Gtk::Stock::REDO,
					_( "Redo" ),
					"",		// this tooltip will be dynamic
					Gtk::AccelKey( GDK_z, Gdk::CONTROL_MASK | Gdk::SHIFT_MASK ),
					sigc::mem_fun( *this, &Lifeograph::handle_redo ) );

	create_action(	m_action_previous_list,
					"ListPrevious",
					Gtk::Stock::GO_UP,
					_( "Previous" ),
					_( "Go to the previous entry in the list" ),
					Gtk::AccelKey( GDK_Page_Up, Gdk::CONTROL_MASK ),
                    sigc::bind( sigc::mem_fun( m_panel_diary->m_treeview_entries,
                                               &WidgetEntryList::go_up ),
                                false ) );

	create_action(	m_action_next_list,
					"ListNext",
					Gtk::Stock::GO_DOWN,
					_( "Next" ),
					_( "Go to the next entry in the list" ),
					Gtk::AccelKey( GDK_Page_Down, Gdk::CONTROL_MASK ),
                    sigc::bind( sigc::mem_fun( m_panel_diary->m_treeview_entries,
                                               &WidgetEntryList::go_down ),
                                false ) );

	create_action(	m_action_previous_browsing,
					"BrowseBack",
					Gtk::Stock::GO_BACK,
					"Back", _( "Go back in the browsing history" ),
					Gtk::AccelKey( GDK_Left, Gdk::MOD1_MASK ),
					sigc::mem_fun( *this, &Lifeograph::go_back ) );

	create_action(	m_action_previous_match,
					"MatchPrevious",
					Gtk::Stock::GO_BACK,
// TRANSLATORS: label and tooltip of the go-to-previous-match-of-filtered-text action
					_( "Previous Match" ),
					_( "Go to previous match" ),
					Gtk::AccelKey( GDK_F3, Gdk::SHIFT_MASK ),
					sigc::mem_fun( m_panel_diary, &PanelDiary::go_prev_match ) );

	create_action(	m_action_next_match,
					"MatchNext",
					Gtk::Stock::GO_FORWARD,
// TRANSLATORS: label and tooltip of the go-to-next-match-of-filtered-text action
					_( "Next Match" ),
					_( "Go to next match" ),
					Gtk::AccelKey( GDK_F3, Gdk::ModifierType( 0 ) ),
					sigc::mem_fun( m_panel_diary, &PanelDiary::go_next_match ) );

	create_action(	m_action_jump2current,
					"Jump",
					Gtk::Stock::JUMP_TO,
					"", "",		// no grachical interface yet!
					Gtk::AccelKey( GDK_j, Gdk::CONTROL_MASK ),
					sigc::mem_fun( m_panel_diary, &PanelDiary::present_current_row ) );
	
	create_action(	m_action_show_prev_session_elem,
					"PrevSessElem",
					Gtk::Stock::JUMP_TO,
					"", "",		// no grachical interface yet!
					Gtk::AccelKey( GDK_l, Gdk::CONTROL_MASK ),
					sigc::mem_fun( m_panel_diary, &PanelDiary::show_prev_session_elem ) );

	create_action(	m_action_focusfilter,
					"FocusFilter",
					Gtk::Stock::FIND,
					"", "",		// no grachical interface yet!
					Gtk::AccelKey( GDK_f, Gdk::CONTROL_MASK ),
					sigc::mem_fun( *this, &Lifeograph::focus_filter ) );

	m_action_today = Gtk::Action::create_with_icon_name(
		"Today",
		"x-office-calendar",
		_( "Today" ),
		_( "Go to today (creates a new entry if there isn't any or when already at today)" ) );
	m_actions->add( m_action_today, Gtk::AccelKey( GDK_d, Gdk::CONTROL_MASK ) );
	m_action_today->set_accel_group( get_accel_group() );
	m_action_today->connect_accelerator();
	m_action_today->property_is_important() = true;
}
